package caslayout.importing.excel;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRichTextString;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.poifs.filesystem.POIFSFileSystem;

import caslayout.importing.common.Assessment;
import caslayout.importing.common.DataRow;
import caslayout.importing.common.ImportValidationException;
import caslayout.importing.common.Score;
import caslayout.importing.common.ScoreCode;
import caslayout.util.GenUtils;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: ExcelImporter.java,v 1.4 2008/10/13 23:58:07 bozyurt Exp $
 */

public class ExcelImporter {
	protected String excelWorkSheetFile;
	protected Assessment assessment;
	protected boolean hasScoreCodesSheet;
	protected boolean hasAssessmentData;
	protected int scoreCodesSheetIdx = -1;
	protected int asDataSheetIdx = -1;
	protected Map<String, Integer> adColIdxMap = new HashMap<String, Integer>(7);

	public ExcelImporter(String excelWorkSheetFile) {
		super();
		this.excelWorkSheetFile = excelWorkSheetFile;
	}

	public void setAssessmentName(String asName) {
		if (assessment != null) {
			assessment.setName(asName);
		}
	}

	public void checkAssessmentValidity() throws ImportValidationException {
		FileInputStream in = null;
		POIFSFileSystem fs = null;

		try {
			in = new FileInputStream(excelWorkSheetFile);
			fs = new POIFSFileSystem(in);
			HSSFWorkbook wb = new HSSFWorkbook(fs);
			int numSheets = wb.getNumberOfSheets();
			if (numSheets < 1 || numSheets > 3) {
				throw new ImportValidationException(
						"Not a valid clinical assessment spreadsheet!");
			}
			HSSFSheet sheet = wb.getSheetAt(0);
			// check the header of the score sheet
			String[] headRow = getRowCells(sheet, 0);
			if (headRow.length < 4
					|| (!headRow[0].equalsIgnoreCase("scoreseq")
							|| !headRow[1].equalsIgnoreCase("scorename")
							|| !headRow[2].equalsIgnoreCase("scoretype") || !headRow[3]
							.equalsIgnoreCase("question"))) {
				throw new ImportValidationException(
						"Score sheet needs to have at least these three columns in the same order:\n "
								+ "'SCORESEQ','SCORENAME','SCORETYPE','QUESTION'");
			}
			int numScores = sheet.getPhysicalNumberOfRows() - 1;

			String[] scGSArr = { "scorename", "question", "scorecode",
					"scorecodetype", "scorelabel", "scorevalue" };

			if (numSheets > 1) {
				sheet = wb.getSheetAt(1);
				headRow = getRowCells(sheet, 0);
				if (headRow.length >= 1) {
					if (headRow[0].equalsIgnoreCase("scorename")) {
						// score codes sheet
						if (headRow.length < 6 || !allMatches(headRow, scGSArr)) {
							throw new ImportValidationException(
									"Score sheet needs to have at least these six columns in the same order:\n"
									+ "'SCORENAME','QUESTION','SCORECODE','SCORECODETYPE','SCORELABEL','SCOREVALUE'");
						}
						this.hasScoreCodesSheet = true;
						this.scoreCodesSheetIdx = 1;
					} else if (headRow[0].equalsIgnoreCase("site")) {
						if ( isValidAssessmentDataHeader(headRow, numScores) ) {
							this.hasAssessmentData = true;
							this.asDataSheetIdx = 1;
							this.adColIdxMap = prepADColumnIdxMap(headRow);
						}
					}
				}
				if (numSheets > 2) {
					sheet = wb.getSheetAt(2);
					headRow = getRowCells(sheet, 0);
					if ( isValidAssessmentDataHeader(headRow, numScores)) {
						this.hasAssessmentData = true;
						this.asDataSheetIdx = 2;
						this.adColIdxMap = prepADColumnIdxMap(headRow);
					}
				}
			}

		} catch (IOException iox) {
			throw new ImportValidationException(iox);
		} finally {
			GenUtils.close(in);
		}
	}

	protected Map<String, Integer> prepADColumnIdxMap(String[] headRow) {
		Map<String, Integer> map = new HashMap<String, Integer>();
		for (int i = 0; i < headRow.length; i++) {
			map.put(headRow[i].toLowerCase(), new Integer(i));
		}
		return map;
	}

	protected boolean isValidAssessmentDataHeader(String[] headRow,
			int numScores) throws ImportValidationException {
		String[] adGSArr = { "site", "subjectid", "subjecttype",
				"experimentname", "timestamp" };
		String[] optionals = { "scoretype" };

		if (headRow.length >= 1) {
			if (headRow[0].equalsIgnoreCase("site")) {
				if (headRow.length < (adGSArr.length + numScores)
						|| !allMatches(headRow, adGSArr, optionals)) {
					throw new ImportValidationException(
							"Score sheet needs to have at least five + # of scores ("
									+ numScores
									+ ") columns:\n"
									+ "'SITE','SUBJECTID','SUBJECTTYPE','EXPERIMENTNAME', 'TIMESTAMP',...");
				}
				return true;
			}
		}
		return false;
	}

	protected boolean allMatches(String[] cols, String[] goldStandard) {
		for (int i = 0; i < goldStandard.length; i++) {
			if (!cols[i].equalsIgnoreCase(goldStandard[i])) {
				return false;
			}
		}
		return true;
	}

	protected boolean allMatches(String[] cols, String[] goldStandard,
			String[] optionals) {
		Set<String> optionalSet = new HashSet<String>(7);
		for (int i = 0; i < optionals.length; i++) {
			optionalSet.add(optionals[i]);
		}

		int idx = 0;
		for (int i = 0; i < goldStandard.length; i++) {
			if (idx >= cols.length) {
				return false;
			}
			if (!cols[idx].equalsIgnoreCase(goldStandard[i])) {
				while (idx < cols.length && optionalSet.contains(cols[idx])) {
					idx++;
				}
				if (!cols[idx].equalsIgnoreCase(goldStandard[i])) {
					return false;
				}
			}
			++idx;
		}
		return true;
	}

	protected boolean contains(String[] arr, String toMatch) {
		for (int i = 0; i < arr.length; i++) {
			if (arr[i].equals(toMatch)) {
				return true;
			}
		}
		return false;
	}

	public Assessment importFromWorkSheet() throws IOException,
			ImportValidationException {
		FileInputStream in = null;
		POIFSFileSystem fs = null;

		try {
			checkAssessmentValidity();
			in = new FileInputStream(excelWorkSheetFile);
			fs = new POIFSFileSystem(in);
			HSSFWorkbook wb = new HSSFWorkbook(fs);
			String asName = new File(excelWorkSheetFile).getName();
			asName = asName.replaceAll("\\.\\w+$", "");
			assessment = new Assessment(asName);
			HSSFSheet sheet = wb.getSheetAt(0);
			prepScores(sheet);

			if (this.hasScoreCodesSheet) {
				sheet = wb.getSheetAt(this.scoreCodesSheetIdx);
				prepScoreCodes(sheet, assessment.getScores());
			}

			//HSSFRow row = sheet1.getRow(0);
			// dumpWorkSheet(sheet1, lastRowNum, row);

			return assessment;

		} finally {
			GenUtils.close(in);
		}
	}

	protected void dumpWorkSheet(HSSFSheet sheet1, int lastRowNum, HSSFRow row) {
		StringBuffer buf = new StringBuffer(150);
		for (Iterator<?> iter = row.cellIterator(); iter.hasNext();) {
			HSSFCell cell = (HSSFCell) iter.next();
			HSSFRichTextString rts = cell.getRichStringCellValue();
			buf.append(rts.toString());
			if (iter.hasNext()) {
				buf.append(',');
			}
		}
		System.out.println(buf.toString());

		// print data also
		for (int i = 1; i <= lastRowNum; i++) {
			row = sheet1.getRow(i);
			System.out.println(row2CSV(row));
		}
	}

	/**
	 *
	 * @return
	 * @throws IOException
	 * @throws ImportValidationException
	 */
	public List<DataRow> loadAssessmentData() throws IOException,
			ImportValidationException {
		if (!hasAssessmentData) {
			return null;
		}
		FileInputStream in = null;
		POIFSFileSystem fs = null;

		try {
			in = new FileInputStream(excelWorkSheetFile);
			fs = new POIFSFileSystem(in);
			HSSFWorkbook wb = new HSSFWorkbook(fs);
			HSSFSheet sheet = wb.getSheetAt(this.asDataSheetIdx);
			int firstRowNum = sheet.getFirstRowNum();
			int lastRowNum = sheet.getLastRowNum();
			// String[] headRow = getRowCells(sheet, firstRowNum);

			Score[] scores = new Score[this.assessment.getScores().size()];

			int idx = 0;
			for (Iterator<Score> it = assessment.getScores().iterator(); it.hasNext();) {
				Score sc = it.next();
				if ( adColIdxMap.get(sc.getName().toLowerCase()) == null) {
					throw new ImportValidationException(
							"The score does not match any assessment data header: "
									+ sc.getName());
				}
				scores[idx++] = sc;
			}


			List<DataRow> dataRows = new ArrayList<DataRow>(sheet.getPhysicalNumberOfRows());
			for (int i = firstRowNum + 1; i <= lastRowNum; i++) {
				String[] row = getRowCells(sheet, i);
				String subjectID = getValue(row, "subjectid");
				String site = getValue(row, "site");
				DataRow dr = new DataRow(subjectID, site);
				dr.setSubjectType( getValue(row,"subjecttype"));
				dr.setExperimentName( getValue(row,"experimentname"));
				dr.setTimeStamp( getValue(row, "timestamp"));
				dr.setScoreTypeCol(row[5]);
				dr.setScores(scores);
				String[] data = new String[ assessment.getScores().size()];
				int j = 0;
				for (Iterator<Score> it = assessment.getScores().iterator(); it.hasNext();) {
					Score sc = (Score) it.next();
					data[j++] = getValue(row, sc.getName().toLowerCase());
				}
				dr.setData(data);
				dataRows.add(dr);
			}

			return dataRows;
		} finally {
			GenUtils.close(in);
		}
	}

	protected String getValue(String[] row, String columnName) {
		Integer idx = this.adColIdxMap.get(columnName);
		return row[ idx.intValue()];
	}

	protected String[] getRowCells(HSSFSheet sheet, int rowIdx) {
		HSSFRow hssfRow = sheet.getRow(rowIdx);
		String[] row = new String[hssfRow.getPhysicalNumberOfCells()];
		int i = 0;
		for (Iterator<?> iter = hssfRow.cellIterator(); iter.hasNext();) {
			HSSFCell cell = (HSSFCell) iter.next();
			HSSFRichTextString rts = cell.getRichStringCellValue();
			row[i++] = rts.toString();
		}
		return row;
	}

	protected void prepScores(HSSFSheet sheet) {
		int firstRowNum = sheet.getFirstRowNum();
		int lastRowNum = sheet.getLastRowNum();
		System.out.println("firstRowNum:" + firstRowNum + " lastRowNum: "
				+ lastRowNum);
		HSSFRow row = sheet.getRow(0);
		assert (row.getPhysicalNumberOfCells() == 3);

		for (int i = 1; i <= lastRowNum; i++) {
			row = sheet.getRow(i);
			int scoreSeq = Integer.parseInt(row.getCell((short) 0)
					.getRichStringCellValue().toString().trim());
			String scoreName = row.getCell((short) 1).getRichStringCellValue()
					.getString().trim();
			String scoreType = row.getCell((short) 2).getRichStringCellValue()
					.getString().trim();
			String question = row.getCell((short) 3).getRichStringCellValue()
					.getString().trim();
			Score sc = new Score(scoreName, question, scoreSeq, scoreType);
			assessment.addScore(sc);
		}
	}

	protected void prepScoreCodes(HSSFSheet sheet, List<Score> scores) {
		int firstRowNum = sheet.getFirstRowNum();
		int lastRowNum = sheet.getLastRowNum();
		System.out.println("firstRowNum:" + firstRowNum + " lastRowNum: "
				+ lastRowNum);
		HSSFRow row = sheet.getRow(0);
		assert (row.getPhysicalNumberOfCells() == 6);
		Map<String, Score> scoreMap = new HashMap<String, Score>();
		for (Iterator<Score> iter = scores.iterator(); iter.hasNext();) {
			Score score = iter.next();
			scoreMap.put(score.getName(), score);
		}

		for (int i = 1; i <= lastRowNum; i++) {
			row = sheet.getRow(i);
			String scoreName = row.getCell((short) 0).getRichStringCellValue()
					.getString().trim();
			String scoreCodeStr = row.getCell((short) 2)
					.getRichStringCellValue().getString().trim();
			String scoreLabel = row.getCell((short) 4).getRichStringCellValue()
					.getString().trim();
			String scoreValue = row.getCell((short) 5).getRichStringCellValue()
					.getString().trim();

			Score score = scoreMap.get(scoreName);
			assert (score != null);
			score.addScoreCode(new ScoreCode(scoreCodeStr, scoreLabel,
					scoreValue));
		}
	}

	public static String row2CSV(HSSFRow row) {
		StringBuffer buf = new StringBuffer(150);
		for (Iterator<?> iter = row.cellIterator(); iter.hasNext();) {
			HSSFCell cell = (HSSFCell) iter.next();
			HSSFRichTextString rts = cell.getRichStringCellValue();
			buf.append(rts);
			if (iter.hasNext()) {
				buf.append(',');
			}
		}
		return buf.toString();
	}

	public Assessment getAssessment() {
		return assessment;
	}

	public boolean getHasAssessmentData() {
		return hasAssessmentData;
	}

	public static void main(String[] args) throws IOException,
			ImportValidationException {
		String xlsFile = "/home/bozyurt/dev/java/birn_testbed/test/excel/SOCIOECONOMICSTATUSSES.xls";
		ExcelImporter importer = new ExcelImporter(xlsFile);

		Assessment as = importer.importFromWorkSheet();
		System.out.println(as);

	}

}
