package caslayout.test;

import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import caslayout.ui.CALMConfig;
import caslayout.ui.db.Assessment;
import caslayout.ui.db.AssessmentItem;
import caslayout.ui.db.AssessmentScore;
import caslayout.ui.db.AssessmentScoreCode;
import caslayout.ui.db.SessionFactoryFactory;
import caslayout.ui.db.services.AssessmentService;
import caslayout.ui.model.AssessmentInfo;
import caslayout.ui.model.AssessmentMaintenanceHelper;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentManTest.java,v 1.8 2008/10/14 23:21:16 bozyurt Exp $
 */
public class AssessmentManTest extends TestCase {
   CALMConfig config;

   public AssessmentManTest(String name) {
      super(name);
   }

   protected void setUp() throws Exception {
      config = new CALMConfig("caslayout.properties");
      Class.forName("oracle.jdbc.driver.OracleDriver");

   }

   protected void shutDown() throws Exception {

   }

   public void testLoad() throws Exception {
      AssessmentInfo asi = AssessmentMaintenanceHelper.loadAssessment(config);

      asi
            .dumpScoreHierarchy(new PrintWriter(new OutputStreamWriter(
                  System.out)));
   }

   public void testCheckAssessment() throws Exception {
      AssessmentService service = AssessmentService.getInstance(config);
      try {

         service.getAssessment("SANS");

         Assessment goldAs = populateAssessment();
         checkResult(service, goldAs);

      } finally {
         if (service != null)
            SessionFactoryFactory.shutdown();
      }

   }

   public void testSaveAssessment() throws Exception {
      Assessment as = populateAssessment();
      AssessmentService service = AssessmentService.getInstance(config);
      try {
         service.setSecurityClassification("FIRST BIRN");

         service.saveAssessment(as);

         Assessment goldAs = populateAssessment();
         checkResult(service, goldAs);

      } finally {
         if (service != null)
            SessionFactoryFactory.shutdown();
      }
   }

   public void checkResult(AssessmentService service, Assessment goldAs)
         throws Exception {
      Assessment as = service.getAssessment("SANS");
      assertTrue(as != null);

      assertEquals(goldAs.getName(), as.getName());
      assertEquals(goldAs.getScores().size(), as.getScores().size());

      Map<String, AssessmentScore> map = toScoreMap(as.getScores());
      for (Object element : goldAs.getScores()) {
         AssessmentScore goldScore = (AssessmentScore) element;
         AssessmentScore score = map.get(goldScore
               .getScoreName());
         assertEquals(goldScore.getScoreName(), score.getScoreName());
         checkScoreCodes(goldScore.getScoreCodes(), score.getScoreCodes());
      }

      checkItems(goldAs.getItems(), as.getItems());
   }

   public void checkScoreCodes(List<AssessmentScoreCode> goldCodes, List<AssessmentScoreCode> codes) throws Exception {
      assertEquals(goldCodes.size(), codes.size());

      Map<String, AssessmentScoreCode> map = toScoreCodeMap(codes);
      for (Object element : goldCodes) {
         AssessmentScoreCode goldAsc = (AssessmentScoreCode) element;
         AssessmentScoreCode asc = map.get(goldAsc
               .getScoreCode());
         assertNotNull(asc);
         assertEquals(goldAsc.getScoreCode(), asc.getScoreCode());
         assertEquals(goldAsc.getScoreName(), asc.getScoreName());
      }
   }

   public void checkItems(List<AssessmentItem> goldItems, List<AssessmentItem> items) throws Exception {
      assertEquals(goldItems.size(), items.size());
      Map<String, AssessmentItem> map = toItemMap(items);
      for (Object element : goldItems) {
         AssessmentItem goldItem = (AssessmentItem) element;
         AssessmentItem item = map
               .get(goldItem.getScoreName());
         assertNotNull(item);
         assertEquals(goldItem.getScoreName(), item.getScoreName());
         assertEquals(goldItem.getItemLeadingText(), item.getItemLeadingText());
      }
   }

   protected Map<String, AssessmentScore> toScoreMap(Set<AssessmentScore> scores) {
      Map<String, AssessmentScore> map = new HashMap<String, AssessmentScore>();
      for (Object element : scores) {
         AssessmentScore score = (AssessmentScore) element;
         map.put(score.getScoreName(), score);
      }
      return map;
   }

   protected Map<String, AssessmentScoreCode> toScoreCodeMap(List<AssessmentScoreCode> scoreCodes) {
      Map<String, AssessmentScoreCode> map = new HashMap<String, AssessmentScoreCode>();
      for (Object element : scoreCodes) {
         AssessmentScoreCode asc = (AssessmentScoreCode) element;
         map.put(asc.getScoreCode(), asc);
      }
      return map;
   }

   protected Map<String, AssessmentItem> toItemMap(List<AssessmentItem> items) {
      Map<String, AssessmentItem> map = new HashMap<String, AssessmentItem>();
      for (Object element : items) {
         AssessmentItem item = (AssessmentItem) element;
         map.put(item.getScoreName(), item);
      }
      return map;
   }

   public void testDeleteAssessment() throws Exception {
      AssessmentService service = AssessmentService.getInstance(config);
      try {

         Assessment as = service.getAssessment("SANS");
         assertTrue(as != null);
         service.deleteAssessment(as, false);

      } finally {
         if (service != null)
            SessionFactoryFactory.shutdown();
      }
   }

   public void testUpdateAssessment() throws Exception {
      AssessmentService service = AssessmentService.getInstance(config);
      try {

         Assessment as = service.getAssessment("SANS");

         Assessment updatedAs = populateUpdatedAssessment(as);

         service.updateAssessment(updatedAs);

         checkResult(service, updatedAs);

      } finally {
         if (service != null)
            SessionFactoryFactory.shutdown();
      }
   }

   protected Assessment populateUpdatedAssessment(Assessment as) {

      Map<String, AssessmentScore> scoreMap = toScoreMap(as.getScores());
      AssessmentScore score = scoreMap
            .get("Affective Flattening or blunting");
      score.setScoreName("Affective Flattening or Blunting");

      AssessmentScore ss13 = new AssessmentScore(as.getID(),
            "Paucity of Expressive Gestures");
      ss13.setScoreSequence(new Integer(4));
      ss13.setScoreType("integer");
      ss13.setScoreLevel(new Integer(2));
      ss13.setParent(score);
      as.addScore(ss13);
      // update the rest of score sequences

      AssessmentScore s = scoreMap.get("Alogia");
      s.setScoreSequence(new Integer(5));
      s = scoreMap.get("Poverty of Speech");
      s.setScoreSequence(new Integer(6));

      return as;
   }

   protected Assessment populateAssessment() {
      Assessment as = new Assessment(null, "SANS");
      AssessmentScore score = new AssessmentScore(as.getID(),
            "Affective Flattening or blunting");

      score.setScoreSequence(new Integer(1));
      score.setScoreType("varchar");
      score.setScoreLevel(new Integer(1));
      AssessmentScore ss11 = new AssessmentScore(as.getID(),
            "Unchanging Facial Expression");
      ss11.setScoreType("integer");
      ss11.setScoreLevel(new Integer(2));
      ss11.setScoreSequence(new Integer(2));
      ss11.setParent(score);

      AssessmentScore ss12 = new AssessmentScore(as.getID(),
            "Decreased Spontaneous Movements");
      ss12.setScoreSequence(new Integer(3));
      ss12.setScoreType("integer");
      ss12.setScoreLevel(new Integer(2));
      ss12.setParent(score);

      AssessmentScore score2 = new AssessmentScore(as.getID(), "Alogia");
      score2.setScoreType("varchar");
      score2.setScoreLevel(new Integer(1));
      score2.setScoreSequence(new Integer(4));
      AssessmentScore ss21 = new AssessmentScore(as.getID(),
            "Poverty of Speech");
      ss21.setScoreType("integer");
      ss21.setScoreLevel(new Integer(2));
      ss21.setScoreSequence(new Integer(5));
      ss21.setParent(score2);

      as.addScore(score);
      as.addScore(ss11);
      as.addScore(ss12);
      as.addScore(score2);
      as.addScore(ss21);

      // add scorecodes

      ss11.addScoreCode(new AssessmentScoreCode(null, ss11.getScoreName(),
            "Not at all", "0", "integer", ""));
      ss11.addScoreCode(new AssessmentScoreCode(null, ss11.getScoreName(),
            "Questionable decrease", "1", "integer", ""));
      ss11.addScoreCode(new AssessmentScoreCode(null, ss11.getScoreName(),
            "Mild", "2", "integer", ""));

      ss12.addScoreCode(new AssessmentScoreCode(null, ss12.getScoreName(),
            "Not at all", "0", "integer", ""));
      ss12.addScoreCode(new AssessmentScoreCode(null, ss12.getScoreName(),
            "Questionable decrease", "1", "integer", ""));
      ss12.addScoreCode(new AssessmentScoreCode(null, ss12.getScoreName(),
            "Mild", "2", "integer", ""));

      // add assessment items

      AssessmentItem item = new AssessmentItem(null, ss11.getScoreName());
      item
            .setItemLeadingText("The subject's face appears wooden, mechanical, frozen.");
      as.addItem(item);
      item = new AssessmentItem(null, ss12.getScoreName());
      item
            .setItemLeadingText("The subject sits quietly throughout the interview and shows few or no spontaneous movements.");
      as.addItem(item);

      return as;
   }

   public static Test suite() {
      TestSuite suite = new TestSuite();
      // suite.addTest( new AssessmentManTest("testLoad") );
      // suite.addTest( new AssessmentManTest("testSaveAssessment") );
      // suite.addTest( new AssessmentManTest("testDeleteAssessment") );
      suite.addTest(new AssessmentManTest("testUpdateAssessment"));
      // suite.addTest( new AssessmentManTest("testCheckAssessment") );
      return suite;
   }

   public static void main(String[] args) {
      junit.textui.TestRunner.run(suite());
   }

}
