package caslayout.ui;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.font.FontRenderContext;
import java.awt.geom.Rectangle2D;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Observable;

import org.apache.log4j.Logger;
import org.jdom.Element;

import caslayout.util.GUIUtils;
import caslayout.util.UIDManager;
import caslayout.util.XMLUtils;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: AbstractDisplayComponent.java,v 1.22 2007/10/08 18:00:40
 *          bozyurt Exp $
 */

public abstract class AbstractDisplayComponent extends Observable implements
      IDisplayComponent {
   /** unique within a document ID used for identification and referencing */
   protected String id;
   /** form component name if applicable */
   protected String name = "";
   /**
    * starting X coordinate of this display element. All coordinates are in grid
    * units.
    */
   protected int x;
   /**
    * starting Y coordinate of this display element. All coordinates are in grid
    * units.
    */
   protected int y;
   /** the width of this display element in grid units. */
   protected int width;
   /** the height of this display element in grid units. */
   protected int height;
   /** the parent of this display element (if any) */
   protected IDisplayComponent parent;
   protected java.util.List<IDisplayComponent> children;
   /** if true displays debugging information on drawing */
   protected boolean debugEnabled = false;
   protected Rectangle2D bounds = null;
   /** The peer on which the actual rendering is done */
   protected Component peer;
   /** if true means this component is selected by user for further manipulation */
   protected boolean selected = false;
   /** model associated with this display component */
   protected Object model;
   /** logical group id (if any) of the component */
   protected String groupID;
   /** class attribute for Cascade Style sheet application to the component */
   protected String cssClass = "";
   /** alignment of the component */
   protected Justification justification;
   /**
    * if true displays associated score in the left right hand corner of the
    * component
    */
   boolean showAssociation;
   /** */
   protected String questionID;
   /* */
   protected boolean showQuestions;

   protected static Logger log = Logger.getLogger("AbstractDisplayComponent");

   public AbstractDisplayComponent() {
      this.id = UIDManager.getInstance().getNextUID();
      justification = Justification.CENTER_JUSTIFICATION;
   }

   public AbstractDisplayComponent(String uid) {
      this.id = uid;
      justification = Justification.CENTER_JUSTIFICATION;
   }

   public AbstractDisplayComponent(int x, int y, Component peer) {
      this(x, y, peer, false, UIDManager.getInstance().getNextUID(), "",
            Justification.CENTER_JUSTIFICATION);
   }

   public AbstractDisplayComponent(int x, int y, Component peer, String id,
         String name, Justification justification) {
      this(x, y, peer, false, id, name, justification);
   }

   public AbstractDisplayComponent(int x, int y, Component peer,
         boolean debugEnabled, String id, String name,
         Justification justification) {
      this.x = x;
      this.y = y;
      this.peer = peer;
      this.debugEnabled = debugEnabled;
      this.id = id;
      this.name = name;
      this.justification = justification;
   }

   public AbstractDisplayComponent(AbstractDisplayComponent other) {
      this.x = other.x;
      this.y = other.y;
      this.peer = other.peer;
      this.debugEnabled = other.debugEnabled;
      this.id = other.id;
      this.name = other.name;
      this.cssClass = other.cssClass;
      this.justification = other.justification;
   }

   public void addChild(IDisplayComponent child) {
      if (children == null) {
         children = new LinkedList<IDisplayComponent>();
      }
      children.add(child);
   }

   public boolean removeChild(IDisplayComponent child) {
      if (children == null)
         return false;
      return children.remove(child);
   }

   public Rectangle2D getBounds(Graphics2D g2) {
      return bounds;
   }

   public void setBounds(Rectangle2D r) {
      bounds = r;
      if (log.isDebugEnabled()) {
         log.debug("setting bounds " + r);
      }
      this.x = (int) bounds.getX();
      this.y = (int) bounds.getY();
      this.width = (int) bounds.getWidth();
      this.height = (int) bounds.getHeight();

   }

   public abstract void draw(Graphics2D g2);

   protected void showGroupID(Graphics2D g2, String groupID,
         Rectangle2D boundingRect, Color theColor) {
      Font font = g2.getFont();
      FontRenderContext frCtx = g2.getFontRenderContext();
      // double labelArea = 0;
      Rectangle2D r = font.getStringBounds(groupID, frCtx);
      // labelArea = r.getWidth() * r.getHeight();
      /** @todo change font size based on the percent area occupied */
      Color oldColor = g2.getColor();
      g2.setColor(theColor);
      // draw to the upper right corner
      int x1 = 0, y1 = 0;
      x1 = (int) (boundingRect.getX() + boundingRect.getWidth() - r.getWidth());
      y1 = (int) (boundingRect.getY() + r.getHeight());

      // log.info("showGroupID x1:"+ x1 + " y1:"+ y1 + " groupID="+ groupID);
      g2.drawString(groupID, x1, y1);

      g2.setColor(oldColor);
   }

   protected void showScoreName(Graphics2D g2, Rectangle2D boundingRect,
         Color theColor) {
      String scoreName = GUIUtils.findAssociatedScoreName(groupID, id);
      // log.info("scoreName=" + scoreName + " groupID=" + groupID + " id=" +
      // id);

      String displayName = null;
      if (scoreName != null)
         displayName = GUIUtils.toDisplaySize(scoreName, 9);
      if (displayName == null)
         return;

      Font oldFont = g2.getFont();
      int size = oldFont.getSize();
      Font newFont = new Font(oldFont.getName(), oldFont.getStyle(), size - 1);
      g2.setFont(newFont);

      FontRenderContext frCtx = g2.getFontRenderContext();

      Rectangle2D r = newFont.getStringBounds(displayName, frCtx);
      Color oldColor = g2.getColor();
      g2.setColor(theColor);

      // draw to the upper left corner
      int x1 = 0, y1 = 0;
      x1 = (int) (boundingRect.getX() + 2);
      y1 = (int) (boundingRect.getY() + r.getHeight());

      g2.drawString(displayName, x1, y1);

      g2.setFont(oldFont);
      g2.setColor(oldColor);
   }

   protected void showQuestion(Graphics2D g2, Rectangle2D boundingRect,
         Color theColor) {
      if (getQuestionID() == null)
         return;
      String questionID = "q" + getQuestionID();
      double width = boundingRect.getWidth();
      double height = boundingRect.getHeight();
      double xMargin = Math.max(width * 0.01, 3);
      double yMargin = Math.max(height * 0.01, 1);
      Color oldColor = g2.getColor();
      g2.setColor(theColor);

      g2.fillRect((int) (boundingRect.getX() + xMargin), (int) (boundingRect
            .getY() + yMargin), (int) (width - 2 * xMargin),
            (int) (height - 2 * yMargin));

      g2.setColor(oldColor);

      int x1 = (int) (boundingRect.getX() + xMargin + 2);
      int y1 = (int) (boundingRect.getY() + height - 2 * yMargin);
      GUIUtils.drawSmallText(g2, questionID, Color.RED, x1, y1);
   }

   public Dimension getPreferredSize() {
      if (bounds == null) {
         getBounds((Graphics2D) peer.getGraphics());
      }
      return new Dimension((int) bounds.getWidth(), (int) bounds.getHeight());
   }

   public void translate(double deltaX, double deltaY) {
      if (children == null) {
         return;
      }
      for (Iterator<IDisplayComponent> iter = children.iterator(); iter.hasNext();) {
         IDisplayComponent child = iter.next();
         child.translate(deltaX, deltaY);
      }
   }

   public void invalidate() {
   /** @todo */
   }

   public void setModel(Object newModel) {
      this.model = newModel;
   }

   public Object getModel() {
      return this.model;
   }

   // ---------------------- setters --------------
   public void setX(int newX) {
      this.x = newX;
   }

   public void setY(int newY) {
      this.y = newY;
   }

   public void setWidth(int newWidth) {
      this.width = newWidth;
   }

   public void setHeight(int newHeight) {
      this.height = newHeight;
   }

   public void setParent(IDisplayComponent newParent) {
      this.parent = newParent;
   }

   public void setPeer(Component newPeer) {
      this.peer = newPeer;
   }

   public void setName(String newName) {
      this.name = newName;
   }

   public void setId(String newId) {
      this.id = newId;
   }

   /** gets the logical group id */
   public String getGroupID() {
      return this.groupID;
   }

   /** sets the logical group id */
   public void setGroupID(String groupID) {
      this.groupID = groupID;
   }

   public void setQuestionID(String newQuestionID) {
      this.questionID = newQuestionID;
   }

   public String getQuestionID() {
      return this.questionID;
   }

   public void setJustification(Justification newJustification) {
      this.justification = newJustification;
   }

   public void setShowAssociations(boolean value) {
      showAssociation = value;
   }

   public void setShowQuestions(boolean value) {
      showQuestions = value;
   }

   public boolean getShowQuestions() {
      return showQuestions;
   }

   // ---------------------- getters --------------
   public int getX() {
      return this.x;
   }

   public int getY() {
      return this.y;
   }

   public int getWidth() {
      return this.width;
   }

   public int getHeight() {
      return this.height;
   }

   public IDisplayComponent getParent() {
      return this.parent;
   }

   public String getId() {
      return this.id;
   }

   public String getName() {
      return this.name;
   }

   public String getCSSClass() {
      return this.cssClass;
   }

   public void setCSSClass(String className) {
      this.cssClass = className;
   }

   public Justification getJustification() {
      return this.justification;
   }

   public boolean getShowAssociations() {
      return showAssociation;
   }

   public Object clone() {
      try {
         AbstractDisplayComponent clonedObj = (AbstractDisplayComponent) super
               .clone();
         if (children != null) {
            for (Iterator<IDisplayComponent> iter = children.iterator(); iter.hasNext();) {
               IDisplayComponent idc = iter.next();
               clonedObj.addChild((IDisplayComponent) idc.clone());
            }

         }
         return clonedObj;
      } catch (CloneNotSupportedException ce) {
         return null;
      }
   }

   public Element toXML(Element root) {
      Element e = new Element("object");
      // peer is transient
      e.addContent(XMLUtils.prepareProperty("x", String.valueOf(x)));
      e.addContent(XMLUtils.prepareProperty("y", String.valueOf(y)));
      e.addContent(XMLUtils.prepareProperty("height", String.valueOf(height)));
      e.addContent(XMLUtils.prepareProperty("width", String.valueOf(width)));
      e.addContent(XMLUtils.prepareProperty("id", id));
      e.addContent(XMLUtils.prepareProperty("name", name));
      if (cssClass != null && cssClass.trim().length() > 0)
         e.addContent(XMLUtils.prepareProperty("cssClass", cssClass));

      if (parent != null)
         e.addContent(XMLUtils.prepareRefID("parent", String.valueOf(parent
               .hashCode())));
      return e;
   }

   public void setSelected(boolean value) {
      this.selected = value;
   }

   public boolean getSelected() {
      return this.selected;
   }

   public boolean isContained(Rectangle2D rect, Graphics2D g2) {
      if (rect == null)
         return false;
      Rectangle2D b = getBounds(g2);
      if (b == null)
         return false;
      return rect.contains(b);
   }

   public boolean contains(Rectangle2D rect, Graphics2D g2) {
      if (rect == null)
         return false;
      Rectangle2D b = getBounds(g2);
      if (b == null)
         return false;
      return b.contains(rect);
   }

   public boolean contains(IDisplayComponent ic) {
      return false;
   }

   public String toString() {
      StringBuffer buf = new StringBuffer();
      buf.append("AbstractDisplayComponent::[");
      buf.append("id=").append(id).append(",height=").append(height);
      buf.append(",width=").append(width);
      buf.append(']');
      return buf.toString();
   }
}
