package caslayout.ui;

import java.awt.*;

import caslayout.util.*;
import java.awt.geom.*;
import java.awt.font.FontRenderContext;
import org.jdom.Element;
import java.util.*;
import java.util.List;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: ButtonDisplayComponent.java,v 1.8 2005/10/07 00:34:44 bozyurt
 *          Exp $
 */

public class ButtonDisplayComponent extends AbstractDisplayComponent {
   protected ButtonType action;
   protected String label;
   protected boolean javascriptOnly;
   protected List<Integer> questionIDs = new LinkedList<Integer>();

   private static Color gray = new Color(173, 173, 173);
   private static Color lightGray = new Color(207, 207, 207);
   private static Color veryLightGray = new Color(245, 245, 245);

   public ButtonDisplayComponent() {
      super();
   }

   public ButtonDisplayComponent(String id) {
      super(id);
   }

   public ButtonDisplayComponent(int x, int y, Component peer) {
      this(x, y, peer, ButtonType.SUBMIT_BUTTON);
   }

   public ButtonDisplayComponent(int x, int y, Component peer, ButtonType action) {
      super(x, y, peer);
      this.action = action;
      this.label = action.getValue();
   }

   public ButtonDisplayComponent(ButtonDisplayComponent other) {
      super(other);
      this.id = UIDManager.getInstance().getNextUID();

      this.action = other.action;
      this.label = other.action.getValue();
   }

   public Object clone() {
      return new ButtonDisplayComponent(this);
   }

   // ---------------------- setters --------------
   public void setAction(ButtonType newAction) {
      this.action = newAction;
      this.label = action.getValue();
   }

   public void setLabel(String newLabel) {
      this.label = newLabel;
   }

   public void setJavascriptOnly(boolean newJavascriptOnly) {
      this.javascriptOnly = newJavascriptOnly;
   }

   public void addQuestionID(Integer questionID) {
      questionIDs.add(questionID);
   }

   void clearQuestionIDs() {
      questionIDs.clear();
   }

   // ---------------------- getters --------------
   public ButtonType getAction() {
      return this.action;
   }

   public String getLabel() {
      return this.label;
   }

   public boolean getJavascriptOnly() {
      return this.javascriptOnly;
   }

   public List<Integer> getQuestionIDs() {
      return questionIDs;
   }

   public void draw(Graphics2D g2) {
      Rectangle2D r = getBounds(g2);
      Color oldColor = g2.getColor();
      Font font = g2.getFont();

      if (getShowQuestions())
         super.showQuestion(g2, r, Color.ORANGE);

      FontRenderContext frCtx = g2.getFontRenderContext();
      String str = (label.length() > 0) ? label : "    ";

      Font newFont = new Font(font.getName(), Font.BOLD, font.getSize());
      Rectangle2D sbr = newFont.getStringBounds(str, frCtx);
      int hgap = 4;
      int vgap = 4;
      int lgap = 4;

      hgap = (int) (r.getWidth() - sbr.getWidth() - 2 * lgap) / 2;
      if (hgap < 0) {
         hgap = 4;
      }
      vgap = (int) (r.getHeight() - sbr.getHeight() - 2 * lgap) / 2;
      if (vgap < 0)
         vgap = 4;
      int bw = (int) sbr.getWidth() + 2 * lgap;
      int bh = (int) sbr.getHeight() + 2 * lgap;
      double x1 = r.getX() + hgap;
      double y1 = r.getY() + (r.getHeight() - bh) / 2;

      Dimension2D compDim = new Dimension((int) bw, (int) bh);
      Point2D point = GUIUtils.calculateAlignmentPosition(r, compDim,
            justification);

      x1 = point.getX();
      y1 = point.getY();

      Rectangle2D.Double buttonBox = new Rectangle2D.Double(x1, y1, bw, bh);
      g2.setPaint(Color.lightGray);
      g2.fill(buttonBox);
      g2.setColor(gray);
      g2.draw(new Rectangle2D.Double(x1, y1, bw, bh));
      g2.setColor(lightGray);

      g2.draw(new Line2D.Double(x1 + 1, y1 + bh - 1, x1 + bw - 1, y1 + bh - 1));
      g2.draw(new Line2D.Double(x1 + bw - 1, y1 + 1, x1 + bw - 1, y1 + bh - 1));
      g2.setColor(veryLightGray);

      g2.draw(new Line2D.Double(x1 + 1, y1 + 1, x1 + bw - 1, y1 + 1));
      g2.draw(new Line2D.Double(x1 + 1, y1 + 1, x1 + 1, y1 + bh - 1));

      g2.setColor(oldColor);

      g2.setFont(newFont);

      g2.drawString(str, (int) x1 + lgap,
            (int) (y1 + sbr.getHeight() + lgap / 2));
      g2.setFont(font);

   }

   public Rectangle2D getBounds(Graphics2D g2) {
      if (super.bounds == null) {
         Rectangle rb = peer.getBounds();
         Rectangle2D.Double r = new Rectangle2D.Double(x + 3, y + 3, rb
               .getWidth() - 6, rb.getHeight() - 6);

         super.bounds = r;
      }
      return super.bounds;
   }

   public static ButtonDisplayComponent initializeFromXML(Element e) {
      String label = XMLUtils.getStringPropertyValue("label", e, "");
      String action = XMLUtils.getStringPropertyValue("action", e, "");
      String id = XMLUtils.getStringPropertyValue("id", e, "");
      String name = XMLUtils.getStringPropertyValue("name", e, "");
      String cssClass = XMLUtils.getStringPropertyValue("cssClass", e, null);

      String jsOnlyStr = XMLUtils.getStringPropertyValue("javascript-only", e,
            null);
      String qidListStr = XMLUtils.getStringPropertyValue("question-ids", e,
            null);

      ButtonDisplayComponent bdc = new ButtonDisplayComponent(id);
      bdc.name = name;
      bdc.action = ButtonType.findButtonType(action);
      bdc.label = label;

      bdc.x = XMLUtils.getPropertyValue("x", e, 0);
      bdc.y = XMLUtils.getPropertyValue("y", e, 0);
      bdc.width = XMLUtils.getPropertyValue("height", e, 0);
      bdc.height = XMLUtils.getPropertyValue("height", e, 0);
      String justificationValue = XMLUtils.getStringPropertyValue(
            "justification", e, null);
      if (justificationValue != null) {
         bdc.setJustification(Justification
               .findJustification(justificationValue));
      }

      bdc.cssClass = cssClass;
      String refID = XMLUtils.getReferencePropertyValue("parent", e, null);
      if (refID != null)
         bdc.parent = (IDisplayComponent) XMLUtils
               .getReferencedObject(refID, e);

      if (jsOnlyStr != null) {
         bdc.setJavascriptOnly(new Boolean(jsOnlyStr).booleanValue());
      }
      if (qidListStr != null) {
         StringTokenizer stok = new StringTokenizer(qidListStr, ",");
         while (stok.hasMoreTokens()) {
            bdc.addQuestionID(new Integer(stok.nextToken()));
         }
      }
      return bdc;
   }

   public Element toXML(Element root) {
      Element e = super.toXML(root);
      e.setAttribute("class", "caslayout.ui.ButtonDisplayComponent");
      e.addContent(XMLUtils.prepareProperty("label", label));
      e.addContent(XMLUtils.prepareProperty("action", action.getValue()));
      e.addContent(XMLUtils.prepareProperty("justification", justification
            .getValue()));
      if (!questionIDs.isEmpty()) {
         e.addContent(XMLUtils.prepareProperty("javascript-only", String
               .valueOf(javascriptOnly)));
         StringBuffer buf = new StringBuffer();
         for (Iterator<Integer> iter = questionIDs.iterator(); iter.hasNext();) {
            Integer qid = iter.next();
            buf.append(qid);
            if (iter.hasNext()) {
               buf.append(',');
            }
         }
         e.addContent(XMLUtils.prepareProperty("question-ids", buf.toString()));
      }
      return e;
   }

}
