package caslayout.ui;

import java.util.*;
import java.util.List;

import java.awt.*;
import java.awt.geom.*;

import org.jdom.*;
import caslayout.util.XMLUtils;
import caslayout.util.UIDManager;
import caslayout.ui.model.AssociationHelper;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: CAContainer.java,v 1.23 2008/10/13 23:58:07 bozyurt Exp $
 */
public class CAContainer extends AbstractDisplayComponent {
   protected List<IDisplayComponent> components = new ArrayList<IDisplayComponent>(1);
   protected ILayoutManager layoutMan;
   /** Specifies the space that a container must leave at each of its edges */
   protected Insets insets = new Insets(3, 3, 3, 3);
   protected boolean isGroup = false;

   public CAContainer() {
      super(UIDManager.getInstance().getNextUID());
   }

   public CAContainer(String id) {
      super(id);
   }

   public CAContainer(int x, int y, ILayoutManager layoutMan) {
      this(x, y, layoutMan, UIDManager.getInstance().getNextUID());
   }

   public CAContainer(int x, int y, ILayoutManager layoutMan, String id) {
      super(x, y, null);
      this.id = id;
      this.layoutMan = layoutMan;
   }

   public CAContainer(ILayoutManager layoutMan) {
      super(UIDManager.getInstance().getNextUID());
      this.layoutMan = layoutMan;
   }

   public CAContainer(CAContainer other) {
      super(other.x, other.y, other.peer,
            UIDManager.getInstance().getNextUID(), other.name,
            Justification.LEFT_JUSTIFICATION);

      this.name = other.name;
      this.isGroup = other.isGroup;

      this.insets = (Insets) other.insets.clone();
      if (other.layoutMan != null) {
         this.layoutMan = (ILayoutManager) other.layoutMan.clone();
      }

      if (!other.components.isEmpty()) {
         // do a deep copy
         this.components = new ArrayList<IDisplayComponent>(other.components.size());
         int idx = 0;
         for (Iterator<IDisplayComponent> iter = other.components.iterator(); iter.hasNext();) {
            IDisplayComponent ic = iter.next();
            if (ic != null) {
               add((IDisplayComponent) ic.clone(), idx);
            } else {
               add(null, idx);
            }
            ++idx;
         }
      }
   }

   public void add(IDisplayComponent ic) {
      addDisplayComponent(ic, null, -1);
   }

   public void add(IDisplayComponent ic, int index) {
      addDisplayComponent(ic, null, index);
   }

   public void add(IDisplayComponent ic, Object constraints) {
      addDisplayComponent(ic, constraints, -1);
   }

   public void add(IDisplayComponent ic, Object constraints, int index) {
      addDisplayComponent(ic, constraints, index);
   }

   public List<IDisplayComponent> getComponents() {
      return components;
   }

   public void setInsets(Insets insets) {
      this.insets = insets;
   }

   public Insets getInsets() {
      return insets;
   }

   public void layout() {
      if (layoutMan != null) {
         layoutMan.layoutContainer(this);
      }
   }

   public void invalidate() {
      if (layoutMan != null) {
         layoutMan.invalidateLayout(this);
      }
      super.invalidate();

   }

   public void draw(Graphics2D g2) {
      if (layoutMan != null) {
         layoutMan.layoutContainer(this);
      }
      Color oldColor = g2.getColor();
      if (isGroup) {
         g2.setColor(Color.magenta);
      }
      for (Iterator<IDisplayComponent> iter = components.iterator(); iter.hasNext();) {
         IDisplayComponent ic = iter.next();
         if (ic == null) {
            continue;
         }
         ic.draw(g2);
      }
      if (isGroup) {
         g2.setColor(oldColor);
      }

   }

   public Rectangle2D getBounds(Graphics2D g2) {
      return bounds;
   }

   public Object hitTest(int mX, int mY, Graphics2D g2) {
      if (layoutMan != null) {
         return layoutMan.hitTest(mX, mY, this, g2);
      }
      return null;
   }

   protected void addDisplayComponent(IDisplayComponent ic, Object constraints,
         int idx) {
      if (ic != null) {
         ic.setParent(this);
      }
      if (idx >= 0) {
         if (idx >= components.size()) {
            int diff = idx - components.size();
            for (int i = 0; i < diff; ++i) {
               components.add(null);
            }
            components.add(ic);
         } else {
            components.set(idx, ic);
         }
      } else {
         components.add(ic);
      }
      if (layoutMan != null) {
         layoutMan.addLayoutComponent(ic, constraints);
      }

      if (log.isDebugEnabled()) {
         for (Iterator<IDisplayComponent> iter = components.iterator(); iter.hasNext();) {
            IDisplayComponent anIC = iter.next();
            log.debug(" " + anIC);
         }
      }
   }

   public int getComponentCount() {
      return components.size();
   }

   public ILayoutManager getLayoutManager() {
      return this.layoutMan;
   }

   public void remove(IDisplayComponent ic) {

      int idx = components.indexOf(ic);
      if (idx != -1) {
         // propagate the deletion to all interested parties
         AssociationHelper.getInstance().componentRemoved(
               (AbstractDisplayComponent) ic);
         components.set(idx, null);
      }

      if (layoutMan != null) {
         layoutMan.removeLayoutComponent(ic);
      }
   }

   /**
    * removes the display component from the grid shifting everything after the
    * removed component, one cell to the left
    *
    * @param ic
    */
   public void removePermanently(IDisplayComponent ic) {
      components.remove(ic);
   }

   public void removePermanently(List<Integer> offsetList) {
      if (offsetList.isEmpty())
         return;

      int startIdx = ((Integer) offsetList.get(0)).intValue();
      int lastIdx = ((Integer) offsetList.get(1)).intValue();

      ListIterator<IDisplayComponent> iter = components.listIterator(startIdx);
      int noItems = lastIdx - startIdx + 1;
      int count = 0;
      while (iter.hasNext()) {
         if (count >= noItems)
            break;
         AbstractDisplayComponent adc = (AbstractDisplayComponent) iter.next();
         if (adc != null)
            log.info("removing " + adc.toString());
         iter.remove();
         ++count;
      }
   }

   /**
    * inserts null values into components list to accomodate for a new empty row
    * at the given starting offset
    *
    * @param offsetList
    */
   public void insertRow(List<Integer> offsetList) {
      if (offsetList.isEmpty())
         return;
      int startIdx = ((Integer) offsetList.get(0)).intValue();
      int lastIdx = ((Integer) offsetList.get(1)).intValue();
      int noItems = lastIdx - startIdx + 1;

      for (int i = 0; i < noItems; i++) {
         components.add(startIdx + i, null);
      }
   }

   public int findComponentIndex(IDisplayComponent ic) {
      return components.indexOf(ic);
   }

   public Dimension getPreferredSize() {
      if (bounds == null) {
         return new Dimension(50, 16);
      } else {
         return new Dimension((int) bounds.getWidth(), (int) bounds.getHeight());
      }
   }

   public Object clone() {
      return new CAContainer(this);
   }

   public void setSelected(boolean value) {
      super.setSelected(value);
      if (layoutMan != null) {
         layoutMan.setSelected(selected);
      }
   }

   public void setIsGroup(boolean newIsGroup) {
      this.isGroup = newIsGroup;
      if (layoutMan != null) {
         Color c = (isGroup) ? Color.magenta : null;
         layoutMan.setGridColor(c);
      }
   }

   public boolean getIsGroup() {
      return this.isGroup;
   }

   public Element toXML(Element root) {
      Element e = super.toXML(root);
      e.setName("container");
      e.setAttribute("class", "caslayout.ui.CAContainer");
      e.setAttribute("isGroup", String.valueOf(isGroup));

      if (layoutMan != null) {
         Element layoutElem = layoutMan.toXML(root);
         e.addContent(layoutElem);
      }

      Element insetsElem = new Element("insets");
      insetsElem.setAttribute("class", "java.awt.Insets");
      insetsElem.addContent(XMLUtils.prepareProperty("left", String
            .valueOf(insets.left)));
      insetsElem.addContent(XMLUtils.prepareProperty("top", String
            .valueOf(insets.top)));
      insetsElem.addContent(XMLUtils.prepareProperty("right", String
            .valueOf(insets.right)));
      insetsElem.addContent(XMLUtils.prepareProperty("bottom", String
            .valueOf(insets.bottom)));
      e.addContent(insetsElem);

      Element compsElem = new Element("collection");
      compsElem.setAttribute("name", "components");
      compsElem.setAttribute("type", "list");
      compsElem.setAttribute("class", "java.util.ArrayList");
      compsElem.setAttribute("size", String.valueOf(components.size()));
      e.addContent(compsElem);
      int i = 0;
      for (Iterator<IDisplayComponent> iter = components.iterator(); iter.hasNext();) {
         IDisplayComponent ic = iter.next();
         if (ic != null) {
            Element icElem = new Element("entry");
            icElem.setAttribute("index", String.valueOf(i));
            icElem.addContent(ic.toXML(root));
            compsElem.addContent(icElem);
         }
         ++i;
      }

      return e;
   }

   public boolean contains(IDisplayComponent ic) {
      if (components.isEmpty()) {
         return false;
      }
      for (Iterator<IDisplayComponent> iter = components.iterator(); iter.hasNext();) {
         IDisplayComponent idc = iter.next();
         if (idc == ic) {
            return true;
         } else if (idc instanceof CAContainer) {
            boolean hasThis = idc.contains(ic);
            if (hasThis) {
               return true;
            }
         }
      }
      return false;
   }

}
