package caslayout.ui;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Insets;
import java.awt.Stroke;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import java.beans.DefaultPersistenceDelegate;
import java.beans.Encoder;
import java.beans.Statement;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

import org.apache.log4j.Logger;
import org.jdom.Element;

import caslayout.util.DebugUtils;
import caslayout.util.XMLUtils;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: CAGridLayout.java,v 1.29 2008/10/13 23:58:07 bozyurt Exp $
 */
public class CAGridLayout implements ILayoutManager, Serializable {
   private static final long serialVersionUID = -8509123058877539984L;
   protected int hGap;
   protected int vGap;
   protected int rows;
   protected int cols;
   protected Color gridColor;
   protected GridCellInfo gridCellInfo;
   protected Map<String, IDisplayComponent> dispCompMap = new HashMap<String, IDisplayComponent>();
   protected boolean selected = false;
   public final static boolean USE_PERCENTAGE = true;
   public final static boolean USE_MULTICELLS = false;

   protected static Logger log = Logger.getLogger("CAGridLayout");

   public CAGridLayout(boolean usePercentages) {
      if (rows < 0 || cols < 0)
         throw new IllegalArgumentException(
               "Both rows and columns should be greater than zero");
      this.rows = 1;
      this.cols = 0;
      this.hGap = 0;
      this.vGap = 0;
      gridCellInfo = new GridCellInfo(rows, cols, this, usePercentages);
   }

   public CAGridLayout(int rows, int cols, boolean usePercentages) {
      this(rows, cols, 0, 0, usePercentages);
   }

   public CAGridLayout(int rows, int cols, int hGap, int vGap,
         boolean usePercentages) {
      if (rows < 0 || cols < 0)
         throw new IllegalArgumentException(
               "Both rows and columns should be greater than zero");
      this.rows = rows;
      this.cols = cols;
      this.hGap = hGap;
      this.vGap = vGap;
      gridCellInfo = new GridCellInfo(rows, cols, this, usePercentages);
   }

   protected CAGridLayout(int rows, int cols, int hGap, int vGap) {
      if (rows < 0 || cols < 0)
         throw new IllegalArgumentException(
               "Both rows and columns should be greater than zero");
      this.rows = rows;
      this.cols = cols;
      this.hGap = hGap;
      this.vGap = vGap;
      gridCellInfo = new GridCellInfo(rows, cols, this);
   }

   public CAGridLayout(CAGridLayout other) {
      this.rows = other.rows;
      this.cols = other.cols;
      this.hGap = other.hGap;
      this.vGap = other.vGap;
      this.gridCellInfo = new GridCellInfo(other.gridCellInfo);
      this.gridCellInfo.parent = this;
   }

   // ---------------------- setters --------------
   public void setHGap(int newHGap) {
      this.hGap = newHGap;
   }

   public void setVGap(int newVGap) {
      this.vGap = newVGap;
   }

   public void setRows(int newRows) {
      this.rows = newRows;
      this.gridCellInfo.setMaxRows(newRows);
   }

   public void setCols(int newCols) {
      this.cols = newCols;
      this.gridCellInfo.setMaxCols(newCols);
   }

   /**
    * returns all non-null CellConstraint objects starting from row 0, column 0
    *
    * @return
    */
   public List<CellConstraint> enumerateCellConstraints() {
      List<CellConstraint> cList = new ArrayList<CellConstraint>();
      for (CellConstraint[] cellRow : this.gridCellInfo.getCellRows()) {
         for (int j = 0; j < cellRow.length; j++) {
            if (cellRow[j] != null) {
               cList.add(cellRow[j]);
            }
         }
      }
      return cList;
   }

   public List<Integer> insertRow(int afterRow) {
      List<Integer> offsetList = this.gridCellInfo.addRow(afterRow);
      return offsetList;
   }

   public List<Integer> removeRow(int aRowIdx) {
      List<Integer> offsetList = this.gridCellInfo.removeRow(aRowIdx);
      return offsetList;
   }

   public void setGridColor(Color newGridColor) {
      this.gridColor = newGridColor;
   }

   public Color getGridColor() {
      return this.gridColor;
   }

   public int getHGap() {
      return this.hGap;
   }

   public int getVGap() {
      return this.vGap;
   }

   public int getRows() {
      return this.rows;
   }

   public int getCols() {
      return this.cols;
   }

   public int getEffectiveRowCount() {
      return gridCellInfo.getRowCount();
   }

   public int getEffectiveColumnCount(int row) {
      return gridCellInfo.getColCount(row);
   }

   public void addLayoutComponent(IDisplayComponent ic, Object constraints) {
      CellConstraint cons = (CellConstraint) constraints;
      if (cons != null) {
         gridCellInfo.setCellConstraint(cons.getRowIdx(), cons.getColIdx(),
               cons);

         StringBuffer sb = new StringBuffer();
         sb.append(cons.getRowIdx()).append('_').append(cons.getColIdx());
         dispCompMap.put(sb.toString(), ic);
      }
   }

   /**
    * in case layoutManager is populated without using
    * <code>addLayoutComponent()</code>. (like deserialization)
    *
    * @param rowIdx
    * @param colIdx
    * @param ic
    */
   public void putToDispCompMap(int rowIdx, int colIdx, IDisplayComponent ic) {
      StringBuffer sb = new StringBuffer();
      sb.append(rowIdx).append('_').append(colIdx);
      dispCompMap.put(sb.toString(), ic);
      // log.info("putToDispCompMap key=" + sb.toString() + " value=" + ic);
   }

   public void removeLayoutComponent(IDisplayComponent ic) {
   // no-op
   }

   public void setCellConstraint(CellConstraint cons) {
      gridCellInfo.setCellConstraint(cons.getRowIdx(), cons.getColIdx(), cons);
   }

   public IDisplayComponent getDisplayComponentAt(int row, int col) {
      StringBuffer sb = new StringBuffer();
      sb.append(row).append('_').append(col);
      log.info("getDisplayComponentAt key=" + sb.toString() + " value="
            + dispCompMap.get(sb.toString()));
      return dispCompMap.get(sb.toString());
   }

   public Location findLocation(IDisplayComponent ic) {
      for (Map.Entry<String, IDisplayComponent> entry : dispCompMap.entrySet()) {
         IDisplayComponent idc = entry.getValue();
         if (idc == ic) {
            StringTokenizer stok = new StringTokenizer(entry.getKey(), "_");
            int row = Integer.parseInt(stok.nextToken());
            int col = Integer.parseInt(stok.nextToken());
            return new Location(row, col, -1);
         }
      }
      return null;
   }

   public CellConstraint getCellConstraint(int row, int col) {
      CellConstraint[] rowArr = (CellConstraint[]) gridCellInfo.getRow(row);
      return rowArr[col];
   }

   public CellConstraint[] getRowColumns(int row) {
      CellConstraint[] rowArr = (CellConstraint[]) gridCellInfo.getRow(row);
      return rowArr;
   }

   public void setGridCellInfo(GridCellInfo newGCI) {
      this.gridCellInfo = newGCI;
   }

   public Dimension preferredLayoutSize(CAContainer parent) {
      Insets insets = parent.getInsets();
      int ncomps = parent.getComponentCount();
      int noRows = rows;
      int noCols = cols;

      if (noRows > 0) {
         noCols = (ncomps + noRows - 1) / noRows;
      } else {
         noRows = (ncomps + noCols - 1) / noCols;
      }

      // determine the height and width of the grid that would would fit
      // the component with the largest preferred size
      int height = 0, width = 0;
      for (Iterator<IDisplayComponent> iter = parent.getComponents().iterator(); iter
            .hasNext();) {
         IDisplayComponent ic = iter.next();
         Dimension d = ic.getPreferredSize();
         if (d.width > width) {
            width = d.width;
         }
         if (d.height > height) {
            width = d.height;
         }
      }
      int lw = insets.left + insets.right + noCols * width + (noCols - 1)
            * hGap;
      int lh = insets.top + insets.bottom + noRows * height + (noRows - 1)
            * vGap;
      return new Dimension(lw, lh);
   }

   public GridLocation layoutGridHitTest(int mX, int mY, CAContainer parent,
         Graphics2D g2) {
      Rectangle2D bounds = parent.getBounds(g2);
      if (!bounds.contains(mX, mY)) {
         return null;
      }
      Insets insets = parent.getInsets();
      double lw = parent.getWidth() - insets.left - insets.right;
      double lh = parent.getHeight() - insets.top - insets.bottom;
      double ch = (lh - (rows - 1) * vGap) / rows;
      double cw = (lw - (cols - 1) * hGap) / cols;

      double x = parent.getX() + insets.left;
      double y = parent.getY() + insets.top;
      int selRow = 0;
      int selCol = 0;

      // check if a horizontal cell boundary is selected
      int rowCount = gridCellInfo.getRowCount();
      double curCellHeight = -1;
      for (int i = 0; i < rowCount; ++i) {
         CellConstraint[] rowCells = gridCellInfo.getRow(i);
         curCellHeight = rowCells[0].calculateHeight(ch, 0, lh);
         if (mY >= y && mY <= y + curCellHeight) {
            selRow = i;
            break;
         }
         y += vGap + curCellHeight;
      }

      CellConstraint[] rowCells = gridCellInfo.getRow(selRow);
      int colCount = gridCellInfo.getColCount(selRow);

      double curCellWidth = -1;
      CellConstraint selectedCell = null;
      for (int i = 0; i < colCount; ++i) {
         curCellWidth = rowCells[i].calculateWidth(cw, hGap, lw);
         if (mX >= x && mX <= x + curCellWidth) {
            selCol = i;
            selectedCell = rowCells[i];
            break;
         }
         x += hGap + rowCells[i].calculateWidth(cw, 0, lw);
      }

      curCellWidth = rowCells[selCol].calculateWidth(cw, hGap, lw);
      curCellHeight = gridCellInfo.getRow(selRow)[0].calculateHeight(ch, 0, lh);
      double rightDist = x + curCellWidth - mX;
      double leftDist = mX - x;
      double topDist = mY - y;
      double bottomDist = y + curCellHeight - mY;
      int selectedGridSide = -1;
      if (Math.min(rightDist, leftDist) < Math.min(topDist, bottomDist)) {
         if (rightDist > leftDist) {
            selectedGridSide = GridLocation.LEFT;
            if (atRowBoundary(selCol, rowCells, true)) {
               return null;
            }
         } else {
            selectedGridSide = GridLocation.RIGHT;
            if (atRowBoundary(selCol, rowCells, false)) {
               return null;
            }
         }
      } else if (Math.min(rightDist, leftDist) > Math.min(topDist, bottomDist)) {
         if (topDist > bottomDist) {
            selectedGridSide = GridLocation.BOTTOM;
            if (atColumnBoundary(selRow, rowCount, false)) {
               return null;
            }
         } else {
            selectedGridSide = GridLocation.TOP;
            if (atColumnBoundary(selRow, rowCount, true)) {
               return null;
            }
         }
      } else {
         if (rightDist > leftDist) {
            selectedGridSide = GridLocation.LEFT;
         } else {
            selectedGridSide = GridLocation.RIGHT;
         }
      }

      GridLocation gl = new GridLocation(selRow, selCol,
            new Rectangle2D.Double(x, y,
                  selectedCell.calculateWidth(cw, 0, lw), selectedCell
                        .calculateHeight(ch, 0, lh)));

      gl.setSelGridSide(selectedGridSide);
      gl.setIdx(gridCellInfo.getIndex(selRow, selCol));
      return gl;
   }

   public static boolean atColumnBoundary(int rowIdx, int rowCount,
         boolean toTop) {
      if (toTop) {
         if (rowIdx == 0) {
            return true;
         }
      }
      if (!toTop) {
         if (rowIdx == rowCount - 1) {
            return true;
         }
      }
      return false;
   }

   public static boolean atRowBoundary(int colIdx, CellConstraint[] rowCells,
         boolean toLeft) {
      if (toLeft) {
         if (colIdx == 0) {
            return true;
         }
         for (int i = 0; i < colIdx; i++) {
            if (!rowCells[i].hasZeroWidth()) {
               return false;
            }
         }
         return true;
      }

      if (!toLeft) {
         if (colIdx == rowCells.length - 1) {
            return true;
         }
         for (int i = colIdx + 1; i < rowCells.length; i++) {
            if (!rowCells[i].hasZeroWidth()) {
               return false;
            }
         }
         return true;
      }
      return false;
   }

   public boolean calcNewCellConstraints(GridLocation gridLoc,
         CAContainer parent, double deltaX, double deltaY) {
      Insets insets = parent.getInsets();
      double lw = parent.getWidth() - insets.left - insets.right;
      double lh = parent.getHeight() - insets.top - insets.bottom;

      CellConstraint cc = gridCellInfo.getConstraint(gridLoc.getRow(), gridLoc
            .getColumn());
      if (cc instanceof PercentCellConstraint) {
         // PercentCellConstraint pcc = (PercentCellConstraint) cc;
         CellConstraint[] rowCells = gridCellInfo.getRow(gridLoc.getRow());
         int selCol = gridLoc.getColumn();
         double absDeltaX = Math.abs(deltaX);
         double absDeltaY = Math.abs(deltaY);

         if (absDeltaX > 0
               && (gridLoc.getSelGridSide() == GridLocation.RIGHT || gridLoc
                     .getSelGridSide() == GridLocation.LEFT)) {
            // for percentcellconstraint only total width of the container is
            // required
            double w1 = rowCells[selCol].calculateWidth(0, 0, lw);
            if (gridLoc.getSelGridSide() == GridLocation.RIGHT) {
               double w2 = rowCells[gridLoc.getColumn() + 1].calculateWidth(0,
                     0, lw);
               if (deltaX > 0 && deltaX > (w2 - 3)) {
                  // System.out.println("(1) deltaX=" + deltaX + " w2 -3=" + (w2
                  // -3) );
                  return false;
               } else if (deltaX < 0 && -deltaX > (w1 - 3)) {
                  // System.out.println("(2) deltaX=" + deltaX + " w1 -3=" + (w1
                  // -3) );
                  return false;
               }
               double newW1Per = PercentCellConstraint.calculatePercentage(w1
                     + deltaX, lw);
               double newW2Per = PercentCellConstraint.calculatePercentage(w2
                     - deltaX, lw);
               ((PercentCellConstraint) rowCells[selCol])
                     .setColPercent(newW1Per);
               ((PercentCellConstraint) rowCells[selCol + 1])
                     .setColPercent(newW2Per);
               return true;
            } else if (gridLoc.getSelGridSide() == GridLocation.LEFT) {
               double w2 = rowCells[gridLoc.getColumn() - 1].calculateWidth(0,
                     0, lw);
               if (deltaX > 0 && deltaX > (w2 - 3)) {
                  return false;
               } else if (deltaX < 0 && -deltaX > (w1 - 3)) {
                  return false;
               }
               double newW1Per = PercentCellConstraint.calculatePercentage(w1
                     - deltaX, lw);
               double newW2Per = PercentCellConstraint.calculatePercentage(w2
                     + deltaX, lw);
               ((PercentCellConstraint) rowCells[selCol])
                     .setColPercent(newW1Per);
               ((PercentCellConstraint) rowCells[selCol - 1])
                     .setColPercent(newW2Per);
               return true;
            }
         }

         if (absDeltaY > 0
               && (gridLoc.getSelGridSide() == GridLocation.BOTTOM || gridLoc
                     .getSelGridSide() == GridLocation.TOP)) {
            double h1 = rowCells[0].calculateHeight(0, 0, lh);
            if (gridLoc.getSelGridSide() == GridLocation.BOTTOM) {
               CellConstraint[] rowCells2 = gridCellInfo.getRow(gridLoc
                     .getRow() + 1);
               double h2 = rowCells2[0].calculateHeight(0, 0, lh);
               if (deltaY > 0 && deltaY > (h2 - 3)) {
                  return false;
               } else if (deltaY < 0 && -deltaY > (h1 - 3)) {
                  return false;
               }
               double newH1Per = PercentCellConstraint.calculatePercentage(h1
                     + deltaY, lh);
               double newH2Per = PercentCellConstraint.calculatePercentage(h2
                     - deltaY, lh);
               setRowPercent(rowCells, newH1Per);
               setRowPercent(rowCells2, newH2Per);
               return true;
            } else if (gridLoc.getSelGridSide() == GridLocation.TOP) {
               CellConstraint[] rowCells2 = gridCellInfo.getRow(gridLoc
                     .getRow() - 1);
               double h2 = rowCells2[0].calculateHeight(0, 0, lh);
               if (deltaY > 0 && deltaY > (h2 - 3)) {
                  return false;
               } else if (deltaY < 0 && -deltaY > (h1 - 3)) {
                  return false;
               }
               double newH1Per = PercentCellConstraint.calculatePercentage(h1
                     - deltaY, lh);
               double newH2Per = PercentCellConstraint.calculatePercentage(h2
                     + deltaY, lh);
               setRowPercent(rowCells, newH1Per);
               setRowPercent(rowCells2, newH2Per);
               return true;
            }
         }
      }
      return false;
   }

   private void setRowPercent(CellConstraint[] rowCells, double newHeightPerc) {
      for (int i = 0; i < rowCells.length; i++) {
         ((PercentCellConstraint) rowCells[i]).setRowPercent(newHeightPerc);
      }
   }

   public void hiliteGridWalls(CAContainer parent, Graphics2D g2,
         GridLocation gridLoc) {
      Insets insets = parent.getInsets();
      Line2D line = null;
      double lw = parent.getWidth() - insets.left - insets.right;
      double lh = parent.getHeight() - insets.top - insets.bottom;
      double cw = (lw - (cols - 1) * hGap) / cols;
      double ch = (lh - (rows - 1) * vGap) / rows;
      // top left corner
      double x1 = parent.getX() + insets.left;
      double y1 = parent.getY() + insets.top;
      Color oldColor = g2.getColor();
      Stroke oldStroke = g2.getStroke();
      g2.setColor(Color.red);
      g2.setStroke(new BasicStroke(2));
      // g2.setXORMode(Color.yellow);

      int rowCount = gridCellInfo.getRowCount();
      for (int i = 0; i < rowCount; ++i) {
         CellConstraint[] rowCells = gridCellInfo.getRow(i);
         double deltaY = vGap + rowCells[0].calculateHeight(ch, vGap, lh);
         if (gridLoc.getRow() == i) {
            CellConstraint cell = rowCells[gridLoc.getColumn()];
            for (int j = 0; j < gridLoc.getColumn(); j++) {
               x1 += hGap + rowCells[j].calculateWidth(cw, hGap, lw);
            }
            double curCellWidth = cell.calculateWidth(cw, hGap, lw);

            if (gridLoc.selGridSide == GridLocation.TOP) {
               g2.draw(line = new Line2D.Double(x1, y1, x1 + curCellWidth, y1));
            } else if (gridLoc.selGridSide == GridLocation.BOTTOM) {
               g2.draw(line = new Line2D.Double(x1, y1 + deltaY, x1
                     + curCellWidth, y1 + deltaY));
            } else if (gridLoc.selGridSide == GridLocation.LEFT) {
               g2.draw(line = new Line2D.Double(x1, y1, x1, y1 + deltaY));
            } else {
               g2.draw(line = new Line2D.Double(x1 + curCellWidth, y1, x1
                     + curCellWidth, y1 + deltaY));
            }

            if (log.isDebugEnabled()) {
               log.debug(this.getClass().toString() + " "
                     + DebugUtils.dumpLine2D(line));
            }
         }

         y1 += vGap + rowCells[0].calculateHeight(ch, vGap, lh);
      }

      g2.setPaintMode();
      g2.setStroke(oldStroke);
      g2.setColor(oldColor);
   }

   public void displayLayoutGrid(CAContainer parent, Graphics2D g2,
         boolean debugEnabled) {
      Insets insets = parent.getInsets();
      Line2D line = null;
      double lw = parent.getWidth() - insets.left - insets.right;
      double lh = parent.getHeight() - insets.top - insets.bottom;
      if (log.isDebugEnabled())
         log.debug("parentHeight=" + parent.getHeight() + " insets="
               + insets.toString());
      double cw = (lw - (cols - 1) * hGap) / cols;
      double ch = (lh - (rows - 1) * vGap) / rows;
      // top left corner
      double x1 = parent.getX() + insets.left;
      double y1 = parent.getY() + insets.top;
      Color oldColor = g2.getColor();

      // log.debug("ch=" + ch);

      /*
       * if ( parent.getId().equals("C7") ) { log.info("ID:" + parent.getId() + "
       * x1=" + x1 + " y1=" + y1); log.info("parentHeight=" + parent.getHeight() + "
       * insets=" + insets.toString()); log.info(parent.getBounds(g2).toString() ); }
       */

      Stroke oldStroke = g2.getStroke();
      if (getSelected()) {
         float[] dashPattern = { 2, 1, 2 };
         BasicStroke stroke = new BasicStroke(1, BasicStroke.CAP_BUTT,
               BasicStroke.JOIN_MITER, 1, dashPattern, 0);
         g2.setStroke(stroke);
         g2.setColor(Color.blue);
      } else if (gridColor != null) {
         g2.setColor(gridColor);
      } else {
         g2.setColor(Color.red);
      }

      if (log.isDebugEnabled()) {
         log.debug(this.getClass().toString() + " parent X=" + parent.getX()
               + " Y=" + parent.getY());
      }
      int rowCount = gridCellInfo.getRowCount();

      log.debug(">> rowCount=" + rowCount);
      int colCount = 0;

      // draw horizontal grid lines
      double[] ycoords = new double[rowCount + 1];
      for (int i = 0; i < rowCount; ++i) {
         ycoords[i] = y1;

         g2.draw(line = new Line2D.Double(x1, y1, x1 + lw, y1));
         if (log.isDebugEnabled()) {
            log.debug(this.getClass().toString() + " "
                  + DebugUtils.dumpLine2D(line));
         }

         CellConstraint[] rowCells = gridCellInfo.getRow(i);
         y1 += vGap + rowCells[0].calculateHeight(ch, vGap, lh);

      }

      // draw bottom border
      y1 = parent.getY() + insets.top + lh;
      ycoords[rowCount] = y1;
      g2.draw(line = new Line2D.Double(x1, y1, x1 + lw, y1));
      if (log.isDebugEnabled()) {
         log.debug(this.getClass().toString() + " "
               + DebugUtils.dumpLine2D(line));
      }

      // foreach row draw the vertical grid lines
      y1 = parent.getY() + insets.top;
      for (int i = 0; i < rowCount; ++i) {
         x1 = parent.getX() + insets.left;
         CellConstraint[] rowCells = gridCellInfo.getRow(i);

         colCount = gridCellInfo.getColCount(i);

         log.debug(">> colCount=" + colCount);
         for (int j = 0; j < colCount; ++j) {
            CellConstraint cons = rowCells[j];
            // for debugging
            if (log.isDebugEnabled()) {
               log.debug("drawing " + cons.toString());
            }

            g2.draw(line = new Line2D.Double(x1, y1, x1, ycoords[i + 1]));

            if (log.isDebugEnabled()) {
               log.debug(this.getClass().toString() + " "
                     + DebugUtils.dumpLine2D(line));
            }

            // don't forget the horizontal gaps if the colSpan is larger
            // than 1 (IBO 7/1/04)
            x1 += hGap + cons.calculateWidth(cw, hGap, lw);
         }

         y1 = ycoords[i + 1];

         if (log.isDebugEnabled()) {
            log.debug("after row " + i + " y1=" + y1);
         }
      }

      if (log.isDebugEnabled()) {
         log.debug("1 y1=" + y1 + " y+ lh ="
               + (parent.getY() + insets.top + lh) + " parent.Y="
               + parent.getY() + " lh=" + lh);
      }

      x1 = parent.getX() + insets.left + lw;
      y1 = parent.getY() + insets.top;
      // draw the right border
      g2.draw(line = new Line2D.Double(x1, y1, x1, y1 + lh));
      if (log.isDebugEnabled()) {
         log.debug(this.getClass().toString() + " "
               + DebugUtils.dumpLine2D(line));
      }

      g2.setColor(oldColor);
      g2.setStroke(oldStroke);
   }

   public Object hitTest(int mX, int mY, CAContainer parent, Graphics2D g2) {
      Rectangle2D bounds = parent.getBounds(g2);
      if (bounds.contains(mX, mY)) {
         Insets insets = parent.getInsets();
         double lw = parent.getWidth() - insets.left - insets.right;
         double lh = parent.getHeight() - insets.top - insets.bottom;

         double cw = (lw - (cols - 1) * hGap) / cols;
         double ch = (lh - (rows - 1) * vGap) / rows;
         double x = parent.getX() + insets.left;
         double y = parent.getY() + insets.top;
         int selRow = 0;
         int selCol = 0;

         int rowCount = gridCellInfo.getRowCount();
         int colCount = 0;

         for (int i = 0; i < rowCount; ++i) {
            CellConstraint[] rowCells = gridCellInfo.getRow(i);

            if (mY >= y && mY <= y + rowCells[0].calculateHeight(ch, 0, lh)) {
               selRow = i;
               break;
            }

            y += vGap + rowCells[0].calculateHeight(ch, 0, lh);
         }

         CellConstraint[] rowCells = gridCellInfo.getRow(selRow);
         colCount = gridCellInfo.getColCount(selRow);

         CellConstraint selectedCell = null;
         for (int i = 0; i < colCount; ++i) {
            if (mX >= x && mX <= x + rowCells[i].calculateWidth(cw, hGap, lw)) {
               selCol = i;
               selectedCell = rowCells[i];
               break;
            }
            x += hGap + rowCells[i].calculateWidth(cw, 0, lw);
         }

         GridLocation gl = new GridLocation(selRow, selCol,
               new Rectangle2D.Double(x, y, selectedCell.calculateWidth(cw, 0,
                     lw), selectedCell.calculateHeight(ch, 0, lh)));

         gl.setIdx(gridCellInfo.getIndex(selRow, selCol));
         log.debug("hitTest:: " + gl);
         return gl;
      }

      return null;
   }

   public void invalidateLayout(CAContainer parent) {
      /** @todo Implement this caslayout.ui.ILayoutManager method */
      throw new java.lang.UnsupportedOperationException(
            "Method invalidateLayout() not yet implemented.");
   }

   public Object clone() {
      return new CAGridLayout(this);
   }

   public static void setPersistenceDelegate(Encoder encoder) {
      encoder.setPersistenceDelegate(CAGridLayout.class,
            new DefaultPersistenceDelegate() {
               protected void initialize(Class<?> type, Object oldInstance,
                     Object newInstance, Encoder out) {
                  super.initialize(type, oldInstance, newInstance, out);
                  CAGridLayout layout = (CAGridLayout) oldInstance;
                  try {
                     GridCellInfo gci = new GridCellInfo(layout.gridCellInfo);
                     System.out.println(">> persisting setGridCellInfo");
                     out.writeStatement(new Statement(oldInstance,
                           "setGridCellInfo", new Object[] { gci }));

                  } catch (Exception x) {
                     x.printStackTrace();
                  }
               }
            });
   }

   public String toString() {
      StringBuffer buf = new StringBuffer(128);
      buf.append("CAGridLayout ::[");
      buf.append("rows=").append(rows).append(",cols=").append(cols);
      buf.append("hGap=").append(hGap).append(",vGap=").append(vGap);
      buf.append(']');
      return buf.toString();
   }

   public void layoutContainer(CAContainer parent) {
      Insets insets = parent.getInsets();
      int ncomps = parent.getComponentCount();
      int noRows = rows;
      int noCols = cols;

      if (ncomps == 0)
         return;

      double lw = parent.getWidth() - insets.left - insets.right;
      double lh = parent.getHeight() - insets.top - insets.bottom;
      double cw = (lw - (noCols - 1) * hGap) / noCols;
      double ch = (lh - (noRows - 1) * vGap) / noRows;

      double x1 = parent.getX() + insets.left;
      double y1 = parent.getY() + insets.top;
      int index = 0;
      int rowCount = gridCellInfo.getRowCount();
      int colCount = 0;

      for (int i = 0; i < rowCount; ++i) {
         /*
          * if ( parent.getId().equals("C1") ) { log.info("layoutContainer ID:" +
          * parent.getId() + " y1:" + y1); }
          */
         x1 = parent.getX() + insets.left;
         CellConstraint[] rowCells = gridCellInfo.getRow(i);

         colCount = gridCellInfo.getColCount(i);
         for (int j = 0; j < colCount; j++) {
            CellConstraint cons = rowCells[j];

            if (index < parent.getComponents().size()) {
               IDisplayComponent ic = (IDisplayComponent) parent
                     .getComponents().get(index);

               if (ic != null) {
                  Rectangle2D.Double r = new Rectangle2D.Double(x1, y1, cons
                        .calculateWidth(cw, 0, lw), cons.calculateHeight(ch, 0,
                        lh));

                  if (log.isDebugEnabled()) {
                     log.debug(">>>> setting bounds for row=" + i + " column="
                           + j + " r=" + r);
                  }
                  ic.setBounds(r);
               }
            }
            index++;
            // x1 += hGap + cw * cons.colSpan;
            x1 += hGap + cons.calculateWidth(cw, hGap, lw);
         }// j
         // y1 += vGap + ch * rowSpan;
         y1 += vGap + rowCells[0].calculateHeight(ch, 0, lh);
      }// i
   }

   public void setSelected(boolean value) {
      this.selected = value;
   }

   public boolean getSelected() {
      return this.selected;
   }

   public static Object initializeFromXML(Element e) {
      int cols = 0, rows = 1, hGap = 3, vGap = 3;
      String refID = e.getAttributeValue("id");

      cols = XMLUtils.getPropertyValue("cols", e, cols);
      rows = XMLUtils.getPropertyValue("rows", e, rows);
      hGap = XMLUtils.getPropertyValue("hGap", e, hGap);
      vGap = XMLUtils.getPropertyValue("vGap", e, vGap);

      CAGridLayout layout = new CAGridLayout(rows, cols, hGap, vGap);
      XMLUtils.registerObject(refID, layout);
      Element objElem = e.getChild("grid-cell-info");
      if (objElem != null) {
         GridCellInfo gci = GridCellInfo.initializeFromXML(objElem);
         layout.setGridCellInfo(gci);
      }
      return layout;
   }

   public Element toXML(Element root) {
      Element e = new Element("layout-man");
      e.setAttribute("id", String.valueOf(this.hashCode()));
      e.setAttribute("class", "caslayout.ui.CAGridLayout");
      e.addContent(XMLUtils.prepareProperty("cols", String.valueOf(cols)));
      e.addContent(XMLUtils.prepareProperty("rows", String.valueOf(rows)));
      e.addContent(XMLUtils.prepareProperty("hGap", String.valueOf(hGap)));
      e.addContent(XMLUtils.prepareProperty("vGap", String.valueOf(vGap)));
      e.addContent(gridCellInfo.toXML(root));
      return e;
   }

   public static class Location {
      int row;
      int col;
      int idx;

      public Location(int row, int col, int idx) {
         this.row = row;
         this.col = col;
         this.idx = idx;
      }

      public int getRow() {
         return row;
      }

      public int getCol() {
         return col;
      }

      public int getIdx() {
         return idx;
      }
   }
}
