package caslayout.ui;

import guilib.common.BaseDialog;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridLayout;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionListener;
import java.awt.geom.Rectangle2D;
import java.beans.Customizer;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import javax.swing.AbstractListModel;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JTextField;

import org.apache.log4j.Logger;

import caslayout.propedit.PropertyEditorPanel;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: CAGridLayoutCustomizer.java,v 1.17 2006/03/03 07:39:01 bozyurt
 *          Exp $
 */

public class CAGridLayoutCustomizer extends JPanel implements Customizer,
      ActionListener, PropertyChangeListener {
   private static final long serialVersionUID = 5020020244770565744L;
   protected CAGridLayout layoutMan;
   /** the container with which the layout manager is associated */
   protected CAContainer container;
   protected PreviewPanel previewPanel;
   protected JComboBox maxRowsCombo;
   protected JComboBox maxColsCombo;
   protected JLabel rowsLabel;
   protected JLabel colsLabel;
   protected JPanel controlPanel;
   protected boolean usePercents = false;
   protected PropertyEditorPanel curPEPanel;
   protected static Logger log = Logger.getLogger("CAGridLayoutCustomizer");
   protected CellConstraintEditorPane ceditorPane;
   protected JButton rowColumnsPercentButton;
   protected JComboBox lowBoundRowCombo;
   protected JComboBox upperBoundRowCombo;
   protected SimpleListModel lbrComboModel;
   protected SimpleListModel ubrComboModel;
   public final static int MAX_ROWS = 50;

   protected PropertyChangeSupport pcs = new PropertyChangeSupport(this);

   public CAGridLayoutCustomizer() {}

   /**
    * Initializes this bean customizer. Must be called only once before the
    * customizer is put into its parent container
    */
   public void init() {

      this.setLayout(new BorderLayout(3, 3));
      this.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));

      controlPanel = prepareControlPanel();
      JPanel lp = prepareLayoutPreviewPanel();
      lp.setPreferredSize(new Dimension(350, 200));
      lp.setMinimumSize(new Dimension(350, 200));

      this.add(lp);
      this.add(controlPanel, BorderLayout.SOUTH);
   }

   protected JPanel prepareLayoutPreviewPanel() {
      JPanel layoutPane = new JPanel(new BorderLayout(3, 3));
      previewPanel = new PreviewPanel(layoutMan, this);
      layoutPane.add(previewPanel);
      layoutPane.setBorder(BorderFactory.createLineBorder(Color.darkGray));
      return layoutPane;
   }

   protected JPanel prepareControlPanel() {
      JPanel cp = new JPanel(new BorderLayout(5, 5));
      JPanel p = new JPanel(new GridLayout(2, 2, 5, 5));

      Vector<Integer> rowIdxs = new Vector<Integer>();
      Vector<Integer> colIdxs = new Vector<Integer>();
      for (int i = 1; i <= MAX_ROWS; i++) {
         rowIdxs.add(new Integer(i));
      }
      for (int i = 1; i <= 30; i++) {
         colIdxs.add(new Integer(i));
      }

      // set the cell editor pane
      CellConstraint cc = layoutMan.getCellConstraint(0, 0);
      if (usePercents) {
         ceditorPane = new PercentConstraintEditorPane(cc);
      } else {
         ceditorPane = new CellSpanConstraintEditorPane(cc);
      }
      // for change notification
      ceditorPane.addPropertyChangeListener(this);
      cp.add(ceditorPane);

      maxRowsCombo = new JComboBox(rowIdxs);
      maxRowsCombo.setSelectedIndex(layoutMan.getRows() - 1);
      maxRowsCombo.addActionListener(this);
      maxColsCombo = new JComboBox(colIdxs);
      maxColsCombo.setSelectedIndex(layoutMan.getCols() - 1);
      maxColsCombo.addActionListener(this);
      p.add(new ComboField("Row:", maxRowsCombo));
      p.add(new ComboField("Col:", maxColsCombo));

      rowsLabel = new JLabel("1", JLabel.LEFT);
      colsLabel = new JLabel("1", JLabel.LEFT);

      p.add(new LabelField("Row:", rowsLabel));
      p.add(new LabelField("Column:", colsLabel));

      if (usePercents) {
         JPanel p1 = new JPanel();
         BoxLayout boxL = new BoxLayout(p1, BoxLayout.Y_AXIS);
         p1.setLayout(boxL);

         rowColumnsPercentButton = new JButton("Update Row");
         JPanel p2 = new JPanel();
         BoxLayout bl = new BoxLayout(p2, BoxLayout.X_AXIS);
         p2.setLayout(bl);

         rowColumnsPercentButton.addActionListener(this);

         lbrComboModel = new SimpleListModel(layoutMan.getRows());
         lowBoundRowCombo = new JComboBox(lbrComboModel);
         lowBoundRowCombo.setSelectedIndex(0);
         ubrComboModel = new SimpleListModel(layoutMan.getRows());
         upperBoundRowCombo = new JComboBox(ubrComboModel);
         upperBoundRowCombo.setSelectedIndex(0);

         p2.add(new JLabel("From "));
         p2.add(lowBoundRowCombo);
         p2.add(new JLabel(" to "));
         p2.add(upperBoundRowCombo);
         p2.add(rowColumnsPercentButton);
         p2.add(Box.createGlue());

         p1.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
         p2.setBorder(BorderFactory
               .createTitledBorder("Set Column percents for row(s)"));

         // p.setBorder( BorderFactory.createLineBorder(Color.GRAY) );
         p.setBorder(BorderFactory
               .createTitledBorder("Adjust row/column numbers"));

         p1.add(p);
         p1.add(Box.createVerticalStrut(3));
         p1.add(p2);

         cp.add(p1, BorderLayout.NORTH);

      } else {
         cp.add(p, BorderLayout.NORTH);
      }

      return cp;
   }

   protected void handleRowColumnsSetting(int rowNumber, int upperBound) {
      // int colCount = layoutMan.getEffectiveColumnCount(rowNumber);
      CellConstraint[] rowArr = layoutMan.getRowColumns(rowNumber);
      double[] colPercents = new double[rowArr.length];
      for (int i = 0; i < rowArr.length; i++) {
         PercentCellConstraint pcc = (PercentCellConstraint) rowArr[i];
         colPercents[i] = pcc.getColPercent();
      }

      ColumnsPercentSelectionDialog dlg = new ColumnsPercentSelectionDialog(
            null, "", colPercents);
      dlg.setLocationRelativeTo(this);

      int rc = dlg.showDialog();
      if (rc == BaseDialog.OK_PRESSED) {
         colPercents = dlg.getColPercents();
         if (upperBound > 0) {
            for (int i = rowNumber; i < upperBound; ++i) {
               rowArr = layoutMan.getRowColumns(i);
               for (int j = 0; j < colPercents.length; j++) {
                  ((PercentCellConstraint) rowArr[j])
                        .setColPercent(colPercents[j]);
               }
            }
         } else {
            for (int i = 0; i < colPercents.length; i++) {
               ((PercentCellConstraint) rowArr[i])
                     .setColPercent(colPercents[i]);
            }
         }
         // this.controlPanel.validate();
         repaint();
      }
      dlg.dispose();
   }

   /**
    *
    * @param bean
    *           the bean to edit
    */
   public void setObject(Object bean) {
      this.layoutMan = (CAGridLayout) bean;
      this.usePercents = (layoutMan.getCellConstraint(0, 0) instanceof PercentCellConstraint);
   }

   public void setContainer(CAContainer container) {
      this.container = container;
   }

   public void addPropertyChangeListener(PropertyChangeListener listener) {
      pcs.addPropertyChangeListener(listener);
   }

   public void removePropertyChangeListener(PropertyChangeListener listener) {
      pcs.removePropertyChangeListener(listener);
   }

   protected void removeRow(int idx) {
      List<Integer> offsetList = layoutMan.removeRow(idx);
      if (container != null) {
         container.removePermanently(offsetList);
      }
      // update the displayed max number of rows
      int selIdx = maxRowsCombo.getSelectedIndex();
      maxRowsCombo.setSelectedIndex(selIdx - 1);
      updateFields();
   }

   protected void insertRow(int afterRow) {
      List<Integer> offsetList = layoutMan.insertRow(afterRow);
      if (container != null) {
         container.insertRow(offsetList);
      }
      // update the displayed max number of rows
      int selIdx = maxRowsCombo.getSelectedIndex();
      maxRowsCombo.setSelectedIndex(selIdx + 1);
      updateFields();
   }

   protected void updateFields() {
      if (lbrComboModel.getSize() != layoutMan.getRows()) {
         lbrComboModel.update(layoutMan.getRows());
      }
      if (ubrComboModel.getSize() != layoutMan.getRows()) {
         ubrComboModel.update(layoutMan.getRows());
      }
      repaint();
   }

   public void actionPerformed(ActionEvent event) {
      log.debug("in actionPerformed " + event.getSource());
      Object es = event.getSource();
      if (es == maxRowsCombo || es == maxColsCombo) {
         int selRow = ((Integer) maxRowsCombo.getSelectedItem()).intValue();
         int selCol = ((Integer) maxColsCombo.getSelectedItem()).intValue();
         if (log.isDebugEnabled()) {
            log.debug("selRow=" + selRow + ", selCol=" + selCol);
         }
         try {
            layoutMan.setCols(selCol);
            layoutMan.setRows(selRow);
            if (lbrComboModel.getSize() != layoutMan.getRows()) {
               lbrComboModel.update(layoutMan.getRows());
            }
            if (ubrComboModel.getSize() != layoutMan.getRows()) {
               ubrComboModel.update(layoutMan.getRows());
            }
            repaint();
         } catch (Exception x) {
            x.printStackTrace();
         }
      } else if (es == rowColumnsPercentButton) {
         int selLowRow = ((Integer) lowBoundRowCombo.getSelectedItem())
               .intValue();
         int selUpperRow = ((Integer) upperBoundRowCombo.getSelectedItem())
               .intValue();
         if (selUpperRow > selLowRow) {
            handleRowColumnsSetting(selLowRow - 1, selUpperRow);
         } else {
            handleRowColumnsSetting(selLowRow - 1, -1);
         }
      }
   }

   public static class SimpleListModel extends AbstractListModel implements
         ComboBoxModel {
      private static final long serialVersionUID = 2133998438687282955L;
      java.util.List<Integer> items = new ArrayList<Integer>();
      Object selectedItem;

      public SimpleListModel(int maxItemNumber) {
         for (int i = 1; i <= maxItemNumber; i++) {
            items.add(new Integer(i));
         }
      }

      public void setSelectedItem(Object anItem) {
         selectedItem = anItem;
      }

      public Object getSelectedItem() {
         return selectedItem;
      }

      public void update(int newMaxItemNumber) {
         items.clear();
         for (int i = 1; i <= newMaxItemNumber; i++) {
            items.add(new Integer(i));
         }
         fireContentsChanged(this, 0, items.size());
      }

      public Object getElementAt(int index) {
         if (index < 0 || index > items.size())
            return null;
         return items.get(index);
      }

      public int getSize() {
         return items.size();
      }
   }

   public static class ComboField extends JPanel {
      private static final long serialVersionUID = -5662303406568285956L;
      JComboBox combo;

      public ComboField(String fieldTitle, JComboBox combo) {
         super();
         this.combo = combo;
         BoxLayout bl = new BoxLayout(this, BoxLayout.X_AXIS);
         setLayout(bl);
         add(new JLabel(fieldTitle));
         add(Box.createGlue());
         add(combo);
      }
   }

   public static class LabelField extends JPanel {
      private static final long serialVersionUID = -1746219729601004785L;
      JLabel label;

      public LabelField(String fieldTitle, JLabel label) {
         super();
         this.label = label;
         BoxLayout bl = new BoxLayout(this, BoxLayout.X_AXIS);
         setLayout(bl);
         add(Box.createGlue());
         add(new JLabel(fieldTitle));
         add(label);
         add(Box.createGlue());

      }
   }

   public static class PreviewPanel extends JPanel {
      private static final long serialVersionUID = -7748132509353006372L;
      Grid grid = new Grid(8, 8);
      CAPanel panel;

      public PreviewPanel(CAGridLayout gl, CAGridLayoutCustomizer customizer) {
         super(null);
         setSize(350, 200);

         panel = new CAPanel(gl, this.getInsets().left, this.getInsets().top,
               this.getWidth(), this.getHeight());
         PreviewPanelMouseHandler mh = new PreviewPanelMouseHandler(this,
               customizer);
         addMouseListener(mh);
         addMouseMotionListener(mh);
      }

      public void paintComponent(Graphics g) {
         Graphics2D g2 = (Graphics2D) g;

         g2.setColor(Color.white);
         Rectangle r = getBounds();
         g2.fillRect(r.x, r.y, r.width, r.height);

         panel.setX(r.x);
         panel.setY(r.y);
         panel.setWidth(r.width);
         panel.setHeight(r.height);
         panel.setBounds(new Rectangle2D.Double(r.x, r.y, r.width, r.height));
         if (log.isDebugEnabled()) {
            log.debug(">>>>*** " + panel.getBounds(g2));
         }

         // if ( grid != null) {
         // grid.draw(g2, r);
         // }

         g2.setColor(Color.black);
         if (grid != null) {
            Rectangle2D b = panel.getBounds(g2);
            if (b == null) {
               b = new Rectangle2D.Double(8, 8, r.width - 8, r.height - 8);
            }
            if (b != null) {
               panel.setBounds(b);
            }
         }
         panel.draw(g2);
      }

   }

   public void propertyChange(PropertyChangeEvent evt) {
      CellConstraint cc = ceditorPane.getConstraint();
      if (log.isDebugEnabled()) {
         log.debug("***** cc=" + cc);
      }
      log.info("***** cc=" + cc);
      layoutMan.setCellConstraint(cc);
      repaint();
   }

   abstract class CellConstraintEditorPane extends JPanel {
      CellConstraint cc;
      PropertyChangeSupport pcs = new PropertyChangeSupport(this);

      public CellConstraintEditorPane(CellConstraint cc) {
         this.cc = cc;
         setLayout(new GridLayout(1, 2, 5, 5));
      }

      public void addPropertyChangeListener(PropertyChangeListener pcl) {
         pcs.addPropertyChangeListener(pcl);
      }

      public void removePropertyChangeListener(PropertyChangeListener pcl) {
         pcs.removePropertyChangeListener(pcl);
      }

      public abstract CellConstraint getConstraint();

      public abstract void setCellConstraint(CellConstraint acc);

   }

   class CellSpanConstraintEditorPane extends CellConstraintEditorPane
         implements ActionListener {
      private static final long serialVersionUID = 525569824490223633L;
      JTextField colSpanField = new JTextField(2);
      JTextField rowSpanField = new JTextField(2);

      public CellSpanConstraintEditorPane(CellConstraint cc) {
         super(cc);
         init();
      }

      void init() {
         JPanel labelPanel = new JPanel(new GridLayout(2, 1));
         JPanel fieldPanel = new JPanel(new GridLayout(2, 1));
         setBorder(BorderFactory
               .createTitledBorder("Adjust row and/or column span for the selected cell"));
         labelPanel.add(new JLabel("Column Span:", JLabel.RIGHT));
         labelPanel.add(new JLabel("Row Span:", JLabel.RIGHT));

         colSpanField.addActionListener(this);
         rowSpanField.addActionListener(this);

         fieldPanel.add(colSpanField);
         fieldPanel.add(rowSpanField);

         if (cc != null) {
            MultipleCellSpanConstraint mcsc = (MultipleCellSpanConstraint) cc;
            colSpanField.setText(String.valueOf(mcsc.getColSpan()));
            rowSpanField.setText(String.valueOf(mcsc.getRowSpan()));
         }
         add(labelPanel);
         add(fieldPanel);
      }

      public void setCellConstraint(CellConstraint acc) {
         MultipleCellSpanConstraint macc = (MultipleCellSpanConstraint) acc;
         this.cc = new MultipleCellSpanConstraint(macc.getRowIdx(), macc
               .getColIdx(), macc.getColSpan(), macc.getRowSpan());

         MultipleCellSpanConstraint mcsc = (MultipleCellSpanConstraint) cc;
         colSpanField.setText(String.valueOf(mcsc.getColSpan()));
         rowSpanField.setText(String.valueOf(mcsc.getRowSpan()));
      }

      public CellConstraint getConstraint() {
         MultipleCellSpanConstraint mcsc = (MultipleCellSpanConstraint) cc;
         int colSpan = getFieldValue(colSpanField);
         if (colSpan > 0)
            mcsc.setColSpan(colSpan);
         int rowSpan = getFieldValue(rowSpanField);
         if (rowSpan > 0)
            mcsc.setRowSpan(rowSpan);

         return mcsc;
      }

      int getFieldValue(JTextField field) {
         int value = -1;
         try {
            value = Integer.parseInt(field.getText());
         } catch (NumberFormatException nfe) {
            value = -1;
         }
         return value;
      }

      public void actionPerformed(ActionEvent e) {
         MultipleCellSpanConstraint mcsc = (MultipleCellSpanConstraint) cc;
         if (e.getSource() == colSpanField) {
            int colSpan = getFieldValue(colSpanField);
            if (colSpan > 0) {
               int oldValue = mcsc.getColSpan();
               mcsc.setColSpan(colSpan);
               pcs.firePropertyChange("colSpan", oldValue, colSpan);
            }
         } else if (e.getSource() == rowSpanField) {
            int rowSpan = getFieldValue(rowSpanField);
            if (rowSpan > 0) {
               int oldValue = mcsc.getRowSpan();
               mcsc.setRowSpan(rowSpan);
               pcs.firePropertyChange("rowSpan", oldValue, rowSpan);
            }
         }
      }
   }

   class PercentConstraintEditorPane extends CellConstraintEditorPane implements
         ActionListener {
      private static final long serialVersionUID = -4346470671045195793L;
      JTextField colPercentField = new JTextField(2);
      JTextField rowPercentField = new JTextField(2);

      public PercentConstraintEditorPane(CellConstraint cc) {
         super(cc);
         init();
      }

      void init() {
         JPanel labelPanel = new JPanel(new GridLayout(2, 1));
         JPanel fieldPanel = new JPanel(new GridLayout(2, 1));
         setBorder(BorderFactory
               .createTitledBorder("Adjust row and/or column percentage for the selected cell"));
         labelPanel.add(new JLabel("Column percentage (%):", JLabel.RIGHT));
         labelPanel.add(new JLabel("Row percentage (%):", JLabel.RIGHT));

         colPercentField.addActionListener(this);
         rowPercentField.addActionListener(this);

         fieldPanel.add(colPercentField);
         fieldPanel.add(rowPercentField);

         if (cc != null) {
            PercentCellConstraint pcc = (PercentCellConstraint) cc;
            colPercentField.setText(String.valueOf(pcc.getColPercent()));
            rowPercentField.setText(String.valueOf(pcc.getRowPercent()));
         }

         add(labelPanel);
         add(fieldPanel);
      }

      public void setCellConstraint(CellConstraint acc) {
         PercentCellConstraint apcc = (PercentCellConstraint) acc;
         this.cc = new PercentCellConstraint(apcc.getRowIdx(),
               apcc.getColIdx(), apcc.getColPercent(), apcc.getRowPercent());

         PercentCellConstraint pcc = (PercentCellConstraint) cc;
         colPercentField.setText(String.valueOf(pcc.getColPercent()));
         rowPercentField.setText(String.valueOf(pcc.getRowPercent()));
      }

      public CellConstraint getConstraint() {
         PercentCellConstraint pcc = (PercentCellConstraint) cc;
         double colPercent = getFieldValue(colPercentField);
         if (colPercent >= 0 && colPercent <= 100)
            pcc.setColPercent(colPercent);
         double rowPercent = getFieldValue(rowPercentField);
         if (rowPercent > 0 && rowPercent < 100)
            pcc.setRowPercent(rowPercent);

         return pcc;
      }

      double getFieldValue(JTextField field) {
         double value = -1;
         try {
            value = Double.parseDouble(field.getText());
            if (value < 0 || value > 100)
               return -1;
         } catch (NumberFormatException nfe) {
            value = -1;
         }
         return value;
      }

      public void actionPerformed(ActionEvent e) {
         PercentCellConstraint pcc = (PercentCellConstraint) cc;
         if (e.getSource() == colPercentField) {
            double colPercent = getFieldValue(colPercentField);
            if (colPercent >= 0 && colPercent <= 100) {
               double oldValue = pcc.getColPercent();
               pcc.setColPercent(colPercent);
               pcs.firePropertyChange("colSpan", new Double(oldValue),
                     new Double(colPercent));
            }
         } else if (e.getSource() == rowPercentField) {
            double rowPercent = getFieldValue(rowPercentField);
            if (rowPercent > 0 && rowPercent < 100) {
               double oldValue = pcc.getRowPercent();
               pcc.setRowPercent(rowPercent);
               pcs.firePropertyChange("rowSpan", new Double(oldValue),
                     new Double(rowPercent));
            }
         }
      }
   }

   public static class PopupMenuHelper implements ActionListener {
      CAGridLayoutCustomizer customizer;
      int selectedRow = -1;

      public PopupMenuHelper(CAGridLayoutCustomizer customizer) {
         this.customizer = customizer;
      }

      public void actionPerformed(ActionEvent e) {
         if (!(e.getSource() instanceof JMenuItem))
            return;
         JMenuItem item = (JMenuItem) e.getSource();
         if (item.getText().equals("Delete the selected row")) {
            handleDeleteRow();
         } else if (item.getText().equals("Add a row after the selected row")) {
            handleInsertRow();
         }
      }

      protected void handleDeleteRow() {
         if (selectedRow == -1) {
            CALMHelper.showError(customizer,
                  "You need to select a row by clicking inside a row first!");
            return;
         }

         int numRows = customizer.layoutMan.getRows();
         if (numRows == 1) {
            CALMHelper.showError(customizer,
                  "A container cannot shrink below a single row!");
            return;
         }

         int rc = JOptionPane.showConfirmDialog(customizer,
               "Do you want to delete " + (selectedRow + 1) + ". row?", "",
               JOptionPane.YES_NO_OPTION);
         if (rc == JOptionPane.YES_OPTION) {
            customizer.removeRow(selectedRow);
            customizer.previewPanel.repaint();
         }
      }

      protected void handleInsertRow() {
         if (selectedRow == -1) {
            CALMHelper.showError(customizer,
                  "You need to select a row by clicking inside a row first!");
            return;
         }
         int numRows = ((Integer) customizer.maxRowsCombo.getSelectedItem())
               .intValue();
         if (numRows == CAGridLayoutCustomizer.MAX_ROWS) {
            CALMHelper.showError(customizer,
                  "A container cannot grow beyond the max number of rows of "
                        + CAGridLayoutCustomizer.MAX_ROWS);
            return;
         }

         int rc = JOptionPane.showConfirmDialog(customizer,
               "Do you want to add a new row after " + (selectedRow + 1)
                     + ". row?", "", JOptionPane.YES_NO_OPTION);
         if (rc == JOptionPane.YES_OPTION) {
            customizer.insertRow(selectedRow);
            customizer.previewPanel.repaint();
         }
      }

      void setSelectedRow(int row) {
         this.selectedRow = row;
      }
   }

   public static class PreviewPanelMouseHandler extends MouseAdapter implements
         MouseMotionListener {
      PreviewPanel prevPanel;
      CAGridLayoutCustomizer customizer;
      JPopupMenu popupMenu;
      PopupMenuHelper popupHelper;
      MouseState dragState = null;

      public PreviewPanelMouseHandler(PreviewPanel prevPanel,
            CAGridLayoutCustomizer customizer) {
         this.prevPanel = prevPanel;
         this.customizer = customizer;
         preparePopupMenu();
      }

      public void preparePopupMenu() {
         popupHelper = new PopupMenuHelper(customizer);
         popupMenu = new JPopupMenu("");
         JMenuItem menuItem = new JMenuItem("Add a row after the selected row");
         menuItem.addActionListener(popupHelper);
         popupMenu.add(menuItem);

         menuItem = new JMenuItem("Delete the selected row");
         menuItem.addActionListener(popupHelper);
         popupMenu.add(menuItem);
      }

      protected boolean showPopup(MouseEvent me) {
         if (me.isPopupTrigger()) {
            popupMenu.show(me.getComponent(), me.getX(), me.getY());
            return true;
         }
         return false;
      }

      public void mouseClicked(MouseEvent e) {
         GridLocation gridLoc = (GridLocation) customizer.layoutMan.hitTest(e
               .getX(), e.getY(), prevPanel.panel, (Graphics2D) prevPanel
               .getGraphics());
         if (gridLoc != null) {
            log.info("Mouse Clicked: " + gridLoc.toString());

            int selRow = gridLoc.getRow();
            int selCol = gridLoc.getColumn();
            popupHelper.setSelectedRow(selRow);

            CellConstraint cc = customizer.layoutMan.getCellConstraint(selRow,
                  selCol);
            try {

               customizer.ceditorPane.setCellConstraint(cc);

               customizer.rowsLabel.setText(String.valueOf(selRow + 1));
               customizer.colsLabel.setText(String.valueOf(selCol + 1));
               customizer.controlPanel.validate();

            } catch (Exception x) {
               x.printStackTrace();
            }
         }
      }

      protected void handleGridResize(int mX, int mY) {
         GridLocation gridLoc = customizer.layoutMan.layoutGridHitTest(mX, mY,
               prevPanel.panel, (Graphics2D) prevPanel.getGraphics());
         System.out.println("gridLoc=" + gridLoc);
         if (gridLoc == null) {
            return;
         }
         int selRow = gridLoc.getRow();
         int selCol = gridLoc.getColumn();

         CellConstraint cc = customizer.layoutMan.getCellConstraint(selRow,
               selCol);
         if (cc instanceof PercentCellConstraint) {
            dragState = new MouseState(mX, mY, gridLoc);

            // customizer.layoutMan.hiliteGridWalls(prevPanel.panel,
            // (Graphics2D) prevPanel.getGraphics(), gridLoc);

         }

      }

      public void mousePressed(MouseEvent e) {
         boolean show = showPopup(e);
         if (!show) {
            boolean controlPressed = (e.getModifiersEx() & MouseEvent.CTRL_DOWN_MASK) == MouseEvent.CTRL_DOWN_MASK;

            if (controlPressed) {
               handleGridResize(e.getX(), e.getY());
            }
         }

      }

      public void mouseReleased(MouseEvent e) {
         showPopup(e);
         dragState = null;
      }

      public void mouseDragged(MouseEvent e) {
         if (dragState == null) {
            return;
         }
         int deltaX = e.getX() - dragState.startX;
         int deltaY = e.getY() - dragState.startY;
         // System.out.println("deltaX=" + deltaX + " deltaY=" + deltaY);
         if (deltaX != 0 || deltaY != 0) {
            boolean changed = customizer.layoutMan.calcNewCellConstraints(
                  dragState.gridLoc, prevPanel.panel, deltaX, deltaY);
            if (changed) {
               // System.out.println("changed.");
               customizer.repaint();
            }
            dragState.startX = e.getX();
            dragState.startY = e.getY();
         }

      }

      public void mouseMoved(MouseEvent e) {
      // no-op
      }

   }// PreviewPanelMouseHandler

   public static class MouseState {
      int startX;
      int startY;
      GridLocation gridLoc;

      public MouseState(int startx, int starty, GridLocation gl) {
         startX = startx;
         startY = starty;
         gridLoc = gl;
      }

   }
}