package caslayout.ui;

import guilib.AppFrame;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.io.BufferedWriter;
import java.io.CharArrayReader;
import java.io.CharArrayWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URL;
import java.util.Enumeration;
import java.util.Vector;

import javax.help.CSH;
import javax.help.HelpBroker;
import javax.help.HelpSet;
import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComboBox;
import javax.swing.JEditorPane;
import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTabbedPane;
import javax.swing.JToolBar;
import javax.swing.UIManager;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.plaf.FontUIResource;

import org.apache.log4j.Logger;

import caslayout.CommandLineInterface;
import caslayout.Constants;
import caslayout.codegen.HTMLGenerator;
import caslayout.ui.actions.AddButtonAdapter;
import caslayout.ui.actions.AddCheckBoxAdapter;
import caslayout.ui.actions.AddContainerAdapter;
import caslayout.ui.actions.AddDropdownAdapter;
import caslayout.ui.actions.AddRadioButtonAdapter;
import caslayout.ui.actions.AddTemplateAdapter;
import caslayout.ui.actions.AddTextAdapter;
import caslayout.ui.actions.AddTextAreaAdapter;
import caslayout.ui.actions.AddTextFieldAdapter;
import caslayout.ui.actions.AsDataUploadAdapter;
import caslayout.ui.actions.BatchSaveAssessmentAdapter;
import caslayout.ui.actions.ImportAdapter;
import caslayout.ui.actions.NewDocumentAdapter;
import caslayout.ui.actions.SaveAssessmentAdapter;
import caslayout.ui.actions.SelectionToolAdapter;
import caslayout.ui.actions.ShowAssociationTipsAdapter;
import caslayout.ui.actions.ShowAssociationsAdapter;
import caslayout.ui.actions.ShowQuestionsAdapter;
import caslayout.ui.actions.StrutsCodeGenAdapter;
import caslayout.ui.assoc.AssessmentMaintenancePanel;
import caslayout.ui.model.AssessmentAssociation;
import caslayout.ui.model.AssessmentInfo;
import caslayout.ui.model.AssessmentMaintenanceHelper;
import caslayout.ui.model.AssociationHelper;
import caslayout.util.GUIUtils;
import caslayout.util.GenUtils;
import caslayout.util.UIDManager;
import caslayout.util.XMLUtils;

import com.digitprop.tonic.TonicLookAndFeel;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: ClinicalAssessmentLayoutManager.java,v 1.64 2007/04/16 17:04:25
 *          bozyurt Exp $
 */

public class ClinicalAssessmentLayoutManager extends AppFrame implements
      ChangeListener {
   private static final long serialVersionUID = 6433670830140885101L;
   protected ScoreLayoutPanel slPanel;
   protected JSplitPane splitPane;

   protected JToolBar toolBar;
   protected JFileChooser openFileChooser;
   protected JFileChooser saveFileChooser;

   protected CALMConfig config;
   protected Vector<String> templateList = new Vector<String>();

   /** the full path of the currently edited layout file */
   protected String curLayoutFile;
   protected static Document curDoc;
   protected PageEditComponent pageEditComp;
   protected HelpDialog helpDlg;
   protected JTabbedPane tabPane;
   protected JPanel previewPanel;
   protected JEditorPane htmlPane;
   protected AssessmentInfo assessmentInfo;
   protected AssessmentMaintenancePanel amp;
   protected JPanel treePanel;
   protected boolean showAssocs = false;
   protected boolean showQuestions = false;
   protected CALMHelper.SavedState savedState;
   protected HelpBroker helpBroker;
   protected static Logger log = Logger
         .getLogger("ClinicalAssessmentLayoutManager");
   public final static String VERSION = "2.0  Release 1 (10/11/2007)";

   public ClinicalAssessmentLayoutManager(int width, int height)
         throws Exception {
      super(width, height);
      setTitle("Clinical Assessment Layout Manager");
      Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
      int sWidth = (int) screenSize.getWidth();
      int sHeight = (int) screenSize.getHeight();
      setBounds(sWidth / 16, sHeight / 16, sWidth * 7 / 8, sHeight * 7 / 8);
      setWidth(sWidth * 7 / 8);
      setHeight(sHeight * 7 / 8);

      init();
   }

   public static Document getCurDoc() {
      return ClinicalAssessmentLayoutManager.curDoc;
   }

   public static void setCurDoc(Document curDoc) {
      ClinicalAssessmentLayoutManager.curDoc = curDoc;
   }

   public ScoreLayoutPanel getSlPanel() {
      return slPanel;
   }

   public Vector<String> getTemplateList() {
      return templateList;
   }

   public CALMConfig getConfig() {
      return config;
   }

   public CALMHelper.SavedState getSavedState() {
      return savedState;
   }

   public PageEditComponent getPageEditComp() {
      return pageEditComp;
   }

   public boolean isShowAssocs() {
      return showAssocs;
   }

   public boolean isShowQuestions() {
      return showQuestions;
   }

   public void setShowAssocs(boolean showAssocs) {
      this.showAssocs = showAssocs;
   }

   public void setShowQuestions(boolean showQuestions) {
      this.showQuestions = showQuestions;
   }

   public AssessmentInfo getAssessmentInfo() {
      return assessmentInfo;
   }

   public void setCurLayoutFile(String curLayoutFile) {
      this.curLayoutFile = curLayoutFile;
   }

   public static Logger getLog() {
      return log;
   }

   public void init() throws Exception {
      config = new CALMConfig("caslayout.properties");

      Class.forName(Constants.ORACLE_JDBC_DRIVER);
      Class.forName(Constants.POSTGRES_JDBC_DRIVER).newInstance();

      JPanel panel = new JPanel(new BorderLayout(1, 1));

      slPanel = new ScoreLayoutPanel("Score Layout", true, this, config);
      try {
         // create an empty single page document
         ClinicalAssessmentLayoutManager.curDoc = new Document("empty", "");
         curDoc.addPage(new Page(1));

         CAPanel caPanel = (CAPanel) slPanel.getCAContainerAt(0);

         curDoc.getFirstPage().setViewPanel(caPanel);

         slPanel.setMinimumSize(new Dimension(500, 300));
         slPanel.setPreferredSize(new Dimension(600, 500));

         slPanel.setBorder(BorderFactory.createLineBorder(Color.magenta));

         // add scrolling
         JScrollPane scrollPane = new JScrollPane(slPanel);

         tabPane = new JTabbedPane();

         tabPane.add(scrollPane, "Layout");
         previewPanel = preparePreviewPanel();
         tabPane.add(previewPanel, "Preview");

         // register a change listener for the tabbed pane
         tabPane.addChangeListener(new ChangeListener() {
            public void stateChanged(ChangeEvent e) {
               JTabbedPane pane = (JTabbedPane) e.getSource();
               int sel = pane.getSelectedIndex();
               if (sel > 0) {
                  try {
                     handlePreview();
                  } catch (Exception x) {
                     x.printStackTrace();
                  }
               }
            }
         });

         panel.add(tabPane, BorderLayout.CENTER);
         toolBar = new JToolBar("");
         toolBar.setRollover(true);
         toolBar.setFloatable(false);

         JMenu toolsMenu = createToolsMenu();
         panel.add(toolBar, BorderLayout.PAGE_START);

         openFileChooser = CALMHelper.createFileChooser(false, System
               .getProperty("user.home"));
         saveFileChooser = CALMHelper.createFileChooser(true, System
               .getProperty("user.home"));
         // prepare saved templates
         String templateDir = config.getTemplateDir();
         String[] templateFiles = GenUtils.getFiles(templateDir, ".xml");
         if (templateFiles != null) {
            for (int i = 0; i < templateFiles.length; i++) {
               templateList.add(templateFiles[i].substring(0, templateFiles[i]
                     .lastIndexOf('.')));
            }
         }
         // load the saved state
         this.savedState = CALMHelper.loadState();

         this.treePanel = prepareAssessmentTreePane();

         splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, treePanel,
               tabPane);
         splitPane.setOneTouchExpandable(true);
         panel.add(splitPane, BorderLayout.CENTER);
         splitPane.setDividerLocation(180);

         // register property change listeners for the form view and
         // assessment view
         slPanel.addPropertyChangeListener(amp);
         amp.addPropertyChangeListener(slPanel);

         AssessmentMaintenanceHelper.initialize(config);

         super.addView(panel);
         super.addMenu(toolsMenu);
         super.addMenu(createOptionsMenu());
         prepareFileMenu();

         if (!AssessmentMaintenanceHelper.isConnectToDB()) {
            disableDBRequiringActions(toolsMenu);
         }

         // prepare help system
         HelpSet hs = null;
         URL hsURL = HelpSet.findHelpSet(ClinicalAssessmentLayoutManager.class
               .getClassLoader(), "calm.hs");
         hs = new HelpSet(null, hsURL);
         helpBroker = hs.createHelpBroker();

      } catch (Exception x) {
         log.error("init", x);
         x.printStackTrace();
      }
   }

   protected void disableDBRequiringActions(JMenu toolsMenu) {
      JMenuItem item = GUIUtils.findMenuItem(Constants.SAVE_ASSESSMENT,
            toolsMenu);
      item.setEnabled(false);
      item = GUIUtils.findMenuItem(Constants.STRUTS_CODEGEN, toolsMenu);
      item.setEnabled(false);
      item = GUIUtils.findMenuItem(Constants.ASSESSMENT_DATA_UPLOAD, toolsMenu);
      item.setEnabled(false);
      item = GUIUtils.findMenuItem(Constants.BATCH_SAVE_ASSESSMENT, toolsMenu);
      item.setEnabled(false);
   }

   protected JPanel prepareAssessmentTreePane() throws Exception {
      // create an AssessmentInfo object either populated from an existing
      // assessment or for a new assessment (name + mandatory fields
      assessmentInfo = AssessmentMaintenanceHelper.createEmptyAssessment();

      JPanel mainPanel = new JPanel(new BorderLayout(3, 3));
      this.amp = new AssessmentMaintenancePanel(assessmentInfo, null, this);
      JScrollPane scrollPane = new JScrollPane(amp);
      mainPanel.add(scrollPane);

      // associate the assessment with the document
      AssociationHelper.getInstance().associateAssessment("assoc-assessment",
            curDoc, this.assessmentInfo);
      // add AssessmentMaintenancePanel as observer (view) of
      // AssociationHelper
      AssociationHelper.getInstance().addObserver(this.amp);

      return mainPanel;
   }

   public void updateAssessmentTreePane(AssessmentInfo newAsInfo)
         throws Exception {
      this.assessmentInfo = newAsInfo;
      this.amp.populateAssessmentTree(this.assessmentInfo);
      // associate the assessment with the document
      AssociationHelper.getInstance().associateAssessment("assoc-assessment",
            curDoc, this.assessmentInfo);

      amp.setTitle(this.assessmentInfo.getName());
      // add AssessmentMaintenancePanel as observer (view) of
      // AssociationHelper
      AssociationHelper.getInstance().addObserver(this.amp);
   }

   protected JPanel preparePreviewPanel() throws IOException {

      this.htmlPane = new JEditorPane();
      htmlPane.setContentType("text/html");
      htmlPane.setEditable(false);
      JScrollPane htmlView = new JScrollPane(htmlPane);
      htmlView.setPreferredSize(new Dimension(750, 550));

      JPanel p = new JPanel(new BorderLayout(1, 1));
      p.add(htmlView);
      return p;
   }

   protected void handlePreview() throws IOException {
      CAPanel caPanel = (CAPanel) slPanel.getCAContainerAt(0);
      HTMLGenerator gen = new HTMLGenerator(caPanel);
      CharArrayWriter out = new CharArrayWriter(3000);
      gen.generate(out);
      CharArrayReader in = new CharArrayReader(out.toCharArray());

      if (log.isDebugEnabled()) {
         persistPreview(out);
         log.debug(out.toString());
      }

      out = null;
      htmlPane.read(in, null);
   }

   private void persistPreview(CharArrayWriter writer) {
      BufferedWriter out = null;
      try {
         out = new BufferedWriter(new FileWriter(config.getTemplateDir()
               + File.separator + "preview_dump.html"));
         out.write(writer.toCharArray());
      } catch (IOException iox) {
         iox.printStackTrace();
      } finally {
         if (out != null) {
            try {
               out.close();
            } catch (Exception x) { /* ignore */
            }
         }
      }
   }

   protected void prepareFileMenu() {
      Action action = new NewDocumentAdapter(this, "New", null,
            "New assessment", new Integer(KeyEvent.VK_N));
      JMenuItem item = new JMenuItem(action);
      super.fileMenu.add(item, 0);

      action = new ImportAdapter(this, "Import", null,
            "Import Clinical Assessment from other formats", new Integer(
                  KeyEvent.VK_I));
      item = new JMenuItem(action);
      super.fileMenu.add(item, 2);

   }

   protected void createToolsMenuItem(Action action, JMenu menu) {
      JMenuItem item = new JMenuItem(action);
      menu.add(item);
      item.setIcon(null);
      toolBar.add(CALMHelper.makeToolbarButton(action));
   }

   protected void createToolsCheckBoxMenuItem(Action action, JMenu menu) {
      JCheckBoxMenuItem item = new JCheckBoxMenuItem(action);
      menu.add(item);
      item.setIcon(null);
      toolBar.add(CALMHelper.makeToolbarButton(action));
   }

   protected JMenu createOptionsMenu() {
      JMenu optionsMenu = new JMenu("Options");
      optionsMenu.setMnemonic(KeyEvent.VK_O);

      Action action = new ShowAssociationsAdapter(this, Constants.SHOW_ASSOCS,
            null, Constants.SHOW_ASSOCS_DESC, new Integer(KeyEvent.VK_A));
      JMenuItem item = new JCheckBoxMenuItem(action);
      optionsMenu.add(item);

      action = new ShowAssociationTipsAdapter(this, Constants.SHOW_ASSOC_TIPS,
            null, Constants.SHOW_ASSOC_TIPS_DESC, new Integer(KeyEvent.VK_T));
      item = new JCheckBoxMenuItem(action);
      ((JCheckBoxMenuItem) item).setState(true);
      optionsMenu.add(item);

      action = new ShowQuestionsAdapter(this, Constants.SHOW_QUESTION, null,
            Constants.SHOW_QUESTION_DESC, new Integer(KeyEvent.VK_Q));
      item = new JCheckBoxMenuItem(action);
      optionsMenu.add(item);

      return optionsMenu;
   }

   protected JMenu createToolsMenu() {
      JMenu toolsMenu = new JMenu("Tools");

      toolsMenu.setMnemonic(KeyEvent.VK_T);
      ImageIcon icon = CALMHelper.makeButtonIcon("select.gif");
      Action action = new SelectionToolAdapter(this,
            Constants.SELECT_COMPONENTS, icon,
            Constants.SELECT_COMPONENTS_DESC, new Integer(KeyEvent.VK_S));
      JMenuItem item = new JMenuItem(action);
      toolsMenu.add(item);
      item.setIcon(null);
      toolBar.add(CALMHelper.makeToolbarButton(action));

      icon = CALMHelper.makeButtonIcon("field_layout.jpg");
      action = new AddContainerAdapter(this, Constants.ADD_CONTAINER, icon,
            Constants.ADD_CONTAINER_DESC, new Integer(KeyEvent.VK_C));
      createToolsMenuItem(action, toolsMenu);

      icon = CALMHelper.makeButtonIcon("text.gif");
      action = new AddTextAdapter(this, Constants.ADD_TEXT, icon,
            Constants.ADD_TEXT_DESC, new Integer(KeyEvent.VK_E));
      createToolsMenuItem(action, toolsMenu);

      icon = CALMHelper.makeButtonIcon("radio_button.jpg");
      action = new AddRadioButtonAdapter(this, Constants.ADD_RADIO, icon,
            Constants.ADD_RADIO_DESC, new Integer(KeyEvent.VK_R));
      createToolsMenuItem(action, toolsMenu);

      icon = CALMHelper.makeButtonIcon("checkbox_icon2.gif");
      action = new AddCheckBoxAdapter(this, Constants.ADD_CHECKBOX, icon,
            Constants.ADD_CHECKBOX, new Integer(KeyEvent.VK_X));
      createToolsMenuItem(action, toolsMenu);

      icon = CALMHelper.makeButtonIcon("edit_field.jpg");
      action = new AddTextFieldAdapter(this, Constants.ADD_TEXTFIELD, icon,
            Constants.ADD_TEXTFIELD_DESC, new Integer(KeyEvent.VK_T));
      createToolsMenuItem(action, toolsMenu);

      icon = CALMHelper.makeButtonIcon("textarea_icon.gif");
      action = new AddTextAreaAdapter(this, Constants.ADD_TEXTAREA, icon,
            Constants.ADD_TEXTAREA_DESC, new Integer(KeyEvent.VK_A));
      createToolsMenuItem(action, toolsMenu);

      icon = CALMHelper.makeButtonIcon("dropdown_icon.jpg");
      action = new AddDropdownAdapter(this, Constants.ADD_DROPDOWN, icon,
            Constants.ADD_DROPDOWN_DESC, new Integer(KeyEvent.VK_D));
      createToolsMenuItem(action, toolsMenu);

      icon = CALMHelper.makeButtonIcon("add_button.jpg");
      action = new AddButtonAdapter(this, Constants.ADD_BUTTON, icon,
            Constants.ADD_BUTTON_DESC, new Integer(KeyEvent.VK_B));
      createToolsMenuItem(action, toolsMenu);

      icon = CALMHelper.makeButtonIcon("add_template.jpg");
      action = new AddTemplateAdapter(this, Constants.ADD_TEMPLATE, icon,
            Constants.ADD_TEMPLATE, new Integer(KeyEvent.VK_Z));
      createToolsMenuItem(action, toolsMenu);

      this.pageEditComp = new PageEditComponent(ClinicalAssessmentLayoutManager
            .getCurDoc().getNumPages());
      this.pageEditComp.addChangeListener(this);
      toolBar.add(this.pageEditComp);

      action = new StrutsCodeGenAdapter(this, Constants.STRUTS_CODEGEN, null,
            Constants.STRUTS_CODEGEN_DESC, new Integer(KeyEvent.VK_G));
      item = new JMenuItem(action);
      toolsMenu.add(item);

      action = new SaveAssessmentAdapter(this, Constants.SAVE_ASSESSMENT, null,
            Constants.SAVE_ASSESSMENT_DESC, new Integer(KeyEvent.VK_D));
      item = new JMenuItem(action);
      toolsMenu.add(item);

      action = new BatchSaveAssessmentAdapter(this,
            Constants.BATCH_SAVE_ASSESSMENT, null,
            Constants.BATCH_SAVE_ASSESSMENT_DESC, new Integer(KeyEvent.VK_X));
      item = new JMenuItem(action);
      toolsMenu.add(item);

      action = new AsDataUploadAdapter(this, Constants.ASSESSMENT_DATA_UPLOAD,
            null, Constants.ASSESSMENT_DATA_UPLOAD_DESC, new Integer(
                  KeyEvent.VK_U));
      item = new JMenuItem(action);
      toolsMenu.add(item);

      return toolsMenu;
   }

   protected JMenu createHelpMenu() {
      helpMenu = super.createHelpMenu();

      helpMenu.setMnemonic(KeyEvent.VK_H);
      JMenuItem item = new JMenuItem("Help Contents", KeyEvent.VK_C);
      helpMenu.add(item);

      // item.addActionListener(new HelpAdapter());
      item.addActionListener(new CSH.DisplayHelpFromSource(helpBroker));

      return helpMenu;
   }

   public void addTemplate(String templateFilepath) {
      String templateDir = config.getTemplateDir();
      String tempScreenName = GUIUtils.prepareTemplateScreenName(
            templateFilepath, templateDir);

      if (tempScreenName != null) {
         if (!templateList.contains(tempScreenName)) {
            templateList.add(tempScreenName);
         }
      }
   }

   protected JPanel prepareTemplateSelector(final String templateDir) {
      JPanel p = new JPanel();
      p.setBorder(BorderFactory.createLineBorder(Color.BLUE));
      p.add(Box.createHorizontalGlue());
      final JComboBox box = new JComboBox(templateList);
      p.add(box);
      JButton button = new JButton("Add");
      p.add(button);
      button.addActionListener(new ActionListener() {
         public void actionPerformed(ActionEvent ae) {
            String tempScreenName = (String) box.getSelectedItem();
            String tempFile = GUIUtils.prepareTemplateFilename(tempScreenName,
                  templateDir);
            try {
               CAPanel template = XMLUtils.loadPanel(tempFile, slPanel);
               slPanel.setSelectedComponent(template);
            } catch (Exception x) {
               CALMHelper.showError(slPanel, x.getMessage(),
                     "Error while loading template");
               x.printStackTrace();
               log.error("prepareTemplateSelector", x);
            }
         }
      });
      return p;
   }

   public void onSave() {
      if (this.curLayoutFile != null) {
         try {
            String filename = this.curLayoutFile;
            if (this.curLayoutFile.endsWith(".calm")) {
               // .calm files cannot saved, one a .calm
               // file is loaded, the changes to it can only be
               // saved as full CALM XML file.
               filename = filename.replaceFirst("\\.calm$", ".xml");
            }
            Document.saveDocument(filename,
                  ClinicalAssessmentLayoutManager.curDoc);
            getStatusBar().setText("The layout is saved to " + filename);
         } catch (IOException iox) {
            CALMHelper.showError(this, iox.getMessage(),
                  "Error while saving the layout");

            log.error("", iox);
            iox.printStackTrace();
         }
      } else {
         onSaveAs();
      }
   }

   public void onSaveAs() {
      if (this.curLayoutFile != null) {
         String filename = this.curLayoutFile;
         if (this.curLayoutFile.endsWith(".calm")) {
            filename = filename.replaceFirst("\\.calm$", ".xml");
         }
         saveFileChooser.setSelectedFile(new File(filename));
      } else {
         if (savedState.getLastDirectory() != null) {
            File f = new File(savedState.getLastDirectory());
            if (f.isDirectory()) {
               saveFileChooser.setCurrentDirectory(f);
            }
         }
      }
      int rc = saveFileChooser.showSaveDialog(this);
      if (rc == JFileChooser.APPROVE_OPTION) {
         try {
            String newFilename = saveFileChooser.getSelectedFile()
                  .getAbsolutePath();
            Document.saveDocument(newFilename,
                  ClinicalAssessmentLayoutManager.curDoc);
            getStatusBar().setText("The layout is saved as " + newFilename);
            this.curLayoutFile = newFilename;

         } catch (IOException iox) {
            CALMHelper.showError(this, iox.getMessage(),
                  "Error while saving the layout");
            log.error("", iox);
            iox.printStackTrace();
         }
      }
   }

   public void onOpen() {
      if (savedState.getLastDirectory() != null) {
         File f = new File(savedState.getLastDirectory());
         if (f.isDirectory()) {
            openFileChooser.setCurrentDirectory(f);
         }
      }
      int rc = openFileChooser.showOpenDialog(this);
      if (rc == JFileChooser.APPROVE_OPTION) {
         BufferedWriter bout = null;
         try {
            File sf = openFileChooser.getSelectedFile();
            Document doc = null;
            if (sf.getName().endsWith(".calm")) {
               doc = CALMHelper.buildDocument(sf.getAbsolutePath(), slPanel,
                     config);
            } else {
               doc = Document.loadDocument(sf.getAbsolutePath(), slPanel);
            }

            this.savedState.setLastDirectory(openFileChooser
                  .getCurrentDirectory().getAbsolutePath());

            if (doc != null) {
               String layoutFile = openFileChooser.getSelectedFile()
                     .getAbsolutePath();

               handleOpenDocument(doc, layoutFile);
            }
         } catch (Exception x) {
            x.printStackTrace();
            CALMHelper.showError(slPanel, x.getMessage(),
                  "Error while loading the layout");

         } finally {
            GenUtils.close(bout);
         }
      }
   }

   public void handleImportCALMCADoc(String calmCAFile) throws Exception {
      Document doc = CALMHelper.buildDocument(calmCAFile, slPanel, config);
      handleOpenDocument(doc, calmCAFile);
   }

   protected void handleOpenDocument(Document doc, String layoutFile)
         throws Exception {
      AssociationHelper assocHelper = AssociationHelper.getInstance();

      UIDManager.getInstance().reset(doc);
      slPanel.setCAContainerAt(doc.getFirstPage().getViewPanel(), 0);
      this.curLayoutFile = layoutFile;
      ClinicalAssessmentLayoutManager.curDoc = doc;
      this.pageEditComp.changePageNumbers(curDoc.getNumPages());

      // show assocs and questions if set
      slPanel.showAssociations(this.showAssocs);
      slPanel.showQuestions(this.showQuestions);
      slPanel.repaint();

      // add AssessmentMaintenancePanel as observer (view) of
      // AssociationHelper
      AssociationHelper.getInstance().addObserver(this.amp);

      AssessmentInfo newAsInfo = null;
      if (assocHelper.getAsAssoc() != null) {
         AssessmentAssociation aa = assocHelper.getAsAssoc();
         if (aa.getLeft() != null) {
            newAsInfo = aa.getLeft();
         }
      }
      if (newAsInfo != null) {
         updateAssessmentTreePane(newAsInfo);
      } else {
         // create an empty AssociationInfo
         assessmentInfo = AssessmentMaintenanceHelper.createEmptyAssessment();
         updateAssessmentTreePane(assessmentInfo);
      }
   }

   public void onClose() {
      CALMHelper.saveState(savedState);

      AssessmentMaintenanceHelper.shutdown();
      super.onClose();
   }

   public void onHelp() {
      helpDlg = new HelpDialog(this, "Help");
      helpDlg.showDialog();
      helpDlg.dispose();
   }

   class HelpAdapter implements ActionListener {
      public void actionPerformed(ActionEvent e) {
         ClinicalAssessmentLayoutManager.this.onHelp();
      }
   }

   // for ChangeListener interface
   public void stateChanged(ChangeEvent e) {
      if (e.getSource() instanceof PageEditComponent) {
         PageEditComponent pec = (PageEditComponent) e.getSource();
         if (pec.getLastAction() == PageEditComponent.CHANGE_PAGE) {
            Page curPage = curDoc.getPageAt(pec.getSelectedPageIdx());
            slPanel.setCAContainerAt(curPage.getViewPanel(), 0);
            slPanel.showAssociations(this.showAssocs);
            slPanel.showQuestions(this.showQuestions);
            slPanel.repaint();
         } else if (pec.getLastAction() == PageEditComponent.ADD_PAGE) {
            log.info("adding new page");

            CAPanel aPanel = slPanel.createDefaultPanel(false);
            Page newPage = null;
            if (pec.getNewPageLocation() == PageEditComponent.BEFORE_CURRENT) {
               newPage = new Page(pec.getSelectedPageIdx() + 1);
               curDoc.addPage(newPage, pec.getSelectedPageIdx());
            } else if (pec.getNewPageLocation() == PageEditComponent.AFTER_CURRENT) {
               if (pec.getSelectedPageIdx() + 2 <= curDoc.getNumPages()) {
                  newPage = new Page(pec.getSelectedPageIdx() + 2);
                  curDoc.addPage(newPage, pec.getSelectedPageIdx() + 1);
               } else {
                  newPage = new Page(curDoc.getNumPages() + 1);
                  curDoc.addPage(newPage);
               }
            } else {
               // as the last page
               newPage = new Page(curDoc.getNumPages() + 1);
               curDoc.addPage(newPage);
            }
            newPage.setViewPanel(aPanel);

            slPanel.setCAContainerAt(newPage.getViewPanel(), 0);
            slPanel.showAssociations(this.showAssocs);
            slPanel.showQuestions(this.showQuestions);
            slPanel.repaint();
         } else if (pec.getLastAction() == PageEditComponent.DELETE_PAGE) {
            int pageIDx = pec.getToBeDeletedPageIDx();

            curDoc.removePage(pageIDx);
            Page curPage = curDoc.getPageAt(pec.getSelectedPageIdx());
            slPanel.setCAContainerAt(curPage.getViewPanel(), 0);
            slPanel.showAssociations(this.showAssocs);
            slPanel.showQuestions(this.showQuestions);
            slPanel.repaint();
         }
      }
   }

   public static void validateCALMDocument(String xmlSchema, String xmlDocFile) {
      try {
         XMLUtils.verifyByXMLSchema(xmlDocFile, xmlSchema);
         System.out.println("Valid CALM document");
      } catch (Exception x) {
         x.printStackTrace();
         System.out.println("Not a valid CALM document");
      }
      System.exit(0);
   }

   public static void usage() {
      System.err
            .println("Usage: ${JAVA} caslayout.ui.ClinicalAssessmentLayoutManager [-schema <xml-schema> \\");
      System.err
            .println("             -xml <CALM-XML-doc>] [-help] [-version]");
      System.err
            .println("or  ${JAVA} caslayout.ui.ClinicalAssessmentLayoutManager -add -xml <CALM-XML-doc>");
      System.err.println("where");
      System.err.println("\t-help - displays this information");
      System.err
            .println("\t-schema <xml-schema>  -xml <CALM-XML-doc> are the CALM XML Schema and the CALM document");
      System.err.println("\t to be validated against the schema");
      System.err
            .println("\t-add to add the provided assessment <CALM-XML-Doc> to the HID database");
      System.err.println("");
      System.exit(1);
   }

   @SuppressWarnings("unchecked")
   public static void setUIFont(FontUIResource fur) {
      Enumeration keys = UIManager.getDefaults().keys();
      while (keys.hasMoreElements()) {
         Object key = keys.nextElement();
         Object value = UIManager.get(key);
         if (value instanceof FontUIResource) {
            UIManager.put(key, fur);
         }
      }
   }

   public static void main(String[] args) {
      try {
         if (args.length == 1) {
            if (args[0].equals("-version")) {
               System.out.println(VERSION);
               System.exit(0);
            } else {
               usage();
            }
         } else if (args.length == 4) {
            String xmlSchema = null;
            String xmlDocFile = null;
            if (args[0].equals("-schema")) {
               xmlSchema = args[1];
               if (args[2].equals("-xml")) {
                  xmlDocFile = args[3];
               } else {
                  usage();
               }
               validateCALMDocument(xmlSchema, xmlDocFile);
            } else if (args[0].equals("-xml")) {
               xmlDocFile = args[1];
               if (args[2].equals("-schema")) {
                  xmlSchema = args[3];
               } else {
                  usage();
               }
               validateCALMDocument(xmlSchema, xmlDocFile);
            }
         } else if (args.length == 3) {
            if (!args[0].equals("-add"))
               usage();
            String xmlDocFile = null;
            if (!args[1].equals("-xml"))
               usage();
            xmlDocFile = args[2];
            CommandLineInterface cli = new CommandLineInterface();
            cli.saveAssessment2DB(xmlDocFile);
            System.exit(0);

         } else if (args.length > 0) {
            usage();
         }

         setUIFont(new FontUIResource("Dialog", Font.PLAIN, 11));
         UIManager.setLookAndFeel(new TonicLookAndFeel());
         ClinicalAssessmentLayoutManager calm = new ClinicalAssessmentLayoutManager(
               1000, 700);

         calm.startup();
      } catch (Exception x) {
         x.printStackTrace();
      }
   }

}
