package caslayout.ui;

import java.awt.Component;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Stack;

import javax.swing.JOptionPane;

import org.apache.log4j.Logger;
import org.jdom.Comment;
import org.jdom.Element;
import org.jdom.filter.Filter;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import caslayout.ui.model.AssessmentAssociation;
import caslayout.ui.model.AssessmentInfo;
import caslayout.ui.model.AssociationHelper;
import caslayout.util.XMLUtils;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: Document.java,v 1.20 2008/10/13 23:58:07 bozyurt Exp $
 */
public class Document {
   protected List<Page> pages = new ArrayList<Page>();
   protected String name;
   protected String description;
   protected String version;
   protected String versionComment;

   protected Map<String, IDisplayComponent> displayCompMap = new HashMap<String, IDisplayComponent>();
   protected Set<String> missingLGComps = new LinkedHashSet<String>(4);
   protected Set<String> missingQGComps = new LinkedHashSet<String>(4);
   protected Set<String> missingAssocComps = new LinkedHashSet<String>(4);
   protected static Logger log = Logger.getLogger(Document.class);

   public Document(String name) {
      this(name, null);
   }

   public Document(String name, String description) {
      this.name = name;
      this.description = description;
   }

   // ---------------------- setters --------------
   public void setName(String newName) {
      this.name = newName;
   }

   public void setDescription(String newDescription) {
      this.description = newDescription;
   }

   public void setVersion(String version) {
      this.version = version;
   }

   public void setVersionComment(String versionComment) {
      this.versionComment = versionComment;
   }

   // ---------------------- getters --------------
   public String getName() {
      return this.name;
   }

   public String getDescription() {
      return this.description;
   }

   public String getVersion() {
      return version;
   }

   public String getVersionComment() {
      return versionComment;
   }

   public void addPage(Page page) {
      pages.add(page);
   }

   public void addPage(Page page, int idx) {
      if (idx < pages.size()) {
         pages.add(idx, page);
      } else if (idx == pages.size()) {
         pages.add(page);
      } else {
         /* do nothing */
      }
   }

   public void removePage(int idx) {
      if (idx < pages.size()) {
         Page page = pages.remove(idx);
         /**
          * @todo remove the components for the removed page from the
          *       displayCompMap
          */
         removeComponents(page.getViewPanel());
      }
   }

   private void removeComponents(CAContainer cont) {
      LogicalGroupRepository lgRep = LogicalGroupRepository.getInstance();
      QuestionGroupRepository qgRep = QuestionGroupRepository.getInstance();
      for (Iterator<IDisplayComponent> iter = cont.getComponents().iterator(); iter
            .hasNext();) {
         IDisplayComponent adc = iter.next();
         iter.remove();
         if (adc == null) {
            continue;
         }
         if (adc instanceof CAContainer) {
            removeComponents((CAContainer) adc);
            cont.removePermanently(adc);
         } else {
            cont.removePermanently(adc);
            displayCompMap.remove(adc.getId());
            LogicalGroup lg = lgRep.findMatching(adc);
            if (lg != null) {
               lgRep.removeGroup(lg.getId());
            }
            QuestionGroup qg = qgRep.findMatching(adc);
            if (qg != null) {
               qgRep.removeGroup(qg.getId());
            }
         }
      }
   }

   public int getNumPages() {
      return pages.size();
   }

   public Page getFirstPage() {
      if (pages.isEmpty()) {
         return null;
      }
      return pages.get(0);
   }

   public Iterator<Page> getPagesIterator() {
      return pages.iterator();
   }

   public Page getPageAt(int idx) {
      if (idx >= 0 && idx < pages.size()) {
         return pages.get(idx);
      }
      return null;
   }

   public int findPageForDisplayComponent(IDisplayComponent ic) {
      // find the root node of the container tree, ic is a member of
      IDisplayComponent parent = ic.getParent();
      if (parent == null) {
         parent = ic;
      }
      while (parent.getParent() != null) {
         parent = parent.getParent();
      }
      for (Iterator<Page> iter = getPagesIterator(); iter.hasNext();) {
         Page page = iter.next();
         if (page.getViewPanel() == parent) {
            return page.getNumber();
         }
      }

      return -1;
   }

   /**
    * Do a depth-first search to find the component with the given id
    *
    * @param adc
    * @param id
    * @param stack
    */
   protected static void findComponentInPage(IDisplayComponent ic,
         IDisplayComponent target, Stack<IDisplayComponent> stack) {
      if (ic == target) {
         stack.push(ic);
         return;
      }
      if (ic instanceof CAContainer) {
         CAContainer c = (CAContainer) ic;
         List<IDisplayComponent> nextLevel = new LinkedList<IDisplayComponent>();
         for (Iterator<IDisplayComponent> iter = c.getComponents().iterator(); iter
               .hasNext();) {
            IDisplayComponent child = iter.next();
            if (child != null) {
               if (!(child instanceof CAContainer)) {
                  if (child == target) {
                     stack.push(child);
                     return;
                  }
               } else {
                  nextLevel.add(child);
               }
            }
         }

         for (Iterator<IDisplayComponent> iter = nextLevel.iterator(); iter
               .hasNext();) {
            CAContainer con = (CAContainer) iter.next();
            findComponentInPage(con, target, stack);
         }
      }
   }

   void populateDisplayComponentMap(IDisplayComponent adc) {
      if (adc == null) {
         return;
      }
      if (adc instanceof CAContainer) {
         CAContainer con = (CAContainer) adc;
         for (Iterator<IDisplayComponent> iter = con.getComponents().iterator(); iter
               .hasNext();) {
            IDisplayComponent child = iter.next();
            if (child != null) {
               if (child instanceof CAContainer) {
                  populateDisplayComponentMap(child);
               } else {
                  displayCompMap.put(child.getId(), child);
                  // log.info("adding to displayCompMap: " +
                  // child.getId());
               }
            }
         }
      } else {
         // log.info("adding to displayCompMap: " + adc.getId());
         displayCompMap.put(adc.getId(), adc);
      }
   }

   public IDisplayComponent findDisplayComponent(String id) {
      return displayCompMap.get(id);
   }

   public void handleMissingCompInLogicalGroup(String missingCompID) {
      missingLGComps.add(missingCompID);
   }

   public void handleMissingCompInQuestionGroup(String missingCompID) {
      missingQGComps.add(missingCompID);
   }

   public void handleMissingCompInAssociation(String missingCompID) {
      missingAssocComps.add(missingCompID);
   }

   protected String createDocumentCleanupWarningMessage() {
      StringBuffer buf = new StringBuffer(128);
      buf.append("The loaded document contained orphan display components!\n");
      buf
            .append("The document is cleaned up. Please save it to make the changes permanent!\n");
      buf.append("Cleanup Summary\n-------------\n");
      if (!missingLGComps.isEmpty()) {
         buf
               .append(missingLGComps.size()
                     + " orphan display component references is removed from logical groups.\n");
      }
      if (!missingQGComps.isEmpty()) {
         buf
               .append(missingQGComps.size()
                     + " orphan display component references is removed from question groups.\n");
      }
      if (!missingAssocComps.isEmpty()) {
         buf
               .append(missingAssocComps.size()
                     + " associations with invalid display component references is removed.\n");
      }
      return buf.toString();
   }

   protected boolean neededCleanup() {
      return (!missingLGComps.isEmpty() || !missingQGComps.isEmpty() || !missingAssocComps
            .isEmpty());
   }

   public static Document loadDocument(String filename, Component parent)
         throws Exception {
      SAXBuilder builder = null;

      builder = new SAXBuilder(false);
      org.jdom.Document doc = builder.build(filename);
      Element clElem = doc.getRootElement();

      String type = clElem.getAttribute("type") != null ? clElem
            .getAttributeValue("type") : "";
      if (!clElem.getName().equals("caslayout")
            || !type.equalsIgnoreCase("document")) {
         CALMHelper.showError(parent, "The XML file " + filename
               + " is not recognized as a proper CASLayout document",
               "Error while loading the layout");
      }
      Element docElem = doc.getRootElement().getChild("document");
      String name = docElem.getAttributeValue("name");
      Element descElem = docElem.getChild("description");
      Document document = new Document(name);
      if (descElem != null) {
         document.setDescription(descElem.getText());
      }

      if (docElem.getAttribute("version") != null) {
         document.setVersion(docElem.getAttributeValue("version"));
      }
      List<?> contentList = docElem.getContent(new VersionCommentFilter());
      if (!contentList.isEmpty()) {
         Comment c = (Comment) contentList.get(0);
         document.setVersionComment(c.getText());
      }

      List<?> pageElems = docElem.getChildren("page");
      for (Iterator<?> iter = pageElems.iterator(); iter.hasNext();) {

         Element pageElem = (Element) iter.next();
         int pageNumber = Integer
               .parseInt(pageElem.getAttributeValue("number"));
         Page page = new Page(pageNumber);
         document.addPage(page);

         Element element = pageElem.getChild("container");

         CAPanel newPanel = CAPanel.initializeFromXML(element);
         page.setViewPanel(newPanel);
         XMLUtils.setPeerForChildren(newPanel, parent);
         XMLUtils.clearCache();

         document.populateDisplayComponentMap(newPanel);
      }

      // initialize the logical groups
      Element lgElem = clElem.getChild("logical-groups");
      if (lgElem != null) {
         loadGroupData(lgElem, document);
      }
      // initialize the question groups
      Element qgElem = clElem.getChild("question-groups");
      if (qgElem != null) {
         loadQuestionGroupData(qgElem, document);
      }
      // initialize the model and bindings
      AssessmentInfo asi = loadModelData(clElem);
      if (asi != null) {
         loadBindings(clElem, asi, document);
      }

      if (document.neededCleanup()) {
         JOptionPane.showMessageDialog(null, document
               .createDocumentCleanupWarningMessage(), "Document Cleanup",
               JOptionPane.WARNING_MESSAGE);
      }

      return document;
   }

   public static void saveDocument(String filename, Document curDoc)
         throws IOException {
      XMLOutputter xmlOut = null;
      BufferedWriter out = null;
      try {
         xmlOut = new XMLOutputter(Format.getPrettyFormat());
         Element rootElem = new Element("caslayout");
         rootElem.setAttribute("type", "document");
         Element docElem = new Element("document");
         docElem.setAttribute("name", new File(filename).getName());
         rootElem.addContent(docElem);
         if (curDoc.getVersion() != null) {
            docElem.setAttribute("version", curDoc.getVersion());
         }
         if (curDoc.getVersionComment() != null) {
            docElem.addContent(new Comment(curDoc.getVersionComment()));
         } else {
            docElem
                  .addContent(new Comment(
                        " $Id: Document.java,v 1.20 2008/10/13 23:58:07 bozyurt Exp $ "));
         }

         Element descriptionElem = new Element("description");
         descriptionElem
               .addContent("A Clinical assessment layout description file");
         docElem.addContent(descriptionElem);

         // save the model data and associations first
         saveModelData(rootElem);
         saveBindings(rootElem);
         saveGroupData(rootElem);
         saveQuestionGroupData(rootElem);

         int idx = 1;
         for (Iterator<Page> iter = curDoc.getPagesIterator(); iter.hasNext();) {
            Page page = iter.next();
            Element pageElem = new Element("page");
            pageElem.setAttribute("number", String.valueOf(idx));
            docElem.addContent(pageElem);
            pageElem.addContent(page.getViewPanel().toXML(pageElem));

            ++idx;
         }

         out = new BufferedWriter(new FileWriter(filename));

         xmlOut.output(rootElem, out);
      } finally {
         if (out != null) {
            try {
               out.close();
            } catch (Exception x) {}
         }
      }
   }

   protected static void loadGroupData(Element parentElem, Document document)
         throws IOException {
      LogicalGroupRepository lgRep = LogicalGroupRepository.getInstance();
      lgRep.populateFromXML(parentElem, document);
   }

   protected static void loadQuestionGroupData(Element parentElem,
         Document document) throws IOException {
      QuestionGroupRepository qgRep = QuestionGroupRepository.getInstance();
      qgRep.populateFromXML(parentElem, document);
   }

   protected static AssessmentInfo loadModelData(Element parentElem)
         throws IOException {
      AssessmentInfo asi = null;
      Element e = parentElem.getChild("assessment");
      if (e == null) {
         return null;
      }
      asi = AssessmentInfo.initializeFromXML(e);
      return asi;
   }

   protected static void saveGroupData(Element parentElem) throws IOException {
      Element lgElem = new Element("logical-groups");
      parentElem.addContent(lgElem);
      LogicalGroupRepository lgRep = LogicalGroupRepository.getInstance();
      for (Iterator<String> iter = lgRep.getGroupIDs().iterator(); iter.hasNext();) {
         String groupID = iter.next();
         LogicalGroup lg = lgRep.get(groupID);
         Element elem = lg.toXML(null);
         lgElem.addContent(elem);
      }
   }

   protected static void saveQuestionGroupData(Element parentElem)
         throws IOException {
      Element qgElem = new Element("question-groups");
      parentElem.addContent(qgElem);
      QuestionGroupRepository qgRep = QuestionGroupRepository.getInstance();
      for (Iterator<String> iter = qgRep.getQuestionIDs().iterator(); iter.hasNext();) {
         String questionID = iter.next();
         QuestionGroup qg = qgRep.get(questionID);
         Element elem = qg.toXML(null);
         qgElem.addContent(elem);
      }
   }

   protected static void saveModelData(Element parentElem) throws IOException {
      AssociationHelper ah = AssociationHelper.getInstance();
      AssessmentAssociation aa = ah.getAsAssoc();
      if (aa == null) {
         return;
      }
      AssessmentInfo asi = aa.getLeft();
      log.info(">>> asi =" + asi.getMandatoryFields().size() + " asi "
            + asi.toString());
      Element e = asi.toXML(null);
      parentElem.addContent(e);
   }

   protected static void loadBindings(Element parentElem, AssessmentInfo asi,
         Document doc) throws IOException {
      Element e = parentElem.getChild("bindings");
      if (e == null) {
         return;
      }
      // AssociationHelper is a singleton, so its state is reset and
      // reinitialized from the XML
      AssociationHelper.initializeFromXML(e, asi, doc);
   }

   protected static void saveBindings(Element parentElem) throws IOException {
      AssociationHelper ah = AssociationHelper.getInstance();
      Element be = ah.toXML(null);
      parentElem.addContent(be);
   }

   private static final class VersionCommentFilter implements Filter {
      private static final long serialVersionUID = 1L;

      public boolean matches(Object o1) {
         if (o1 instanceof Comment) {
            Comment c = (Comment) o1;
            if (c.getText().indexOf("$Id") != -1) {
               return true;
            }
         }
         return false;
      }
   }// ;

}
