package caslayout.ui;

import java.util.*;
import java.util.List;

import java.awt.*;
import java.awt.font.*;
import java.awt.geom.*;

import org.jdom.*;
import caslayout.util.*;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: DropdownDisplayComponent.java,v 1.5 2005/09/09 01:36:07 bozyurt
 *          Exp $
 */

public class DropdownDisplayComponent extends AbstractDisplayComponent {
   protected Map<String, String> labelValueMap;
   protected boolean populatedFromDatabase;
   /* parametrized SQL query used to populate the dropdown dynamically */
   protected String sqlQuery;

   public DropdownDisplayComponent() {
      super();
      labelValueMap = new LinkedHashMap<String, String>(11);
   }

   public DropdownDisplayComponent(String id) {
      super(id);
      labelValueMap = new LinkedHashMap<String, String>(11);
   }

   public DropdownDisplayComponent(Map<String, String> labelValueMap, int x,
         int y, Component peer) {
      super(x, y, peer);
      this.labelValueMap = labelValueMap;
   }

   public DropdownDisplayComponent(DropdownDisplayComponent other) {
      super(other);
      this.id = UIDManager.getInstance().getNextUID();

      this.labelValueMap = new LinkedHashMap<String, String>(
            other.labelValueMap);
      getBounds((Graphics2D) peer.getGraphics());
   }

   public Object clone() {
      return new DropdownDisplayComponent(this);
   }

   public void addLabelValue(String label, String value) {
      labelValueMap.put(label, value);
   }

   public Map<String, String> getLabelValueMap() {
      return labelValueMap;
   }

   // ---------------------- setters --------------
   public void setPopulatedFromDatabase(boolean newPopulatedFromDatabase) {
      this.populatedFromDatabase = newPopulatedFromDatabase;
   }

   public void setSqlQuery(String newSqlQuery) {
      this.sqlQuery = newSqlQuery;
   }

   // ---------------------- getters --------------
   public boolean getPopulatedFromDatabase() {
      return this.populatedFromDatabase;
   }

   public String getSqlQuery() {
      return this.sqlQuery;
   }

   public Rectangle2D getBounds(Graphics2D g2) {
      if (super.bounds == null) {
         Rectangle rb = peer.getBounds();
         Rectangle2D.Double r = new Rectangle2D.Double(x + 3, y + 3, rb
               .getWidth() - 6, rb.getHeight() - 6);

         super.bounds = r;
      }
      return super.bounds;
   }

   public void draw(Graphics2D g2) {
      int margin = 2;
      double boxHeight, boxWidth;

      Rectangle2D r = getBounds(g2);

      Color oldColor = g2.getColor();
      if (selected) {
         g2.setColor(Color.blue);
      } else if (groupID != null) {
         g2.setColor(Color.magenta);
         super.showGroupID(g2, groupID, r, Color.cyan);
      }

      if (getShowQuestions())
         super.showQuestion(g2, r, Color.ORANGE);

      if (getShowAssociations())
         super.showScoreName(g2, r, Color.cyan);

      Font font = g2.getFont();
      FontRenderContext frCtx = g2.getFontRenderContext();
      Rectangle2D strRect = font.getStringBounds("AHy", frCtx);

      boxHeight = Math.min(r.getHeight(), strRect.getHeight() + 2);
      boxWidth = Math.min(r.getWidth(), 80);

      Dimension2D compDim = new Dimension((int) boxWidth, (int) boxHeight);
      Point2D point = GUIUtils.calculateAlignmentPosition(r, compDim,
            justification);

      Rectangle2D.Double boundingBox = new Rectangle2D.Double(point.getX(),
            point.getY(), boxWidth, boxHeight);

      g2.setPaint(Color.lightGray);
      g2.fill(boundingBox);

      Rectangle2D.Double fieldBox = new Rectangle2D.Double(point.getX()
            + margin, point.getY() + margin, boxWidth - 2 * margin, boxHeight
            - 2 * margin);

      g2.setPaint(Color.white);
      g2.fill(fieldBox);
      if (selected) {
         g2.setColor(Color.blue);
      } else {
         g2.setColor(oldColor);
      }
      g2.draw(fieldBox);

      // draw the downward triangle and box to the right
      double bbMargin = 3;
      double bbWidth = 10;
      double bbHeight = boxHeight - 2 * margin;
      double bbX1 = point.getX() + margin + boxWidth - 2 * margin - bbWidth;
      double bbY1 = point.getY() + margin;
      Rectangle2D.Double buttonBox = new Rectangle2D.Double(bbX1, bbY1,
            bbWidth, bbHeight);
      GeneralPath gp = new GeneralPath();
      gp.moveTo((float) (bbX1 + bbMargin), (float) (bbY1 + bbMargin));
      gp.lineTo((float) (bbX1 + bbWidth - bbMargin), (float) (bbY1 + bbMargin));
      gp.lineTo((float) (bbX1 + bbWidth / 2),
            (float) (bbY1 + bbHeight - bbMargin));
      gp.lineTo((float) (bbX1 + bbMargin), (float) (bbY1 + bbMargin));

      g2.draw(buttonBox);
      g2.draw(gp);

      g2.setColor(oldColor);
   }

   public static DropdownDisplayComponent initializeFromXML(Element e) {
      String id = XMLUtils.getStringPropertyValue("id", e, "");
      String cssClass = XMLUtils.getStringPropertyValue("cssClass", e, null);
      DropdownDisplayComponent ddc = new DropdownDisplayComponent(id);

      ddc.x = XMLUtils.getPropertyValue("x", e, 0);
      ddc.y = XMLUtils.getPropertyValue("y", e, 0);
      ddc.width = XMLUtils.getPropertyValue("height", e, 0);
      ddc.height = XMLUtils.getPropertyValue("height", e, 0);
      String justificationValue = XMLUtils.getStringPropertyValue(
            "justification", e, null);
      if (justificationValue != null) {
         ddc.setJustification(Justification
               .findJustification(justificationValue));
      }
      ddc.cssClass = cssClass;
      String refID = XMLUtils.getReferencePropertyValue("parent", e, null);
      if (refID != null) {
         ddc.parent = (IDisplayComponent) XMLUtils
               .getReferencedObject(refID, e);
      }

      Element opsElem = e.getChild("options");
      List<?> children = opsElem.getChildren("option");
      for (Iterator<?> iter = children.iterator(); iter.hasNext();) {
         Element elem = (Element) iter.next();
         String label = elem.getAttributeValue("label");
         String value = elem.getAttributeValue("value");
         ddc.labelValueMap.put(label, value);

      }

      String pfdStr = XMLUtils.getStringPropertyValue("populatedFromDatabase",
            e, "false");
      ddc.setPopulatedFromDatabase(new Boolean(pfdStr).booleanValue());
      String sqlQuery = XMLUtils.getStringPropertyValue("sqlQuery", e, null);
      if (sqlQuery != null) {
         ddc.setSqlQuery(XMLUtils.decodeXML(sqlQuery));
      }

      return ddc;
   }

   public Element toXML(Element root) {
      Element e = super.toXML(root);
      e.setAttribute("class", "caslayout.ui.DropdownDisplayComponent");
      Element opsElem = new Element("options");
      e.addContent(opsElem);
      for (Map.Entry<String, String> entry : labelValueMap.entrySet()) {
         String label = entry.getKey();
         String value = entry.getValue();
         Element oe = new Element("option");
         oe.setAttribute("label", label);
         oe.setAttribute("value", value);
         opsElem.addContent(oe);
      }
      e.addContent(XMLUtils.prepareProperty("justification", justification
            .getValue()));
      e.addContent(XMLUtils.prepareProperty("populatedFromDatabase", String
            .valueOf(populatedFromDatabase)));
      if (sqlQuery != null) {
         e.addContent(XMLUtils.prepareProperty("sqlQuery", XMLUtils
               .toXML(sqlQuery)));
      }
      return e;
   }

}
