package caslayout.ui;

import java.util.*;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: GridCellHelper.java,v 1.4 2008/10/13 23:58:07 bozyurt Exp $
 */
public class GridCellHelper {
   protected GridCellHelper() {}

   public static void adjustRowPercentages(PercentCellConstraint[] ra,
         double rowPercent) {
      for (int i = 0; i < ra.length; i++) {
         ra[i].setRowPercent(rowPercent);
      }
   }

   /**
    * finds the non-zero minimum
    *
    * @param vec
    *           a vector of doubles to check for min
    * @return non-zero minimum
    */
   public static double findMin(double[] vec) {
      double min = Double.MAX_VALUE;
      for (int i = 0; i < vec.length; i++) {
         if (vec[i] == 0)
            continue;
         if (vec[i] < min)
            min = vec[i];
      }
      return min;
   }

   public static double findMedianRowPercent(List<CellConstraint[]> cellRows) {
      double[] rowPercents = new double[cellRows.size()];
      int i = 0;
      for (Iterator<CellConstraint[]> iter = cellRows.iterator(); iter.hasNext();) {
         PercentCellConstraint[] rowArr = (PercentCellConstraint[]) iter.next();
         rowPercents[i++] = rowArr[0].getRowPercent();
      }
      Arrays.sort(rowPercents);
      int midPoint = (int) (rowPercents.length / 2.0);
      if (midPoint >= rowPercents.length)
         midPoint = rowPercents.length - 1;
      return rowPercents[midPoint];
   }

   public static double[] getRelativeRowRatios(List<CellConstraint[]> cellRows) {
      double[] ratios = new double[cellRows.size()];
      int idx = 0;
      for (Iterator<CellConstraint[]> iter = cellRows.iterator(); iter.hasNext();) {
         PercentCellConstraint[] rowArr =  (PercentCellConstraint[]) iter.next();
         ratios[idx++] = rowArr[0].getRowPercent() / 100.0;
      }

      return ratios;
   }

   public static double[] getRelativeColumnRatios(PercentCellConstraint[] ra) {
      double[] ratios = new double[ra.length];
      for (int i = 0; i < ra.length; i++) {
         // ratios[i] = refPercent / ra[i].getColPercent();
         ratios[i] = ra[i].getColPercent() / 100.0;
      }
      return ratios;
   }

   /**
    *
    * @param ra
    * @param newColCount
    * @return
    */
   public static double[] calculateNewColPercents(PercentCellConstraint[] ra,
         int newColCount) {
      double[] ratios = getRelativeColumnRatios(ra);
      double[] newPercents = new double[newColCount];
      double sum = 0;
      if (ra.length >= newColCount) {
         // remove columns from the end
         for (int i = 0; i < newColCount; i++) {
            sum += ratios[i];
         }
         for (int i = 0; i < newColCount; i++) {
            newPercents[i] = 100.0 * (ratios[i] / sum);
         }
      } else {
         double minRatio = findMin(ratios);
         sum = (newColCount - ra.length) * minRatio;
         // add columns of equivalent width to the end
         for (int i = 0; i < ra.length; i++) {
            sum += ratios[i];
         }
         for (int i = 0; i < ra.length; i++) {
            newPercents[i] = 100.0 * (ratios[i] / sum);
         }
         for (int i = ra.length; i < newColCount; i++) {
            newPercents[i] = 100.0 * (minRatio / sum);
         }
      }
      return newPercents;
   }

   public static double[] calculateNewRowPercents(List<CellConstraint[]> cellRows, int newRowCount) {
      double[] ratios = getRelativeRowRatios(cellRows);
      double[] newPercents = new double[newRowCount];
      double sum = 0;
      if (cellRows.size() >= newRowCount) {
         for (int i = 0; i < newRowCount; i++) {
            sum += ratios[i];
         }
         for (int i = 0; i < newRowCount; i++) {
            newPercents[i] = 100.0 * (ratios[i] / sum);
         }
      } else {
         double minRatio = findMin(ratios);
         sum = (newRowCount - cellRows.size()) * minRatio;
         for (int i = 0; i < cellRows.size(); i++) {
            sum += ratios[i];
         }
         for (int i = 0; i < cellRows.size(); i++) {
            newPercents[i] = 100.0 * (ratios[i] / sum);
         }
         for (int i = cellRows.size(); i < newRowCount; i++) {
            newPercents[i] = 100.0 * (minRatio / sum);
         }
      }

      return newPercents;
   }

   public static double[] calculateNewColPercents(PercentCellConstraint[] ra,
         PercentCellConstraint changedCell) {
      double[] ratios = getRelativeColumnRatios(ra);
      double[] newPercents = new double[ra.length];
      double sum = 0;

      ratios[changedCell.getColIdx()] = 0;
      for (int i = 0; i < ra.length; i++) {
         sum += ratios[i];
      }
      double newPortion = 1.0 - changedCell.getColPercent() / 100;
      double scale;

      if (sum >= 0.01) {
         scale = newPortion / sum;
      } else {
         scale = 1;
         if (newPortion > 0 && ra.length > 1) {
            // divide the space equally between the previously zero width
            // columns
            for (int i = 0; i < ra.length; i++) {
               if (i != changedCell.getColIdx()) {
                  ratios[i] = newPortion / (ra.length - 1);
               }
            }
         }
      }

      for (int i = 0; i < ra.length; i++) {
         ratios[i] *= scale;
      }
      ratios[changedCell.getColIdx()] = changedCell.getColPercent() / 100;

      for (int i = 0; i < ra.length; i++) {
         newPercents[i] = 100.0 * ratios[i];
      }

      return newPercents;
   }

   public static double[] calculateNewRowPercents(List<CellConstraint[]> cellRows,
         PercentCellConstraint changedCell) {
      double[] ratios = getRelativeRowRatios(cellRows);
      double[] newPercents = new double[cellRows.size()];
      double sum = 0;

      ratios[changedCell.getRowIdx()] = 0;

      for (int i = 0; i < cellRows.size(); i++) {
         sum += ratios[i];
      }

      double newPortion = 1.0 - changedCell.getRowPercent() / 100;
      double scale = newPortion / sum;
      for (int i = 0; i < cellRows.size(); i++) {
         ratios[i] *= scale;
      }
      ratios[changedCell.getRowIdx()] = changedCell.getRowPercent() / 100;

      for (int i = 0; i < cellRows.size(); i++) {
         newPercents[i] = 100.0 * ratios[i];
      }

      return newPercents;
   }

   public static void applyColumnPercents(PercentCellConstraint[] ra,
         double[] colPercents) {
      for (int i = 0; i < ra.length; i++) {
         ra[i].setColPercent(colPercents[i]);
      }
   }

   public static void applyRowPercents(List<CellConstraint[]> cellRows, double[] rowPercents) {
      int idx = 0;
      for (Iterator<CellConstraint[]> iter = cellRows.iterator(); iter.hasNext();) {
         PercentCellConstraint[] ra = (PercentCellConstraint[]) iter.next();
         for (int i = 0; i < ra.length; i++) {
            ra[i].setRowPercent(rowPercents[idx]);
         }
         ++idx;
      }
   }

   public static PercentCellConstraint[] adjustColumns(int rowIdx,
         PercentCellConstraint[] ra, int newColCount) {
      PercentCellConstraint[] newRa = new PercentCellConstraint[newColCount];
      if (newColCount > ra.length) {
         for (int i = 0; i < ra.length; i++) {
            newRa[i] = ra[i];
         }
         double rowPercent = ra[0].getRowPercent();
         for (int i = 0; i < newColCount - ra.length; i++) {
            newRa[i + ra.length] = new PercentCellConstraint(rowIdx, i
                  + ra.length, 10, rowPercent);
         }
      } else {
         for (int i = 0; i < newColCount; i++) {
            newRa[i] = ra[i];
         }
      }

      return newRa;
   }

}