package caslayout.ui;

import java.util.*;

import org.apache.log4j.*;
import org.jdom.*;
import caslayout.util.*;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: GridCellInfo.java,v 1.16 2008/10/13 23:58:07 bozyurt Exp $
 */
public class GridCellInfo {
   int maxCols, maxRows;
   List<CellConstraint[]> cellRows;
   CAGridLayout parent = null;
   protected static Logger log = Logger.getLogger("GridCellInfo");

   public GridCellInfo() {
      log.debug("*** in GridCellInfo 1");
   }

   public GridCellInfo(int maxRows, int maxCols, CAGridLayout parent,
         boolean usePercentConstraint) {
      if (log.isDebugEnabled()) {
         log.debug("*** in GridCellInfo 2");
      }
      this.parent = parent;
      cellRows = new ArrayList<CellConstraint[]>(maxRows);
      this.maxCols = maxCols;
      this.maxRows = maxRows;
      if (usePercentConstraint) {
         for (int i = 0; i < maxRows; i++) {
            PercentCellConstraint[] rowArr = new PercentCellConstraint[maxCols];
            for (int j = 0; j < maxCols; j++) {
               rowArr[j] = new PercentCellConstraint(i, j, 100.0 / maxCols,
                     100.0 / maxRows);
            }
            cellRows.add(rowArr);
         }
      } else {
         for (int i = 0; i < maxRows; i++) {
            MultipleCellSpanConstraint[] rowArr = new MultipleCellSpanConstraint[maxCols];
            for (int j = 0; j < maxCols; j++) {
               rowArr[j] = new MultipleCellSpanConstraint(i, j, 1, 1);
            }
            cellRows.add(rowArr);
         }
      }
   }

   protected GridCellInfo(int maxRows, int maxCols, CAGridLayout parent) {
      if (log.isDebugEnabled()) {
         log.debug("*** in GridCellInfo 4");
      }
      this.parent = parent;
      cellRows = new ArrayList<CellConstraint[]>(maxRows);
      this.maxCols = maxCols;
      this.maxRows = maxRows;
   }

   public GridCellInfo(GridCellInfo other) {
      if (log.isDebugEnabled()) {
         log.debug("*** in GridCellInfo 3");
      }
      this.parent = other.parent;
      cellRows = new ArrayList<CellConstraint[]>(other.maxRows);
      this.maxCols = other.maxCols;
      this.maxRows = other.maxRows;

      boolean isPercentConstraint = false;
      if (other.cellRows.get(0)[0] instanceof PercentCellConstraint) {
         isPercentConstraint = true;
      }
      for (int i = 0; i < other.cellRows.size(); i++) {
         int colCount = other.getColCount(i);

         CellConstraint otherRowArr[] = other.getRow(i);
         if (isPercentConstraint) {
            PercentCellConstraint[] rowArr = new PercentCellConstraint[maxCols];
            for (int j = 0; j < colCount; j++) {
               PercentCellConstraint otherConstraint = (PercentCellConstraint) otherRowArr[j];
               rowArr[j] = new PercentCellConstraint(otherConstraint.rowIdx,
                     otherConstraint.colIdx, otherConstraint.colPercent,
                     otherConstraint.rowPercent);
            }
            cellRows.add(rowArr);
         } else {
            MultipleCellSpanConstraint[] rowArr = new MultipleCellSpanConstraint[maxCols];
            for (int j = 0; j < colCount; j++) {
               MultipleCellSpanConstraint otherConstraint = (MultipleCellSpanConstraint) otherRowArr[j];
               rowArr[j] = new MultipleCellSpanConstraint(
                     otherConstraint.rowIdx, otherConstraint.colIdx,
                     otherConstraint.colSpan, otherConstraint.rowSpan);
            }
            cellRows.add(rowArr);
         }
      }// i
   }

   /**
    *
    * @return
    */
   public int getMaxRows() {
      return maxRows;
   }

   /**
    * sets the max number of rows in the grid
    *
    * @param newMaxRows
    */
   public void setMaxRows(int newMaxRows) {
      if (cellRows == null)
         return;
      if (newMaxRows == maxRows)
         return;
      if (newMaxRows > maxRows) {
         if (getGridRowCount() >= newMaxRows) {
            if (maxRows == 0)
               maxRows = newMaxRows;
            return;
         }
         int diff = newMaxRows - maxRows;
         appendRows(diff);
      } else {
         int diff = maxRows - newMaxRows;

         removeRowsFromEnd(diff);
      }
      maxRows = newMaxRows;
   }

   /**
    * returns the low offset and high offset bounding the removed components
    * will only work with PercentConstraints
    *
    * @param aRowIdx
    * @return
    */
   public List<Integer> removeRow(int aRowIdx) {
      CellConstraint[] rowArr = cellRows.get(aRowIdx);
      List<Integer> offsetList = new ArrayList<Integer>();
      int startOffset = aRowIdx * maxCols;
      int endOffset = aRowIdx * maxCols + rowArr[rowArr.length - 1].getColIdx();
      offsetList.add(new Integer(startOffset));
      offsetList.add(new Integer(endOffset));

      cellRows.remove(aRowIdx);
      maxRows--;

      double[] newPercents = GridCellHelper.calculateNewRowPercents(cellRows,
            cellRows.size());
      GridCellHelper.applyRowPercents(cellRows, newPercents);

      return offsetList;
   }

   public List<Integer> addRow(int afterRowIdx) {
      List<Integer> offsetList = new ArrayList<Integer>();
      PercentCellConstraint[] rowArr = new PercentCellConstraint[maxCols];

      double rowPercent = GridCellHelper.findMedianRowPercent(cellRows);
      for (int i = 0; i < rowArr.length; i++) {
         rowArr[i] = new PercentCellConstraint(afterRowIdx + 1, i,
               100.0 / maxCols, rowPercent);
      }
      int startOffset = (afterRowIdx + 1) * maxCols;
      int endOffset = (afterRowIdx + 1) * maxCols + maxCols - 1;
      offsetList.add(new Integer(startOffset));
      offsetList.add(new Integer(endOffset));

      cellRows.add(afterRowIdx + 1, rowArr);
      maxRows++;

      double[] newPercents = GridCellHelper.calculateNewRowPercents(cellRows,
            cellRows.size());
      GridCellHelper.applyRowPercents(cellRows, newPercents);

      return offsetList;
   }

   protected void removeRowsFromEnd(int noRowsTobeRemoved) {
      int remRows = noRowsTobeRemoved;
      ListIterator<CellConstraint[]> it = cellRows
            .listIterator(cellRows.size()); // -1

      while (remRows > 0 && it.hasPrevious()) {

         CellConstraint[] rowArr = it.previous();
         int rowIdx = rowArr[0].getRowIdx();

         if (rowArr instanceof MultipleCellSpanConstraint[]) {
            MultipleCellSpanConstraint[] mcscArr = (MultipleCellSpanConstraint[]) rowArr;
            if (mcscArr[0].rowSpan <= remRows) {
               it.remove();
               remRows -= mcscArr[0].rowSpan;
            } else {
               int newRowSpan = mcscArr[0].rowSpan - remRows;
               for (int i = 0; i < rowArr.length; i++) {
                  mcscArr[i].setRowSpan(newRowSpan);
               }
               remRows = 0;
            }
         } else if (rowArr instanceof PercentCellConstraint[]) {
            // we can just remove the last noRowsTobeRemoved
            it.remove();
            remRows--;

            // clean the display components in the removed row (if any)
            for (int i = 0; i < rowArr.length; i++) {
               IDisplayComponent ic = parent.getDisplayComponentAt(rowIdx,
                     rowArr[i].getColIdx());
               if (ic != null)
                  ((CAContainer) ic.getParent()).removePermanently(ic);
            }

         }
      }// while

      if (isPercentConstraint()) {
         double[] newPercents = GridCellHelper.calculateNewRowPercents(
               cellRows, cellRows.size());
         GridCellHelper.applyRowPercents(cellRows, newPercents);
      }

   }

   protected boolean isPercentConstraint() {
      if (cellRows.isEmpty())
         return false;
      CellConstraint[] rowArr = cellRows.get(0);
      if (rowArr.length == 0)
         return false;
      return (rowArr[0] instanceof PercentCellConstraint);
   }

   public int getMaxCols() {
      return maxCols;
   }

   /**
    * Sets the maximum number of columns for the grid
    *
    * @param newMaxCols
    */
   public void setMaxCols(int newMaxCols) {
      if (cellRows == null)
         return;
      if (newMaxCols == maxCols)
         return;

      if (isPercentConstraint()) {
         setMaxColsForPercentConstraints(newMaxCols);
         maxCols = newMaxCols;
         return;
      }

      if (newMaxCols > maxCols) {
         for (int i = 0; i < cellRows.size(); i++) {
            CellConstraint[] rowArr = cellRows.get(i);
            int colCount = getColCount(i);
            // CHECK!!
            if (colCount >= newMaxCols)
               continue;
            int extraColCount = newMaxCols - maxCols;
            MultipleCellSpanConstraint[] newRowArr = new MultipleCellSpanConstraint[newMaxCols];
            System.arraycopy(rowArr, 0, newRowArr, 0, colCount);
            for (int j = 0; j < extraColCount; j++) {
               newRowArr[colCount + j] = new MultipleCellSpanConstraint(i,
                     colCount + j, 1, 1);
            }
            cellRows.set(i, newRowArr);
         }
      } else {
         for (int i = 0; i < cellRows.size(); i++) {
            CellConstraint[] rowArr = cellRows.get(i);
            int colCount = getColCount(i);
            if (colCount <= maxCols) {
               // just replace the row with the column adjusted one
               MultipleCellSpanConstraint[] newRowArr = new MultipleCellSpanConstraint[newMaxCols];
               int copiedColumnCount = (colCount > newMaxCols) ? newMaxCols
                     : colCount;
               System.arraycopy(rowArr, 0, newRowArr, 0, copiedColumnCount);
               int extraColCount = newMaxCols - colCount;
               for (int j = 0; j < extraColCount; j++) {
                  newRowArr[colCount + j] = new MultipleCellSpanConstraint(i,
                        colCount + j, 1, 1);
               }

               cellRows.set(i, newRowArr);
               // need to adjust the colspan for the last column to satisfy grid
               // column count (maxCol) constraint
               satisfyColumnGridCountConstraint(newRowArr, getGridColCount(i),
                     newMaxCols);

            } else {
               int remCols = newMaxCols - maxCols;
               MultipleCellSpanConstraint[] newRowArr = new MultipleCellSpanConstraint[newMaxCols];
               int newColCount = colCount;
               for (int j = colCount - 1; j >= 0; j--) {
                  MultipleCellSpanConstraint mcsc = (MultipleCellSpanConstraint) rowArr[j];
                  if (mcsc.colSpan <= remCols) {
                     remCols -= mcsc.colSpan;
                     newColCount--;
                  } else {
                     mcsc.setColSpan(mcsc.getColSpan() - remCols);
                     remCols = 0;
                  }
                  if (remCols == 0)
                     break;
               }// j
               System.arraycopy(rowArr, 0, newRowArr, 0, newColCount);
               int extraColCount = newMaxCols - newColCount;
               for (int j = 0; j < extraColCount; j++) {
                  newRowArr[colCount + j] = new MultipleCellSpanConstraint(i,
                        colCount + j, 1, 1);
               }
               cellRows.set(i, newRowArr);

            }
         }
      }

      maxCols = newMaxCols;
   }

   protected void satisfyColumnGridCountConstraint(
         MultipleCellSpanConstraint[] rowArr, int colGridCount, int newMaxCols) {
      int diff = colGridCount - newMaxCols;
      if (diff <= 0)
         return;
      for (int i = rowArr.length - 1; i >= 0; i--) {
         if (diff == 0)
            break;
         if (rowArr[i] != null) {
            if (rowArr[i].colSpan > diff) {
               rowArr[i].colSpan -= diff;
               break;
            } else {
               diff -= rowArr[i].colSpan;
               rowArr[i] = null;
            }
         }
      }
   }

   protected void setMaxColsForPercentConstraints(int newMaxCols) {
      for (int i = 0; i < cellRows.size(); i++) {
         PercentCellConstraint[] rowArr = (PercentCellConstraint[]) cellRows
               .get(i);
         int oldColCount = getColCount(i);

         double[] newPercents = GridCellHelper.calculateNewColPercents(rowArr,
               newMaxCols);
         PercentCellConstraint[] newRa = GridCellHelper.adjustColumns(i,
               rowArr, newMaxCols);
         GridCellHelper.applyColumnPercents(newRa, newPercents);
         cellRows.set(i, newRa);

         if (oldColCount > newMaxCols) {
            // remove the orphan components
            for (int j = newMaxCols; j < oldColCount; j++) {
               IDisplayComponent ic = parent.getDisplayComponentAt(i, j);
               if (ic != null) {
                  log.info(" removing *** from container "
                        + ((CAContainer) ic.getParent()).getId());
                  ((CAContainer) ic.getParent()).removePermanently(ic);
               }
            }
         }
      }
   }

   public List<CellConstraint[]> getCellRows() {
      return cellRows;
   }

   public void setCellRows(List<CellConstraint[]> newCellRows) {
      cellRows = newCellRows;
   }

   /**
    *
    * @param row
    * @param col
    * @param constraint
    */
   protected void setPercentCellConstraint(int row, int col,
         PercentCellConstraint constraint) {
      PercentCellConstraint[] rowArr = (PercentCellConstraint[]) cellRows
            .get(row);

      double[] newColPercents = GridCellHelper.calculateNewColPercents(rowArr,
            constraint);
      GridCellHelper.applyColumnPercents(rowArr, newColPercents);

      double[] newRowPercents = GridCellHelper.calculateNewRowPercents(
            cellRows, constraint);
      GridCellHelper.applyRowPercents(cellRows, newRowPercents);
   }

   public void setCellConstraint(int row, int col, CellConstraint constraint) {

      if (constraint instanceof PercentCellConstraint) {
         setPercentCellConstraint(row, col, (PercentCellConstraint) constraint);
         return;
      }

      MultipleCellSpanConstraint cons = (MultipleCellSpanConstraint) constraint;

      int colSpan = cons.colSpan;
      int rowSpan = cons.rowSpan;

      if (maxRows < row + rowSpan) {
         rowSpan = cons.rowSpan = maxRows - row;
      }
      if (maxCols < col + colSpan) {
         colSpan = cons.colSpan = maxCols - col;
      }

      // if row number is greater than the current row size append additional
      // rows
      if (row >= cellRows.size()) {
         int diff = row - cellRows.size() + 1;
         appendRows(diff);
      }

      log.info(">>>> " + cellRows.get(row));
      CellConstraint[] rowArr = cellRows.get(row);
      rowArr[col] = cons;
      if (colSpan > 1) {
         int gridColCount = getGridColCount(row);
         if (gridColCount > maxCols) {
            // adjust remaining column cells
            int remCol = maxCols - (col + colSpan);
            log.info("remCol=" + remCol);

            // shift the remaining columns to the left
            int colIdx = col + 1;
            for (int i = 0; i < remCol; ++i) {
               if (rowArr[col + i + colSpan] != null) {
                  rowArr[col + i + 1] = rowArr[col + i + colSpan];
                  rowArr[col + i + 1].colIdx = colIdx;
                  rowArr[col + i + colSpan] = null;
                  ++colIdx;
               }
            }
            for (int i = col + remCol + 1; i < rowArr.length; i++) {
               rowArr[i] = null;
            }

         }
      }

      cellRows.set(row, rowArr);

      int curGridCount = getGridColCount(row);
      if (curGridCount < maxCols) {
         // the total number of effective columns is less than maxCols
         // so add columns with colspan of 1 to the end of the row
         int diff = maxCols - curGridCount;
         int curColCount = getColCount(row);
         for (int i = 0; i < diff; i++) {
            rowArr[curColCount + i] = new MultipleCellSpanConstraint(row,
                  curColCount + i);
         }
      }

      // make each cell in the row to have the same rowSpan
      for (int i = 0; i < rowArr.length; i++) {
         if (rowArr[i] != null) {
            MultipleCellSpanConstraint mcsc = (MultipleCellSpanConstraint) rowArr[i];
            mcsc.rowSpan = rowSpan;
         }
      }

      int curGridRowCount = getGridRowCount();

      if (curGridRowCount < maxRows) {
         // the total number of effective rows is less than maxRows
         // so append rows to have effectively maxRows rows.
         int diff = maxRows - curGridRowCount;
         appendRows(diff);
      } else if (curGridRowCount > maxRows) {

         // find the location of curGridRowCount in cellRows
         int maxIdx = 0;
         int rowCount = getRowCount();
         int cumRowCount = 0;
         for (int i = 0; i < rowCount; i++) {
            MultipleCellSpanConstraint[] ra = (MultipleCellSpanConstraint[]) cellRows
                  .get(i);
            if (cumRowCount + ra[0].getRowSpan() <= maxRows) {
               cumRowCount += ra[0].getRowSpan();
               maxIdx = i;
            } else {
               break;
            }
         }// for
         for (int i = 0; i < rowCount - maxIdx - 1; i++) {
            cellRows.remove(maxIdx + 1);
         }
      }

      // adjust the rowIdxs if necessary
      for (int i = 0; i < cellRows.size(); i++) {
         CellConstraint[] ra = cellRows.get(i);
         for (int j = 0; j < ra.length; j++) {
            if (ra[j] == null)
               break;
            ra[j].rowIdx = i;
         }
      }

      if (log.isDebugEnabled()) {
         dumpArray(row, rowArr);
      }
      log.info("getColCount() " + getColCount(row));
      log.info("getGridColCount() " + getGridColCount(row));
   }

   void dumpArray(int row, CellConstraint[] arr) {
      log.debug("dumping row " + row);
      for (int i = 0; i < arr.length; i++) {
         if (arr[i] != null) {
            log.debug("row =" + i + " " + arr[i].toString());
         }
      }
   }

   protected void appendRows(int numRows) {
      int gridRowCount = getGridRowCount();
      CellConstraint[] rowArr = cellRows.get(0);
      if (rowArr[0] instanceof PercentCellConstraint) {
         double newRowPercent;
         if (gridRowCount > 1) {
            double[] ratios = GridCellHelper.getRelativeRowRatios(cellRows);
            double minRatio = GridCellHelper.findMin(ratios);
            double sum = minRatio * numRows;
            for (int i = 0; i < ratios.length; i++) {
               sum += ratios[i];
            }
            newRowPercent = 100 * (minRatio / sum);
            int i = 0;
            for (Iterator<CellConstraint[]> iter = cellRows.iterator(); iter
                  .hasNext();) {
               PercentCellConstraint[] ra = (PercentCellConstraint[]) iter
                     .next();
               GridCellHelper.adjustRowPercentages(ra, ratios[i++] / sum
                     * 100.0);
            }

         } else {
            newRowPercent = 100.0 / (gridRowCount + numRows);

            for (Iterator<CellConstraint[]> iter = cellRows.iterator(); iter
                  .hasNext();) {
               PercentCellConstraint[] ra = (PercentCellConstraint[]) iter
                     .next();
               GridCellHelper.adjustRowPercentages(ra, newRowPercent);
            }
         }
         for (int i = 0; i < numRows; i++) {
            rowArr = new PercentCellConstraint[maxCols];
            for (int j = 0; j < rowArr.length; j++) {
               rowArr[j] = new PercentCellConstraint(gridRowCount + i, j,
                     100.0 / maxCols, newRowPercent);
            }
            cellRows.add(rowArr);
         }
      } else {
         for (int i = 0; i < numRows; i++) {
            rowArr = new MultipleCellSpanConstraint[maxCols];
            for (int j = 0; j < rowArr.length; j++) {
               rowArr[j] = new MultipleCellSpanConstraint(gridRowCount + i, j);
            }
            cellRows.add(rowArr);
         }
      }
   }

   /**
    * Returns the total of colSpan values for non null cells.
    *
    * @param row
    *           the index of the row
    * @return the total of colSpan values for non null cells
    */
   protected int getGridColCount(int row) {
      CellConstraint[] rowArr = cellRows.get(row);
      int gridColCount = 0;
      for (int i = 0; i < rowArr.length; i++) {
         if (rowArr[i] != null) {
            MultipleCellSpanConstraint mcsc = (MultipleCellSpanConstraint) rowArr[i];
            gridColCount += mcsc.colSpan;
         }
      }
      return gridColCount;
   }

   /**
    * Returns the total of rowSpan values for all rows in <code>cellRows</code>
    * list.
    *
    * @return the total of rowSpan values for all rows in <code>cellRows</code>
    *         list
    */
   protected int getGridRowCount() {
      int gridRowCount = 0;
      if (cellRows.isEmpty())
         return 0;
      CellConstraint[] rowArr = cellRows.get(0);
      if (rowArr[0] instanceof PercentCellConstraint) {
         return cellRows.size();
      }

      for (int i = 0; i < cellRows.size(); i++) {
         rowArr = cellRows.get(i);
         MultipleCellSpanConstraint mcsc = (MultipleCellSpanConstraint) rowArr[0];
         gridRowCount += mcsc.rowSpan;
      }
      return gridRowCount;
   }

   public int getRowCount() {
      return cellRows.size();
   }

   /**
    *
    * @param row
    *           the index of the row for which the actual column count is
    *           returned
    * @return
    */
   public int getColCount(int row) {
      CellConstraint[] rowArr = cellRows.get(row);
      for (int i = 0; i < rowArr.length; i++) {
         if (rowArr[i] == null)
            return i;
      }
      return rowArr.length;
   }

   public CellConstraint getConstraint(int row, int col) {
      if (row >= cellRows.size())
         return null;
      CellConstraint[] rowArr = cellRows.get(row);
      if (col >= rowArr.length)
         return null;
      return rowArr[col];
   }

   public CellConstraint[] getRow(int row) {
      return cellRows.get(row);
   }

   /**
    * Given a row and column index, returns the flatten cell location index
    *
    * @param row
    * @param col
    * @return
    */
   public int getIndex(int row, int col) {
      int index = 0;
      for (int i = 0; i < row; i++) {
         index += getColCount(i);
      }
      return index + col;
   }

   public static GridCellInfo initializeFromXML(Element e) {
      int maxRows = 0, maxCols = 0;
      List<CellConstraint[]> cellRows = null;
      maxRows = XMLUtils.getPropertyValue("maxRows", e, maxRows);
      maxCols = XMLUtils.getPropertyValue("maxCols", e, maxCols);
      XMLUtils.ListInfo listInfo = XMLUtils.prepareList("cellRows", e);
      if (listInfo != null) {
         cellRows = new ArrayList<CellConstraint[]>(listInfo.getList().size());
         for(Object elem : listInfo.getList())
            cellRows.add( (CellConstraint[]) elem);
         //cellRows = listInfo.getList();
      }
      // parent
      CAGridLayout parent = null;
      String refID = XMLUtils.getReferencePropertyValue("parent", e, null);
      if (refID != null)
         parent = (CAGridLayout) XMLUtils.getReferencedObject(refID, e);
      GridCellInfo gci = new GridCellInfo(maxRows, maxCols, parent);
      //
      checkAndApplyConstraints(gci, cellRows, refID);

      gci.setCellRows(cellRows);
      if (log.isDebugEnabled()) {
         log.debug(gci.toString());
      }
      return gci;
   }

   protected static void checkAndApplyConstraints(GridCellInfo gci,
         List<CellConstraint[]> cellRows, String refID) {
      if (cellRows.isEmpty())
         return;
      // only for percentage constraints
      CellConstraint[] ra = cellRows.get(0);
      if (!(ra[0] instanceof PercentCellConstraint))
         return;

      double orp = -1;
      double totRowPercent = 0;

      for (Iterator<CellConstraint[]> iter = cellRows.iterator(); iter
            .hasNext();) {
         CellConstraint[] rowArr = iter.next();
         double totColPercent = 0;
         double ocp = -1;
         if (gci.getMaxCols() > rowArr.length) {
            log.info("maxCols does not match the rowArr length for refID = "
                  + refID);
            gci.setMaxCols(rowArr.length);
         }

         for (int i = 0; i < rowArr.length; i++) {
            PercentCellConstraint pcc = (PercentCellConstraint) rowArr[i];
            if (ocp < 0 || orp < 0) {
               if (ocp < 0)
                  ocp = pcc.getColPercent();
               if (orp < 0)
                  orp = pcc.getRowPercent();
            }
            totColPercent += pcc.getColPercent();
         }// for i

         totRowPercent += ((PercentCellConstraint) rowArr[0]).getRowPercent();

         if (Math.abs(totColPercent - 100.0) > 0.001) {
            log.info("totColPercent was " + totColPercent
                  + " for GridLayout with refID=" + refID);

            double cp = 100.0 / gci.getMaxCols();
            for (int i = 0; i < rowArr.length; i++) {
               PercentCellConstraint pcc = (PercentCellConstraint) rowArr[i];
               pcc.setColPercent(cp);
            }
         }

      }// iter
      if (Math.abs(totRowPercent - 100.0) > 0.001) {
         log.info("totRowPercent was " + totRowPercent
               + " for GridLayout with refID=" + refID);

         double rp = 100.0 / gci.getMaxRows();
         for (Iterator<CellConstraint[]> iter = cellRows.iterator(); iter
               .hasNext();) {
            CellConstraint[] rowArr = iter.next();

            for (int i = 0; i < rowArr.length; i++) {
               PercentCellConstraint pcc = (PercentCellConstraint) rowArr[i];
               pcc.setRowPercent(rp);
            }
         }
      }

   }

   public Element toXML(Element root) {
      Element e = new Element("grid-cell-info");
      e.setAttribute("class", "caslayout.ui.GridCellInfo");
      e
            .addContent(XMLUtils.prepareProperty("maxRows", String
                  .valueOf(maxRows)));
      e
            .addContent(XMLUtils.prepareProperty("maxCols", String
                  .valueOf(maxCols)));

      e.addContent(XMLUtils.prepareRefID("parent", String.valueOf(parent
            .hashCode())));

      Element cellRowsElem = new Element("collection");
      cellRowsElem.setAttribute("name", "cellRows");
      cellRowsElem.setAttribute("type", "list");
      cellRowsElem.setAttribute("class", "java.util.ArrayList");
      cellRowsElem.setAttribute("size", String.valueOf(cellRows.size()));
      e.addContent(cellRowsElem);
      int idx = 0;
      for (Iterator<CellConstraint[]> iter = cellRows.iterator(); iter
            .hasNext();) {
         CellConstraint[] rowArr = iter.next();
         Element entry = new Element("entry");
         entry.setAttribute("index", String.valueOf(idx));
         Element rowArrElem = new Element("array");
         if (rowArr[0] instanceof PercentCellConstraint) {
            rowArrElem.setAttribute("class", PercentCellConstraint.class
                  .getName());
         } else {
            rowArrElem.setAttribute("class", MultipleCellSpanConstraint.class
                  .getName());
         }
         rowArrElem.setAttribute("length", String.valueOf(rowArr.length));
         entry.addContent(rowArrElem);
         cellRowsElem.addContent(entry);
         for (int i = 0; i < rowArr.length; ++i) {
            if (rowArr[i] != null)
               rowArrElem.addContent(rowArr[i].toXML(root));
         }
         ++idx;
      }
      return e;
   }

   public String toString() {
      StringBuffer buf = new StringBuffer();
      buf.append(" GridCellInfo::[");
      buf.append("maxRows=").append(maxRows).append("maxCols=").append(maxCols);
      for (Iterator<CellConstraint[]> iter = cellRows.iterator(); iter
            .hasNext();) {
         CellConstraint[] rowArr = iter.next();
         if (rowArr != null) {
            for (int i = 0; i < rowArr.length; i++) {
               if (rowArr[i] != null)
                  buf.append(rowArr[i].toString());
            }
         }
      }
      buf.append(']');
      return buf.toString();
   }

}
