package caslayout.ui;

import java.io.*;
import java.util.*;

import java.awt.event.*;
import javax.swing.*;

import org.apache.log4j.*;
import caslayout.util.*;
import guilib.common.*;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: PopupMenuHelper.java,v 1.14 2008/10/13 23:58:07 bozyurt Exp $
 */
public class PopupMenuHelper implements ActionListener {
   protected List<IDisplayComponent> selectedObjects;
   protected IDisplayComponent selectedDisplayComponent;
   protected ScoreLayoutPanel parent;

   protected CALMConfig config;
   protected JFileChooser fileChooser;
   protected static Logger log = Logger.getLogger("PopupMenuHelper");

   public PopupMenuHelper(ScoreLayoutPanel parent, CALMConfig config) {
      this.parent = parent;
      this.config = config;

      String templateDir = config.getTemplateDir();
      File f = new File(templateDir);
      if (f.isDirectory()) {
         fileChooser = new JFileChooser(f);
      } else {
         fileChooser = new JFileChooser(System.getProperty("user.home"));
      }
      fileChooser.setFileFilter(new XMLFileFilter());
   }

   public void setSelectedObjects(List<IDisplayComponent> list) {
      selectedObjects = list;
   }

   public void setSelectedDisplayComponent(
         IDisplayComponent selectedDisplayComponent) {
      log.info("setSelectedDisplayComponent() " + selectedDisplayComponent);
      this.selectedDisplayComponent = selectedDisplayComponent;
   }

   public void actionPerformed(ActionEvent e) {
      if (!(e.getSource() instanceof JMenuItem))
         return;
      JMenuItem item = (JMenuItem) e.getSource();
      if (item.getText().equals("Create a template")) {
         log.info("creating a template....");
         handleCreateTemplate();
      } else if (item.getText().equals("Delete Selection")) {
         handleDeleteSelection();
      } else if (item.getText().equals("Copy Selection")) {
         handleCopySelection();
      } else if (item.getText().equals("Logically Group Selection")) {
         handleLogicalGrouping();
      } else if (item.getText().equals("UnGroup Selection")) {
         handleLogicalUnGrouping();
      } else if (item.getText().equals("Group Selection as a Question")) {
         handleQuestionGrouping();
      } else if (item.getText().equals("Ungroup Selected Question")) {
         handleQuestionUnGrouping();
      }

   }

   protected void handleDeleteSelection() {
      if (selectedObjects != null && !selectedObjects.isEmpty()) {
         CAContainer topmost = getTopMostContainer();
         if (topmost == null) {
            JOptionPane.showMessageDialog(parent,
                  "Can only delete fully selected containers", "Warning",
                  JOptionPane.WARNING_MESSAGE);
            return;
         }
         if (topmost.getParent() != null) {
            CAContainer topmostParent = (CAContainer) topmost.getParent();
            topmostParent.remove(topmost);
            parent.repaint();
         }
      } else if (selectedDisplayComponent != null) {
         // remove single component
         log.info(">> removing selected display element ");
         IDisplayComponent p = selectedDisplayComponent.getParent();
         if (p != null && (p instanceof CAContainer)) {
            CAContainer container = (CAContainer) p;
            container.remove(selectedDisplayComponent);
            parent.repaint();
         }
         parent.setSelectedComponent(null);
         selectedDisplayComponent = null;
      }
   }

   protected void handleCopySelection() {
      if (selectedObjects != null && !selectedObjects.isEmpty()) {
         CAContainer topmost = getTopMostContainer();
         if (topmost == null) {
            JOptionPane.showMessageDialog(parent,
                  "Can only copy fully selected containers", "Warning",
                  JOptionPane.WARNING_MESSAGE);
            parent.curState.clearSelection();
            return;
         }
         if (topmost.getParent() != null) {
            CAContainer copy = (CAContainer) topmost.clone();
            parent.setSelectedComponent(copy);
            selectedDisplayComponent = null;
            parent.curState.clearSelection();
            GUIUtils.showStatusMessage(parent,
                  "Click in an empty cell to paste");
         }

      } else if (selectedDisplayComponent != null) {
         IDisplayComponent copy = (IDisplayComponent) selectedDisplayComponent
               .clone();
         // set the clone as the selected display element, so that it can be
         // pasted
         parent.setSelectedComponent(copy);
         selectedDisplayComponent = null;
         GUIUtils.showStatusMessage(parent, "Click in an empty cell to paste");
      } else {
         log.info("selectedDisplayComponent == null");
      }
   }

   protected void handleQuestionGrouping() {
      if (selectedObjects == null || selectedObjects.isEmpty()) {
         parent.curState.clearSelection();
         return;
      }

      QuestionGroup qg = QuestionGroupRepository.getInstance()
            .createQuestionGroup();
      for (Iterator<IDisplayComponent> iter = selectedObjects.iterator(); iter
            .hasNext();) {
         IDisplayComponent idc = iter.next();
         List<IDisplayComponent> groupableList = QuestionGroup
               .getGroupableComponents(idc);
         for (Iterator<IDisplayComponent> it = groupableList.iterator(); it
               .hasNext();) {
            IDisplayComponent idComponent = it.next();
            qg.addElement(idComponent);
         }
      }
      if (qg.getSize() == 0) {
         JOptionPane.showMessageDialog(parent, "Nothing to group", "Warning",
               JOptionPane.WARNING_MESSAGE);
         return;
      } else {
         // popup a dialog box to get the meta data associated with this
         // question
         /**
          * @todo popup a dialog box to get the meta data associated with this
          *       question
          */
         QuestionGroupDialog dlg = new QuestionGroupDialog(null,
               "Question Meta Data");
         dlg.setLocationRelativeTo(this.parent);
         int rc = dlg.showDialog();

         if (rc == BaseDialog.OK_PRESSED) {
            String questionType = dlg.getQuestionType();
            qg.addMetadata("type", questionType);
            if (questionType == QuestionGroupDialog.MULTI_ANSWER) {
               qg.addMetadata("minAnswer", String.valueOf(dlg.getMinAnswer()));
               qg.addMetadata("maxAnswer", String.valueOf(dlg.getMaxAnswer()));
            }
         } else {
            qg.addMetadata("type", QuestionGroupDialog.REGULAR);
         }
         dlg.dispose();
         // add the question group to the repository
         QuestionGroupRepository.getInstance().add(qg);
      }
      parent.curState.clearSelection();
   }

   protected void handleLogicalGrouping() {
      if (selectedObjects != null && !selectedObjects.isEmpty()) {
         LogicalGroup lg = LogicalGroupRepository.getInstance()
               .createLogicalGroup();
         for (Iterator<IDisplayComponent> iter = selectedObjects.iterator(); iter
               .hasNext();) {
            IDisplayComponent idc = iter.next();
            List<IDisplayComponent> groupableList = LogicalGroup
                  .getGroupableComponents(idc);
            for (Iterator<IDisplayComponent> it = groupableList.iterator(); it
                  .hasNext();) {
               IDisplayComponent idComponent = it.next();
               lg.addElement(idComponent);
            }
         }
         if (lg.getSize() == 0) {
            JOptionPane.showMessageDialog(parent, "Nothing to group",
                  "Warning", JOptionPane.WARNING_MESSAGE);
            return;
         } else {
            // add the logical group to the repository
            LogicalGroupRepository.getInstance().add(lg);
         }
         parent.curState.clearSelection();
      }
   }

   protected void handleQuestionUnGrouping() {
      if (selectedObjects == null || selectedObjects.isEmpty()) {
         parent.curState.clearSelection();
         return;
      }
      // first find all groupable objects
      Map<IDisplayComponent, IDisplayComponent> map = new LinkedHashMap<IDisplayComponent, IDisplayComponent>(
            17);
      for (Iterator<IDisplayComponent> iter = selectedObjects.iterator(); iter
            .hasNext();) {
         IDisplayComponent idc = iter.next();
         List<IDisplayComponent> groupableList = QuestionGroup
               .getGroupableComponents(idc);
         for (Iterator<IDisplayComponent> it = groupableList.iterator(); it
               .hasNext();) {
            IDisplayComponent idComp = it.next();
            map.put(idComp, idComp);
         }
      }// for
      if (map.isEmpty()) {
         parent.curState.clearSelection();
         return;
      }
      // check if all of the selected objects are in the same question group
      QuestionGroup theGroup = null;
      int size = 0;
      for (Iterator<IDisplayComponent> iter = map.keySet().iterator(); iter
            .hasNext();) {
         IDisplayComponent idc = iter.next();
         QuestionGroup lg = QuestionGroupRepository.getInstance().findMatching(
               idc);
         if (lg == null)
            continue;
         if (theGroup == null)
            theGroup = lg;
         if (lg != theGroup) {
            JOptionPane
                  .showMessageDialog(
                        parent,
                        "All selected components must belong to the same question group",
                        "Warning", JOptionPane.WARNING_MESSAGE);
            parent.curState.clearSelection();
            return;
         } else {
            size++;
         }
      }
      if (theGroup == null) {
         JOptionPane.showMessageDialog(parent, "Nothing to ungroup", "Warning",
               JOptionPane.WARNING_MESSAGE);
         parent.curState.clearSelection();
         return;
      }

      if (size == theGroup.getSize()) {
         theGroup.ungroup();
         QuestionGroupRepository.getInstance().removeGroup(theGroup.getId());
      } else {
         JOptionPane.showMessageDialog(parent,
               "You need to select all elements of a question to ungroup",
               "Warning", JOptionPane.WARNING_MESSAGE);
      }

      parent.curState.clearSelection();
   }

   protected void handleLogicalUnGrouping() {
      if (selectedObjects != null && !selectedObjects.isEmpty()) {
         // first find all groupable objects
         Map<IDisplayComponent, IDisplayComponent> map = new LinkedHashMap<IDisplayComponent, IDisplayComponent>(
               17);
         for (Iterator<IDisplayComponent> iter = selectedObjects.iterator(); iter
               .hasNext();) {
            IDisplayComponent idc = iter.next();
            List<IDisplayComponent> groupableList = LogicalGroup.getGroupableComponents(idc);
            for (Iterator<IDisplayComponent> it = groupableList.iterator(); it.hasNext();) {
               IDisplayComponent idComp = it.next();
               map.put(idComp, idComp);
            }
         }// for
         if (map.isEmpty()) {
            parent.curState.clearSelection();
            return;
         }

         // check if all of the selected objects are in a logical group
         LogicalGroup theGroup = null;
         int size = 0;
         for (Iterator<IDisplayComponent> iter = map.keySet().iterator(); iter
               .hasNext();) {
            IDisplayComponent idc = iter.next();
            LogicalGroup lg = LogicalGroupRepository.getInstance()
                  .findMatching(idc);
            if (lg == null)
               continue;
            if (theGroup == null)
               theGroup = lg;
            if (lg != theGroup) {
               JOptionPane
                     .showMessageDialog(
                           parent,
                           "All selected components must belong to the same logical group",
                           "Warning", JOptionPane.WARNING_MESSAGE);
               parent.curState.clearSelection();
               return;
            } else {
               size++;
            }
         }
         if (theGroup == null) {
            JOptionPane.showMessageDialog(parent, "Nothing to ungroup",
                  "Warning", JOptionPane.WARNING_MESSAGE);
            parent.curState.clearSelection();
            return;
         }

         if (size == theGroup.getSize()) {
            theGroup.ungroup();
            LogicalGroupRepository.getInstance().removeGroup(theGroup.getId());
         } else {
            JOptionPane.showMessageDialog(parent,
                  "You need to select all elements of a group to ungroup",
                  "Warning", JOptionPane.WARNING_MESSAGE);
         }
      }
      parent.curState.clearSelection();
   }

   protected void handleCreateTemplate() {
      if (selectedObjects != null && !selectedObjects.isEmpty()) {
         CAContainer topmost = getTopMostContainer();
         if (topmost != null) {
            if (topmost instanceof CAPanel) {
               CAPanel template = (CAPanel) topmost;
               try {
                  String templateDir = config.getTemplateDir();
                  if (new File(templateDir).isDirectory()) {
                     fileChooser.setCurrentDirectory(new File(templateDir));
                  }
                  int rc = fileChooser.showSaveDialog(parent);
                  if (rc == JFileChooser.APPROVE_OPTION) {
                     String templateFileName = fileChooser.getSelectedFile()
                           .getAbsolutePath();
                     XMLUtils.savePanel(templateFileName, template);

                     ClinicalAssessmentLayoutManager calm = (ClinicalAssessmentLayoutManager) parent
                           .getOwner();
                     calm.addTemplate(templateFileName);
                  }
               } catch (IOException iox) {
                  CALMHelper.showError(parent, iox.getMessage(),
                        "Error while saving template");
                  iox.printStackTrace();
                  log.error("", iox);
               }
            } else {
               JOptionPane.showMessageDialog(parent,
                     "Only CAPanel objects can be used as templates!",
                     "Warning", JOptionPane.WARNING_MESSAGE);
            }
         } else {
            CALMHelper.showError(parent, "Cannot find the topmost container!");
         }
         parent.curState.clearSelection();
      }
   }

   /**
    * Finds and returns the topmost container in the selected object list. Only
    * one topmost container is allowed for a template.
    *
    * @return null if there is no topmost container ( e.g. only display
    *         components) or multiple topmost containers
    */
   protected CAContainer getTopMostContainer() {
      CAContainer topMost = null;
      Map<IDisplayComponent, IDisplayComponent> icMap = new HashMap<IDisplayComponent, IDisplayComponent>();
      for (Iterator<IDisplayComponent> iter = selectedObjects.iterator(); iter
            .hasNext();) {
         IDisplayComponent ic = iter.next();
         icMap.put(ic, ic);
      }

      for (Iterator<IDisplayComponent> iter = selectedObjects.iterator(); iter
            .hasNext();) {
         IDisplayComponent ic = iter.next();
         CAContainer container = getTopmostParent(ic, icMap);
         if (container == null) {
            // every display component needs to have a topmost component
            return null;
         }
         if (topMost == null) {
            topMost = container;
         } else {
            if (topMost != container) {
               // every display component needs to have the same topmost
               // component
               return null;
            }
         }
      }
      return topMost;
   }

   protected CAContainer getTopmostParent(IDisplayComponent ic,
         Map<IDisplayComponent, IDisplayComponent> icMap) {
      IDisplayComponent parent = ic.getParent();
      if (parent == null) {
         if (ic instanceof CAContainer) {
            return (CAContainer) ic;
         } else {
            return null;
         }
      } else if (icMap.get(parent) == null) {
         if (ic instanceof CAContainer) {
            return (CAContainer) ic;
         } else {
            return null;
         }
      } else {
         if (icMap.get(parent.getParent()) != null) {
            return getTopmostParent(parent, icMap);
         } else {
            if (parent instanceof CAContainer) {
               return (CAContainer) parent;
            } else {
               return null;
            }
         }
      }
   }

}
