package caslayout.ui;

import java.util.*;

import org.jdom.*;


/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: QuestionGroup.java,v 1.9 2008/10/13 23:58:07 bozyurt Exp $
 */
public class QuestionGroup {
  protected String id;
  protected LinkedList<IDisplayComponent> elements = new LinkedList<IDisplayComponent>();
  protected Map<String, String> metaDataMap = new LinkedHashMap<String, String>(7);
  protected Map<String, Integer> scoreIDMap = new HashMap<String, Integer>();
  public final static String REGULAR = "regular";
  public final static String MULTI_ANSWER = "multi-answer";


  public QuestionGroup(String id) {
    this.id = id;
  }

  public void addElement(IDisplayComponent idc) {
    // a display component needs to be free to be in a question group
    if ( idc.getQuestionID() != null)
      return;

    idc.setQuestionID(id);
    elements.add( idc );
  }

  public String getId() { return this.id; }

  public int getSize() { return elements.size(); }

  public List<IDisplayComponent> getElements() { return elements; }

  public Map<String, Integer> getScoreIDMap() { return scoreIDMap; }

  public int getScoreID(String scoreName) {
    return scoreIDMap.get(scoreName).intValue();
  }

  public void setScoreID(String scoreName, int id) {
    scoreIDMap.put(scoreName, new Integer(id));
  }

  public void addMetadata(String name, String value) {
    metaDataMap.put(name, value);
  }

  public String getMetaDataValue(String key) {
    return metaDataMap.get(key);
  }

  public boolean isMultiAnswer() {
    String value = metaDataMap.get("type");
    return value.equals(MULTI_ANSWER);
  }

  public int getMinAnswer() {
    String value = metaDataMap.get("minAnswer");
    if (value == null)
      return 1;
    return Integer.parseInt( value);
  }

  public int getMaxAnswer() {
   String value = metaDataMap.get("maxAnswer");
   if (value == null)
     return 1;
   return Integer.parseInt( value);
 }


  public void ungroup() {
     for (Iterator<IDisplayComponent> iter = elements.iterator(); iter.hasNext(); ) {
       IDisplayComponent idc = iter.next();
       idc.setQuestionID(null);
     }
     elements.clear();
  }


  public static List<IDisplayComponent> getGroupableComponents(IDisplayComponent parent) {
   List<IDisplayComponent> groupableList = new LinkedList<IDisplayComponent>();
   getGroupableComponents(parent, groupableList);
   return groupableList;
 }


 protected static void getGroupableComponents(IDisplayComponent parent, List<IDisplayComponent> groupableList) {
    if ( parent instanceof CAContainer) {
      CAContainer con = (CAContainer) parent;
      for (IDisplayComponent element : con.getComponents()) {
        AbstractDisplayComponent child = (AbstractDisplayComponent) element;
        if ( child instanceof RadioButtonDisplayComponent ||
             child instanceof CheckBoxDisplayComponent)
        {
          groupableList.add(child);
         } else if ( child instanceof CAContainer) {
          getGroupableComponents(child, groupableList);
        }
      }
    } else {
      groupableList.add(parent);
    }
  }

  public Element toXML(Element root) {
    Element e = new Element("question");
    e.setAttribute("id", id);

    for(Map.Entry<String, String> entry : metaDataMap.entrySet()) {
      Element me = new Element("meta");
      me.setAttribute("name", (String) entry.getKey());
      me.setAttribute("value", (String) entry.getValue());
      e.addContent(me);
    }

    for (Iterator<IDisplayComponent> iter = elements.iterator(); iter.hasNext(); ) {
      AbstractDisplayComponent adc = (AbstractDisplayComponent)iter.next();
      Element ce = new Element("display-comp");
      ce.setAttribute("id", adc.getId());
      e.addContent(ce);
    }
    return e;
  }


  public static QuestionGroup initializeFromXML(Element e, Document document) {
    String id = e.getAttributeValue("id");
    QuestionGroup qg = new QuestionGroup(id);
    List<?> children = e.getChildren("display-comp");
    for (Iterator<?> iter = children.iterator(); iter.hasNext(); ) {
      Element child = (Element)iter.next();
      String refID = child.getAttributeValue("id");
      IDisplayComponent idc = document.findDisplayComponent(refID);
     if ( idc != null) {
        qg.addElement(idc);
        idc.setQuestionID(id);
     } else {
       document.handleMissingCompInQuestionGroup(refID);
     }
    }
    if ( children.isEmpty()) {
    	// an orphan question without any display elements
    	return null;
    }
    children = e.getChildren("meta");
    for (Iterator<?> iter = children.iterator(); iter.hasNext(); ) {
      Element child = (Element)iter.next();
      String name = child.getAttributeValue("name");
      String value = child.getAttributeValue("value");
      qg.addMetadata(name, value);
    }
    return qg;
  }

  public String toString() {
    StringBuffer sb = new StringBuffer(128);
    sb.append("QuestionGroup::[");
    sb.append("id=").append(id);
    if ( !scoreIDMap.isEmpty()) {
    	sb.append(",scores[");
    	for (Iterator<String> iter = scoreIDMap.keySet().iterator(); iter.hasNext();) {
    		String scoreName = iter.next();
    		sb.append(scoreName);
    		if ( iter.hasNext()) {
    			sb.append(',');
    		}
		}
    	sb.append("]");
    }
    sb.append(']');
    return sb.toString();
  }

}
