package caslayout.ui;

import guilib.AppFrame;
import guilib.common.BaseDialog;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.geom.Line2D;
import java.awt.geom.Rectangle2D;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;

import org.apache.log4j.Logger;
import org.jdom.Element;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import caslayout.propedit.PropertyEditorPanel;
import caslayout.ui.model.Association;
import caslayout.ui.model.AssociationHelper;
import caslayout.ui.model.MandatoryFieldAssociation;
import caslayout.ui.model.ScoreAssociation;
import caslayout.util.GUIUtils;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: ScoreLayoutPanel.java,v 1.38 2008/10/13 23:58:07 bozyurt Exp $
 */
public class ScoreLayoutPanel extends JPanel implements MouseListener,
      PropertyChangeListener, MouseMotionListener {
   private static final long serialVersionUID = -3106890607327801599L;
   protected Grid grid;
   protected java.util.List<CAContainer> displayComponents = new ArrayList<CAContainer>();
   protected String title;
   protected CAPanel panel;
   protected CurrentState curState;
   protected AppFrame owner;
   protected JPopupMenu popupMenu;
   protected PopupMenuHelper popupHelper;
   protected CALMConfig config;
   protected SelectedComponentInfo compForAssociation;
   protected PropertyChangeSupport pcs;
   /** bound property for labels selected for association */
   protected Object selectedLabels;
   protected String toolTipText = "";
   protected boolean showAssociationTips = true;
   /** the current state as used by the mouse event to choose proper action */
   protected int state = 0;

   PrintWriter logWriter = null;
   protected final static boolean DUMP_BOUNDARY_INFO = false;

   protected static Logger log = Logger.getLogger("ScoreLayoutPanel");

   public ScoreLayoutPanel(String title, boolean enableDebugging,
         AppFrame owner, CALMConfig config) {
      super(true);
      pcs = new PropertyChangeSupport(this);
      this.title = title;
      this.owner = owner;
      curState = new CurrentState();

      setBackground(Color.white);
      setLayout(null);
      grid = new Grid(10, 10);
      this.addMouseListener(this);
      this.addMouseMotionListener(this);

      this.setToolTipText(this.toolTipText);

      CAGridLayout gl = new CAGridLayout(1, 1, 3, 3, false);

      panel = new CAPanel(gl, this.getInsets().left, this.getInsets().top, this
            .getWidth()
            - getInsets().left - getInsets().right, this.getHeight()
            - getInsets().top - getInsets().bottom);
      displayComponents.add(panel);
      popupHelper = new PopupMenuHelper(this, config);
      preparePopupMenu();
      if (DUMP_BOUNDARY_INFO) {
         try {
            logWriter = new PrintWriter(new FileWriter(System
                  .getProperty("user.home")
                  + File.separator + ".calm_boundaries.log"), true);
         } catch (Exception x) {
            x.printStackTrace();
            logWriter = null;
         }
      }
   }

   public CAPanel createDefaultPanel(boolean usePercentages) {
      CAGridLayout gl = new CAGridLayout(1, 1, 4, 4, usePercentages);
      CAPanel panel = new CAPanel(gl, this.getInsets().left,
            this.getInsets().top, this.getWidth() - getInsets().left
                  - getInsets().right, this.getHeight() - getInsets().top
                  - getInsets().bottom);

      return panel;
   }

   public CAPanel createChildPanel(ILayoutManager lm, boolean usePercentages) {
      CAPanel panel = new CAPanel(lm, this.getInsets().left,
            this.getInsets().top, this.getWidth() - getInsets().left
                  - getInsets().right, this.getHeight() - getInsets().top
                  - getInsets().bottom);

      return panel;
   }

   public void setTitle(String newTitle) {
      this.title = newTitle;
   }

   public String getTitle() {
      return this.title;
   }

   public void setSelectedComponent(IDisplayComponent newSelectedComponent) {
      curState.setSelectedComponent(newSelectedComponent);
      // log.info(" new selected component:" + newSelectedComponent);
   }

   public AppFrame getOwner() {
      return owner;
   }

   public IDisplayComponent getSelectedComponent() {
      return curState.getSelectedComponent();
   }

   public CurrentState getCurrentState() {
      return curState;
   }

   public void saveLayout(String filename) throws IOException {
      XMLOutputter xmlOut = null;
      BufferedWriter out = null;
      try {
         out = new BufferedWriter(new FileWriter(filename));
         xmlOut = new XMLOutputter(Format.getPrettyFormat());
         Element rootElem = new Element("caslayout");
         rootElem.addContent(panel.toXML(rootElem));
         xmlOut.output(rootElem, out);
      } finally {
         if (out != null)
            try {
               out.close();
            } catch (Exception x) {}
      }
   }

   protected void setPeerForChildren(AbstractDisplayComponent adc,
         Component peer) {
      if (adc == null)
         return;
      if (adc instanceof CAContainer) {
         CAContainer con = (CAContainer) adc;
         for (Iterator<IDisplayComponent> iter = con.components.iterator(); iter
               .hasNext();) {
            AbstractDisplayComponent child = (AbstractDisplayComponent) iter
                  .next();
            if (child != null)
               setPeerForChildren(child, peer);
         }
      } else {
         adc.peer = peer;
      }
   }

   public void showAssociationTips(boolean value) {
      this.showAssociationTips = value;
   }

   /**
    * toggles associated score hint in the display components.
    *
    * @param value
    *           if true show associated score names in the display components
    */
   public void showAssociations(boolean value) {
      // this.showAssocs = value;
      for (Iterator<CAContainer> iter = displayComponents.iterator(); iter
            .hasNext();) {
         IDisplayComponent ic = iter.next();
         if (ic != null)
            showAssociations(value, ic);
      }
   }

   protected void showAssociations(boolean value, IDisplayComponent ic) {
      if (ic == null)
         return;
      if (ic instanceof CAContainer) {
         CAContainer con = (CAContainer) ic;
         for (IDisplayComponent idc : con.getComponents()) {
            if (idc != null) {
               showAssociations(value, idc);
            }
         }
      } else {
         ic.setShowAssociations(value);
      }
   }

   public void showQuestions(boolean value) {
      for (Iterator<CAContainer> iter = displayComponents.iterator(); iter
            .hasNext();) {
         IDisplayComponent ic = iter.next();
         if (ic != null)
            showQuestions(value, ic);
      }
   }

   protected void showQuestions(boolean value, IDisplayComponent ic) {
      if (ic == null)
         return;
      if (ic instanceof CAContainer) {
         CAContainer con = (CAContainer) ic;
         for (IDisplayComponent idc : con.getComponents()) {
            if (idc != null) {
               showQuestions(value, idc);
            }
         }
      } else {
         ic.setShowQuestions(value);
      }
   }

   public void addPropertyChangeListener(PropertyChangeListener pcl) {
      pcs.addPropertyChangeListener(pcl);
   }

   public void removePropertyChangeListener(PropertyChangeListener pcl) {
      pcs.removePropertyChangeListener(pcl);
   }

   /**
    * bound property
    *
    * @param newCompForAssociation
    */
   public void setCompForAssociation(SelectedComponentInfo newCompForAssociation) {
      SelectedComponentInfo oldValue = this.compForAssociation;
      this.compForAssociation = newCompForAssociation;
      pcs.firePropertyChange("compForAssociation", oldValue,
            newCompForAssociation);
   }

   public SelectedComponentInfo getCompForAssociation() {
      return this.compForAssociation;
   }

   // bound property
   public void setSelectedLabels(Object newLabels) {
      Object oldValue = this.selectedLabels;
      this.selectedLabels = newLabels;
      pcs.firePropertyChange("selectedLabels", oldValue, newLabels);
   }

   public Object getSelectedLabels() {
      return this.selectedLabels;
   }

   public void propertyChange(PropertyChangeEvent pce) {

      if (pce.getPropertyName().equals("selectedModelData")) {
         this.state = CurrentState.ASSOCIATE;
         // log.info("In ASSOCIATE mode");
      } else if (pce.getPropertyName().equals("associateLabels")) {
         this.state = CurrentState.ASSOCIATE_LABELS;
         // log.info("In ASSOCIATE LABELS mode");
      } else {
         this.state = CurrentState.NONE;
         invalidate();
         repaint();
      }
   }

   public int getDisplayComponentCount() {
      return displayComponents.size();
   }

   public CAContainer getCAContainerAt(int idx) {
      if (idx < 0 || idx > displayComponents.size())
         return null;
      return displayComponents.get(idx);
   }

   public void setCAContainerAt(CAContainer container, int idx) {
      displayComponents.set(idx, container);
   }

   public String getToolTipText(MouseEvent me) {
      if (!showAssociationTips) {
         return null;
      }
      IDisplayComponent ic = componentHitTest(me.getX(), me.getY(), 2);

      if (ic != null) {
         Association assoc = AssociationHelper.getInstance()
               .findScoreAssociationForDisplayComponent(ic);
         if (assoc != null) {
            this.toolTipText = prepareToolTipText(assoc);
            return this.toolTipText;
         }
      }
      this.toolTipText = null;
      return null;
   }

   private String prepareToolTipText(Association assoc) {
      StringBuffer buf = new StringBuffer();
      if (assoc instanceof ScoreAssociation) {
         ScoreAssociation sa = (ScoreAssociation) assoc;
         buf.append("associated with '");
         buf.append(sa.getLeft().getScoreName()).append('\'');
      } else {
         MandatoryFieldAssociation mfa = (MandatoryFieldAssociation) assoc;
         buf.append("associated with mandatory field '");
         buf.append(mfa.getLeft()).append('\'');
      }

      if (buf.length() == 0) {
         return null;
      }
      return buf.toString();
   }

   public void paintComponent(Graphics g) {
      Graphics2D g2 = (Graphics2D) g;
      g2.setColor(Color.white);
      Rectangle r = getBounds();
      Insets insets = getInsets();

      r.x = insets.left;
      r.y = insets.top;
      r.height -= (insets.top + insets.bottom);
      r.width -= (insets.left + insets.right);

      g2.fillRect(r.x, r.y, r.width, r.height);

      panel.setX(r.x);
      panel.setY(r.y);
      panel.setWidth(r.width);
      panel.setHeight(r.height);
      if (log.isDebugEnabled()) {
         log.debug(">> r = " + r);
      }

      Rectangle2D panelBounds = new Rectangle2D.Double(r.x, r.y, r.width,
            r.height);
      panel.setBounds(panelBounds);

      /*
       * no grid is shown anymore if ( grid != null) { grid.draw(g2, r); }
       */

      if (DUMP_BOUNDARY_INFO) {
         dumpComponentBounds(g2, logWriter, true);
      }
      Rectangle2D maxBound = getMaxBound(g2);
      double scaleX = (maxBound.getWidth() > 0) ? panelBounds.getWidth()
            / maxBound.getWidth() : 0;
      double scaleY = (maxBound.getHeight() > 0) ? panelBounds.getHeight()
            / maxBound.getHeight() : 0;

      boolean doScale = (scaleX > 0 && scaleY > 0)
            && (scaleX != 1.0 && scaleY != 1.0);

      g2.setColor(Color.black);
      for (Iterator<CAContainer> iter = displayComponents.iterator(); iter
            .hasNext();) {
         IDisplayComponent ic = iter.next();
         if (grid != null) {
            Rectangle2D b = ic.getBounds(g2);
            if (b == null) {
               b = new Rectangle2D.Double(8, 8, r.width - 8, r.height - 8);
            }
            if (b != null) {
               if (doScale) {
                  if (log.isDebugEnabled()) {
                     log.debug("scaleX=" + scaleX + "scaleY=" + scaleY);
                  }
                  // orig
                  scale(scaleX, scaleY, b);
               }

               if (!(ic instanceof CAContainer)) {
                  grid.snap(b);
               }
               ic.setBounds(b);
            }
         }
         ic.draw(g2);
      }
   }

   protected void scale(double scaleX, double scaleY, Rectangle2D bounds) {
      double sw = bounds.getWidth() * scaleX;
      double sh = bounds.getHeight() * scaleY;
      double sx = bounds.getX() * scaleX;
      double sy = bounds.getY() * scaleY;
      bounds.setRect(sx, sy, sw, sh);
   }

   protected void scale(Graphics2D g2, double scaleX, double scaleY,
         Rectangle2D bounds, IDisplayComponent ic) {
      double sw = bounds.getWidth() * scaleX;
      double sh = bounds.getHeight() * scaleY;
      double sx = bounds.getX() * scaleX;
      double sy = bounds.getY() * scaleY;
      bounds.setRect(sx, sy, sw, sh);
      if (ic instanceof CAContainer) {
         CAContainer con = (CAContainer) ic;
         for (IDisplayComponent item : con.getComponents()) {
            if (item == null)
               continue;
            Rectangle2D r = item.getBounds(g2);
            scale(g2, scaleX, scaleY, r, item);
         }
      }
   }

   /**
    * calculates and returns the dimension of the Rectangle that will contain
    * all the display components at the size they are persisted
    *
    * @param g2
    *           Graphichs2D
    * @return
    */
   protected Rectangle2D getMaxBound(Graphics2D g2) {
      double width = 0, height = 0;
      for (Iterator<CAContainer> iter = displayComponents.iterator(); iter
            .hasNext();) {
         IDisplayComponent ic = iter.next();
         Rectangle2D r = ic.getBounds(g2);

         double w = r.getX() + r.getWidth();
         double h = r.getY() + r.getHeight();
         if (width < w) {
            width = w;
         }
         if (height < h) {
            height = h;
         }
      }

      return new Rectangle2D.Double(0, 0, width, height);
   }

   protected void findContainerBound(CAContainer container, Graphics2D g2,
         Rectangle2D br) {
      for (IDisplayComponent ic : container.getComponents()) {
         if (ic == null)
            continue;
         Rectangle2D r = ic.getBounds(g2);
         if (ic instanceof CAContainer) {
            findContainerBound((CAContainer) ic, g2, br);
         } else {
            double w = r.getX() + r.getWidth();
            double h = r.getY() + r.getHeight();
            if (br.getWidth() < w) {
               br.setRect(br.getX(), br.getY(), w, br.getHeight());
            }
            if (br.getHeight() < h) {
               br.setRect(br.getX(), br.getY(), br.getWidth(), h);
            }
         }
      }

   }

   protected void dumpComponentBounds(Graphics2D g2, PrintWriter out,
         boolean showBanner) {
      if (out == null)
         return;
      if (showBanner) {
         out.println("\n------------------------------------------------");
      }
      for (Iterator<CAContainer> iter = displayComponents.iterator(); iter
            .hasNext();) {
         IDisplayComponent ic = iter.next();
         Rectangle2D r = ic.getBounds(g2);
         String pid = ic.getParent() != null ? ((AbstractDisplayComponent) ic
               .getParent()).getId() : "none";
         out.println("ID=" + ic.getId() + " PID=" + pid + " x:" + r.getX()
               + " y:" + r.getY() + " w:" + r.getWidth() + " h:"
               + r.getHeight());
         if (ic instanceof CAContainer) {
            CAContainer con = (CAContainer) ic;
            for (IDisplayComponent item : con.getComponents()) {
               if (item == null)
                  continue;
               dumpComponentBounds(item, g2, out);

            }
         }

      }
   }

   protected void dumpComponentBounds(IDisplayComponent ic, Graphics2D g2,
         PrintWriter out) {
      Rectangle2D r = ic.getBounds(g2);
      String pid = ic.getParent() != null ? ((AbstractDisplayComponent) ic
            .getParent()).getId() : "none";
      out.println("ID=" + ic.getId() + " PID=" + pid + " x:" + r.getX() + " y:"
            + r.getY() + " w:" + r.getWidth() + " h:" + r.getHeight());
      if (ic instanceof CAContainer) {
         CAContainer con = (CAContainer) ic;
         for (IDisplayComponent item : con.getComponents()) {
            if (item == null)
               continue;
            dumpComponentBounds(item, g2, out);
         }
      }
   }

   class Result {
      CAContainer container;
      Rectangle2D minBounds;

      public Result(CAContainer container, Rectangle2D minBounds) {
         this.container = container;
         this.minBounds = minBounds;
      }
   }

   class CHTResult {
      CAContainer container;
      IDisplayComponent ic;

      public CHTResult(CAContainer cont) {
         this.container = cont;
      }
   }

   protected IDisplayComponent componentHitTest(int mX, int mY, int width) {
      Graphics2D g2 = (Graphics2D) getGraphics();
      width = Math.max(width, 2);
      CHTResult result = new CHTResult(null);
      Rectangle2D hitRect = new Rectangle2D.Double(mX - width / 2, mY - width
            / 2, width, width);
      for (Iterator<CAContainer> iter = displayComponents.iterator(); iter
            .hasNext();) {
         IDisplayComponent ic = iter.next();
         if (ic == null)
            continue;
         if (ic instanceof CAContainer) {
            CAContainer con = (CAContainer) ic;
            result.container = con;
            componentHitTest(g2, hitRect, result);
            if (result.ic != null) {
               return result.ic;
            }
         } else {
            if (ic.contains(hitRect, g2)) {
               return ic;
            }
         }
      }
      return null;
   }

   void componentHitTest(Graphics2D g2, Rectangle2D hitRect, CHTResult result) {
      for (IDisplayComponent ic : result.container.getComponents()) {
         if (ic == null)
            continue;
         if (ic instanceof CAContainer) {
            CAContainer child = (CAContainer) ic;
            result.container = child;
            componentHitTest(g2, hitRect, result);
         } else {
            if (ic.contains(hitRect, g2)) {
               result.ic = ic;
               return;
            }
         }
      }
   }

   /**
    * Finds the smallest container that contains the mouse click location
    * recursively going down the container hierarchy.
    *
    * @param mX
    * @param mY
    * @return
    */
   protected CAContainer containerHitTest(int mX, int mY) {
      Graphics2D g2 = (Graphics2D) getGraphics();
      Rectangle2D minBounds = new Rectangle2D.Double(0, 0, Integer.MAX_VALUE,
            Integer.MAX_VALUE);
      CAContainer theContainer = null;
      Result result = new Result(theContainer, minBounds);
      for (Iterator<CAContainer> iter = displayComponents.iterator(); iter
            .hasNext();) {
         IDisplayComponent ic = iter.next();
         if (ic instanceof CAContainer) {
            CAContainer con = (CAContainer) ic;
            Rectangle2D b = con.getBounds(g2);
            if (b.contains(mX, mY) && minBounds.contains(b)) {
               minBounds = b;
               theContainer = con;
               result = new Result(theContainer, minBounds);

               containerHitTest(mX, mY, con, result);
            }
         }
      }
      return result.container;
   }

   protected void containerHitTest(int mX, int mY, CAContainer parent,
         Result result) {
      Graphics2D g2 = (Graphics2D) getGraphics();
      for (IDisplayComponent ic : parent.getComponents()) {
         if (ic instanceof CAContainer) {
            CAContainer con = (CAContainer) ic;
            Rectangle2D b = con.getBounds(g2);
            if (b.contains(mX, mY) && result.minBounds.contains(b)) {
               result.minBounds = b;
               result.container = con;

               containerHitTest(mX, mY, con, result);
            }
         }
      }
   }

   public void preparePopupMenu() {
      popupMenu = new JPopupMenu("Operate on Selection");
      JMenuItem menuItem = new JMenuItem("Create a template");
      menuItem.addActionListener(popupHelper);
      popupMenu.add(menuItem);

      menuItem = new JMenuItem("Delete Selection");
      menuItem.addActionListener(popupHelper);
      popupMenu.add(menuItem);

      menuItem = new JMenuItem("Copy Selection");
      menuItem.addActionListener(popupHelper);
      popupMenu.add(menuItem);

      menuItem = new JMenuItem("Logically Group Selection");
      menuItem.addActionListener(popupHelper);
      popupMenu.add(menuItem);
      menuItem = new JMenuItem("UnGroup Selection");
      menuItem.addActionListener(popupHelper);
      popupMenu.add(menuItem);

      menuItem = new JMenuItem("Group Selection as a Question");
      menuItem.addActionListener(popupHelper);
      popupMenu.add(menuItem);
      menuItem = new JMenuItem("Ungroup Selected Question");
      menuItem.addActionListener(popupHelper);
      popupMenu.add(menuItem);
   }

   protected void enableDisablePopupMenuItems(boolean[] enabledArr) {
      for (int i = 0; i < enabledArr.length; i++) {
         ((JMenuItem) popupMenu.getComponent(i)).setEnabled(enabledArr[i]);
      }
   }

   protected boolean showPopup(MouseEvent me) {
      if (me.isPopupTrigger()) {
         if (curState.command == null) {

            // check if the cursor is over a display element which is not a
            // container
            // in that case show popup menu
            CAContainer con = containerHitTest(me.getX(), me.getY());
            if (con == null)
               return false;

            Graphics2D g2 = (Graphics2D) getGraphics();
            Object location = con.hitTest(me.getX(), me.getY(), g2);
            if (location != null && location instanceof GridLocation) {
               GridLocation gl = (GridLocation) location;

               IDisplayComponent idc = null;
               if (gl.idx < con.getComponents().size()) {
                  idc = (IDisplayComponent) con.getComponents().get(gl.idx);
               }
               if (idc == null)
                  return false;
               //
               enableDisablePopupMenuItems(new boolean[] { false, true, true,
                     true, true, true, true });

               popupHelper.setSelectedDisplayComponent(idc);
               popupMenu.show(me.getComponent(), me.getX(), me.getY());
            }
            return true;
         }
         if (curState.command instanceof SelectionCommand) {
            SelectionCommand sc = (SelectionCommand) curState.command;

            ((JMenuItem) popupMenu.getComponent(0)).setEnabled(!sc
                  .getSelectedObjects().isEmpty());

            popupMenu.show(me.getComponent(), me.getX(), me.getY());
            return true;
         }
      }
      return false;
   }

   /**
    * single left mouse click
    *
    * @param x
    * @param y
    * @return true if the selected component is edited
    */
   protected boolean handleQuickEdit(int x, int y) {
      // log.info("in handleQuickEdit");
      CAContainer con = containerHitTest(x, y);
      if (con == null)
         return false;
      Graphics2D g2 = (Graphics2D) getGraphics();
      Object location = con.hitTest(x, y, g2);
      if (location == null || !(location instanceof GridLocation)) {
         return false;
      }
      GridLocation gl = (GridLocation) location;
      IDisplayComponent idc = null;
      if (gl.idx < con.getComponents().size()) {
         idc = (IDisplayComponent) con.getComponents().get(gl.idx);
      }
      if (idc == null)
         return false;
      if (idc instanceof TextDisplayComponent) {
         // log.info("handleQuickEdit TextDisplayComponent");
         TextDisplayComponent tdc = (TextDisplayComponent) idc;
         MultilineStringEditor mse = new MultilineStringEditor();
         // set the text of the textdisplay component so that the customizer
         // can
         // edit it
         mse.setValue(tdc.getLabel());
         Component comp = mse.getCustomEditor();
         GUIUtils.handleMessageDialog(comp, "Multi-line Text", x, y);

         repaint();
         return true;
      }
      return false;
   }

   protected List<CAContainer> getContainerHierarchy(CAContainer con) {
      List<CAContainer> list = new ArrayList<CAContainer>();
      list.add(con);
      CAContainer parent = (CAContainer) con.getParent();
      while (parent != null) {
         list.add(parent);
         parent = (CAContainer) parent.getParent();
      }
      return list;
   }

   protected void handleDoubleClick(int x, int y, boolean controlPressed) {
      CAContainer con = containerHitTest(x, y);
      if (con == null)
         return;
      Graphics2D g2 = (Graphics2D) getGraphics();
      Object location = con.hitTest(x, y, g2);
      if (location == null || !(location instanceof GridLocation)) {
         return;
      }
      GridLocation gl = (GridLocation) location;
      IDisplayComponent idc = null;
      if (gl.idx < con.getComponents().size()) {
         idc = (IDisplayComponent) con.getComponents().get(gl.idx);
      }

      try {
         PropertyEditorPanel pePanel = null;
         if (idc == null || controlPressed) {
            List<CAContainer> conHierarchy = getContainerHierarchy(con);
            ContainerSelectionDialog csDlg = new ContainerSelectionDialog(
                  this.owner, "Select Container", conHierarchy);
            csDlg.setLocation(x, y);
            int rc = csDlg.showDialog();
            if (rc == BaseDialog.OK_PRESSED) {
               con = csDlg.getSelectedContainer();
               // show the Grid Layout Customization
               CAGridLayoutCustomizer customizer = new CAGridLayoutCustomizer();
               customizer.setObject(((CAPanel) con).getLayoutMan());
               customizer.setContainer(con);
               customizer.init();

               CommonDialog dlg = new CommonDialog(this.owner, "Edit Layout",
                     customizer);
               dlg.setLocation(x, y);
               rc = dlg.showDialog();
               if (rc == BaseDialog.OK_PRESSED) {
                  repaint();
               }
               dlg.dispose();
            }
            csDlg.dispose();
         } else {
            // a display component is clicked, just show the properties
            if (idc instanceof DropdownDisplayComponent) {
               DropdownDisplayComponent ddc = (DropdownDisplayComponent) idc;
               DropdownEditDialog dlg = null;
               if (ddc.getPopulatedFromDatabase() == true) {
                  dlg = new DynamicDropdownEditDialog(this.owner,
                        "Edit Dynamic Dropdown", ddc);
               } else {
                  dlg = new DropdownEditDialog(this.owner, "Edit dropdown", ddc);
               }
               dlg.setLocation(x, y);
               int rc = dlg.showDialog();
               if (rc == BaseDialog.OK_PRESSED) {
                  repaint();
               }
               dlg.dispose();
            } else {
               pePanel = new PropertyEditorPanel(idc);
               PropertyDialog propDlg = new PropertyDialog(this.owner, "",
                     pePanel);
               propDlg.setLocation(x, y);

               int rc = propDlg.showDialog();
               if (rc == BaseDialog.OK_PRESSED) {
                  repaint();
               }
               propDlg.dispose();
            }
         }
      } catch (Exception e) {
         e.printStackTrace();
      }
   }

   protected void handleSelectForAssociation(int x, int y) {
      CAContainer con = containerHitTest(x, y);
      if (con == null)
         return;
      Graphics2D g2 = (Graphics2D) getGraphics();
      Object location = con.hitTest(x, y, g2);
      if (location != null && location instanceof GridLocation) {
         GridLocation gl = (GridLocation) location;
         IDisplayComponent idc = null;
         if (gl.idx < con.getComponents().size()) {
            idc = (IDisplayComponent) con.getComponents().get(gl.idx);
         }
         if (idc != null) {
            SelectedComponentInfo sci = new SelectedComponentInfo(
                  (AbstractDisplayComponent) idc);
            this.state = CurrentState.NONE;
            setCompForAssociation(sci);
         }
      }
   }

   protected void clearState(boolean pasteLastComp) {
      if (curState.getMultiplePaste() && pasteLastComp) {
         curState.setMultiplePaste(false);
      }
      if (!curState.getMultiplePaste()) {
         curState.setObjectUsed(true);
         curState.clearSelection();
         // since we are not in multiselect mode cleanup the reference to the
         // 'selected' component
         setSelectedComponent(null);
      }
   }

   public void mouseClicked(MouseEvent e) {
      // log.info(" mouseClicked:: x= "+ e.getX() + " y="+ e.getY());

      boolean controlPressed = (e.getModifiersEx() & MouseEvent.CTRL_DOWN_MASK) == MouseEvent.CTRL_DOWN_MASK;
      boolean shiftPressed = (e.getModifiersEx() & MouseEvent.SHIFT_DOWN_MASK) == MouseEvent.SHIFT_DOWN_MASK;

      if (e.getClickCount() == 2 && !shiftPressed) {
         // log.info("handling double click");
         handleDoubleClick(e.getX(), e.getY(), controlPressed);
         return;
      }

      if (state == CurrentState.ASSOCIATE) {
         handleSelectForAssociation(e.getX(), e.getY());
         return;
      }

      if (controlPressed && shiftPressed) {
         if (e.getClickCount() == 1) {
            if (handleQuickEdit(e.getX(), e.getY())) {
               return;
            }
         }
      }

      if (controlPressed && !curState.getMultiplePaste()) {
         // allow multiple paste
         curState.setMultiplePaste(true);
         // log.info("*** multiple paste is set");
      }

      IDisplayComponent selectedComp = getSelectedComponent();

      if (selectedComp == null) {
         log.info("nothing is selected");
         curState.setMultiplePaste(false);
         return;
      }

      boolean pasteLastComp = false;
      if (curState.getMultiplePaste() && !controlPressed) {
         // cleanup state
         pasteLastComp = true;
         clearState(true);
         repaint();
         return;
      }

      if (!(selectedComp instanceof CAContainer)) {
         // a display component is selected before, this click operation will
         // add it to the pointed location
         CAContainer con = containerHitTest(e.getX(), e.getY());
         if (con != null) {
            Graphics2D g2 = (Graphics2D) getGraphics();
            Object location = con.hitTest(e.getX(), e.getY(), g2);
            if (location != null && location instanceof GridLocation) {
               GridLocation gl = (GridLocation) location;
               // log.info("%%%%% Adding display component:" + gl);
               Dimension dim = selectedComp.getPreferredSize();
               Rectangle2D ab = new Rectangle2D.Double(e.getX(), e.getY(), dim
                     .getWidth(), dim.getHeight());
               selectedComp.setBounds(ab);

               con.add(selectedComp, gl.getIdx());
               clearState(pasteLastComp);
               repaint();
            }
         }
      } else {
         CAContainer con = containerHitTest(e.getX(), e.getY());
         if (con != null) {
            Graphics2D g2 = (Graphics2D) getGraphics();
            Object location = con.hitTest(e.getX(), e.getY(), g2);
            if (location != null && location instanceof GridLocation) {
               GridLocation gl = (GridLocation) location;
               // log.info(gl);
               CAContainer selCon = (CAContainer) selectedComp;
               Rectangle2D ab = gl.getBounds();

               Insets is = selCon.getInsets();
               Rectangle2D ib = new Rectangle2D.Double(is.left, is.top, 0, 0);
               ab.add(ib);
               selectedComp.setBounds(ab);
               con.add(selectedComp, gl.getIdx());
               clearState(pasteLastComp);
               repaint();
            }
         }
      }
   }

   public void mousePressed(MouseEvent e) {

      if (curState.getState() != CurrentState.SELECTION) {
         boolean shown = showPopup(e);
         if (!shown) {
            boolean ctrlPressed = (e.getModifiersEx() & MouseEvent.CTRL_DOWN_MASK) == MouseEvent.CTRL_DOWN_MASK;
            if (ctrlPressed) {
               handleGridResize(e.getX(), e.getY());
            }
         }
         return;
      }
      curState.startPos = new Point(e.getX(), e.getY());
      Graphics2D g2 = (Graphics2D) getGraphics();
      curState.origColor = g2.getColor();
   }

   protected void handleGridResize(int mX, int mY) {
      CAContainer con = containerHitTest(mX, mY);
      if (con != null) {
         Graphics2D g2 = (Graphics2D) getGraphics();
         // TODO assumption only CAGridLayout manager is used relax later!
         CAGridLayout layoutMan = (CAGridLayout) con.getLayoutManager();
         GridLocation gridLoc = layoutMan.layoutGridHitTest(mX, mY, con, g2);
         System.out.println("gridLoc=" + gridLoc);
         if (gridLoc == null) {
            return;
         }
         int selRow = gridLoc.getRow();
         int selCol = gridLoc.getColumn();

         CellConstraint cc = layoutMan.getCellConstraint(selRow, selCol);
         if (cc instanceof PercentCellConstraint) {
            curState.setGridSelState(new CurrentState.GridSelectState(mX, mY,
                  gridLoc, con));
         }
      }
   }

   protected Rectangle2D prepareSelectionRect(Point start, Point end) {
      double width = Math.abs(end.x - start.x);
      double height = Math.abs(end.y - start.y);
      // find the left upper hand corner
      Point left = new Point(start.x, start.y);
      if (left.x > end.x)
         left.x = end.x;
      if (left.y > end.y)
         left.y = end.y;
      return new Rectangle2D.Double(left.x, left.y, width, height);
   }

   public void mouseReleased(MouseEvent e) {
      // log.info(">> mouseReleased");
      curState.setGridSelState(null);

      if (curState.getState() != CurrentState.SELECTION) {
         showPopup(e);
         return;
      }
      Graphics2D g2 = (Graphics2D) getGraphics();
      Rectangle2D rect = null;
      if (curState.endPos != null) {
         g2.setXORMode(Color.yellow);

         drawRect(g2, curState.startPos.x, curState.startPos.y,
               curState.endPos.x, curState.endPos.y);
         rect = prepareSelectionRect(curState.startPos, curState.endPos);
      }
      SelectionCommand cmd = (SelectionCommand) curState.command;
      if (cmd == null) {
         cmd = new SelectionCommand(this, rect);
         curState.setCommand(cmd);
      } else {
         cmd.setRect(rect);
      }
      try {
         // log.info(">> mouseReleased: cmd.execute");
         cmd.execute();
      } catch (CommandException ce) {
         ce.printStackTrace();
      }
      g2.setPaintMode();
      curState.startPos = null;
      curState.endPos = null;
      curState.setState(CurrentState.NONE);

      // the user wants to use a selection of text fields to populate score
      // codes
      if (ScoreLayoutPanel.this.state == CurrentState.ASSOCIATE_LABELS) {
         if (!cmd.getSelectedObjects().isEmpty()) {
            ArrayList<IDisplayComponent> selectedObjects = new ArrayList<IDisplayComponent>(
                  cmd.getSelectedObjects());
            ScoreLayoutPanel.this.setSelectedLabels(selectedObjects);

            ScoreLayoutPanel.this.state = CurrentState.NONE;
            // clear the selection ?

         }
      } else { //
         popupHelper.setSelectedObjects(cmd.getSelectedObjects());
      }
   }

   public void mouseEntered(MouseEvent e) {
   // no-op
   }

   public void mouseExited(MouseEvent e) {
   // no-op
   }

   protected void drawRect(Graphics2D g2, int x1, int y1, int x2, int y2) {
      g2.draw(new Line2D.Double(x1, y1, x2, y1));
      g2.draw(new Line2D.Double(x1, y2, x2, y2));
      g2.draw(new Line2D.Double(x1, y1, x1, y2));
      g2.draw(new Line2D.Double(x2, y1, x2, y2));
   }

   public void mouseDragged(MouseEvent e) {
      if (curState.getState() != CurrentState.SELECTION) {
         if (curState.getGridSelState() != null) {
            CurrentState.GridSelectState gls = curState.getGridSelState();
            int deltaX = e.getX() - gls.getStartX();
            int deltaY = e.getY() - gls.getStartY();
            if (deltaX != 0 || deltaY != 0) {
               if (gls.getSelected() != null) {
                  // TODO assumption only CAGridLayout manager is used relax
                  // later!
                  CAGridLayout layoutMan = (CAGridLayout) gls.getSelected()
                        .getLayoutManager();

                  boolean changed = layoutMan.calcNewCellConstraints(gls
                        .getGridLoc(), gls.getSelected(), deltaX, deltaY);
                  if (changed) {
                     // System.out.println("changed.");
                     repaint();
                  }
                  gls.setStartX(e.getX());
                  gls.setStartY(e.getY());
               }
            }
         }
         return;
      }
      if (curState.startPos != null) {
         Graphics2D g2 = (Graphics2D) getGraphics();
         g2.setXORMode(Color.yellow);

         if (curState.endPos != null) {
            drawRect(g2, curState.startPos.x, curState.startPos.y,
                  curState.endPos.x, curState.endPos.y);
         }

         curState.endPos = new Point(e.getX(), e.getY());
         drawRect(g2, curState.startPos.x, curState.startPos.y,
               curState.endPos.x, curState.endPos.y);
      }
   }

   public void mouseMoved(MouseEvent e) {
   // no-op
   }

}// ;
