package caslayout.ui;

import java.awt.Font;
import java.util.Iterator;
import java.util.LinkedList;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: StyledTextIterator.java,v 1.6 2008/10/13 23:58:07 bozyurt Exp $
 */

public class StyledTextIterator implements java.util.Iterator<StyledText> {
  protected java.util.List<StyledText> lineParts = new LinkedList<StyledText>();
  protected Iterator<StyledText> iterator;
  protected Font defaultFont;
  protected String lastToken = null;
  protected int idx = 0;
  public final static int NONE = 0;
  public final static int BOLD = 1;
  public final static int ITALIC = 2;
  public final static int UNDERLINE = 3;



    public StyledTextIterator(String line, Font defaultFont) {
       this.defaultFont = defaultFont;
       parse(line);
       iterator = lineParts.iterator();
    }

    protected void parse(String line) {
      char[] lineBuf = line.toCharArray();
      String token;
      int style = NONE;
      do {
        token = getToken(lineBuf);
        if ( token == null)
           break;
        if ( token.equalsIgnoreCase("<b>") ) {
           style = BOLD;
        } else if ( token.equalsIgnoreCase("<i>")) {
           style = ITALIC;
        } else if ( token.equalsIgnoreCase("<u>")) {
            style = UNDERLINE;
        } else {
          String textPart = null;
          Font f = null;
          switch(style) {
            case BOLD:
              textPart = parseTextPart(token, lineBuf, "</b>");
              f = new Font(defaultFont.getName(), Font.BOLD, defaultFont.getSize() );
              lineParts.add( new StyledText( textPart, f, StyledText.NONE) );
              break;
            case ITALIC:
              textPart = parseTextPart(token, lineBuf, "</i>");
              f = new Font(defaultFont.getName(), Font.ITALIC, defaultFont.getSize() );
              lineParts.add( new StyledText( textPart, f, StyledText.NONE) );
              break;
            case UNDERLINE:
              textPart = parseTextPart(token, lineBuf, "</u>");
              f = new Font(defaultFont.getName(), Font.PLAIN, defaultFont.getSize() );

              lineParts.add( new StyledText( textPart, f, StyledText.UNDERLINE) );
              break;
            default:
              textPart = parseTextPart(token, lineBuf, null);
              lineParts.add( new StyledText( textPart, defaultFont, StyledText.NONE) );
          }
          style = NONE;
        }
      } while ( token != null);
      if ( !lineParts.isEmpty() ) {
         // trim the extra white space at the last styled text part
         StyledText st = lineParts.get( lineParts.size() - 1);
         st.text = st.getText().trim();
      }
    }

   protected String parseTextPart(String token, char[] lineBuf, String endToken) {
     StringBuffer part = new StringBuffer();
     part.append(token);

     token = getToken(lineBuf);
     while (token != null &&
              ( (endToken == null && !hasStyle(token)) ||
                ( endToken != null && !endToken.equalsIgnoreCase(token) ) ) )
       {
         part.append(' ').append(token);
         token = getToken(lineBuf);

       }
     if ( token != null && endToken == null) {
       // push the last token back
       pushBack(token);
     }
     part.append(' ');
     return part.toString();
   }
   protected void pushBack(String token) {
     lastToken = token;
   }

    protected boolean hasStyle(String token) {
      return token.equalsIgnoreCase("<b>") ||
           token.equalsIgnoreCase("<i>") ||
           token.equalsIgnoreCase("<u>");
    }

    protected String getToken(char[] lineBuf) {
      if ( lastToken != null) {
        String tok = lastToken;
        lastToken = null;
        return tok;
      }
      StringBuffer buf = new StringBuffer();
       eatSpaces(lineBuf);
       if ( idx >= lineBuf.length)
          return null;

       while ( idx < lineBuf.length && ( !Character.isWhitespace(lineBuf[idx])) ) {
         if (lineBuf[idx] == '<') {
           if ( buf.length() > 0)
             return buf.toString();

           while (idx < lineBuf.length && lineBuf[idx] != '>') {
              if ( !Character.isWhitespace(lineBuf[idx]) ) {
                   buf.append(lineBuf[idx++]);
              }
           }
           if ( lineBuf[idx] == '>') {
             buf.append(lineBuf[idx++]);
             return buf.toString();
           } else {
             buf = new StringBuffer();
             ++idx;
             continue;
           }
         }
         buf.append(lineBuf[ idx++ ]);
       }
       return buf.toString();
    }

    protected void eatSpaces(char[] lineBuf) {
      while( idx < lineBuf.length &&
             (lineBuf[idx] == ' ' || lineBuf[idx] == '\t') ) {
        ++idx;
      }
    }


    public int getTextPartCount() { return lineParts.size(); }

    public boolean hasNext() {
      return iterator.hasNext();
    }

    public StyledText next() {
      return iterator.next();
    }

    public void remove() { /* not used */}


    public static void main(String[] args) {
        Font f = new Font("Helvetica", Font.PLAIN, 12);
        StyledTextIterator sti =  new StyledTextIterator("<b>1.</b><i> Question 1:</i> some question", f);
        while ( sti.hasNext() ) {
          StyledText st = (StyledText) sti.next();
          System.out.println("st="+ st);
        }
    }

}


