package caslayout.ui;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics2D;
import java.awt.geom.Dimension2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.beans.DefaultPersistenceDelegate;
import java.beans.Encoder;
import java.beans.Expression;

import org.jdom.Element;

import caslayout.util.GUIUtils;
import caslayout.util.UIDManager;
import caslayout.util.XMLUtils;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: TextDisplayComponent.java,v 1.16 2006/03/29 02:25:03 bozyurt
 *          Exp $
 */
public class TextDisplayComponent extends AbstractDisplayComponent {
   protected MultilineString label;

   public TextDisplayComponent() {
      super();
      justification = Justification.LEFT_JUSTIFICATION;
   }

   public TextDisplayComponent(String id) {
      super(id);
      justification = Justification.LEFT_JUSTIFICATION;
   }

   public TextDisplayComponent(String labelText, int x, int y, Component peer) {
      super(x, y, peer);
      this.label = new MultilineString();
      this.label.setText(labelText);
      justification = Justification.LEFT_JUSTIFICATION;
   }

   public TextDisplayComponent(MultilineString label, int x, int y,
         Component peer) {
      super(x, y, peer);
      this.label = label;
      justification = Justification.LEFT_JUSTIFICATION;
   }

   public TextDisplayComponent(MultilineString label, int x, int y,
         Component peer, String id) {
      super(x, y, peer, id, "", Justification.LEFT_JUSTIFICATION);
      this.label = label;
      getBounds((Graphics2D) peer.getGraphics());
      justification = Justification.LEFT_JUSTIFICATION;
   }

   public TextDisplayComponent(TextDisplayComponent other) {
      super(other);
      this.id = UIDManager.getInstance().getNextUID();

      // super(other.x, other.y, other.peer,
      // UIDManager.getInstance().getNextUID(), other.name,
      // other.justification);
      this.label = new MultilineString(other.label);
      getBounds((Graphics2D) peer.getGraphics());
   }

   public void draw(Graphics2D g2) {
      Rectangle2D r = getBounds(g2);
      Rectangle2D labelBounds = label.getBounds(g2);
      Font oldFont = null;

      if (getShowQuestions())
         super.showQuestion(g2, r, Color.ORANGE);

      if (r.getWidth() <= labelBounds.getWidth()
            || r.getHeight() <= labelBounds.getHeight()) {
         oldFont = g2.getFont();
         int size = oldFont.getSize();
         Font newFont = new Font(oldFont.getName(), oldFont.getStyle(),
               size - 1);
         g2.setFont(newFont);
         labelBounds = label.getBounds(g2);
      }

      Dimension2D compDim = new Dimension((int) labelBounds.getWidth(),
            (int) labelBounds.getHeight());
      Point2D point = GUIUtils.calculateAlignmentPosition(r, compDim,
            justification);

      if (selected) {
         label.setTextColor(Color.blue);
      } else if (groupID != null) {
         label.setTextColor(Color.magenta);
      } else {
         label.setTextColor(Color.black);
      }

      label.draw(g2, new Rectangle2D.Double(point.getX(), point.getY(),
            labelBounds.getWidth(), labelBounds.getHeight()));

      if (oldFont != null) {
         g2.setFont(oldFont);
      }
   }

   public Rectangle2D getBounds(Graphics2D g2) {
      if (bounds == null) {
         bounds = label.getBounds(g2);
      }
      return super.bounds;
   }

   public MultilineString getLabel() {
      return label;
   }

   public void setLabel(MultilineString newLabel) {
      this.label = newLabel;
   }

   public void setText(String txt) {
      if (label == null) {
         label = new MultilineString(txt);
      } else {
         label.setText(txt);
      }
   }

   public void setJustification(Justification newJustification) {
      super.setJustification(newJustification);
      label.setJustification(newJustification);
   }

   public Object clone() {
      return new TextDisplayComponent(this);
   }

   public static void setPresistenceDelegate(Encoder encoder) {
      encoder.setPersistenceDelegate(TextDisplayComponent.class,
            new DefaultPersistenceDelegate() {

               @SuppressWarnings("unused")
               protected Expression instantiate(Class<?> type, Object oldInstance) {
                  TextDisplayComponent tdc = (TextDisplayComponent) oldInstance;
                  Expression expr = new Expression(oldInstance, oldInstance
                        .getClass(), "new", new Object[] { tdc });
                  try {
                     System.out.println("<<<<>>>>>> expr=" + expr.toString());
                  } catch (Exception x) {
                     x.printStackTrace();
                  }
                  return expr;
               }

            });
   }

   public static Object initializeFromXML(Element e) {
      String text = XMLUtils.getStringPropertyValue("text", e, null);
      String id = XMLUtils.getStringPropertyValue("id", e, "");
      String cssClass = XMLUtils.getStringPropertyValue("cssClass", e, null);

      TextDisplayComponent tdc = new TextDisplayComponent(id);
      if (text != null) {
         tdc.setText(XMLUtils.convertBR2EOL(text));
      }
      tdc.x = XMLUtils.getPropertyValue("x", e, 0);
      tdc.y = XMLUtils.getPropertyValue("y", e, 0);
      tdc.width = XMLUtils.getPropertyValue("height", e, 0);
      tdc.height = XMLUtils.getPropertyValue("height", e, 0);
      String refID = XMLUtils.getReferencePropertyValue("parent", e, null);
      tdc.name = XMLUtils.getStringPropertyValue("name", e, "");

      String justificationValue = XMLUtils.getStringPropertyValue(
            "justification", e, null);
      if (justificationValue != null) {
         tdc.setJustification(Justification
               .findJustification(justificationValue));
      }

      tdc.cssClass = cssClass;
      if (refID != null)
         tdc.parent = (IDisplayComponent) XMLUtils
               .getReferencedObject(refID, e);
      return tdc;
   }

   public Element toXML(Element root) {
      Element e = super.toXML(root);
      e.setAttribute("class", "caslayout.ui.TextDisplayComponent");
      if (label != null)
         e.addContent(XMLUtils.prepareProperty("text", XMLUtils
               .convertEOL2BR(label.getText())));
      e.addContent(XMLUtils.prepareProperty("justification", justification
            .getValue()));
      return e;
   }
}
