package caslayout.ui.actions;

import guilib.common.BaseDialog;

import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.GridLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.SwingWorker;

import caslayout.importing.common.Assessment;
import caslayout.importing.common.DataRow;
import caslayout.importing.ui.AsDataImportDialog;
import caslayout.ui.CALMHelper;
import caslayout.ui.ClinicalAssessmentLayoutManager;
import caslayout.ui.db.Protocol;
import caslayout.ui.db.services.AssessmentDataService;
import caslayout.ui.db.services.AssessmentExistsException;
import caslayout.util.GenUtils;

/**
 * @author I. Burak Ozyurt
 * @version $Id: AsDataUploadAdapter.java,v 1.5 2008/10/14 23:21:16 bozyurt Exp $
 */
public class AsDataUploadAdapter extends AbstractAction implements
      PropertyChangeListener {
   private final ClinicalAssessmentLayoutManager manager;
   private static SimpleDateFormat daf = new SimpleDateFormat("MM/dd/yyyy");
   private static SimpleDateFormat tsf = new SimpleDateFormat(
         "MM/dd/yyyy HH:mm:ss");
   private static SimpleDateFormat tmf = new SimpleDateFormat(
         "MM/dd/yyyy HH:mm");
   private static SimpleDateFormat thf = new SimpleDateFormat("MM/dd/yyyy HH");

   private ProgressDialog progDlg;
   private String errorFile;
   private String[] errorHeaders;

   private static final long serialVersionUID = 1L;

   static {
      daf.setLenient(false);
      tsf.setLenient(false);
      tmf.setLenient(false);
      thf.setLenient(false);
   }

   public AsDataUploadAdapter(ClinicalAssessmentLayoutManager manager,
         String text, ImageIcon icon, String desc, Integer mnemonic) {
      super(text, icon);
      this.manager = manager;
      putValue(SHORT_DESCRIPTION, desc);
      putValue(MNEMONIC_KEY, mnemonic);
   }

   public void actionPerformed(ActionEvent e) {
      AssessmentDataService ads = AssessmentDataService.getInstance(manager
            .getConfig());
      AsDataImportDialog dlg = new AsDataImportDialog(manager,
            "Assessment Data Import", ads);
      int rc = dlg.showDialog();
      if (rc == BaseDialog.OK_PRESSED) {
         List<DataRow> dataRowList = dlg.getDataRowList();
         if (dataRowList != null && !dataRowList.isEmpty()) {
            Assessment as = dlg.getAssessment();
            Map<Object,Object> protocolMap = ads.getProtocolMap();
            Protocol protocol = findMatchingProtocol(protocolMap, dlg
                  .getSelectedProtocol());

            String importedFile = dlg.getImportedFile();
            this.errorFile = importedFile
                  .replaceFirst("\\.\\w+$", "-error.csv");
            this.errorHeaders = dlg.getAsDataHeaders();

            AsDataUploadProcess process = new AsDataUploadProcess(protocol, as,
                  dataRowList, ads);

            this.progDlg = new ProgressDialog(manager,
                  "Assessment Data Upload Progress", "Initializing...");

            process.addPropertyChangeListener(this);
            manager.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

            process.execute();

            progDlg.showDialog();
         }
      }
      dlg.dispose();
   }

   class AsDataUploadProcess extends SwingWorker<Object,Object> {
      Protocol protocol;
      Assessment as;
      List<DataRow> dataRowList;
      AssessmentDataService ads;
      String note;

      public AsDataUploadProcess(Protocol protocol, Assessment as,
            List<DataRow> dataRowList, AssessmentDataService ads) {
         super();
         this.protocol = protocol;
         this.as = as;
         this.dataRowList = dataRowList;
         this.ads = ads;
      }

      public Object doInBackground() throws Exception {
         setProgress(0);
         int count = 0;
         int badOnes = 0;
         int size = dataRowList.size();
         List<ErrorInfo> errorInfoList = new ArrayList<ErrorInfo>(10);
         for (Iterator<DataRow> it = dataRowList.iterator(); it.hasNext();) {
            DataRow dataRow = null;
            try {
               dataRow = (DataRow) it.next();
               Date visitDate = toDate(dataRow.getTimeStamp());
               Date segmentTimestamp = toDate(dataRow.getTimeStamp());
               System.out.println("Assessment Name:" + as.getName());
               ads.addAssessmentData(dataRow, as.getName(), "clinical visit",
                     dataRow.getSubjectType(), visitDate, segmentTimestamp,
                     protocol);
            } catch (Exception x) {
               x.printStackTrace();
               ++badOnes;
               if (x instanceof AssessmentExistsException) {
                  errorInfoList.add(new ErrorInfo(dataRow,
                        "record exists in the database"));
               } else {
                  errorInfoList.add(new ErrorInfo(dataRow, x.getMessage()));
               }
            }
            ++count;
            double percentComplete = count / (double) size * 100;
            setProgress((int) percentComplete);
            StringBuilder buf = new StringBuilder();
            buf.append("Upload ").append((int) percentComplete).append(
                  "% complete. Uploaded:");
            buf.append(count - badOnes);
            if (badOnes > 0) {
               buf.append(" Error cases:").append(badOnes);
            }
            setNote(buf.toString());
         }

         if (!errorInfoList.isEmpty()) {
            BufferedWriter out = null;
            try {
               out = new BufferedWriter(new FileWriter(errorFile));
               StringBuilder buf = new StringBuilder(400);
               for (int i = 0; i < errorHeaders.length; i++) {
                  buf.append(errorHeaders[i]);
                  if ((i + 1) < errorHeaders.length) {
                     buf.append(',');
                  }
               }
               out.write(buf.toString());
               out.newLine();
               for (Iterator<ErrorInfo> it = errorInfoList.iterator(); it.hasNext();) {
                  ErrorInfo ei = it.next();
                  out.write(ei.toString());
                  out.newLine();
               }
            } catch (IOException x) {
               x.printStackTrace();
               CALMHelper.showError(manager, x.getMessage());
            } finally {
               GenUtils.close(out);
            }

         }
         StringBuilder buf = new StringBuilder();
         buf.append("Uploaded ").append(count - badOnes);
         buf.append(" assessments to the database");
         if (badOnes > 0) {
            buf.append(", ").append(badOnes);
            buf.append(" records failed.");
            buf.append("\nThe failed records are written to ");
            buf.append(errorFile).append(".");
         }

         CALMHelper.showMessage(manager, buf.toString(),
               "Upload Result Summary");

         return null;
      }

      public void done() {
         Toolkit.getDefaultToolkit().beep();
         manager.setCursor(null);
         progDlg.dispose();
      }

      public String getNote() {
         return note;
      }

      public void setNote(String note) {
         this.note = note;
         super.firePropertyChange("note", null, note);
      }

   }// ;

   static class ErrorInfo {
      DataRow dr;
      String errMessage;

      public ErrorInfo(DataRow dr, String errMessage) {
         super();
         this.dr = dr;
         this.errMessage = errMessage;
      }

      public String toString() {
         StringBuilder buf = new StringBuilder(400);
         buf.append('"').append(quoteForCSV(dr.getSite())).append("\",");
         buf.append('"').append(quoteForCSV(dr.getSubjectID())).append("\",");
         buf.append('"').append(quoteForCSV(dr.getSubjectType())).append("\",");
         buf.append('"').append(quoteForCSV(dr.getExperimentName())).append(
               "\",");
         buf.append('"').append(quoteForCSV(dr.getTimeStamp())).append("\",");
         // buf.append('"').append( quoteForCSV(dr.getScoreTypeCol())
         // ).append("\",");
         for (int i = 0; i < dr.getData().length; i++) {
            buf.append('"').append(quoteForCSV(dr.getData()[i].toString()));
            buf.append("\",");
         }
         buf.append('"').append(quoteForCSV(errMessage)).append('"');
         return buf.toString();
      }
   }

   public static String quoteForCSV(String s) {
      return s.replaceAll("\"", "\"\"");
   }

   public static Date toDate(String ts) throws ParseException {
      Date date = null;
      if (ts == null || ts.trim().length() == 0) {
         throw new ParseException("Missing date!", 0);
      }
      try {
         if (ts.indexOf(':') != -1) {
            int numMatch = GenUtils.numberOfMatch(ts, ':');
            switch (numMatch) {
            case 1:
               date = tmf.parse(ts);
               break;
            case 2:
               date = tsf.parse(ts);
               break;
            default:
               throw new ParseException("Not a valid date: " + ts, 0);
            }
         } else {
            date = thf.parse(ts);
         }
      } catch (ParseException pe) {
         date = daf.parse(ts);
      }
      return date;
   }

   protected Protocol findMatchingProtocol(Map<Object, Object> protocolMap, String protocolID) {
      Protocol theOne = null;
      int maxVersion = Integer.MIN_VALUE;
      for (Iterator<Object> it = protocolMap.values().iterator(); it.hasNext();) {
         Protocol pr = (Protocol) it.next();
         if (pr.getProtocolKey().getProtocolId().equals(protocolID)) {
            if (maxVersion < pr.getProtocolKey().getProtocolVersion()
                  .intValue()) {
               maxVersion = pr.getProtocolKey().getProtocolVersion().intValue();
               theOne = pr;
            }
         }
      }
      return theOne;
   }

   public static class ProgressDialog extends BaseDialog {
      private static final long serialVersionUID = 1L;
      JProgressBar pbar;
      JLabel noteLabel;
      JPanel panel;

      public ProgressDialog(Frame owner, String title, String note) {
         super(owner, title, new String[] { "OK" });
         setPreferredSize(new Dimension(300, 125));
         pbar = new JProgressBar(0, 100);
         pbar.setValue(0);
         pbar.setStringPainted(true);
         if (note == null)
            note = "";
         noteLabel = new JLabel(note);
         panel = new JPanel(new GridLayout(2, 1, 6, 6));
         panel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
         panel.add(noteLabel);
         panel.add(pbar);

         getContentPane().add(panel);
         pack();

      }

      public void setValue(int progress) {
         pbar.setValue(progress);
      }

      public void setNote(String note) {
         System.out.println("setting text: " + note);
         noteLabel.setText(note);
         panel.invalidate();
      }

   }

   public void propertyChange(PropertyChangeEvent evt) {
      if ("progress" == evt.getPropertyName()) {
         int progress = ((Integer) evt.getNewValue()).intValue();
         this.progDlg.setValue(progress);
      } else if ("note" == evt.getPropertyName()) {
         System.out.println("Updating node");
         this.progDlg.setNote((String) evt.getNewValue());
      }
   }
}
