package caslayout.ui.assoc;

import guilib.common.BaseDialog;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.GridLayout;
import java.awt.HeadlessException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Observable;
import java.util.Observer;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.JTree;
import javax.swing.border.TitledBorder;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import org.apache.log4j.Logger;

import caslayout.Constants;
import caslayout.propedit.PropertyEditorPanel;
import caslayout.ui.AbstractDisplayComponent;
import caslayout.ui.CAContainer;
import caslayout.ui.CALMHelper;
import caslayout.ui.Document;
import caslayout.ui.DropdownDisplayComponent;
import caslayout.ui.IDisplayComponent;
import caslayout.ui.LogicalGroup;
import caslayout.ui.RadioButtonDisplayComponent;
import caslayout.ui.SelectedComponentInfo;
import caslayout.ui.TextDisplayComponent;
import caslayout.ui.TextFieldDisplayComponent;
import caslayout.ui.model.AssessmentInfo;
import caslayout.ui.model.Association;
import caslayout.ui.model.AssociationHelper;
import caslayout.ui.model.ItemInfo;
import caslayout.ui.model.MandatoryFieldAssociation;
import caslayout.ui.model.NodeInfo;
import caslayout.ui.model.ScoreAssociation;
import caslayout.ui.model.ScoreCodeInfo;
import caslayout.ui.model.ScoreInfo;
import caslayout.ui.model.SecurityClassificationInfo;
import caslayout.util.GUIUtils;

/**
 * Maintains Assessment - Score hierarchy in a tree layout
 *
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentMaintenancePanel.java,v 1.26 2007/10/08 18:00:41
 *          bozyurt Exp $
 */
public class AssessmentMaintenancePanel extends JPanel implements
      PropertyChangeListener, Observer {
   private static final long serialVersionUID = -5587026794498037984L;
   protected JTree asTree;
   protected DefaultTreeModel treeModel;
   protected DefaultMutableTreeNode modelRoot;

   protected AssessmentInfo assessmentInfo;
   protected Document document;
   protected JFrame parentFrame = null;
   protected JScrollPane modelPane;
   protected SelectedModelDataInfo selectedModelData;
   protected ModelTreeMouseAdapter mouseAdapter;
   /**
    * bound property used to associate selected label(s) with a score's
    * scorecodes or the question
    */
   protected boolean associateLabels = false;

   protected PropertyChangeSupport pcs;
   /**
    * whenever user wants to associate labels, this is set to the selected score
    * info in the assessment tree
    */
   protected ScoreInfo theScoreInfo;
   /**
    * whenever user wants to assign a label text as the leading of an
    * assessmentitem db record, this is set
    */
   protected ItemInfo theItemInfo;
   protected static Logger log = Logger.getLogger("AssessmentMaintenancePanel");

   public AssessmentMaintenancePanel(AssessmentInfo assessmentInfo,
         Document document, JFrame parentFrame) {
      this.assessmentInfo = assessmentInfo;
      this.document = document;
      this.parentFrame = parentFrame;
      this.setLayout(new BorderLayout(3, 3));
      pcs = new PropertyChangeSupport(this);
      init();
   }

   public void init() {
      if (assessmentInfo != null) {
         populateAssessmentTree(assessmentInfo);

         mouseAdapter = new ModelTreeMouseAdapter(asTree, treeModel,
               this.assessmentInfo, this);
         asTree.addMouseListener(mouseAdapter);
         this.modelPane = new JScrollPane(asTree);
         asTree.expandRow(1);
         modelPane.setBorder(BorderFactory.createTitledBorder(assessmentInfo
               .getName()));
         this.add(modelPane);
      }
   }

   public void setTitle(String title) {
      ((TitledBorder) this.modelPane.getBorder()).setTitle(title);
      modelPane.repaint();
      if (treeModel != null) {
         NodeInfo ni = (NodeInfo) modelRoot.getUserObject();
         ni.update();
         treeModel.nodeChanged(modelRoot);
      }
   }

   public void populateAssessmentTree(AssessmentInfo assessmentInfo) {
      if (asTree != null) {
         modelRoot.removeAllChildren();
         modelRoot.setUserObject(new NodeInfo(assessmentInfo));
         treeModel.setRoot(modelRoot);
         treeModel.nodeChanged(modelRoot);
         if (this.assessmentInfo != assessmentInfo) {
            this.assessmentInfo = assessmentInfo;
            // need to update mouse adapter's assessment info also
            mouseAdapter.setAssessmentInfo(this.assessmentInfo);
         }
         asTree.setCellRenderer(new NodeRenderer(AssociationHelper
               .getInstance()));
      } else {

         modelRoot = new DefaultMutableTreeNode(new NodeInfo(assessmentInfo));
         treeModel = new DefaultTreeModel(modelRoot);
         asTree = new JTree(treeModel);

         asTree.getSelectionModel().setSelectionMode(
               TreeSelectionModel.SINGLE_TREE_SELECTION);

         asTree.setCellRenderer(new NodeRenderer(AssociationHelper
               .getInstance()));
      }

      // add static mandatory fields (namely date, time, informant fields) to
      // the tree root
      DefaultMutableTreeNode timeNode = new DefaultMutableTreeNode(
            new NodeInfo("Time"));
      DefaultMutableTreeNode dateNode = new DefaultMutableTreeNode(
            new NodeInfo("Date"));

      DefaultMutableTreeNode informantNode = new DefaultMutableTreeNode(
            new NodeInfo("InformantID"));
      DefaultMutableTreeNode informantRelNode = new DefaultMutableTreeNode(
            new NodeInfo("InformantRelation"));
      DefaultMutableTreeNode clinicalRaterNode = new DefaultMutableTreeNode(
            new NodeInfo("ClinicalRater"));

      modelRoot.add(dateNode);
      modelRoot.add(timeNode);
      modelRoot.add(informantNode);
      modelRoot.add(informantRelNode);
      modelRoot.add(clinicalRaterNode);

      Map<ScoreInfo, Object> remMap = new LinkedHashMap<ScoreInfo, Object>();
      Map<ScoreInfo, Object> curLevelMap = new LinkedHashMap<ScoreInfo, Object>();
      for (Iterator<ScoreInfo> iter = assessmentInfo.getScores().iterator(); iter
            .hasNext();) {
         ScoreInfo si = iter.next();
         if (si.getParent() == null) {
            curLevelMap.put(si, si);
         } else {
            remMap.put(si, si.getParent());
         }
      }
      // for (Iterator iter = curLevelMap.entrySet().iterator();
      // iter.hasNext();) {
      for (Map.Entry<ScoreInfo, Object> entry : curLevelMap.entrySet()) {
         ScoreInfo si = (ScoreInfo) entry.getKey();
         DefaultMutableTreeNode node = new DefaultMutableTreeNode(new NodeInfo(
               si));
         modelRoot.add(node);
         entry.setValue(node);
      }

      do {
         Map<ScoreInfo, Object> nextLevelMap = new LinkedHashMap<ScoreInfo, Object>();
         remMap = populateModelTreeLevel(remMap, nextLevelMap);
         // for (Iterator iter = nextLevelMap.entrySet().iterator(); iter
         // .hasNext();) {
         for (Map.Entry<ScoreInfo, Object> entry : nextLevelMap.entrySet()) {
            ScoreInfo si = (ScoreInfo) entry.getKey();
            DefaultMutableTreeNode parentNode = (DefaultMutableTreeNode) curLevelMap
                  .get(si.getParent());
            DefaultMutableTreeNode node = new DefaultMutableTreeNode(
                  new NodeInfo(si));
            entry.setValue(node);
            parentNode.add(node);
         }
         curLevelMap = nextLevelMap;
      } while (!remMap.isEmpty());

      checkAndFixNodeSequence();

      treeModel.reload();
   }

   @SuppressWarnings("unchecked")
   protected void checkAndFixNodeSequence() {
      Enumeration enumer = modelRoot.preorderEnumeration();
      int count = 1;
      while (enumer.hasMoreElements()) {
         DefaultMutableTreeNode node = (DefaultMutableTreeNode) enumer
               .nextElement();
         NodeInfo ni = (NodeInfo) node.getUserObject();
         int preOrderIdx = count - (Constants.NUM_MANDATORY_FIELDS + 1);
         if (ni.getData() instanceof ScoreInfo) {
            ScoreInfo si = (ScoreInfo) ni.getData();
            // System.out.println("si scoresequence =" +
            // si.getScoreSequence() + " preorder value for node " +
            // preOrderIdx);
            if (si.getScoreSequence() != preOrderIdx) {
               System.out.println("** Incorrect " + si.getScoreName()
                     + " scoresequence! Changed from " + si.getScoreSequence()
                     + " to " + preOrderIdx);
               si.setScoreSequence(preOrderIdx);
            }
         }
         ++count;
      }
   }

   protected Map<ScoreInfo, Object> populateModelTreeLevel(
         Map<ScoreInfo, Object> remMap, Map<ScoreInfo, Object> curLevelMap) {
      Map<ScoreInfo, Object> rMap = new LinkedHashMap<ScoreInfo, Object>();
      for (Iterator<ScoreInfo> iter = remMap.keySet().iterator(); iter
            .hasNext();) {
         ScoreInfo si = iter.next();
         if (remMap.get(si.getParent()) == null) {
            curLevelMap.put(si, si);
         } else {
            rMap.put(si, si.getParent());
         }
      }
      return rMap;
   }

   // for Observer interface
   public void update(Observable o, Object arg) {
      this.asTree.repaint();
   }

   public void addPropertyChangeListener(PropertyChangeListener pcl) {
      pcs.addPropertyChangeListener(pcl);
   }

   public void removePropertyChangeListener(PropertyChangeListener pcl) {
      pcs.removePropertyChangeListener(pcl);
   }

   /**
    * prepopulates the score codes with the selected labels as score codes. The
    * score code values are initialized starting from 0 if the score type is
    * integer, otherwise no prepopulation is done
    *
    * @param labelComps
    */

   protected void handleSelectedLabels(List<IDisplayComponent> labelComps) {
      if (theScoreInfo == null && theItemInfo == null)
         return;
      if (labelComps.isEmpty())
         return;
      if (GUIUtils.areAllTextOrRadioDisplayComponents(labelComps)) {

         if (theScoreInfo != null && theItemInfo == null) {
            if (theScoreInfo.getScoreCodes() != null
                  && !theScoreInfo.getScoreCodes().isEmpty()) {
               /** @todo */
            } else {
               int idx = 0;
               for (Iterator<IDisplayComponent> iter = labelComps.iterator(); iter
                     .hasNext();) {
                  Object item = iter.next();
                  if (item instanceof TextDisplayComponent) {
                     TextDisplayComponent tdc = (TextDisplayComponent) item;
                     String scText = tdc.getLabel().getText().trim();
                     String scValue = theScoreInfo.getScoreType().equals(
                           "integer") ? String.valueOf(idx) : "";
                     ScoreCodeInfo sci = new ScoreCodeInfo(theScoreInfo
                           .getScoreName(), scValue, scText, theScoreInfo
                           .getScoreType(), theScoreInfo.getScoreTypeFormat());
                     theScoreInfo.addScoreCode(sci);
                     ++idx;
                  } else if (item instanceof RadioButtonDisplayComponent) {
                     RadioButtonDisplayComponent rdc = (RadioButtonDisplayComponent) item;
                     if (rdc.getText() != null
                           || rdc.getText().trim().length() > 0) {
                        String scText = rdc.getText().trim();
                        String scValue = theScoreInfo.getScoreType().equals(
                              "integer") ? String.valueOf(idx) : "";
                        ScoreCodeInfo sci = new ScoreCodeInfo(theScoreInfo
                              .getScoreName(), scValue, scText, theScoreInfo
                              .getScoreType(), theScoreInfo
                              .getScoreTypeFormat());
                        theScoreInfo.addScoreCode(sci);
                        ++idx;
                     }
                  }
               } // for
            } // else

            ScoreCodeDialog dlg = new ScoreCodeDialog(null, "Score Codes",
                  theScoreInfo);
            dlg.setLocationRelativeTo(asTree);
            int rc = dlg.showDialog();
            if (rc == BaseDialog.OK_PRESSED) {
               List<ScoreCodeInfo> scoreCodes = dlg.getScoreCodes();
               if (theScoreInfo.getScoreCodes() != null) {
                  theScoreInfo.getScoreCodes().clear();
               }
               for (ScoreCodeInfo sci : scoreCodes) {
                  theScoreInfo.addScoreCode(sci);
               }
            }
            dlg.dispose();
            theScoreInfo = null;
         } // theScoreInfo != null
         else if (theItemInfo != null && theScoreInfo != null) {
            TextDisplayComponent tdc = (TextDisplayComponent) labelComps.get(0);
            theItemInfo.setItemLeadingText(caslayout.util.GenUtils
                  .cleanHTMLTags(tdc.getLabel().getText()));

            ItemInfoDialog iDlg = new ItemInfoDialog(null,
                  "Add Assessment Item", theItemInfo);
            iDlg.setLocationRelativeTo(asTree);
            int rc = iDlg.showDialog();
            if (rc == BaseDialog.OK_PRESSED) {
               theScoreInfo.getAssessment().addItem(theItemInfo);
            }
            iDlg.dispose();
            theItemInfo = null;
            theScoreInfo = null;
         }
      }
   }

   protected boolean validMandatoryFieldAssocComponent(IDisplayComponent adc) {
      return (adc instanceof TextFieldDisplayComponent || adc instanceof DropdownDisplayComponent);
   }

   @SuppressWarnings("unchecked")
   public void propertyChange(PropertyChangeEvent pce) {
      if (pce.getPropertyName().equals("selectedLabels")) {
         handleSelectedLabels((List<IDisplayComponent>) pce.getNewValue());
      } else if (pce.getPropertyName().equals("compForAssociation")) {
         // associate the selected display comp with the score
         SelectedComponentInfo sci = (SelectedComponentInfo) pce.getNewValue();

         if (selectedModelData == null) {
            log.info("selectedModelData was null");
            return;
         }

         associateDispCompWithScore(sci);
      } else if (pce.getPropertyName().equals("questionTextInfo")) {
         /** @todo */
      }
   }

   private void associateDispCompWithScore(SelectedComponentInfo sci)
         throws HeadlessException {
      // start association
      ScoreAssociation sa = null;
      MandatoryFieldAssociation mfa = null;
      if (selectedModelData.hasScoreInfo()) {
         LogicalGroup lg = sci.findLogicalGroup();

         ScoreInfo si = selectedModelData.getScoreInfo();
         if ((si.getScoreCodes() == null || si.getScoreCodes().isEmpty())
               && lg != null) {
            CALMHelper
                  .showError(
                        this,
                        "Logical Group needs to be associated with a score having score codes",
                        "Error");
            return;

         }
         sa = (ScoreAssociation) AssociationHelper.getInstance().getAssocMap()
               .get(si);
         if (sa == null) {
            if (lg != null) {
               sa = new ScoreAssociation(si.getScoreName(),
                     Association.BIDIRECTIONAL, si, lg);
               sa.setScoreValue(lg);
            } else {
               sa = new ScoreAssociation(si.getScoreName(),
                     Association.BIDIRECTIONAL, si, sci.getSelectedObject());
               sa.setScoreValue(sci.getSelectedObject());
            }
         }

      } else { // mandatory fields

         if (!validMandatoryFieldAssocComponent(sci.getSelectedObject())) {
            CALMHelper
                  .showError(
                        this,
                        "You can only associate a text or dropdown field with a mandatory field!",
                        "Error");
            return;
         }

         String mandatoryFieldName = (String) selectedModelData
               .getMandatoryField().getData();

         mfa = AssociationHelper.getInstance().findMandatoryFieldAssociation(
               mandatoryFieldName);
         if (mfa == null) {
            mfa = new MandatoryFieldAssociation(mandatoryFieldName,
                  Association.BIDIRECTIONAL, mandatoryFieldName,
                  (AbstractDisplayComponent) sci.getSelectedObject());
         }
      }

      AssociationDialog dlg = null;
      if (selectedModelData.hasScoreInfo()) {
         dlg = new AssociationDialog(null, "Association", selectedModelData
               .getScoreInfo(), sci, sa);
      } else {
         dlg = new AssociationDialog(null, "Association",
               (String) selectedModelData.getMandatoryField().getData(), sci,
               mfa);
      }
      int rc = dlg.showDialog();
      if (rc == BaseDialog.OK_PRESSED) {
         if (selectedModelData.hasScoreInfo()) {
            // this step completes the score association
            sa = dlg.getScoreAssociation();
            AssociationHelper.getInstance().addScoreAssociation(sa);
         } else {
            AssociationHelper.getInstance().addMandatoryFieldAssociation(mfa);

            // set the fieldinfo metadata for the dynamic dropdown
            if (mfa.getRight() instanceof DropdownDisplayComponent) {
               DropdownDisplayComponent ddc = (DropdownDisplayComponent) mfa
                     .getRight();
               if (ddc.getPopulatedFromDatabase() == true) {
                  this.assessmentInfo.setMandatoryFieldMetaData(mfa.getLeft(),
                        "query", ddc.getSqlQuery());
               }
            }
         }
      }
      dlg.dispose();
      return;
   }

   // bound property
   public void setSelectedModelData(SelectedModelDataInfo newSelectedModelData) {
      SelectedModelDataInfo oldValue = this.selectedModelData;
      this.selectedModelData = newSelectedModelData;
      pcs.firePropertyChange("selectedModelData", oldValue,
            newSelectedModelData);
   }

   public SelectedModelDataInfo getSelectedModelData() {
      return this.selectedModelData;
   }

   // bound property
   public void setAssociateLabels(boolean newAssociateLabels) {
      // always use the negation of the new value, to force
      // property change event firing!
      boolean oldValue = !newAssociateLabels;
      this.associateLabels = newAssociateLabels;
      pcs.firePropertyChange("associateLabels", oldValue, newAssociateLabels);
   }

   public boolean getAssociateLabels() {
      return this.associateLabels;
   }

   protected void populateFromViewPanel(DefaultMutableTreeNode parentNode,
         IDisplayComponent idc) {
      if (idc instanceof CAContainer) {
         DefaultMutableTreeNode node = new DefaultMutableTreeNode(
               new FormNodeInfo(idc));
         parentNode.add(node);
         CAContainer c = (CAContainer) idc;
         for (IDisplayComponent ic : c.getComponents()) {
            if (ic != null) {
               populateFromViewPanel(node, ic);
            }
         }
      } else {
         DefaultMutableTreeNode node = new DefaultMutableTreeNode(
               new FormNodeInfo(idc));
         parentNode.add(node);
      }
   }

   @SuppressWarnings("unchecked")
   public static int getNodeSequence(DefaultMutableTreeNode rootNode,
         DefaultMutableTreeNode theNode) {
      Enumeration enumer = rootNode.preorderEnumeration();
      int count = 1;
      while (enumer.hasMoreElements()) {
         DefaultMutableTreeNode node = (DefaultMutableTreeNode) enumer
               .nextElement();
         if (node == theNode) {
            return count;
         }
         ++count;
      }
      return -1;
   }

   public static class NodeRenderer extends DefaultTreeCellRenderer {
      private static final long serialVersionUID = 8591003612705272022L;
      AssociationHelper ah;
      ImageIcon leafIcon;
      ImageIcon parentIcon;

      public NodeRenderer(AssociationHelper ah) {
         this.ah = ah;
         leafIcon = GUIUtils.createImageIcon("/images/child_object.gif");
         parentIcon = GUIUtils.createImageIcon("/images/parent_node.gif");
      }

      public Component getTreeCellRendererComponent(JTree tree, Object value,
            boolean sel, boolean expanded, boolean leaf, int row,
            boolean hasFocus) {
         DefaultMutableTreeNode node = (DefaultMutableTreeNode) value;
         NodeInfo ni = (NodeInfo) node.getUserObject();
         // Color oldSelectionColor = getTextSelectionColor();
         // Color oldNonTextSelectionColor = getTextNonSelectionColor();

         setTextSelectionColor(Color.white);
         setTextNonSelectionColor(Color.black);

         if (ni.getData() instanceof ScoreInfo) {
            setTextNonSelectionColor(Color.GREEN);
            setTextSelectionColor(Color.GREEN);

            ScoreInfo si = (ScoreInfo) ni.getData();
            // check if there is an association
            if (ah.getAssocMap().get(si) == null) {
               setTextSelectionColor(Color.white);
               setTextNonSelectionColor(Color.black);
            }
         } else if (ni.getData() instanceof String) {
            setTextNonSelectionColor(Color.GREEN);
            setTextSelectionColor(Color.GREEN);
            if (ah.findMandatoryFieldAssociation((String) ni.getData()) == null) {
               setTextSelectionColor(Color.white);
               setTextNonSelectionColor(Color.black);
            }
         }

         Component c = super.getTreeCellRendererComponent(tree, value, sel,
               expanded, leaf, row, hasFocus);
         if (leaf) {
            this.setIcon(leafIcon);
         } else {
            this.setIcon(parentIcon);
         }
         return c;
      }
   }

   class ModelTreeMouseAdapter extends MouseAdapter implements ActionListener {
      JTree asTree;
      DefaultTreeModel asTreeModel;
      JPopupMenu popupMenu;
      ScoreInfo selectedScoreInfo;
      NodeInfo selectedMandatoryField;
      AssessmentInfo asi;
      JMenuItem[] menuItems;
      AssessmentMaintenancePanel amp;
      int asItemAssocConfirmAnswer = -1;
      int scoreCodeAssocConfirmAnswer = -1;

      public ModelTreeMouseAdapter(JTree asTree, DefaultTreeModel asTreeModel,
            AssessmentInfo asi, AssessmentMaintenancePanel amp) {
         this.asTree = asTree;
         this.asTreeModel = asTreeModel;
         this.asi = asi;
         this.amp = amp;
         preparePopupMenu();
      }

      public void setAssessmentInfo(AssessmentInfo newAssessmentInfo) {
         this.asi = newAssessmentInfo;
      }

      protected JMenuItem prepMenuItem(String label) {
         JMenuItem menuItem = new JMenuItem(label);
         menuItem.addActionListener(this);
         popupMenu.add(menuItem);
         return menuItem;
      }

      public void preparePopupMenu() {
         menuItems = new JMenuItem[10];
         popupMenu = new JPopupMenu("");

         menuItems[0] = prepMenuItem("Add Score");
         menuItems[1] = prepMenuItem("Add Score Codes");
         menuItems[2] = prepMenuItem("Add Assessment Item");
         menuItems[3] = prepMenuItem("Update Assessment Item");

         menuItems[4] = prepMenuItem("Properties");
         menuItems[5] = prepMenuItem("Update");
         menuItems[6] = prepMenuItem("Update Score Codes");

         menuItems[7] = prepMenuItem("Delete Score");
         menuItems[8] = prepMenuItem("Associate");
         menuItems[9] = prepMenuItem("Change Assessment Name");

      }

      protected void enableDisableMenuItems(boolean[] disabledList) {
         for (int i = 0; i < disabledList.length; i++) {
            if (disabledList[i]) {
               menuItems[i].setEnabled(false);
            } else {
               menuItems[i].setEnabled(true);
            }
         }
      }

      public void mouseReleased(MouseEvent me) {
         showPopup(me);
      }

      public void mousePressed(MouseEvent me) {
         showPopup(me);
      }

      protected void showPopup(MouseEvent me) {
         if (me.isPopupTrigger()) {
            TreePath path = asTree.getPathForLocation(me.getX(), me.getY());
            if (path != null) {
               DefaultMutableTreeNode node = (DefaultMutableTreeNode) path
                     .getLastPathComponent();
               NodeInfo ni = (NodeInfo) node.getUserObject();
               if (ni.getData() instanceof AssessmentInfo) {
                  popupMenu.show(me.getComponent(), me.getX(), me.getY());

               } else if (ni.getData() instanceof String) {
                  popupMenu.show(me.getComponent(), me.getX(), me.getY());
                  selectedMandatoryField = ni;
               } else {
                  selectedScoreInfo = (ScoreInfo) ni.getData();
                  log.info("selectedScoreInfo=" + selectedScoreInfo);
                  popupMenu.show(me.getComponent(), me.getX(), me.getY());
               }
            }
         }
      }

      public void actionPerformed(ActionEvent e) {
         if (!(e.getSource() instanceof JMenuItem))
            return;
         JMenuItem item = (JMenuItem) e.getSource();
         if (item.getText().equals("Associate")) {
            handleAssociate();
         } else if (item.getText().equals("Add Score")) {
            handleAddScore();
         } else if (item.getText().equals("Add Subscore")) {
            handleAddSubScore();
         } else if (item.getText().equals("Properties")) {
            handleShowProperties();
         } else if (item.getText().equals("Add Score Codes")) {
            handleAddScoreCodes();
         } else if (item.getText().equals("Add Assessment Item")) {
            handleAddAssessmentItem();
         } else if (item.getText().equals("Update Assessment Item")) {
            handleUpdateAssessmentItem();
         } else if (item.getText().equals("Update")) {
            handleUpdate();
         } else if (item.getText().equals("Update Score Codes")) {
            handleUpdateScoreCodes();
         } else if (item.getText().equals("Delete Score")) {
            handleDeleteScore();
         } else if (item.getText().equals("Change Assessment Name")) {
            handleChangeAssessmentName();
         }
      }

      protected DefaultMutableTreeNode checkSelection() {
         DefaultMutableTreeNode selectedNode = (DefaultMutableTreeNode) asTree
               .getLastSelectedPathComponent();
         if (selectedNode == null) {
            CALMHelper.showError(asTree, "You need to select a node first!");
            return null;
         }
         if (!(selectedNode.getUserObject() instanceof NodeInfo)) {
            CALMHelper.showError(asTree,
                  "You need to select a score or assessment tree node!");
            return null;
         }
         return selectedNode;
      }

      protected void handleAddScoreCodes() {
         DefaultMutableTreeNode selectedNode = checkSelection();
         NodeInfo ni = (NodeInfo) selectedNode.getUserObject();

         if (ni.getData() instanceof ScoreInfo) {
            ScoreInfo si = (ScoreInfo) ni.getData();

            int option = this.scoreCodeAssocConfirmAnswer;
            if (this.scoreCodeAssocConfirmAnswer == -1) {
               // ask the user if s(he) wants to associate labels from the
               // form
               ConfirmPanel cp = new ConfirmPanel(
                     "Do you want to select labels for score codes from the form view?");

               option = JOptionPane.showConfirmDialog(asTree, cp,
                     "Label Selection", JOptionPane.YES_NO_OPTION);
               if (cp.rememberAnswer()) {
                  this.scoreCodeAssocConfirmAnswer = option;
               }
            }

            // ask the user if s(he) wants to associate labels from the form

            if (option == JOptionPane.NO_OPTION) {
               ScoreCodeDialog dlg = new ScoreCodeDialog(null, "Score Codes",
                     si);
               dlg.setLocationRelativeTo(asTree);
               int rc = dlg.showDialog();
               if (rc == BaseDialog.OK_PRESSED) {
                  List<ScoreCodeInfo> scoreCodes = dlg.getScoreCodes();
                  if (si.getScoreCodes() != null) {
                     si.getScoreCodes().clear();
                  }
                  for (ScoreCodeInfo sci : scoreCodes) {
                     si.addScoreCode(sci);
                  }
               }
               dlg.dispose();
            } else {
               // yes option, so start communication with the form view
               AssessmentMaintenancePanel.this.theScoreInfo = si;
               AssessmentMaintenancePanel.this.setAssociateLabels(true);
            }
         }
      }

      protected void handleDeleteScore() {
         DefaultMutableTreeNode selectedNode = checkSelection();
         if (selectedNode == null)
            return;
         NodeInfo ni = (NodeInfo) selectedNode.getUserObject();
         if (selectedNode.getChildCount() > 0) {
            CALMHelper.showError(asTree,
                  "You can only delete scores without sub scores!");
            return;
         }

         if (ni.getData() instanceof ScoreInfo) {
            ScoreInfo si = (ScoreInfo) ni.getData();
            int rc = JOptionPane.showConfirmDialog(asTree,
                  "Do you want to delete " + si.getScoreName()
                        + " permanently?", "Score Deletion",
                  JOptionPane.YES_NO_OPTION);
            if (rc == JOptionPane.NO_OPTION) {
               return;
            }
            si.getAssessment().removeScore(si);
            removeCurrentNode();
         }
      }

      @SuppressWarnings("unchecked")
      protected void adjustScoreSequences() {
         DefaultMutableTreeNode rootNode = (DefaultMutableTreeNode) treeModel
               .getRoot();
         Enumeration enumer = rootNode.preorderEnumeration();
         int count = 1;
         final int BIAS = 3;
         while (enumer.hasMoreElements()) {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode) enumer
                  .nextElement();
            NodeInfo ni = (NodeInfo) node.getUserObject();
            if (ni.getData() instanceof ScoreInfo) {
               ScoreInfo si = (ScoreInfo) ni.getData();
               si.setScoreSequence(count - BIAS);
            }
            ++count;
         }
      }

      protected void handleAddScore() {
         DefaultMutableTreeNode selectedNode = checkSelection();
         if (selectedNode == null)
            return;
         NodeInfo ni = (NodeInfo) selectedNode.getUserObject();

         if (ni.getData() instanceof AssessmentInfo) {
            // popup a dialog to get ScoreInfo
            ScoreInfoDialog sDlg = new ScoreInfoDialog(null, "New Score", null);
            sDlg.setLocationRelativeTo(asTree);
            int rc = sDlg.showDialog();
            if (rc == BaseDialog.OK_PRESSED) {
               ScoreInfo si = sDlg.getScoreInfo();
               // add the new node to the tree
               DefaultMutableTreeNode newNode = addNode(selectedNode,
                     new NodeInfo(si), true);
               int nodeSequence = AssessmentMaintenancePanel.getNodeSequence(
                     (DefaultMutableTreeNode) treeModel.getRoot(), newNode);
               prepareScoreInfo(si, null, nodeSequence
                     - (Constants.NUM_MANDATORY_FIELDS + 1));
               // set the score level
               si.setScoreLevel(selectedNode.getLevel() + 1);
               adjustScoreSequences();
            }
            sDlg.dispose();
         } else if (ni.getData() instanceof ScoreInfo) {
            ScoreInfo parentSi = (ScoreInfo) ni.getData();
            ScoreInfoDialog sDlg = new ScoreInfoDialog(null, "New Score", null);
            sDlg.setLocationRelativeTo(asTree);
            int rc = sDlg.showDialog();
            if (rc == BaseDialog.OK_PRESSED) {
               ScoreInfo si = sDlg.getScoreInfo();
               // add the new node to the tree
               DefaultMutableTreeNode newNode = addNode(
                     (DefaultMutableTreeNode) selectedNode, new NodeInfo(si),
                     true);
               int nodeSequence = AssessmentMaintenancePanel.getNodeSequence(
                     (DefaultMutableTreeNode) treeModel.getRoot(), newNode);
               prepareScoreInfo(si, parentSi, nodeSequence
                     - (Constants.NUM_MANDATORY_FIELDS + 1));
               si.setAssessment(this.asi);
               // set the score level
               si.setScoreLevel(selectedNode.getLevel() + 1);
               adjustScoreSequences();
            }
            sDlg.dispose();
         }
      }

      protected void prepareScoreInfo(ScoreInfo child, ScoreInfo parent,
            int nodeSequence) {
         child.setAssessment(this.asi);
         child.setScoreSequence(nodeSequence);
         child.setParent(parent);
         log.info("adding score info " + child.toString() + " to asi "
               + asi.toString());
         if (parent != null)
            log.info("parent=" + parent.toString());
         this.asi.addScoreInfo(child);
      }

      protected void handleAddSubScore() {
         DefaultMutableTreeNode selectedNode = checkSelection();
         if (selectedNode == null)
            return;

         NodeInfo ni = (NodeInfo) selectedNode.getUserObject();
         if (ni.getData() instanceof ScoreInfo) {
            ScoreInfo parentSi = (ScoreInfo) ni.getData();
            ScoreInfoDialog sDlg = new ScoreInfoDialog(null, "New Score", null);
            sDlg.setLocationRelativeTo(asTree);
            int rc = sDlg.showDialog();
            if (rc == BaseDialog.OK_PRESSED) {
               ScoreInfo si = sDlg.getScoreInfo();

               DefaultMutableTreeNode newNode = addNode(selectedNode,
                     new NodeInfo(si), true);
               int nodeSequence = AssessmentMaintenancePanel.getNodeSequence(
                     (DefaultMutableTreeNode) treeModel.getRoot(), newNode);
               prepareScoreInfo(si, parentSi, nodeSequence
                     - (Constants.NUM_MANDATORY_FIELDS + 1));

               adjustScoreSequences();
               // set the score level
               si.setScoreLevel(selectedNode.getLevel() + 1);
            }
            sDlg.dispose();
         }
      }

      protected void handleShowProperties() {
         DefaultMutableTreeNode selectedNode = checkSelection();
         if (selectedNode == null)
            return;
         NodeInfo ni = (NodeInfo) selectedNode.getUserObject();
         if (ni.getData() instanceof ScoreInfo) {
            ScoreInfo si = (ScoreInfo) ni.getData();
            try {
               PropertyEditorPanel pePanel = new PropertyEditorPanel(si);
               pePanel.setReadOnly(true);

               JOptionPane.showMessageDialog(AssessmentMaintenancePanel.this,
                     pePanel, "Properties", JOptionPane.PLAIN_MESSAGE);
            } catch (Exception x) {
               CALMHelper.showError(AssessmentMaintenancePanel.this, x
                     .getMessage());
            }
         }
      }

      protected void handleUpdate() {
         DefaultMutableTreeNode selectedNode = checkSelection();
         if (selectedNode == null)
            return;
         NodeInfo ni = (NodeInfo) selectedNode.getUserObject();

         TreePath parentPath = asTree.getSelectionPath();
         if (ni.getData() instanceof ScoreInfo) {
            ScoreInfo si = (ScoreInfo) ni.getData();
            ScoreInfoDialog sDlg = new ScoreInfoDialog(null, "Update Score", si);
            sDlg.setLocationRelativeTo(asTree);
            int rc = sDlg.showDialog();
            if (rc == BaseDialog.OK_PRESSED) {
               // the passed score info is updated and returned
               si = sDlg.getScoreInfo();
               ni.update();
               treeModel.nodeChanged((TreeNode) parentPath
                     .getLastPathComponent());
            }
            sDlg.dispose();
         } else if (ni.getData() instanceof AssessmentInfo) {
            AssessmentInfo asi = (AssessmentInfo) ni.getData();
            // popup a dialog to get new assessment name
            String newAssessmentName = JOptionPane.showInputDialog(asTree,
                  "Change the name of the Assessment:", asi.getName());
            if (newAssessmentName != null
                  && newAssessmentName.trim().length() > 0) {
               asi.setName(newAssessmentName.trim());
               ni.update();
               treeModel.nodeChanged(selectedNode);
               AssessmentMaintenancePanel.this.setTitle(asi.getName());
            }
         }
      }

      protected void handleUpdateScoreCodes() {
         DefaultMutableTreeNode selectedNode = checkSelection();
         if (selectedNode == null)
            return;
         NodeInfo ni = (NodeInfo) selectedNode.getUserObject();

         if (ni.getData() instanceof ScoreInfo) {
            ScoreInfo si = (ScoreInfo) ni.getData();
            ScoreCodeDialog sDlg = new ScoreCodeDialog(null,
                  "Update Score Codes", si);
            sDlg.setLocationRelativeTo(asTree);
            int rc = sDlg.showDialog();
            if (rc == BaseDialog.OK_PRESSED) {
               // the passed score info is updated and returned
               List<ScoreCodeInfo> scoreCodes = sDlg.getScoreCodes();
               if (si.getScoreCodes() != null) {
                  si.getScoreCodes().clear();
               }
               for (ScoreCodeInfo sci : scoreCodes) {
                  si.addScoreCode(sci);
               }
            }
            sDlg.dispose();
         }
      }

      class ConfirmPanel extends JPanel {
         private static final long serialVersionUID = -6133226454596728721L;
         JCheckBox rememberAnswer = new JCheckBox("Remember Answer");

         public ConfirmPanel(String text) {
            this.setLayout(new GridLayout(2, 1, 5, 5));
            this.add(new JLabel(text));
            this.add(rememberAnswer);
         }

         public boolean rememberAnswer() {
            return rememberAnswer.isSelected();
         }
      }

      protected void handleUpdateAssessmentItem() {
         DefaultMutableTreeNode selectedNode = checkSelection();
         if (selectedNode == null)
            return;
         NodeInfo ni = (NodeInfo) selectedNode.getUserObject();
         if (ni.getData() instanceof ScoreInfo) {
            ScoreInfo si = (ScoreInfo) ni.getData();
            ItemInfo itemInfo = si.getAssessment()
                  .findAsItem(si.getScoreName());

            if (itemInfo != null) {
               ItemInfo newItem = new ItemInfo(itemInfo.getAssessmentID(),
                     itemInfo.getScoreName());
               newItem.setItemLeadingText(itemInfo.getItemLeadingText());
               newItem.setItemTrailingText(itemInfo.getItemTrailingText());
               ItemInfoDialog iDlg = new ItemInfoDialog(null,
                     "Update Assessment Item", newItem);
               iDlg.setLocationRelativeTo(asTree);
               int rc = iDlg.showDialog();
               if (rc == BaseDialog.OK_PRESSED) {
                  itemInfo.setItemLeadingText(newItem.getItemLeadingText());
                  itemInfo.setItemTrailingText(newItem.getItemTrailingText());
               }
               iDlg.dispose();
            }
         }
      }

      protected void handleAddAssessmentItem() {
         DefaultMutableTreeNode selectedNode = checkSelection();
         if (selectedNode == null)
            return;
         NodeInfo ni = (NodeInfo) selectedNode.getUserObject();

         // TreePath parentPath = asTree.getSelectionPath();
         if (ni.getData() instanceof ScoreInfo) {
            ScoreInfo si = (ScoreInfo) ni.getData();
            ItemInfo itemInfo = new ItemInfo();
            itemInfo.setScoreName(si.getScoreName());
            int option = this.asItemAssocConfirmAnswer;

            if (this.asItemAssocConfirmAnswer == -1) {
               // ask the user if s(he) wants to associate labels from the
               // form
               ConfirmPanel cp = new ConfirmPanel(
                     "Do you want to select question text from the form view?");

               option = JOptionPane.showConfirmDialog(asTree, cp,
                     "Label Selection", JOptionPane.YES_NO_OPTION);
               if (cp.rememberAnswer()) {
                  this.asItemAssocConfirmAnswer = option;
               }
            }

            if (option == JOptionPane.NO_OPTION) {
               ItemInfoDialog iDlg = new ItemInfoDialog(null,
                     "Add Assessment Item", itemInfo);
               iDlg.setLocationRelativeTo(asTree);
               int rc = iDlg.showDialog();
               if (rc == BaseDialog.OK_PRESSED) {
                  si.getAssessment().addItem(itemInfo);
               }
               iDlg.dispose();
            } else {
               // yes option, so start communication with the form view
               AssessmentMaintenancePanel.this.theItemInfo = itemInfo;
               AssessmentMaintenancePanel.this.theScoreInfo = si;
               AssessmentMaintenancePanel.this.setAssociateLabels(true);
            }

         }
      }

      protected void handleChangeAssessmentName() {
         String asName = JOptionPane.showInputDialog(asTree,
               "New assessment name:", "Change Assessment Name",
               JOptionPane.PLAIN_MESSAGE);
         if (asName != null) {
            this.asi.setName(asName);
            amp.setTitle(asName);
         }
      }

      protected DefaultMutableTreeNode addNode(DefaultMutableTreeNode parent,
            NodeInfo child, boolean shouldBeVisible) {
         DefaultMutableTreeNode childNode = new DefaultMutableTreeNode(child);
         asTreeModel.insertNodeInto(childNode, parent, parent.getChildCount());
         if (shouldBeVisible) {
            asTree.scrollPathToVisible(new TreePath(childNode.getPath()));
         }

         return childNode;
      }

      protected void removeCurrentNode() {
         TreePath curSelection = asTree.getSelectionPath();
         if (curSelection == null)
            return;
         DefaultMutableTreeNode curNode = (DefaultMutableTreeNode) curSelection
               .getLastPathComponent();
         DefaultMutableTreeNode parent = (DefaultMutableTreeNode) curNode
               .getParent();
         if (parent != null) {
            treeModel.removeNodeFromParent(curNode);
         }
      }

      protected void handleAssociate() {
         DefaultMutableTreeNode selectedNode = checkSelection();
         if (selectedNode == null)
            return;
         NodeInfo ni = (NodeInfo) selectedNode.getUserObject();
         if (ni.getData() instanceof ScoreInfo) {
            ScoreInfo si = (ScoreInfo) ni.getData();
            ScoreAssociation sa = AssociationHelper.getInstance()
                  .findScoreAssociation(si);
            if (sa != null) {
               int rc = JOptionPane.showConfirmDialog(amp,
                     "Removing existing association first?", "",
                     JOptionPane.OK_CANCEL_OPTION);
               if (rc == JOptionPane.OK_OPTION) {
                  // remove the existing association
                  AssociationHelper.getInstance().removeScoreAssociation(sa);
               } else {
                  // just exit the method
                  return;
               }
            }

            // notify all the listeners via the bound property
            AssessmentMaintenancePanel.this
                  .setSelectedModelData(new SelectedModelDataInfo(si));

         } else if (ni.getData() instanceof String) {
            // for a mandatory field
            SelectedModelDataInfo smdi = new SelectedModelDataInfo(ni);
            // notify all the listeners via the bound property
            AssessmentMaintenancePanel.this.setSelectedModelData(smdi);
         }
      }

   } // ModelTreeMouseAdapter

   public static class ScoreInfoDialog extends BaseDialog {
      private static final long serialVersionUID = 6756384157259501917L;
      ScoreInfo scoreInfo;
      JTextField scoreNameField;
      JTextArea descriptionField;
      JComboBox typeField;
      JTextField typeFormatField;
      JComboBox secClassCombo;

      public ScoreInfoDialog(Frame owner, String title, ScoreInfo si) {
         super(owner, title, new String[] { "OK", "Cancel" });

         JPanel labelPanel = new JPanel(new GridLayout(4, 1));
         JPanel fieldPanel = new JPanel(new GridLayout(4, 1));

         labelPanel.add(new JLabel("Score Name:"));
         // labelPanel.add( new JLabel("Description:"));
         labelPanel.add(new JLabel("Score Type:"));
         labelPanel.add(new JLabel("Score Format:"));
         labelPanel.add(new JLabel("Security Classification:"));
         String[] types = new String[] { "integer", "varchar", "float",
               "boolean", "timestamp" };

         typeField = new JComboBox(types);
         scoreNameField = new JTextField(30);
         typeFormatField = new JTextField(30);
         // descriptionField = new JTextArea(5,30);

         List<SecurityClassificationInfo> secClassList = AssessmentInfo
               .getSecurityClassifications();

         String[] scList = new String[secClassList.size()];
         int i = 0;
         for (SecurityClassificationInfo sci : secClassList) {
            scList[i++] = sci.getSecurityClassification();

         }
         secClassCombo = new JComboBox(scList);

         fieldPanel.add(scoreNameField);
         // fieldPanel.add( descriptionField );
         fieldPanel.add(typeField);
         fieldPanel.add(typeFormatField);
         fieldPanel.add(secClassCombo);

         JPanel mainPanel = new JPanel(new GridLayout(1, 2));
         mainPanel.setBorder(BorderFactory.createTitledBorder("Score Info"));

         mainPanel.add(labelPanel);
         mainPanel.add(fieldPanel);

         getContentPane().add(mainPanel);

         if (si != null) {
            scoreNameField.setText(si.getScoreName());
            typeField.setSelectedIndex(findIndex(types, si.getScoreType()));
            typeFormatField.setText(si.getScoreTypeFormat());
            secClassCombo.setSelectedIndex(findIndex(scList, si
                  .getSecurityClassification()));
            this.scoreInfo = si;
         }

         setSize(new Dimension(265, 190));
      }

      protected int findIndex(String[] list, String value) {
         for (int i = 0; i < list.length; i++) {
            if (list[i].equals(value))
               return i;
         }
         return -1;
      }

      public ScoreInfo getScoreInfo() {
         return this.scoreInfo;
      }

      public void actionPerformed(ActionEvent e) {
         this.returnCode = BaseDialog.NONE;
         if (e.getSource() instanceof JButton) {
            JButton button = (JButton) e.getSource();
            if (button.getText().equalsIgnoreCase("ok")) {

               if (scoreNameField.getText().trim().length() == 0) {
                  CALMHelper.showError(this,
                        "You need to specify the score name!");
                  return;
               }

               if (this.scoreInfo == null) {
                  // creates a new score info
                  scoreInfo = new ScoreInfo(scoreNameField.getText().trim());

               }
               String newScoreName = scoreNameField.getText().trim();
               if (!newScoreName.equals(scoreInfo.getScoreName())) {
                  scoreInfo.setScoreName(newScoreName);
                  for (ScoreCodeInfo sci : scoreInfo.getScoreCodes()) {
                     sci.setScoreName(scoreInfo.getScoreName());
                  }
               }
               scoreInfo.setScoreType((String) typeField.getSelectedItem());
               scoreInfo.setScoreTypeFormat((String) typeFormatField.getText()
                     .trim());
               scoreInfo.setSecurityClassification((String) secClassCombo
                     .getSelectedItem());

               // TODO: First, go into ScoreInfo, and add in score type
               // format and timestamp.
               // Then, look for all uses of ScoreType and of typeField,
               // and make them be
               // able to handle timestamps.

               this.returnCode = OK_PRESSED;
            }
         }
         setVisible(false);
      }
   }
}
