package caslayout.ui.builder;

import java.awt.Dimension;
import java.awt.geom.Rectangle2D;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import caslayout.ui.AbstractDisplayComponent;
import caslayout.ui.ButtonDisplayComponent;
import caslayout.ui.ButtonType;
import caslayout.ui.CAGridLayout;
import caslayout.ui.CALMHelper;
import caslayout.ui.CALMConfig;
import caslayout.ui.CAPanel;
import caslayout.ui.CheckBoxDisplayComponent;
import caslayout.ui.Document;
import caslayout.ui.DropdownDisplayComponent;
import caslayout.ui.DynamicDropdownEditDialog;
import caslayout.ui.IDisplayComponent;
import caslayout.ui.Justification;
import caslayout.ui.LogicalGroup;
import caslayout.ui.LogicalGroupRepository;
import caslayout.ui.MultilineString;
import caslayout.ui.PercentCellConstraint;
import caslayout.ui.QuestionGroup;
import caslayout.ui.QuestionGroupRepository;
import caslayout.ui.RadioButtonDisplayComponent;
import caslayout.ui.ScoreLayoutPanel;
import caslayout.ui.TextAreaDisplayComponent;
import caslayout.ui.TextDisplayComponent;
import caslayout.ui.TextFieldDisplayComponent;
import caslayout.ui.model.AssessmentInfo;
import caslayout.ui.model.Association;
import caslayout.ui.model.AssociationHelper;
import caslayout.ui.model.FieldInfo;
import caslayout.ui.model.ItemInfo;
import caslayout.ui.model.MandatoryFieldAssociation;
import caslayout.ui.model.ScoreAssociation;
import caslayout.ui.model.ScoreCodeInfo;
import caslayout.ui.model.ScoreInfo;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: CALMDocBuilder.java,v 1.3 2008/10/13 23:58:07 bozyurt Exp $
 */
public class CALMDocBuilder implements BuilderConstants {
   protected Assessment assessment;
   protected AssessmentInfo assessmentInfo;
   protected Document doc;
   CALMConfig config;
   ScoreLayoutPanel slPanel;
   LogicalGroupRepository lgRep;
   QuestionGroupRepository qgRep;
   AssociationHelper assocHelper;
   Map<String, IDisplayComponent> mf2DispCompMap = new HashMap<String, IDisplayComponent>(
         11);

   public CALMDocBuilder(Assessment assessment, CALMConfig config) {
      super();
      this.assessment = assessment;
      this.config = config;
      slPanel = new ScoreLayoutPanel("Score Layout", true, null, config);
   }

   public CALMDocBuilder(Assessment assessment, CALMConfig config,
         ScoreLayoutPanel slPanel) {
      super();
      this.assessment = assessment;
      this.config = config;
      this.slPanel = slPanel;
   }

   public Document build() {
      lgRep = LogicalGroupRepository.getInstance();
      qgRep = QuestionGroupRepository.getInstance();
      qgRep.clear();
      lgRep.clear();
      assocHelper = AssociationHelper.getInstance();
      assocHelper.reset();

      this.assessmentInfo = buildAssessmentInfo();

      doc = new Document("empty", "");
      assocHelper.associateAssessment("", doc, assessmentInfo);

      String asTitle = assessment.getLabel();
      createCoverPage(asTitle);
      buildMandatoryFieldAssociations();

      // page numbers start at one
      int pageIdx = 2;
      int questionIdx = 0;
      for (Iterator<Page> iter = assessment.getPages().iterator(); iter
            .hasNext();) {
         Page page = (Page) iter.next();
         CAPanel pagePanel = slPanel.createDefaultPanel(false);
         pagePanel.setBounds(new Rectangle2D.Double(0, 0, 400, 400));

         caslayout.ui.Page p = new caslayout.ui.Page(pageIdx);
         p.setViewPanel(pagePanel);
         doc.addPage(p);

         Dimension dim = pagePanel.getPreferredSize();

         CAGridLayout gl = new CAGridLayout(3, 1, CAGridLayout.USE_PERCENTAGE);
         gl.setCellConstraint(new PercentCellConstraint(0, 0, 100.0, 10.0));
         gl.setCellConstraint(new PercentCellConstraint(1, 0, 100.0, 80.0));
         gl.setCellConstraint(new PercentCellConstraint(2, 0, 100.0, 10.0));

         CAPanel container = new CAPanel(gl, 0, 0, (int) dim.getWidth(),
               (int) dim.getHeight());
         pagePanel.add(container);
         dim = container.getPreferredSize();

         if (page.getTitle() != null) {
            // set title
            prepareTitle(page.getTitle(), container, 0);
         }

         int skipActionSize = page.getSkipActions().size();
         gl = new CAGridLayout(page.getQuestions().size() + skipActionSize, 1,
               CAGridLayout.USE_PERCENTAGE);
         CAPanel mainPanel = new CAPanel(gl, 0, 0, (int) dim.getWidth(),
               (int) dim.getHeight());
         container.add(mainPanel, 1);

         int idx = 0;
         for (Iterator<IQuestion> it = page.getQuestions().iterator(); it
               .hasNext();) {
            IQuestion q = it.next();
            if (q instanceof SingleScoreQuestion) {
               SingleScoreQuestion scq = (SingleScoreQuestion) q;
               buildQuestion(mainPanel, idx, scq, slPanel, questionIdx);
            } else {
               MultiScoreQuestion msq = (MultiScoreQuestion) q;
               buildMultipleScoreQuestion(mainPanel, idx, msq, slPanel,
                     questionIdx);
            }
            ++idx;
            SkipAction sa = page.getSkipActionAfter(q);
            if (sa != null) {
               buildSkipAction(mainPanel, idx, sa, slPanel);
               ++idx;
            }
            ++questionIdx;
         }
         boolean lastPage = !iter.hasNext();
         CAPanel navPanel = prepareNavigationPanel(dim, lastPage);
         container.add(navPanel, 2);

         ++pageIdx;
      }
      return doc;
   }

   protected CAPanel prepareNavigationPanel(Dimension dim, boolean lastPage) {
      CAGridLayout gl = new CAGridLayout(1, 3, CAGridLayout.USE_PERCENTAGE);

      gl.setCellConstraint(new PercentCellConstraint(0, 0, 75.0, 100.0));
      gl.setCellConstraint(new PercentCellConstraint(0, 1, 12.5, 100.0));
      gl.setCellConstraint(new PercentCellConstraint(0, 2, 12.5, 100.0));
      CAPanel fp = new CAPanel(gl, 0, 0, (int) dim.getWidth(), (int) dim
            .getHeight());
      ButtonDisplayComponent bdc = new ButtonDisplayComponent(0, 0, slPanel,
            ButtonType.PREVIOUS_BUTTON);
      fp.add(bdc, 1);
      if (lastPage) {
         bdc = new ButtonDisplayComponent(0, 0, slPanel,
               ButtonType.SUBMIT_BUTTON);
      } else {
         bdc = new ButtonDisplayComponent(0, 0, slPanel, ButtonType.NEXT_BUTTON);
      }
      fp.add(bdc, 2);
      return fp;

   }

   protected void prepareTitle(String title, CAPanel container, int cellIdx) {
      MultilineString mlString = new MultilineString(title);
      TextDisplayComponent tdc = new TextDisplayComponent(mlString, 0, 0,
            slPanel);
      tdc.setJustification(Justification.CENTER_JUSTIFICATION);
      container.add(tdc, cellIdx);
   }

   protected void createCoverPage(String title) {
      caslayout.ui.Page p = new caslayout.ui.Page(0);
      doc.addPage(p);
      CAPanel pagePanel = slPanel.createDefaultPanel(false);
      pagePanel.setBounds(new Rectangle2D.Double(0, 0, 400, 400));
      p.setViewPanel(pagePanel);
      slPanel.setCAContainerAt(pagePanel, 0);

      Dimension dim = pagePanel.getPreferredSize();
      CAGridLayout gl = new CAGridLayout(3, 1, CAGridLayout.USE_PERCENTAGE);
      gl.setCellConstraint(new PercentCellConstraint(0, 0, 100.0, 10.0));
      gl.setCellConstraint(new PercentCellConstraint(1, 0, 100.0, 80.0));
      gl.setCellConstraint(new PercentCellConstraint(2, 0, 100.0, 10.0));
      CAPanel container = new CAPanel(gl, 0, 0, (int) dim.getWidth(), (int) dim
            .getHeight());
      if (title != null) {
         prepareTitle(title, container, 0);
      }

      pagePanel.add(container, 0);
      dim = container.getPreferredSize();
      // prepare mandatory input fields
      gl = new CAGridLayout(5, 2, CAGridLayout.USE_PERCENTAGE);
      for (int i = 0; i < 5; i++) {
         gl.setCellConstraint(new PercentCellConstraint(i, 0, 55.0, 20.0));
         gl.setCellConstraint(new PercentCellConstraint(i, 1, 45.0, 20.0));
      }
      CAPanel mainPanel = new CAPanel(gl, 0, 0, (int) dim.getWidth(), (int) dim
            .getHeight());
      IDisplayComponent idc = null;
      mainPanel.add(createDisplayComponent(TEXT,
            Justification.RIGHT_JUSTIFICATION, null,
            "<b>Assessment Date (mm/dd/yyyy):</b>"), 0);
      idc = createDisplayComponent(TEXT_FIELD,
            Justification.LEFT_JUSTIFICATION, null, null);
      mf2DispCompMap.put("date", idc);
      mainPanel.add(idc, 1);
      mainPanel.add(createDisplayComponent(TEXT,
            Justification.RIGHT_JUSTIFICATION, null,
            "<b>Assessment Start Time (hh:mm):</b>"), 2);
      idc = createDisplayComponent(TEXT_FIELD,
            Justification.LEFT_JUSTIFICATION, null, null);
      mf2DispCompMap.put("time", idc);
      mainPanel.add(idc, 3);

      mainPanel
            .add(createDisplayComponent(TEXT,
                  Justification.RIGHT_JUSTIFICATION, null,
                  "<b>Informant ID:</b>"), 4);
      idc = createDisplayComponent(TEXT_FIELD,
            Justification.LEFT_JUSTIFICATION, null, null);
      mf2DispCompMap.put("informantid", idc);
      mainPanel.add(idc, 5);

      mainPanel.add(createDisplayComponent(TEXT,
            Justification.RIGHT_JUSTIFICATION, null,
            "<b>Informant Relation:</b>"), 6);
      idc = createDisplayComponent(DROPDOWN, Justification.LEFT_JUSTIFICATION,
            "informant", null);
      mf2DispCompMap.put("informantrelation", idc);
      mainPanel.add(idc, 7);

      mainPanel.add(createDisplayComponent(TEXT,
            Justification.RIGHT_JUSTIFICATION, null, "<b>Rater:</b>"), 8);
      idc = createDisplayComponent(DROPDOWN, Justification.LEFT_JUSTIFICATION,
            "dynamic", null);
      mf2DispCompMap.put("clinicalrater", idc);
      mainPanel.add(idc, 9);

      container.add(mainPanel, 1);

      // prepare footer
      gl = new CAGridLayout(1, 2, CAGridLayout.USE_PERCENTAGE);
      gl.setCellConstraint(new PercentCellConstraint(0, 0, 85.0, 100.0));
      gl.setCellConstraint(new PercentCellConstraint(0, 1, 15.0, 100.0));
      CAPanel footerPanel = new CAPanel(gl, 0, 0, (int) dim.getWidth(),
            (int) dim.getHeight());
      ButtonDisplayComponent bdc = new ButtonDisplayComponent(0, 0, slPanel,
            ButtonType.NEXT_BUTTON);
      footerPanel.add(bdc, 1);
      container.add(footerPanel, 2);

   }

   protected IDisplayComponent createDisplayComponent(String compType,
         Justification justification, String dropdownType, String label) {
      if (compType.equals(TEXT_FIELD)) {
         TextFieldDisplayComponent tf = new TextFieldDisplayComponent(0, 0,
               slPanel);
         if (justification != null) {
            tf.setJustification(justification);
         }
         return tf;
      } else if (compType.equals(DROPDOWN)) {
         if (dropdownType.equals("informant")) {
            Map<String, String> labelValueMap = CALMHelper
                  .getInformantOptionsMap();
            DropdownDisplayComponent ddc = new DropdownDisplayComponent(
                  labelValueMap, 0, 0, slPanel);
            if (justification != null) {
               ddc.setJustification(justification);
            }
            return ddc;
         } else if (dropdownType.equals("dynamic")) {
            DropdownDisplayComponent ddc = new DropdownDisplayComponent(
                  new HashMap<String, String>(7), 0, 0, slPanel);
            ddc.setPopulatedFromDatabase(true);
            ddc.setSqlQuery(DynamicDropdownEditDialog.DEFAULT_QUERY);
            if (justification != null) {
               ddc.setJustification(justification);
            }
            return ddc;
         } else {
            // TODO
            return null;
         }

      } else if (compType.equals(TEXT)) {
         MultilineString mlString = new MultilineString(label);
         TextDisplayComponent tdc = new TextDisplayComponent(mlString, 0, 0,
               slPanel);
         if (justification != null) {
            tdc.setJustification(justification);
         }
         return tdc;
      }
      return null;
   }

   protected void buildMandatoryFieldAssociations() {
      AbstractDisplayComponent adc = (AbstractDisplayComponent) mf2DispCompMap
            .get("date");
      MandatoryFieldAssociation mfa = new MandatoryFieldAssociation("date",
            Association.BIDIRECTIONAL, "date", adc);
      assocHelper.addMandatoryFieldAssociation(mfa);
      adc = (AbstractDisplayComponent) mf2DispCompMap.get("time");
      mfa = new MandatoryFieldAssociation("time", Association.BIDIRECTIONAL,
            "time", adc);
      assocHelper.addMandatoryFieldAssociation(mfa);

      adc = (AbstractDisplayComponent) mf2DispCompMap.get("informantid");
      mfa = new MandatoryFieldAssociation("informantid",
            Association.BIDIRECTIONAL, "informantid", adc);
      assocHelper.addMandatoryFieldAssociation(mfa);

      adc = (AbstractDisplayComponent) mf2DispCompMap.get("informantrelation");
      mfa = new MandatoryFieldAssociation("informantrelation",
            Association.BIDIRECTIONAL, "informantrelation", adc);
      assocHelper.addMandatoryFieldAssociation(mfa);

      adc = (AbstractDisplayComponent) mf2DispCompMap.get("clinicalrater");
      mfa = new MandatoryFieldAssociation("clinicalrater",
            Association.BIDIRECTIONAL, "clinicalrater", adc);
      assocHelper.addMandatoryFieldAssociation(mfa);

   }

   protected AssessmentInfo buildAssessmentInfo() {
      AssessmentInfo asi = new AssessmentInfo(this.assessment.getName());
      // assumption: one-to-one corresponce between a question and a score
      int sequence = 1;
      for (Iterator<Page> iter = assessment.getPages().iterator(); iter.hasNext();) {
         Page page = iter.next();
         for (Iterator<IQuestion> it = page.getQuestions().iterator(); it.hasNext();) {
            IQuestion q = it.next();
            if (q instanceof SingleScoreQuestion) {
               SubQuestion sq = (SubQuestion) q;
               buildScoreInfo(asi, sequence, sq);
               ++sequence;
            } else {
               MultiScoreQuestion msq = (MultiScoreQuestion) q;
               for (Iterator<SubQuestion> it2 = msq.getSubQuestions().iterator(); it2
                     .hasNext();) {
                  SubQuestion sq = it2.next();
                  buildScoreInfo(asi, sequence, sq);
                  ++sequence;
               }
            }
         }
      }
      // set mandatory fields also
      asi.addMandatoryField(new FieldInfo("date", "string"));
      asi.addMandatoryField(new FieldInfo("time", "string"));
      asi.addMandatoryField(new FieldInfo("informantid", "string"));
      asi.addMandatoryField(new FieldInfo("informantrelation", "string"));
      FieldInfo fi = new FieldInfo("clinicalrater", "string");
      fi.addMetadata("query", DynamicDropdownEditDialog.DEFAULT_QUERY);
      asi.addMandatoryField(fi);

      return asi;
   }

   private void buildScoreInfo(AssessmentInfo asi, int sequence, SubQuestion sq) {
      ScoreInfo si = new ScoreInfo(sq.getBindingScore());
      si.setScoreLevel(1);
      si.setAssessment(asi);
      si.setScoreType(sq.getScoreType());
      si.setSecurityClassification(sq.getSecurityClass());
      if (sq.getScoreSequence() != null) {
         si.setScoreSequence(sq.getScoreSequence().intValue());
      } else {
         si.setScoreSequence(sequence);
      }
      if (sq.getScoreTypeFormat() != null) {
         si.setScoreTypeFormat(sq.getScoreTypeFormat());
      } else {
         si.setScoreTypeFormat("");
      }

      buildScoreCodes(si, sq.getAnswerGroup());
      asi.addScoreInfo(si);
      // initialize assessment items also
      ItemInfo ii = new ItemInfo(new Long(0), si.getScoreName());
      ii.setItemLeadingText(sq.getTextBefore());
      asi.addItem(ii);
   }

   protected void buildScoreCodes(ScoreInfo si, AnswerGroup ag) {
      // assumption: if multiple choice, only one can be selected
      if (ag.getAnswers().size() > 1) {
         for (Iterator<Answer> iter = ag.getAnswers().iterator(); iter.hasNext();) {
            Answer ans = iter.next();
            ScoreCodeInfo sci = new ScoreCodeInfo(si.getScoreName(), ans
                  .getCode(), ans.getLabel(), si.getScoreType(), "");
            si.addScoreCode(sci);
         }
      }
   }

   protected void buildSkipAction(CAPanel parent, int cellIdx, SkipAction sa,
         ScoreLayoutPanel slPanel) {
      CAGridLayout gl = null;
      gl = new CAGridLayout(1, 1, CAGridLayout.USE_PERCENTAGE);
      Dimension dim = parent.getPreferredSize();
      CAPanel container = new CAPanel(gl, 0, 0, (int) dim.getWidth(), (int) dim
            .getHeight());
      parent.add(container, cellIdx);
      ButtonDisplayComponent bdc = new ButtonDisplayComponent(0, 0, slPanel,
            ButtonType.SKIP_BUTTON);
      bdc.setJustification(Justification.RIGHT_JUSTIFICATION);
      bdc.setJavascriptOnly(sa.isJavaScriptOnly());
      bdc.setLabel(sa.getButtonLabel());
      for (Iterator<Integer> iter = sa.getQuestionIdList2Skip().iterator(); iter
            .hasNext();) {
         Integer qid = iter.next();
         // since internal representation of question ids start from 0
         bdc.addQuestionID(new Integer(qid.intValue() - 1));
      }

      container.add(bdc, 0);
   }

   protected void buildMultipleScoreQuestion(CAPanel parent, int cellIdx,
         MultiScoreQuestion msq, ScoreLayoutPanel slPanel, int questionIdx) {
      QuestionGroup qg = new QuestionGroup(String.valueOf(questionIdx));
      qgRep.add(qg);

      qg.addMetadata("type", "multi-answer");
      qg.addMetadata("minAnswer", String.valueOf(msq.getMinAnswer()));
      qg.addMetadata("maxAnswer", String.valueOf(msq.getMaxAnswer()));

      CAGridLayout gl = null;
      gl = new CAGridLayout(3, 1, CAGridLayout.USE_PERCENTAGE);
      gl.setCellConstraint(new PercentCellConstraint(0, 0, 100.0, 5.0));
      gl.setCellConstraint(new PercentCellConstraint(1, 0, 100.0, 90.0));
      gl.setCellConstraint(new PercentCellConstraint(2, 0, 100.0, 5.0));
      Dimension dim = parent.getPreferredSize();
      CAPanel container = new CAPanel(gl, 0, 0, (int) dim.getWidth(), (int) dim
            .getHeight());
      parent.add(container, cellIdx);

      // add question title
      MultilineString mlString = new MultilineString(msq.getTextBefore());
      TextDisplayComponent tdc = new TextDisplayComponent(mlString, 0, 0,
            slPanel);
      qg.addElement(tdc);
      container.add(tdc, 0);
      gl.putToDispCompMap(0, 0, tdc);

      gl = new CAGridLayout(msq.getSubQuestions().size(), 1,
            CAGridLayout.USE_PERCENTAGE);
      dim = container.getPreferredSize();
      CAPanel sqPanel = new CAPanel(gl, 0, 0, (int) dim.getWidth(), (int) dim
            .getHeight());
      container.add(sqPanel, 1);

      int sqCellIdx = 0;
      for (Iterator<SubQuestion> iter = msq.getSubQuestions().iterator(); iter.hasNext();) {
         SubQuestion sq = iter.next();
         buildSubQuestion(sqPanel, sqCellIdx, sq, slPanel, qg);
         sqCellIdx++;
      }

      // prepare multi-answer button panel
      dim = container.getPreferredSize();
      CAGridLayout gl2 = new CAGridLayout(1, 3, CAGridLayout.USE_PERCENTAGE);
      gl2.setCellConstraint(new PercentCellConstraint(0, 0, 70.0, 100.0));
      gl2.setCellConstraint(new PercentCellConstraint(0, 1, 15.0, 100.0));
      gl2.setCellConstraint(new PercentCellConstraint(0, 1, 15.0, 100.0));
      CAPanel buttonPanel = new CAPanel(gl2, 0, 0, (int) dim.getWidth(),
            (int) dim.getHeight());
      ButtonDisplayComponent bdc = new ButtonDisplayComponent(0, 0, slPanel,
            ButtonType.ADD_BUTTON);
      buttonPanel.add(bdc, 1);
      gl2.putToDispCompMap(0, 1, bdc);
      qg.addElement(bdc);
      bdc = new ButtonDisplayComponent(0, 0, slPanel,
            ButtonType.REMOVE_LAST_BUTTON);
      buttonPanel.add(bdc, 2);
      gl2.putToDispCompMap(0, 2, bdc);
      qg.addElement(bdc);
      container.add(buttonPanel, 2);

   }

   protected void buildQuestion(CAPanel parent, int cellIdx,
         SingleScoreQuestion q, ScoreLayoutPanel slPanel, int questionIdx) {

      QuestionGroup qg = new QuestionGroup(String.valueOf(questionIdx));
      qgRep.add(qg);

      qg.addMetadata("type", "regular");
      if (q.getType() == SingleScoreQuestion.MULTIPLE_ANSWER) {
         qg.addMetadata("type", "multi-answer");
         qg.addMetadata("minAnswer", String.valueOf(q.getMinAnswer()));
         qg.addMetadata("maxAnswer", String.valueOf(q.getMaxAnswer()));
      }
      CAGridLayout gl = null;
      Location loc = q.answerGroup.loc;
      if (loc == Location.EAST || loc == Location.WEST) {
         gl = new CAGridLayout(1, 2, CAGridLayout.USE_PERCENTAGE);
      } else if (loc == Location.NORTH || loc == Location.SOUTH) {
         gl = new CAGridLayout(2, 1, CAGridLayout.USE_PERCENTAGE);
      }

      Dimension dim = parent.getPreferredSize();
      CAPanel container = new CAPanel(gl, 0, 0, (int) dim.getWidth(), (int) dim
            .getHeight());
      parent.add(container, cellIdx);
      // qg.addElement(container);
      MultilineString mlString = new MultilineString(q.getTextBefore());
      TextDisplayComponent tdc = new TextDisplayComponent(mlString, 0, 0,
            slPanel);
      qg.addElement(tdc);

      if (loc == Location.EAST || loc == Location.SOUTH) {
         container.add(tdc, 0);
         gl.putToDispCompMap(0, 0, tdc);
      } else {
         if (loc == Location.WEST) {
            gl.putToDispCompMap(0, 1, tdc);
         } else if (loc == Location.NORTH) {
            gl.putToDispCompMap(1, 0, tdc);
         }
         container.add(tdc, 1);
      }
      int cIdx = (loc == Location.WEST || loc == Location.NORTH) ? 0 : 1;
      buildAnswerGroup(container, cIdx, q.answerGroup, slPanel, questionIdx, q
            .getExpressionStr());
   }

   protected void buildSubQuestion(CAPanel parent, int cellIdx, SubQuestion q,
         ScoreLayoutPanel slPanel, QuestionGroup qg) {
      CAGridLayout gl = null;
      Location loc = q.answerGroup.loc;
      if (loc == Location.EAST || loc == Location.WEST) {
         gl = new CAGridLayout(1, 2, CAGridLayout.USE_PERCENTAGE);
      } else if (loc == Location.NORTH || loc == Location.SOUTH) {
         gl = new CAGridLayout(2, 1, CAGridLayout.USE_PERCENTAGE);
      }

      Dimension dim = parent.getPreferredSize();
      CAPanel container = new CAPanel(gl, 0, 0, (int) dim.getWidth(), (int) dim
            .getHeight());
      parent.add(container, cellIdx);
      MultilineString mlString = new MultilineString(q.textBefore);
      TextDisplayComponent tdc = new TextDisplayComponent(mlString, 0, 0,
            slPanel);
      qg.addElement(tdc);

      if (loc == Location.EAST || loc == Location.SOUTH) {
         container.add(tdc, 0);
         gl.putToDispCompMap(0, 0, tdc);
      } else {
         if (loc == Location.WEST) {
            gl.putToDispCompMap(0, 1, tdc);
         } else if (loc == Location.NORTH) {
            gl.putToDispCompMap(1, 0, tdc);
         }
         container.add(tdc, 1);
      }
      int cIdx = (loc == Location.WEST || loc == Location.NORTH) ? 0 : 1;
      buildAnswerGroup(container, cIdx, q.answerGroup, slPanel, Integer
            .parseInt(qg.getId()), null);
   }

   protected void buildAnswerGroup(CAPanel parent, int cellIdx, AnswerGroup ag,
         ScoreLayoutPanel slPanel, int questionIdx, String exprStr) {
      CAGridLayout gl = null;
      LogicalGroup lg = null;
      if (ag.isMultipleSelection()) {
         // create logical group
         lg = new LogicalGroup(String.valueOf(questionIdx));
         lgRep.add(lg);
      }

      QuestionGroup qg = qgRep.get(String.valueOf(questionIdx));
      assert (qg != null);

      int rowIdx = 0;
      int colIdx = 0;
      boolean rowWise = false;
      if (ag.getOrientation() == Orientation.VERTICAL) {
         gl = new CAGridLayout(ag.getAnswers().size(), 1,
               CAGridLayout.USE_PERCENTAGE);
         rowWise = true;
      } else {
         gl = new CAGridLayout(1, ag.getAnswers().size(),
               CAGridLayout.USE_PERCENTAGE);
      }
      Dimension dim = parent.getPreferredSize();
      CAPanel container = null;

      if (ag.getOwner().getType() == SingleScoreQuestion.MULTIPLE_ANSWER) {
         CAGridLayout gl2 = new CAGridLayout(1, 3, CAGridLayout.USE_PERCENTAGE);
         gl2.setCellConstraint(new PercentCellConstraint(0, 0, 70.0, 100.0));
         gl2.setCellConstraint(new PercentCellConstraint(0, 1, 15.0, 100.0));
         gl2.setCellConstraint(new PercentCellConstraint(0, 1, 15.0, 100.0));
         CAPanel mainPanel = new CAPanel(gl2, 0, 0, (int) dim.getWidth(),
               (int) dim.getHeight());
         ButtonDisplayComponent bdc = new ButtonDisplayComponent(0, 0, slPanel,
               ButtonType.ADD_BUTTON);
         mainPanel.add(bdc, 1);
         gl2.putToDispCompMap(0, 1, bdc);
         qg.addElement(bdc);
         bdc = new ButtonDisplayComponent(0, 0, slPanel,
               ButtonType.REMOVE_LAST_BUTTON);
         mainPanel.add(bdc, 2);
         gl2.putToDispCompMap(0, 2, bdc);
         qg.addElement(bdc);
         dim = mainPanel.getPreferredSize();
         container = new CAPanel(gl, 0, 0, (int) dim.getWidth(), (int) dim
               .getHeight());
         parent.add(mainPanel, cellIdx);
         mainPanel.add(container, 0);
         gl2.putToDispCompMap(0, 0, container);
      } else {
         container = new CAPanel(gl, 0, 0, (int) dim.getWidth(), (int) dim
               .getHeight());
         parent.add(container, cellIdx);
      }

      IDisplayComponent controlField = null;
      for (Iterator<Answer> iter = ag.getAnswers().iterator(); iter.hasNext();) {
         Answer ans = iter.next();
         IDisplayComponent idc = buildAnswerComp(ans, ag.getCompType(),
               slPanel, exprStr);
         if (lg != null) {
            lg.addElement(idc);
         } else {
            controlField = idc;
         }
         gl.putToDispCompMap(rowIdx, colIdx, idc);
         container.add(idc);
         qg.addElement(idc);
         if (rowWise) {
            rowIdx++;
         } else {
            colIdx++;
         }
      }

      // create association
      SubQuestion q = ag.getOwner();
      ScoreInfo si = assessmentInfo.findScore(q.getBindingScore());
      assert (si != null);
      ScoreAssociation sa = (ScoreAssociation) assocHelper.getAssocMap()
            .get(si);
      if (sa == null) {
         if (lg != null) {
            sa = new ScoreAssociation(si.getScoreName(),
                  Association.BIDIRECTIONAL, si, lg);
            sa.setScoreValue(lg);
         } else {
            sa = new ScoreAssociation(si.getScoreName(),
                  Association.BIDIRECTIONAL, si, controlField);
            sa.setScoreValue(controlField);
         }
         assocHelper.addScoreAssociation(sa);
      }
   }

   protected IDisplayComponent buildAnswerComp(Answer ans, String compType,
         ScoreLayoutPanel slPanel, String exprStr) {
      if (compType.equals(TEXT_FIELD)) {
         TextFieldDisplayComponent tf = new TextFieldDisplayComponent(0, 0,
               slPanel);
         ComponentConfig cc = ans.getComponentConfig();
         if (cc != null) {
            setCommonParams(cc, tf);

            tf.setMaxFieldLength(cc.getMaxFieldLength());
            tf.setFieldLength(cc.getFieldLength());
         }
         if (exprStr != null) {
            tf.getExpression().setExprString(exprStr);
         }
         return tf;
      } else if (compType.equals(RADIO_BUTTON)) {
         RadioButtonDisplayComponent rbf = new RadioButtonDisplayComponent(ans
               .getLabel(), 0, 0, slPanel);
         ComponentConfig cc = ans.getComponentConfig();
         if (cc != null) {
            setCommonParams(cc, rbf, Justification.CENTER_JUSTIFICATION);
         }
         return rbf;
      } else if (compType.equals(TEXTAREA)) {
         int numCols = DEFAULT_NUM_COLS;
         int numRows = DEFAULT_NUM_ROWS;
         ComponentConfig cc = ans.getComponentConfig();
         if (cc != null) {
            numRows = cc.getNumRows();
            numCols = cc.getNumCols();
         }
         TextAreaDisplayComponent tadc = new TextAreaDisplayComponent(0, 0,
               slPanel, numCols, numRows);
         if (cc != null) {
            setCommonParams(cc, tadc);
         }

         return tadc;
      } else if (compType.equalsIgnoreCase(DROPDOWN)) {
         ComponentConfig cc = ans.getComponentConfig();
         assert (cc != null);
         LinkedHashMap<String, String> labelValueMap = cc.getLabelValueMap();
         DropdownDisplayComponent ddc = new DropdownDisplayComponent(
               labelValueMap, 0, 0, slPanel);
         setCommonParams(cc, ddc);

         return ddc;
      } else if (compType.equalsIgnoreCase(CHECKBOX)) {
         CheckBoxDisplayComponent cbdc = new CheckBoxDisplayComponent(ans
               .getLabel(), 0, 0, slPanel);
         ComponentConfig cc = ans.getComponentConfig();
         if (cc != null) {
            setCommonParams(cc, cbdc, Justification.CENTER_JUSTIFICATION);
         }
         return cbdc;

      } else {
         throw new RuntimeException("Not a supported component type:"
               + compType);
      }
   }

   protected void setCommonParams(ComponentConfig cc, IDisplayComponent idc) {
      setCommonParams(cc, idc, Justification.LEFT_JUSTIFICATION);
   }

   protected void setCommonParams(ComponentConfig cc, IDisplayComponent idc,
         Justification defaultJustification) {
      idc.setJustification(cc.getJustification() != null ? Justification
            .findJustification(cc.getJustification()) : defaultJustification);
      if (cc.getCssClass() != null) {
         idc.setCSSClass(cc.getCssClass());
      }
   }

   public ScoreLayoutPanel getSlPanel() {
      return slPanel;
   }
}
