package caslayout.ui.builder;

import java.io.File;
import java.io.StringReader;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.jdom.Element;
import org.jdom.input.SAXBuilder;

import caslayout.ui.calc.ExpressionLexer;
import caslayout.ui.calc.ExpressionParser;
import caslayout.util.GenUtils;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: CALMDocReader.java,v 1.3 2008/10/13 23:58:07 bozyurt Exp $
 */
public class CALMDocReader {
   protected String calmDOCXMLFile;
   protected String schemaFile;

   public CALMDocReader(String calmDOCXMLFile, String schemaFile)
         throws Exception {
      super();
      if (!calmDOCXMLFile.endsWith(".calm")) {
         throw new Exception("Not a valid CALM document file!");
      }
      this.calmDOCXMLFile = calmDOCXMLFile;
      this.schemaFile = schemaFile;
   }

   public Assessment load() throws Exception {
      SAXBuilder builder = null;
      if (schemaFile != null) {
         File f = new File(schemaFile);

         builder = new SAXBuilder("org.apache.xerces.parsers.SAXParser", true); // validation
         builder.setFeature("http://apache.org/xml/features/validation/schema",
               true);

         builder
               .setProperty(
                     "http://apache.org/xml/properties/schema/external-noNamespaceSchemaLocation",
                     f.toURI().toURL().toString());
      } else {
         builder = new SAXBuilder(false);
      }
      org.jdom.Document doc = builder.build(this.calmDOCXMLFile);
      Element calmElem = doc.getRootElement();
      if (!calmElem.getName().equals("calm")) {
         throw new Exception(
               "Not a valid CALM document file: A CALM XML document must start with <calm> tag!");
      }
      Element asElem = calmElem.getChild("assessment");
      assert (asElem != null);
      String name = asElem.getChildTextTrim("name");
      assert (name != null);
      String label = null;
      if (asElem.getChild("label") != null) {
         label = asElem.getChildText("label");
      } else {
         label = name;
      }

      Assessment as = new Assessment(name, label);
      List<?> pageElems = asElem.getChild("pages").getChildren("page");
      for (Iterator<?> iter = pageElems.iterator(); iter.hasNext();) {
         Element pageElem = (Element) iter.next();
         Page page = preparePage(pageElem, name);
         as.addPage(page);
      }

      return as;
   }

   protected Page preparePage(Element pageElem, String asName) throws Exception {
      int pageNo = Integer.parseInt(pageElem.getAttributeValue("no"));
      String title = asName;
      if (pageElem.getChild("title") != null) {
         title = pageElem.getChildText("title");
      }
      Page p = new Page(pageNo, title);
      // skip actions
      List<?> saElems = pageElem.getChildren("skip");
      for (Iterator<?> iter = saElems.iterator(); iter.hasNext();) {
         Element saElem = (Element) iter.next();
         boolean javascriptOnly = false;
         if (saElem.getAttribute("javascriptOnly") != null) {
            javascriptOnly = saElem.getAttributeValue("javascriptOnly")
                  .equalsIgnoreCase("true");
         }
         String afterQuestionId = saElem.getAttributeValue("afterQuestionId");
         String buttonLabel = "Skip";
         if (saElem.getAttribute("label") != null) {
            buttonLabel = saElem.getAttributeValue("label");
         }
         SkipAction sa = new SkipAction(afterQuestionId, javascriptOnly,
               buttonLabel);
         String qidExpr = saElem.getChildTextTrim("questionIds");
         sa.prepareQuestionIdList(qidExpr);
         p.addSkipAction(sa);
      }

      // questions
      List<IQuestion> qList = new ArrayList<IQuestion>();
      List<?> qElems = pageElem.getChildren("q");
      for (Iterator<?> iter = qElems.iterator(); iter.hasNext();) {
         Element qElem = (Element) iter.next();
         String id = qElem.getAttributeValue("id");
         String bindingScore = qElem.getAttributeValue("bindingScore");
         String scoreType = qElem.getAttributeValue("scoreType");
         String text = bindingScore;
         if (qElem.getChild("text") != null) {
            text = qElem.getChildText("text");
         }
         SingleScoreQuestion q = new SingleScoreQuestion(id, bindingScore,
               scoreType);
         if (qElem.getAttribute("scoreTypeFormat") != null) {
            String formatStr = qElem.getAttributeValue("scoreTypeFormat");
            validateDateFormatString(formatStr);
            q.setScoreTypeFormat(formatStr);
         }

         parseQuestionType(qElem, q);

         q.setTextBefore(text);
         // for calculated fields
         parseExpression(qElem, bindingScore, scoreType, q);

         qList.add(q);
         parseAnswerGroup(qElem, bindingScore, scoreType, q);
      }

      // multi-score questions
      List<?> msqElems = pageElem.getChildren("msq");
      for (Iterator<?> iter = msqElems.iterator(); iter.hasNext();) {
         Element msqElem = (Element) iter.next();
         String id = msqElem.getAttributeValue("id");
         String text = msqElem.getChildText("text");
         MultiScoreQuestion msq = new MultiScoreQuestion(id);
         if (msqElem.getAttribute("maxAnswer") != null) {
            msq.setMaxAnswer(GenUtils.toInt(msqElem
                  .getAttributeValue("maxAnswer"), 5));
         }
         if (msqElem.getAttribute("minAnswer") != null) {
            msq.setMinAnswer(GenUtils.toInt(msqElem
                  .getAttributeValue("minAnswer"), 1));
         }
         if (msq.getMinAnswer() > msq.getMaxAnswer()) {
            msq.setMinAnswer(1);
         }
         msq.setTextBefore(text);
         List<?> sqElems = msqElem.getChildren("subq");
         for (Iterator<?> it = sqElems.iterator(); it.hasNext();) {
            Element sqElem = (Element) it.next();
            text = sqElem.getChildText("text");
            String bindingScore = sqElem.getAttributeValue("bindingScore");
            String scoreType = sqElem.getAttributeValue("scoreType");
            SubQuestion sq = new SubQuestion(bindingScore, scoreType);
            sq.setTextBefore(text);

            if (sqElem.getAttribute("scoreTypeFormat") != null) {
               String formatStr = sqElem.getAttributeValue("scoreTypeFormat");
               validateDateFormatString(formatStr);
               sq.setScoreTypeFormat(formatStr);
            }
            parseAnswerGroup(sqElem, bindingScore, scoreType, sq);
            msq.addSubQuestion(sq);
         }
         qList.add(msq);
      }

      Collections.sort(qList, new Comparator<IQuestion>() {
         public int compare(IQuestion o1, IQuestion o2) {
            return o1.getOrderIndex() - o1.getOrderIndex();
         }

      });
      for (Iterator<IQuestion> iter = qList.iterator(); iter.hasNext();) {
         IQuestion q = iter.next();
         p.addQuestion(q);
      }

      return p;
   }

   protected void parseExpression(Element qElem, String bindingScore,
         String scoreType, SingleScoreQuestion q) throws Exception {
      if (qElem.getChild("expression") != null) {
         if (q.getType() != SingleScoreQuestion.SINGLE_ANSWER) {
            throw new Exception("A calculated field '" + bindingScore
                  + "' cannot have multiple answers!");
         }
         if (!scoreType.equalsIgnoreCase("float")
               && !scoreType.equalsIgnoreCase("int")) {
            throw new Exception("A calculated field '" + bindingScore
                  + "' must be numeric!");
         }
         String expr = qElem.getChildTextTrim("expression");
         validateExpressionSyntax(expr);
         q.setExpressionStr(expr);
      }
   }

   protected void parseAnswerGroup(Element qElem, String bindingScore,
         String scoreType, SubQuestion q) throws Exception {
      Element agElem = qElem.getChild("answerGroup");
      assert (agElem != null);
      String location = agElem.getAttributeValue("location");
      String orientation = agElem.getAttributeValue("orientation");
      String compType = agElem.getAttributeValue("compType");
      AnswerGroup ag = new AnswerGroup(Location.find(location), Orientation
            .find(orientation), compType, scoreType, q);
      q.setAnswerGroup(ag);

      List<?> ansElems = agElem.getChildren("answer");

      if ((q instanceof SingleScoreQuestion)
            && ((SingleScoreQuestion) q).getExpressionStr() != null
            && (!compType.equals(BuilderConstants.TEXT_FIELD) || !ansElems
                  .isEmpty())) {
         throw new Exception("A calculated field '" + bindingScore
               + "' must be of compType 'text' and must have no <answer> tags!");
      }
      if (ansElems == null || ansElems.isEmpty()) {

         if (compType.equals(BuilderConstants.TEXT_FIELD)
               || compType.equals(BuilderConstants.TEXTAREA)) {
            ag.addAnswer(new Answer("", ""));
         } else {
            throw new Exception("Needs a list of <answer> tags for question "
                  + q.getId());
         }
      } else {
         for (Iterator<?> it = ansElems.iterator(); it.hasNext();) {
            Element ansElem = (Element) it.next();
            String label = ansElem.getAttributeValue("label");
            String code = ansElem.getAttributeValue("code");
            if (code == null) {
               code = label;
            }
            Answer answer = new Answer(label, code);
            ag.addAnswer(answer);
            Element compElem = ansElem.getChild("component");
            if (compElem != null) {
               prepareComponentConfig(compType, answer, compElem);
            }
         }
      }
   }

   protected void parseQuestionType(Element qElem, SingleScoreQuestion q) {
      if (qElem.getAttribute("type") != null) {
         String typeStr = qElem.getAttributeValue("type");
         if (typeStr.equals("multiple-answer")) {
            q.setType(SubQuestion.MULTIPLE_ANSWER);
            if (qElem.getAttribute("maxAnswer") != null) {
               q.setMaxAnswer(GenUtils.toInt(qElem
                     .getAttributeValue("maxAnswer"), 5));
            }
            if (qElem.getAttribute("minAnswer") != null) {
               q.setMinAnswer(GenUtils.toInt(qElem
                     .getAttributeValue("minAnswer"), 1));
            }
            if (q.getMinAnswer() > q.getMaxAnswer()) {
               q.setMinAnswer(1);
            }
         } else {
            q.setType(SingleScoreQuestion.SINGLE_ANSWER);
         }

      }
   }

   protected void validateDateFormatString(String formatStr)
         throws ParseException {
      SimpleDateFormat sdf = new SimpleDateFormat(formatStr);
      String formatedDate = sdf.format(new Date());
      sdf.parse(formatedDate);
   }

   protected void validateExpressionSyntax(String exprStr) throws Exception {
      if (!exprStr.endsWith(";")) {
         exprStr += ";";
      }
      StringReader in = new StringReader(exprStr);
      ExpressionLexer lexer = new ExpressionLexer(in);
      ExpressionParser parser = new ExpressionParser(lexer);
      parser.expr();
   }

   protected void prepareComponentConfig(String compType, Answer answer,
         Element compElem) throws Exception {
      String justification = (compElem.getAttribute("justification") != null) ? compElem
            .getAttributeValue("justification")
            : null;
      String cssClass = (compElem.getAttribute("cssClass") != null) ? compElem
            .getAttributeValue("cssClass") : null;

      ComponentConfig cc = new ComponentConfig("", justification, cssClass);
      answer.setComponentConfig(cc);
      if (compType.equalsIgnoreCase(BuilderConstants.DROPDOWN)) {
         Element lvSetElem = compElem.getChild("labelValueSet");
         if (lvSetElem == null) {
            throw new Exception(
                  "For a drop-down component a <labelValueSet> tag is needed!");
         }
         List<?> lvElems = lvSetElem.getChildren("lv");
         for (Iterator<?> it2 = lvElems.iterator(); it2.hasNext();) {
            Element lvElem = (Element) it2.next();
            String label = lvElem.getAttributeValue("label");
            String value = (lvElem.getAttribute("value") != null) ? lvElem
                  .getAttributeValue("value") : label;
            cc.addLabelValue(label, value);
         }
      } else if (compType.equalsIgnoreCase(BuilderConstants.TEXT_FIELD)) {
         if (compElem.getAttribute("fieldLength") != null) {
            cc.setFieldLength(Integer.parseInt(compElem
                  .getAttributeValue("fieldLength")));
         }
         if (compElem.getAttribute("maxFieldLength") != null) {
            cc.setMaxFieldLength(Integer.parseInt(compElem
                  .getAttributeValue("maxFieldLength")));
         }
         // TODO
         assert (cc.getMaxFieldLength() >= cc.getFieldLength());

      } else if (compType.equalsIgnoreCase(BuilderConstants.TEXTAREA)) {
         if (compElem.getAttribute("numCols") != null) {
            cc.setNumCols(Integer.parseInt(compElem
                  .getAttributeValue("numCols")));
         }
         if (compElem.getAttribute("numRows") != null) {
            cc.setNumRows(Integer.parseInt(compElem
                  .getAttributeValue("numRows")));
         }
      }
   }

}
