package caslayout.ui.db.services;

import java.sql.Timestamp;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.hibernate.HibernateException;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;

import caslayout.importing.common.DataRow;
import caslayout.importing.common.Score;
import caslayout.ui.CALMConfig;
import caslayout.ui.db.AbstractVO;
import caslayout.ui.db.AnimalSpecies;
import caslayout.ui.db.Assessment;
import caslayout.ui.db.AssessmentBoolean;
import caslayout.ui.db.AssessmentFloat;
import caslayout.ui.db.AssessmentInformant;
import caslayout.ui.db.AssessmentInteger;
import caslayout.ui.db.AssessmentStatus;
import caslayout.ui.db.AssessmentTimestamp;
import caslayout.ui.db.AssessmentVarchar;
import caslayout.ui.db.BaseVO;
import caslayout.ui.db.DataClassification;
import caslayout.ui.db.DatabaseUser;
import caslayout.ui.db.Experiment;
import caslayout.ui.db.Protocol;
import caslayout.ui.db.ResearchGroup;
import caslayout.ui.db.Segment;
import caslayout.ui.db.SegmentKey;
import caslayout.ui.db.SessionFactoryFactory;
import caslayout.ui.db.Site;
import caslayout.ui.db.StoredAssessment;
import caslayout.ui.db.Study;
import caslayout.ui.db.StudyKey;
import caslayout.ui.db.SubjExperiment;
import caslayout.ui.db.Subject;
import caslayout.ui.db.TableID;
import caslayout.ui.db.Visit;
import caslayout.ui.db.VisitKey;
import caslayout.util.IFunctor;

/**
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentDataService.java,v 1.1.2.3 2007/07/28 01:58:54
 *          bozyurt Exp $
 */

public class AssessmentDataService {
	protected CALMConfig config;
	protected String securityClassification;
	protected Map<Object, Object> tableIDMap;
	protected Map<Object, Object> informantRelationMap;
	protected Map<Object, Object> assessmentStatusMap = null;
	protected Map<Object, Object> dataClassMap;
	protected Map<Object, Object> researchGroupMap;
	protected Map<Object, Object> siteMap;
	protected Map<Object, Object> animalSpeciesMap;
	public final static String MISSING_VALUE_CODE = "-99999";
	public final static String MISSING_VALUE_MSG = "missing value";

	protected static AssessmentDataService instance = null;

	protected AssessmentDataService(CALMConfig config) {
		this.config = config;
	}

	public static synchronized AssessmentDataService getInstance(
			CALMConfig config) {
		if (instance == null) {
			instance = new AssessmentDataService(config);
		}
		return instance;
	}

	public static synchronized AssessmentDataService getInstance() {
		return instance;
	}

	protected synchronized Map<Object, Object> getTableIDMap() {
		if (tableIDMap == null) {
			IFunctor functor = new IFunctor() {
				public Object evaluate(Object argObj) {
					TableID tableID = (TableID) argObj;
					return tableID.getTableName().toLowerCase();
				}
			};
			tableIDMap = getPersistentDataMap("TableID", functor);
		}
		return tableIDMap;
	}

	protected synchronized Map<Object, Object> getInformantRelationMap() {
		if (informantRelationMap == null) {
			IFunctor functor = new IFunctor() {
				public Object evaluate(Object argObj) {
					AssessmentInformant ir = (AssessmentInformant) argObj;
					return ir.getInformantRelation();
				}
			};
			informantRelationMap = getPersistentDataMap("AssessmentInformant",
					functor);
		}
		return informantRelationMap;
	}

	protected synchronized Map<Object, Object> getAssessmentStatusMap() {
		if (assessmentStatusMap == null) {
			IFunctor functor = new IFunctor() {
				public Object evaluate(Object argObj) {
					AssessmentStatus as = (AssessmentStatus) argObj;
					return as.getName();
				}
			};
			assessmentStatusMap = getPersistentDataMap("AssessmentStatus",
					functor);
		}
		return assessmentStatusMap;
	}

	protected synchronized Map<Object, Object> getResearchGroupMap() {
		if (researchGroupMap == null) {
			IFunctor functor = new IFunctor() {
				public Object evaluate(Object argObj) {
					ResearchGroup rg = (ResearchGroup) argObj;
					return rg.getName().toLowerCase();
				}
			};
			researchGroupMap = getPersistentDataMap("ResearchGroup", functor);
		}
		return researchGroupMap;
	}

	protected synchronized Map<Object, Object> getAnimalSpeciesMap() {
		if (animalSpeciesMap == null) {
			IFunctor functor = new IFunctor() {
				public Object evaluate(Object argObj) {
					AnimalSpecies as = (AnimalSpecies) argObj;
					return as.getName().toLowerCase();
				}
			};
			animalSpeciesMap = getPersistentDataMap("AnimalSpecies", functor);
		}
		return animalSpeciesMap;
	}

	protected synchronized Map<Object, Object> getSiteMap() {
		if (siteMap == null) {
			IFunctor functor = new IFunctor() {
				public Object evaluate(Object argObj) {
					Site site = (Site) argObj;
					return site.getSiteName().toLowerCase();
				}
			};
			siteMap = getPersistentDataMap("Site", functor);
		}
		return siteMap;
	}

	protected synchronized Map<Object, Object> getDataClassMap() {
		if (dataClassMap == null) {
			IFunctor functor = new IFunctor() {
				public Object evaluate(Object argObj) {
					DataClassification dc = (DataClassification) argObj;
					return dc.getName().toLowerCase();
				}
			};
			dataClassMap = getPersistentDataMap("DataClassification", functor);
		}
		return dataClassMap;
	}

	protected synchronized Map<Object, Object> getPersistentDataMap(String domainObjName,
			IFunctor functor) {
		SessionFactory sessionFactory = SessionFactoryFactory
				.getSessionFactory(config);
		Session session = null;
		try {
			session = sessionFactory.openSession();
			List<?> list = session.createQuery("from " + domainObjName).list();
			Map<Object, Object> map = new HashMap<Object, Object>();
			for (Iterator<?> it = list.iterator(); it.hasNext();) {
				Object item = it.next();
				Object key = functor.evaluate(item);

				map.put(key, item);
			}
			return map;
		} finally {
			if (session != null)
				session.close();
		}
	}

	public List<?> getExperiments() {
		SessionFactory sessionFactory = SessionFactoryFactory
				.getSessionFactory(config);
		Session session = null;
		List<?> expList = null;
		try {
			session = sessionFactory.openSession();
			expList = session.createQuery("from Experiment").list();
			return expList;
		} finally {
			if (session != null)
				session.close();
		}
	}

	public Map<Object, Object> getProtocolMap() {
		IFunctor functor = new IFunctor() {
			public Object evaluate(Object argObj) {
				Protocol pr = (Protocol) argObj;
				return pr.getProtocolKey().getProtocolId() + "_"
						+ pr.getProtocolKey().getProtocolVersion();
			}
		};
		Map<Object, Object> protocolMap = getPersistentDataMap("Protocol", functor);
		return protocolMap;
	}

	public Map<Object, Object> getAssessmentMap() {
		IFunctor functor = new IFunctor() {
			public Object evaluate(Object argObj) {
				Assessment as = (Assessment) argObj;
				return as.getName();
			}
		};
		return getPersistentDataMap("Assessment", functor);
	}

	public void deleteSubjectVisits(String experimentName, String visitType)
			throws Exception {
		SessionFactory sessionFactory = SessionFactoryFactory
				.getSessionFactory(config);

		Session session = null;
		Transaction trx = null;
		try {
			session = sessionFactory.openSession();
			trx = session.beginTransaction();
			Experiment exp = (Experiment) session.createQuery(
					"from Experiment e where e.name = :expName").setParameter(
					"expName", experimentName).uniqueResult();

			List<?> subjectList = session.createQuery(
					"from SubjExperiment s where s.experiment = :exp")
					.setParameter("exp", exp).list();
			int i = 0;
			Set<SubjExperiment> excludedSubjects = new HashSet<SubjExperiment>(11);
			// excludedSubjects.add("001071753990");
			// excludedSubjects.add("000742488013");
			// excludedSubjects.add("001012126889");
			// excludedSubjects.add("001025290760");
			// excludedSubjects.add("001084262659");
			// excludedSubjects.add("001094450869");
			for (Iterator<?> it = subjectList.iterator(); it.hasNext();) {
				SubjExperiment se = (SubjExperiment) it.next();
				if (excludedSubjects.contains(se.getSubjectID())) {
					continue;
				}
				System.out.println("deleting " + se.getSubjectID());
				deleteVisits(se.getSubjectID(), experimentName, visitType);

				session.delete(se);
				i++;
			}

			trx.commit();
		} catch (Throwable t) {
			if (trx != null)
				trx.rollback();
			throw new Exception(t);
		} finally {
			if (session != null)
				session.close();
		}
	}

	public void deleteVisits(String subjectID, String experimentName,
			String visitType) throws Exception {
		SessionFactory sessionFactory = SessionFactoryFactory
				.getSessionFactory(config);

		Session session = null;
		Transaction trx = null;
		try {
			session = sessionFactory.openSession();
			trx = session.beginTransaction();

			deleteVisits(session, subjectID, experimentName, visitType);

			trx.commit();
		} catch (Throwable t) {
			if (trx != null)
				trx.rollback();
			throw new Exception(t);
		} finally {
			if (session != null)
				session.close();
		}
	}

	protected void deleteVisits(Session session, String subjectID,
			String experimentName, String visitType) throws Exception,
			HibernateException {
		List<?> visits = getStudySegmentsForSubject(subjectID, experimentName,
				visitType);

		for (Iterator<?> it = visits.iterator(); it.hasNext();) {
			Visit v = (Visit) it.next();
			for (Iterator<Study> it2 = v.getStudies().iterator(); it2.hasNext();) {
				Study study = it2.next();
				for (Iterator<Segment> it3 = study.getSegments().iterator(); it3
						.hasNext();) {
					Segment seg = it3.next();
					session.delete(seg);
				}
				session.delete(study);
			}
			session.delete(v);
		}
	}

	public void deleteAssessmentDataFor(String assessmentName) throws Exception {
		SessionFactory sessionFactory = SessionFactoryFactory
				.getSessionFactory(config);

		Session session = null;
		Transaction trx = null;
		try {
			session = sessionFactory.openSession();
			trx = session.beginTransaction();
			Assessment as = (Assessment) session.createQuery(
					"from Assessment a where a.name = :name").setParameter(
					"name", assessmentName).uniqueResult();
			if (as != null) {
				List<?> saList = session.createQuery(
						"from StoredAssessment s where s.assessmentID =:asID")
						.setParameter("asID", as.getID()).list();
				for (Iterator<?> it = saList.iterator(); it.hasNext();) {
					StoredAssessment sa = (StoredAssessment) it.next();
					session.createQuery(
							"delete from AssessmentInteger a where "
									+ "a.storedAssessmentID = :saID")
							.setParameter("saID", sa.getUniqueID())
							.executeUpdate();

					session.createQuery(
							"delete from AssessmentVarchar a where "
									+ "a.storedAssessmentID = :saID")
							.setParameter("saID", sa.getUniqueID())
							.executeUpdate();
					session.createQuery(
							"delete from AssessmentFloat a where "
									+ "a.storedAssessmentID = :saID")
							.setParameter("saID", sa.getUniqueID())
							.executeUpdate();
					session.createQuery(
							"delete from AssessmentTimestamp a where "
									+ "a.storedAssessmentID = :saID")
							.setParameter("saID", sa.getUniqueID())
							.executeUpdate();
					session.createQuery(
							"delete from AssessmentBoolean a where "
									+ "a.storedAssessmentID = :saID")
							.setParameter("saID", sa.getUniqueID())
							.executeUpdate();
					session.delete(sa);
				}
			}
			trx.commit();
		} catch (Throwable t) {
			if (trx != null)
				trx.rollback();
			throw new Exception(t);
		} finally {
			if (session != null)
				session.close();
		}
	}

	public Visit getVisitWithStudySegmentForSubject(Session session,
			DataRow dataRow, String experimentName, String visitType,
			String researchGroupName, Date visitDate, Date segmentTimestamp,
			Protocol protocol, Assessment as) throws Exception {
		Visit visit = null;

		Experiment exp = (Experiment) session.createQuery(
				"from Experiment e where e.name like :name").setParameter(
				"name", experimentName + "%").uniqueResult();

		Map<Object, Object> siteMap = getSiteMap();
		Site primarySite = null;
		for (Iterator<Object> it = siteMap.values().iterator(); it.hasNext();) {
			Site site = (Site) it.next();
			if (site.getIsPrimary().booleanValue()) {
				primarySite = site;
				break;
			}
		}

		List<?> seList = session.createQuery(
				"from SubjExperiment s "
						+ "where s.experiment=:expID and s.subjectID= :sid")
				.setParameter("expID", exp).setParameter("sid",
						dataRow.getSubjectID()).list();

		if (seList.isEmpty()) {
			DatabaseUser dbUser = getDatabaseUser(session);
			List<?> subjectList = session.createQuery(
					"from Subject s where s.subjectID=:sid").setParameter(
					"sid", dataRow.getSubjectID()).list();
			Subject subject = null;
			if (subjectList.isEmpty()) {
				subject = new Subject();
				subject.setSubjectID(dataRow.getSubjectID());
				subject.setExtensionName("humanSubject");
				setCommonProperties(dbUser, "nc_humansubject", subject);
				Site site = (Site) getSiteMap().get(
						dataRow.getSite().toLowerCase());
				subject.setSiteID(site.getUniqueID());
				// if the site is not the local site set remote to
				if (primarySite != null) {
					if (primarySite == site) {
						subject.setRemote(new Boolean(false));
					} else {
						subject.setRemote(new Boolean(true));
					}
				} else {
					subject.setRemote(new Boolean(false));
				}
				AnimalSpecies ansp = (AnimalSpecies) getAnimalSpeciesMap().get(
						"researchsubject");
				subject.setAnimalSpecies(ansp);

				session.save(subject);
			} else {
				subject = (Subject) subjectList.get(0);
			}

			SubjExperiment se = new SubjExperiment();
			se.setExperiment(exp);
			se.setSubjectID(dataRow.getSubjectID());
			ResearchGroup rg = (ResearchGroup) getResearchGroupMap().get(
					researchGroupName.trim().toLowerCase());
			if (rg == null) {
				throw new Exception("Not a valid research group:"
						+ researchGroupName);
			}
			se.setResearchGroup(rg);
			setCommonProperties(dbUser, "nc_subjexperiment", se);
			session.save(se);
			// create a visit, a study (named FMRI) and a segment
			Long visitID = new Long(1);
			visit = createVisitSegmentStudy(visitID, dataRow, visitType,
					visitDate, segmentTimestamp, protocol, session, exp,
					dbUser, as);
		} else {
			List<?> visitList = session
					.createQuery(
							"from Visit v where v.visitKey.subjectID = :sid "
									+ "and v.visitKey.experimentID=:expID and v.visitType=:vtype"
									+ " and v.timeStamp = :ts").setParameter(
							"sid", dataRow.getSubjectID()).setParameter(
							"expID", exp.getUniqueID()).setParameter("vtype",
							visitType).setParameter("ts", visitDate).list();
			if (visitList.isEmpty()) {
				DatabaseUser dbUser = getDatabaseUser(session);

				Long visitID = getNewVisitID(session, exp, dataRow);
				visit = createVisitSegmentStudy(visitID, dataRow, visitType,
						visitDate, segmentTimestamp, protocol, session, exp,
						dbUser, as);

			} else {
				assert (visitList.size() == 1);
				visit = (Visit) visitList.get(0);
				// make sure a study exists
				List<?> studyList = session
						.createQuery(
								"from Study s where s.studyKey.visitID=:vid "
										+ "and s.studyKey.subjectID=:sid and s.studyKey.experimentID=:expID "
										+ "and s.name=:name")
						.setParameter("vid", visit.getVisitKey().getVisitID())
						.setParameter("sid", visit.getVisitKey().getSubjectID())
						.setParameter("expID",
								visit.getVisitKey().getExperimentID())
						.setParameter("name", "FMRI").list();
				if (studyList.isEmpty()) {
					Study study = addStudy(session, visit, "FMRI");
					addSegment(session, study, segmentTimestamp, protocol, as);
				} else {
					assert (studyList.size() == 1);
					Study study = (Study) studyList.get(0);
					visit.addStudy(study);
					addSegment(session, study, segmentTimestamp, protocol, as);
				}
			}
		}
		return visit;

	}

	public Visit getVisitWithStudySegmentForSubject(DataRow dataRow,
			String experimentName, String visitType, String researchGroupName,
			Date visitDate, Date segmentTimestamp, Protocol protocol,
			Assessment as) throws Exception {
		SessionFactory sessionFactory = SessionFactoryFactory
				.getSessionFactory(config);

		Session session = null;
		Visit visit = null;
		Transaction trx = null;
		try {
			session = sessionFactory.openSession();
			trx = session.beginTransaction();
			visit = getVisitWithStudySegmentForSubject(session, dataRow,
					experimentName, visitType, researchGroupName, visitDate,
					segmentTimestamp, protocol, as);

			trx.commit();
			return visit;
		} catch (Throwable t) {
			if (trx != null)
				trx.rollback();
			throw new Exception(t);
		} finally {
			if (session != null)
				session.close();
		}
	}

	protected Segment addSegment(Session session, Study study,
			Date segmentTimestamp, Protocol protocol, Assessment as)
			throws HibernateException, Exception {
		StudyKey skey = study.getStudyKey();
		Segment segment = null;

		List<?> segList = session
				.createQuery(
						"from Segment s where s.segmentKey.subjectID=:sid "
								+ "and s.segmentKey.experimentID=:expID and s.segmentKey.visitID=:vid "
								+ "order by s.segmentKey.segmentID")
				.setParameter("sid", skey.getSubjectID()).setParameter("expID",
						skey.getExperimentID()).setParameter("vid",
						skey.getVisitID()).list();
		Long segmentID = null;
		if (segList.isEmpty()) {
			segmentID = new Long(1);
		} else {
			List<?> saList = session
					.createQuery(
							"from StoredAssessment s where s.assessmentID = :asID "
									+ "and s.experimentID = :expID and s.visitID = :vid "
									+ "and s.subjectID = :sid").setParameter(
							"asID", as.getID()).setParameter("expID",
							skey.getExperimentID()).setParameter("vid",
							skey.getVisitID()).setParameter("sid",
							skey.getSubjectID()).list();

			Map<Long, StoredAssessment> map = new HashMap<Long, StoredAssessment>(7);

			if (!saList.isEmpty()) {
				for (Iterator<?> it = saList.iterator(); it.hasNext();) {
					StoredAssessment sa = (StoredAssessment) it.next();
					map.put(sa.getSegmentID(), sa);
				}
			}

			for (Iterator<?> it = segList.iterator(); it.hasNext();) {
				Segment seg = (Segment) it.next();
				if (seg.getTimeStamp().compareTo(segmentTimestamp) == 0) {
					if (seg.getProtocol().getUniqueID().equals(
							protocol.getUniqueID())) {
						// a segment with this timestamp and protocol exists,
						study.addSegment(seg);
						return seg;
					}
					/*
					 * if (map.get(seg.getSegmentKey().getSegmentID()) == null) { //
					 * a segment with this timestamp exists and // has no
					 * assessment of this type, so just return it return seg; }
					 */
				}
			}

			Segment lastSeg = (Segment) segList.get(segList.size() - 1);
			segmentID = new Long(lastSeg.getSegmentKey().getSegmentID()
					.longValue() + 1);
		}
		segment = createSegment(session, study, segmentID, protocol,
				segmentTimestamp);
		return segment;
	}

	protected Segment createSegment(Session session, Study study,
			Long segmentID, Protocol protocol, Date segTS)
			throws HibernateException, Exception {
		StudyKey skey = study.getStudyKey();
		DatabaseUser dbUser = getDatabaseUser(session);

		Segment segment = new Segment();
		SegmentKey sgKey = new SegmentKey(skey.getVisitID(), skey
				.getExperimentID(), skey.getSubjectID(), segmentID);
		segment.setSegmentKey(sgKey);
		segment.setProtocol(protocol);
		segment.setIsBad(new Boolean(false));
		segment.setIsTimeInterval(new Boolean(false));
		segment.setStudyID(study.getStudyKey().getStudyID());
		segment.setTimeStamp(new Timestamp(segTS.getTime()));
		setCommonProperties(dbUser, "nc_expsegment", segment);
		Long uniqueID = SessionFactoryFactory.getNextSequenceNumber(config
				.getDbType(), session);
		segment.setUniqueID(uniqueID);

		session.save(segment);
		study.addSegment(segment);
		return segment;
	}

	protected Study addStudy(Session session, Visit visit, String studyName)
			throws HibernateException, Exception {
		VisitKey vkey = visit.getVisitKey();
		List<?> studyList = session
				.createQuery(
						"from Study s where s.studyKey.visitID=:vid "
								+ "and s.studyKey.subjectID=:sid and s.studyKey.experimentID=:expID "
								+ "order by s.studyKey.studyID").setParameter(
						"vid", vkey.getVisitID()).setParameter("sid",
						vkey.getSubjectID()).setParameter("expID",
						vkey.getExperimentID()).list();
		Long studyID = null;
		if (studyList.isEmpty()) {
			studyID = new Long(1);
		} else {
			Study lastStudy = (Study) studyList.get(studyList.size() - 1);
			studyID = new Long(
					lastStudy.getStudyKey().getStudyID().longValue() + 1);
		}
		Study study = createStudy(session, visit, studyID, studyName);
		visit.addStudy(study);
		return study;
	}

	protected Study createStudy(Session session, Visit visit, Long studyID,
			String studyName) throws HibernateException, Exception {
		VisitKey vkey = visit.getVisitKey();
		DatabaseUser dbUser = getDatabaseUser(session);
		Study study = new Study();
		StudyKey sKey = new StudyKey(vkey.getVisitID(), vkey.getExperimentID(),
				vkey.getSubjectID(), studyID);
		study.setStudyKey(sKey);
		study.setName(studyName);
		study.setTimeStamp(visit.getTimeStamp());
		study.setIsTimeInterval(new Boolean(false));
		setCommonProperties(dbUser, "nc_expstudy", study);
		Long uniqueID = SessionFactoryFactory.getNextSequenceNumber(config
				.getDbType(), session);
		study.setUniqueID(uniqueID);
		session.save(study);
		return study;
	}

	protected Long getNewVisitID(Session session, Experiment exp,
			DataRow dataRow) {
		List<?> visitList = session.createQuery(
				"from Visit v where v.visitKey.subjectID = :sid "
						+ "and v.visitKey.experimentID=:expID").setParameter(
				"sid", dataRow.getSubjectID()).setParameter("expID",
				exp.getUniqueID()).list();
		if (visitList.isEmpty()) {
			return new Long(1);
		} else {
			long max = Long.MIN_VALUE;
			for (Iterator<?> it = visitList.iterator(); it.hasNext();) {
				Visit v = (Visit) it.next();
				max = Math.max(max, v.getVisitKey().getVisitID().longValue());
			}
			return new Long(max + 1);
		}
	}

	protected Visit createVisitSegmentStudy(Long visitID, DataRow dataRow,
			String visitType, Date visitDate, Date segmentTimeStamp,
			Protocol protocol, Session session, Experiment exp,
			DatabaseUser dbUser, Assessment as) throws HibernateException {
		Visit visit;
		visit = new Visit();
		VisitKey vKey = new VisitKey(visitID, exp.getUniqueID(), dataRow
				.getSubjectID());
		visit.setVisitKey(vKey);
		visit.setVisitType(visitType);
		visit.setIsTimeInterval(new Boolean(false));
		visit.setTimeStamp(new Timestamp(visitDate.getTime()));
		setCommonProperties(dbUser, "nc_expcomponent", visit);
		Long uniqueID = SessionFactoryFactory.getNextSequenceNumber(config
				.getDbType(), session);
		visit.setUniqueID(uniqueID);
		session.save(visit);

		// create a study
		Study study = new Study();
		StudyKey sKey = new StudyKey(visitID, exp.getUniqueID(), dataRow
				.getSubjectID(), new Long(1));
		study.setStudyKey(sKey);
		study.setName("FMRI");
		study.setTimeStamp(new Timestamp(visitDate.getTime()));
		study.setIsTimeInterval(new Boolean(false));
		setCommonProperties(dbUser, "nc_expstudy", study);
		uniqueID = SessionFactoryFactory.getNextSequenceNumber(config
				.getDbType(), session);
		study.setUniqueID(uniqueID);

		session.save(study);

		visit.addStudy(study);
		// create a segment
		Segment segment = new Segment();
		SegmentKey sgKey = new SegmentKey(visitID, exp.getUniqueID(), dataRow
				.getSubjectID(), new Long(1));
		segment.setSegmentKey(sgKey);
		segment.setProtocol(protocol);
		segment.setIsBad(new Boolean(false));
		segment.setIsTimeInterval(new Boolean(false));
		segment.setStudyID(study.getStudyKey().getStudyID());
		segment.setTimeStamp(new Timestamp(segmentTimeStamp.getTime()));
		setCommonProperties(dbUser, "nc_expsegment", segment);
		uniqueID = SessionFactoryFactory.getNextSequenceNumber(config
				.getDbType(), session);
		segment.setUniqueID(uniqueID);

		session.save(segment);
		study.addSegment(segment);
		return visit;
	}

	protected void setCommonProperties(DatabaseUser dbUser, String tableName,
			AbstractVO domainObj) {
		TableID tableID;
		tableID = (TableID) getTableIDMap().get(tableName);
		domainObj.setTableID(tableID.getTableID());
		domainObj.setOwner(dbUser.getUniqueID());
		domainObj.setModUser(dbUser.getUniqueID());
		domainObj.setModTime(new Date());
	}

	public List<?> getStudySegmentsForSubject(String subjectID,
			String experimentName, String visitType) throws Exception {
		SessionFactory sessionFactory = SessionFactoryFactory
				.getSessionFactory(config);
		Session session = null;
		List<?> visitList = null;
		Transaction trx = null;
		try {
			session = sessionFactory.openSession();
			trx = session.beginTransaction();
			Experiment exp = (Experiment) session.createQuery(
					"from Experiment e where e.name like :name").setParameter(
					"name", experimentName + "%").uniqueResult();

			// get subjexperiment

			List<?> seList = session
					.createQuery(
							"from SubjExperiment s "
									+ "where s.experiment=:expID and s.subjectID= :sid")
					.setParameter("expID", exp).setParameter("sid", subjectID)
					.list();
			if (!seList.isEmpty()) {
				// get all visits of type
				visitList = session
						.createQuery(
								"from Visit v where v.visitKey.subjectID = :sid "
										+ "and v.visitKey.experimentID=:expID and v.visitType=:vtype")
						.setParameter("sid", subjectID).setParameter("expID",
								exp.getUniqueID()).setParameter("vtype",
								visitType).list();

				// now get all studies and their corresponding segments also
				if (!visitList.isEmpty()) {
					retrieveStudySegmentsForVisitsOfASubject(session, visitList);
				}
			}
			trx.commit();

			return visitList;
		} catch (Throwable t) {
			if (trx != null)
				trx.rollback();
			throw new Exception(t);
		} finally {

			if (session != null)
				session.close();
		}
	}

	protected void retrieveStudySegmentsForVisitsOfASubject(Session session,
			List<?> visits) {
		Visit aVisit = (Visit) visits.get(0);
		String subjectID = aVisit.getVisitKey().getSubjectID();
		Long expID = aVisit.getVisitKey().getExperimentID();

		List<?> studies = session
				.createQuery(
						"from Study s where s.studyKey.subjectID=:sid and "
								+ "s.studyKey.experimentID = :expID order by s.studyKey.studyID")
				.setParameter("sid", subjectID).setParameter("expID", expID)
				.list();

		List<?> segments = session.createQuery(
				"from Segment s where s.segmentKey.subjectID=:sid "
						+ "and s.segmentKey.experimentID=:expID "
						+ "order by s.segmentKey.segmentID, s.studyID")
				.setParameter("sid", subjectID).setParameter("expID", expID)
				.list();

		Map<Long, Visit> visitMap = new HashMap<Long, Visit>(7);
		for (Iterator<?> it = visits.iterator(); it.hasNext();) {
			Visit visit = (Visit) it.next();
			visitMap.put(visit.getVisitKey().getVisitID(), visit);
		}

		Map<Long, Study> studyMap = new HashMap<Long, Study>(7);
		for (Iterator<?> it = studies.iterator(); it.hasNext();) {
			Study study = (Study) it.next();
			Visit visit = visitMap
					.get(study.getStudyKey().getVisitID());
			if (visit != null) {
				visit.addStudy(study);
				studyMap.put(study.getStudyKey().getStudyID(), study);
			}
		}

		for (Iterator<?> it = segments.iterator(); it.hasNext();) {
			Segment segment = (Segment) it.next();
			Study study = studyMap.get(segment.getStudyID());
			if (study != null) {
				study.addSegment(segment);
			}
		}
	}

	public void addAssessmentData(DataRow dataRow, String assessmentName,
			String visitType, String researchGroupName, Date visitDate,
			Date segmentTimestamp, Protocol protocol) throws AssessmentExistsException, Exception {
		SessionFactory sessionFactory = SessionFactoryFactory
				.getSessionFactory(config);
		Session session = null;
		Transaction trx = null;
		try {
			session = sessionFactory.openSession();
			trx = session.beginTransaction();
			Assessment as = (Assessment) session.createQuery(
					"from Assessment a where a.name = :name").setParameter(
					"name", assessmentName).uniqueResult();

			Visit visit = getVisitWithStudySegmentForSubject(session, dataRow,
					dataRow.getExperimentName(), visitType, researchGroupName,
					visitDate, segmentTimestamp, protocol, as);
			Study study = (Study) visit.getStudies().iterator().next();
			Segment segment = (Segment) study.getSegments().iterator().next();

			if (asssessmentEntryExists(session, segment, as)) {
				throw new AssessmentExistsException();
			}

			addAssessmentData(session, segment, dataRow, as);

			trx.commit();
		} catch (Throwable t) {
			if (trx != null)
				trx.rollback();
			if ( t instanceof Exception) {
				throw (Exception) t;
			} else {
				throw new Exception(t);
			}
		} finally {
			if (session != null)
				session.close();
		}

	}

	public void addAssessmentData(Session session, Segment segment,
			DataRow dataRow, Assessment as) throws Exception {
		DatabaseUser dbUser = getDatabaseUser(session);
		Map<Object, Object> tableNameMap = getTableIDMap();
		DataClassification missingValue = (DataClassification) getDataClassMap()
				.get("missing value");

		StoredAssessment sa = saveStoredAssessmentRec(segment, session, as,
				dbUser, tableNameMap);

		insertAsDataRecords(dataRow, session, as, dbUser, missingValue, sa);
	}

	public boolean asssessmentEntryExists(Session session, Segment theSegment,
			Assessment as) throws Exception {
		Date ts = new Date(theSegment.getTimeStamp().getTime());

		List<?> results = session
				.createQuery(
						"from StoredAssessment sa, Segment s where "
								+ "sa.assessmentID = :asid and sa.timeStamp = :ts and sa.segmentID = s.segmentKey.segmentID "
								+ "and sa.subjectID = s.segmentKey.subjectID "
								+ "and sa.visitID = s.segmentKey.visitID and sa.experimentID = s.segmentKey.experimentID "
								+ "and sa.subjectID= :sid and sa.experimentID=:exid and sa.visitID= :vid "
								+ "and sa.segmentID= :segID").setParameter(
						"asid", as.getID()).setParameter("ts", ts)
				.setParameter("sid", theSegment.getSegmentKey().getSubjectID())
				.setParameter("exid",
						theSegment.getSegmentKey().getExperimentID())
				.setParameter("vid", theSegment.getSegmentKey().getVisitID())
				.setParameter("segID",
						theSegment.getSegmentKey().getSegmentID()).list();
		if (results.isEmpty()) {
			return false;
		}
		for (Iterator<?> it = results.iterator(); it.hasNext();) {
			Object[] records = (Object[]) it.next();
			// StoredAssessment sa = (StoredAssessment) records[0];
			Segment s = (Segment) records[1];
			if (s.getSegmentKey().equals(theSegment.getSegmentKey())) {
				return true;
			}
		}

		return false;
	}

	public void addAssessmentData(Segment segment, DataRow dataRow,
			String assessmentName) throws Exception, HibernateException {

		SessionFactory sessionFactory = SessionFactoryFactory
				.getSessionFactory(config);
		Session session = null;
		Transaction trx = null;
		try {
			session = sessionFactory.openSession();
			trx = session.beginTransaction();
			Assessment as = (Assessment) session.createQuery(
					"from Assessment a where a.name = :name").setParameter(
					"name", assessmentName).uniqueResult();

			addAssessmentData(session, segment, dataRow, as);

			trx.commit();
		} catch (Throwable t) {
			if (trx != null)
				trx.rollback();
			throw new Exception(t);
		} finally {
			if (session != null)
				session.close();
		}

	}

	StoredAssessment saveStoredAssessmentRec(Segment segment, Session session,
			Assessment as, DatabaseUser dbUser, Map<Object, Object> tableNameMap)
			throws HibernateException {
		StoredAssessment sa = new StoredAssessment();
		sa.setAssessmentID(as.getID());
		sa.setExperimentID(segment.getSegmentKey().getExperimentID());
		sa.setInformantID(segment.getSegmentKey().getSubjectID());
		AssessmentInformant ir = (AssessmentInformant) getInformantRelationMap()
				.get("self");
		sa.setInformantRelation(ir);
		sa.setModTime(new Date());
		sa.setIsValidated(new Boolean(true));
		sa.setSegmentID(segment.getSegmentKey().getSegmentID());
		sa.setVisitID(segment.getSegmentKey().getVisitID());
		sa.setSubjectID(segment.getSegmentKey().getSubjectID());
		sa.setIsTimeinterval(new Boolean(false));
		TableID tableID = (TableID) tableNameMap.get("nc_storedassessment");
		sa.setTableID(tableID.getUniqueID());
		sa.setOwner(dbUser.getUniqueID());
		sa.setModUser(dbUser.getUniqueID());
		AssessmentStatus status = (AssessmentStatus) getAssessmentStatusMap()
				.get("one entry complete data");
		sa.setStatus(status);
		sa.setTimeStamp(segment.getTimeStamp());

		session.save(sa);
		return sa;
	}

	protected DatabaseUser getDatabaseUser(Session session)
			throws HibernateException, Exception {
		List<?> dbUserList = session
				.createQuery(
						"from DatabaseUser d where d.name = :name and d.isGroup = :group")
				.setParameter("name", config.getUser().toUpperCase())
				.setParameter("group", new Boolean(false)).list();
		if (dbUserList.isEmpty()) {
			throw new Exception("No database user record for user "
					+ config.getUser());
		}
		DatabaseUser dbUser = null;
		for (Iterator<?> iterator = dbUserList.iterator(); iterator.hasNext();) {
			DatabaseUser dbu = (DatabaseUser) iterator.next();
			if (dbu.getUserClass().getName().equalsIgnoreCase("admin")
					&& dbu.getUserStatus().getUserStatus().equalsIgnoreCase(
							"active")) {
				dbUser = dbu;
				break;
			}
		}
		if (dbUser == null) {
			throw new Exception("No authorized database user record for user "
					+ config.getUser());
		}
		return dbUser;
	}

	protected boolean checkSetClassification(String dataValue, BaseVO asTable,
			Map<Object, Object> dataClassMap) {
		if (dataValue.equals(MISSING_VALUE_CODE) || dataValue.startsWith("NC:")) {
			DataClassification missingValue = (DataClassification) dataClassMap
					.get(MISSING_VALUE_MSG);
			DataClassification dc = null;
			if (dataValue.startsWith("NC:")) {
				String reason = dataValue.substring(3).trim();
				dc = (DataClassification) dataClassMap.get(reason);
			}
			if (dc == null) {
				dc = missingValue;
			}
			if (asTable instanceof AssessmentInteger) {
				((AssessmentInteger) asTable).setClassification(dc
						.getUniqueID());
			} else if (asTable instanceof AssessmentVarchar) {
				((AssessmentVarchar) asTable).setClassification(dc
						.getUniqueID());
			} else if (asTable instanceof AssessmentFloat) {
				((AssessmentFloat) asTable).setClassification(dc.getUniqueID());
			} else if (asTable instanceof AssessmentTimestamp) {
				((AssessmentTimestamp) asTable).setClassification(dc
						.getUniqueID());
			} else if (asTable instanceof AssessmentBoolean) {
				((AssessmentBoolean) asTable).setClassification(dc
						.getUniqueID());
			}
			return true;
		}
		return false;
	}

	protected void insertAsDataRecords(DataRow dataRow, Session session,
			Assessment as, DatabaseUser dbUser,
			DataClassification missingValue, StoredAssessment sa)
			throws Exception {
		Map<Object, Object> tableNameMap = getTableIDMap();
		Map<Object, Object> dataClassMap = getDataClassMap();
		TableID tableID;
		Score[] scores = dataRow.getScores();
		Object[] data = dataRow.getData();
		for (int i = 0; i < scores.length; i++) {
			Score score = scores[i];
			BaseVO asData = null;
			tableID = null;
			boolean noData = false;
			if (score.getScoreType().equals("integer")) {
				asData = new AssessmentInteger();
				AssessmentInteger ai = (AssessmentInteger) asData;
				tableID = (TableID) tableNameMap.get("nc_assessmentinteger");
				ai.setTableID(tableID.getUniqueID());
				if (!checkSetClassification(data[i].toString(), ai, dataClassMap)) {
					ai.setDataValue(new Long(data[i].toString()));
				} else {
					noData = true;
				}
				ai.setStoredAssessmentID(sa.getUniqueID());
			} else if (score.getScoreType().equals("varchar")) {
				asData = new AssessmentVarchar();
				AssessmentVarchar av = (AssessmentVarchar) asData;
				tableID = (TableID) tableNameMap.get("nc_assessmentvarchar");
				av.setTableID(tableID.getUniqueID());
				av.setStoredAssessmentID(sa.getUniqueID());
				if (!checkSetClassification(data[i].toString(), av, dataClassMap)) {
					av.setDataValue(data[i].toString());
				} else {
					noData = true;
				}
			} else if (score.getScoreType().equals("float")) {
				asData = new AssessmentFloat();
				AssessmentFloat af = (AssessmentFloat) asData;
				tableID = (TableID) tableNameMap.get("nc_assessmentfloat");
				af.setTableID(tableID.getUniqueID());
				af.setStoredAssessmentID(sa.getUniqueID());
				if (!checkSetClassification(data[i].toString(), af, dataClassMap)) {
					af.setDataValue(new Float(data[i].toString()));
				} else {
					noData = true;
				}
			} else if (score.getScoreType().equals("timestamp")) {
				asData = new AssessmentTimestamp();
				AssessmentTimestamp at = (AssessmentTimestamp) asData;
				tableID = (TableID) tableNameMap.get("nc_assessmenttimestamp");
				at.setTableID(tableID.getUniqueID());
				at.setStoredAssessmentID(sa.getUniqueID());
				if (!checkSetClassification(data[i].toString(), at, dataClassMap)) {
					// TODO
					at.setDataValue((Timestamp) data[i]);
				} else {
					noData = true;
				}
			} else if (score.getScoreType().equals("boolean")) {
				asData = new AssessmentBoolean();
				AssessmentBoolean ab = (AssessmentBoolean) asData;
				tableID = (TableID) tableNameMap.get("nc_assessmentboolean");
				ab.setTableID(tableID.getUniqueID());
				ab.setStoredAssessmentID(sa.getUniqueID());
				if (!checkSetClassification(data[i].toString(), ab, dataClassMap)) {
					ab.setDataValue(new Boolean(data[i].toString()));
				} else {
					noData = true;
				}
			} else {
				throw new Exception("Not supported score type:"
						+ score.getScoreType());
			}
			asData.setAssessmentID(as.getID());
			asData.setIsvalidated(new Boolean(true));
			asData.setIsranked(new Boolean(false));
			asData.setScorename(score.getName());
			asData.setScoretype(score.getScoreType());
			asData.setSubjectID(sa.getSubjectID());
			asData.setScoreOrder(new Long(1));
			asData.setEntryID(new Long(1));
			if (!noData) {
			  asData.setTextValue(data[i].toString());
			} else {
				asData.setTextValue("");
			}
			asData.setOwner(dbUser.getUniqueID());
			asData.setModUser(dbUser.getUniqueID());
			asData.setModTime(new Date());

			session.save(asData);
		}
	}

	public static String prepareInClause(List<?> list, IFunctor functor) {
		StringBuilder buf = new StringBuilder();
		buf.append("(");
		for (Iterator<?> it = list.iterator(); it.hasNext();) {
			Object item = (Object) it.next();
			buf.append(functor.evaluate(item));
			if (it.hasNext())
				buf.append(",");
		}
		buf.append(')');
		return buf.toString();
	}
}
