package caslayout.ui.db.services;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;

import caslayout.ui.CALMConfig;
import caslayout.ui.db.Assessment;
import caslayout.ui.db.AssessmentItem;
import caslayout.ui.db.AssessmentScore;
import caslayout.ui.db.AssessmentScoreCode;
import caslayout.ui.db.OntologyConcept;
import caslayout.ui.db.SessionFactoryFactory;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentService.java,v 1.1.2.1 2007/07/18 01:42:10 bozyurt
 *          Exp $
 */
public class AssessmentService {
   protected CALMConfig config;
   protected String securityClassification;

   protected static AssessmentService instance = null;
   protected static Logger log = Logger.getLogger(AssessmentService.class);

   protected AssessmentService(CALMConfig config) {
      this.config = config;
   }

   public static synchronized AssessmentService getInstance(CALMConfig config) {
      if (instance == null) {
         instance = new AssessmentService(config);
      }
      return instance;
   }

   public static synchronized AssessmentService getInstance() {
      return instance;
   }

   public List<?> getAllAssessments() {
      SessionFactory sessionFactory = SessionFactoryFactory
            .getSessionFactory(config);
      Session session = null;
      try {
         session = sessionFactory.openSession();
         Transaction trx = session.beginTransaction();
         List<?> asList = session.createQuery(
               "from Assessment a order by name asc").list();
         trx.commit();
         return asList;
      } finally {
         if (session != null)
            session.close();
      }
   }

   public Assessment getAssessment(String name) {
      Assessment as = null;
      SessionFactory sessionFactory = SessionFactoryFactory
            .getSessionFactory(config);
      Session session = null;
      try {
         session = sessionFactory.openSession();
         Transaction trx = session.beginTransaction();
         List<?> asList = session.createQuery(
               "from Assessment a where a.name like :name").setParameter(
               "name", name + "%").list();
         if (!asList.isEmpty()) {
            as = (Assessment) asList.get(0);
            List<?> items = loadItems(as, session);
            for (Iterator<?> iter = items.iterator(); iter.hasNext();) {
               AssessmentItem item = (AssessmentItem) iter.next();
               as.addItem(item);
            }
            List<?> scoreCodes = loadScoreCodes(as, session);
            Map<String, List<AssessmentScoreCode>> scoreCodeMap = groupByScoreName(scoreCodes);
            Set<AssessmentScore> scores = as.getScores();
            for (AssessmentScore score : scores) {
               List<AssessmentScoreCode> codeList = scoreCodeMap.get(score
                     .getScoreName());
               if (codeList == null) {
                  continue;
               }
               for (AssessmentScoreCode asc : codeList) {
                  score.addScoreCode(asc);
               }
            }
         }

         trx.commit();
      } finally {
         if (session != null)
            session.close();
      }
      return as;
   }

   /**
    * checks if there is any data for this assessment, if so throws an
    * Exception.
    *
    * @param as
    * @param deleteDataAlso
    * @throws Exception
    */
   public void deleteAssessment(Assessment as, boolean deleteDataAlso)
         throws Exception {

      boolean hasScores = !as.getScores().isEmpty();
      AssessmentScore aScore = null;
      if (hasScores) {
         aScore = (AssessmentScore) as.getScores().iterator().next();
      }
      SessionFactory sessionFactory = SessionFactoryFactory
            .getSessionFactory(config);
      Session session = null;
      Transaction trx = null;
      try {
         session = sessionFactory.openSession();
         trx = session.beginTransaction();
         if (hasAssessmentData(as, session)) {
            if (!deleteDataAlso) {
               throw new Exception(
                     "You cannot delete an assessment containing data!");
            } else {
               deleteAssessmentData(as, session);
            }
         }

         session.createQuery(
               "delete AssessmentScoreCode sc where sc.assessmentID = :asID")
               .setParameter("asID", as.getID()).executeUpdate();

         session.createQuery(
               "delete AssessmentItem i where i.assessmentID = :asID")
               .setParameter("asID", as.getID()).executeUpdate();

         session.createQuery(
               "delete AssessmentScore s where s.assessment = :as")
               .setParameter("as", as).executeUpdate();

         session.delete(as);

         if (hasScores) {
            deleteOntologyConcept(session, aScore.getAssessmentOntology(),
                  aScore.getAssessmentConcept());
         }
         trx.commit();
      } catch (Throwable t) {
         trx.rollback();
         throw new Exception(t);
      } finally {
         if (session != null)
            session.close();
      }
   }

   /**
    *
    * @param as
    * @throws Exception
    */
   public void updateAssessmentItems(Assessment as) throws Exception {
      SessionFactory sessionFactory = SessionFactoryFactory
            .getSessionFactory(config);
      Session session = null;
      Transaction trx = null;
      try {
         session = sessionFactory.openSession();
         trx = session.beginTransaction();
         Long userID = getDatabaseUser(session, config.getUser());
         if (userID == null) {
            throw new Exception("not a valid database user:" + config.getUser());
         }
         List<?> dbAiList = session.createQuery(
               "from AssessmentItem a where a.assessmentID = :asID")
               .setParameter("asID", as.getID()).list();
         Map<String, AssessmentItem> dbAIMap = new HashMap<String, AssessmentItem>();
         Map<String, AssessmentItem> aiMap = new HashMap<String, AssessmentItem>();
         for (Iterator<?> iter = dbAiList.iterator(); iter.hasNext();) {
            AssessmentItem dbAi = (AssessmentItem) iter.next();
            dbAIMap.put(dbAi.getScoreName(), dbAi);
         }

         for (AssessmentItem ai : as.getItems()) {
            aiMap.put(ai.getScoreName(), ai);
         }

         for (AssessmentItem item : as.getItems()) {
            AssessmentItem dbAI = dbAIMap.get(item.getScoreName());
            if (dbAI == null) {
               setItemTableColumnValues(item, userID, session);
               item.setAssessmentID(as.getID());
               log.info(item.toString());
               session.save(item);
            } else {
               if (!dbAI.getItemLeadingText().equals(item.getItemLeadingText())) {
                  // needs update
                  log.info("updating " + item);
                  session.update(item);
               }
            }
         }// iter
         for (Iterator<?> iter = dbAiList.iterator(); iter.hasNext();) {
            AssessmentItem dbAi = (AssessmentItem) iter.next();
            if (aiMap.get(dbAi.getScoreName()) == null) {
               // remove it
               log.info("deleting " + dbAi);
               session
                     .createQuery(
                           "delete from AssessmentItem a where a.assessmentID = :asID and a.scoreName = :sn")
                     .setParameter("asID", as.getID()).setParameter("sn",
                           dbAi.getScoreName()).executeUpdate();
            }
         }

         trx.commit();

      } catch (Throwable t) {
         trx.rollback();
         throw new Exception(t);
      } finally {
         if (session != null)
            session.close();
      }
   }

   protected void deleteAssessmentData(Assessment as, Session session) {
      deleteFromAsDataTable("AssessmentInteger", as, session);
      deleteFromAsDataTable("AssessmentVarchar", as, session);
      deleteFromAsDataTable("AssessmentFloat", as, session);
      deleteFromAsDataTable("AssessmentTimestamp", as, session);
      deleteFromAsDataTable("AssessmentBoolean", as, session);
      deleteFromAsDataTable("AssessmentData", as, session);

      session.createQuery(
            "delete from StoredAssessment s where s.assessmentID = :asID")
            .setParameter("asID", as.getID()).executeUpdate();
   }

   protected void deleteFromAsDataTable(String asDataVOName, Assessment as,
         Session session) {
      StringBuffer hqlBuf = new StringBuffer();
      hqlBuf.append("delete from ").append(asDataVOName);
      hqlBuf.append(" a where a.assessmentID = :asID");

      int numDeleted = session.createQuery(hqlBuf.toString()).setParameter(
            "asID", as.getID()).executeUpdate();

      System.out.println(hqlBuf.toString() + "\nNum deleted: " + numDeleted);
   }

   protected void updateAssessmentItems(Session session, List<AssessmentItem> items,
         Assessment as, Long userID) throws Exception {
      List<?> oldItems;
      List<AssessmentItem> newItems = new LinkedList<AssessmentItem>();
      List<AssessmentItem> removedItems = new LinkedList<AssessmentItem>();
      List<AssessmentItem> changedItems = new LinkedList<AssessmentItem>();
      Map<String, AssessmentItem> itemsMap = toItemMap(items);
      Map<String, AssessmentItem> oldItemsMap = toItemMap(items);

      oldItems = session.createQuery(
            "from AssessmentItem i where i.assessmentID = :asID").setParameter(
            "asID", as.getID()).list();

      for (Iterator<?> iter = oldItems.iterator(); iter.hasNext();) {
         AssessmentItem oldItem = (AssessmentItem) iter.next();
         AssessmentItem newItem = itemsMap.get(oldItem.getScoreName());
         if (newItem == null) {
            removedItems.add(oldItem);
         } else if (!newItem.isSame(oldItem)) {
            changedItems.add(newItem);
         }
      }
      for (Iterator<AssessmentItem> iter = items.iterator(); iter.hasNext();) {
         AssessmentItem newItem = iter.next();
         AssessmentItem oldItem = oldItemsMap.get(newItem.getScoreName());
         if (oldItem == null) {
            newItems.add(newItem);
         }
      }
      for (Iterator<AssessmentItem> iter = removedItems.iterator(); iter
            .hasNext();) {
         AssessmentItem item = iter.next();
         session
               .createQuery(
                     "from AssessmentItem i where i.assessmentID = :asID and i.scoreName = :sn")
               .setParameter("asID", item.getAssessmentID()).setParameter("sn",
                     item.getScoreName()).executeUpdate();
      }
      for (Iterator<AssessmentItem> iter = newItems.iterator(); iter.hasNext();) {
         AssessmentItem item = iter.next();
         setItemTableColumnValues(item, userID, session);
         session.save(item);
      }
      for (Iterator<AssessmentItem> iter = changedItems.iterator(); iter
            .hasNext();) {
         AssessmentItem item = iter.next();
         session.update(item);
      }
   }

   public void updateAssessment(Assessment as) throws Exception {
      SessionFactory sessionFactory = SessionFactoryFactory
            .getSessionFactory(config);

      Session session = null;
      Transaction trx = null;
      try {
         session = sessionFactory.openSession();
         trx = session.beginTransaction();

         Long userID = getDatabaseUser(session, config.getUser());
         if (userID == null) {
            throw new Exception("not a valid database user:" + config.getUser());
         }

         List<?> oldScores = session.createQuery(
               "from AssessmentScore s where s.assessment = :as").setParameter(
               "as", as).list();
         List<?> oldScoreCodes = session.createQuery(
               "from AssessmentScoreCode sc where sc.assessmentID = :asID")
               .setParameter("asID", as.getID()).list();
         OntologyConcept oc = checkAndInsertOntologyConcept(session, as, as
               .getOwner());

         List<AssessmentScore> newScores = new LinkedList<AssessmentScore>();
         List<AssessmentScore> removedScores = new LinkedList<AssessmentScore>();
         List<AssessmentScore> changedScores = new LinkedList<AssessmentScore>();

         List<AssessmentScoreCode> newScoreCodes = new LinkedList<AssessmentScoreCode>();
         List<AssessmentScoreCode> removedScoreCodes = new LinkedList<AssessmentScoreCode>();
         List<AssessmentScoreCode> changedScoreCodes = new LinkedList<AssessmentScoreCode>();
         Map<String, List<AssessmentScoreCode>> oldScoreCodeMap = groupByScoreName(oldScoreCodes);

         for (Iterator<?> iter = oldScores.iterator(); iter.hasNext();) {
            AssessmentScore oldScore = (AssessmentScore) iter.next();
            AssessmentScore newScore = findScore(oldScore.getScoreName(),
                  new ArrayList<AssessmentScore>(as.getScores()));
            if (newScore == null) {
               removedScores.add(oldScore);
               populateRemovedScoreCodes(oldScore, removedScoreCodes,
                     oldScoreCodeMap);
            } else {
               List<AssessmentScoreCode> oldScoreCodeList = oldScoreCodeMap
                     .get(newScore.getScoreName());
               if (oldScoreCodeList != null) {
                  populateChangedScoreCodes(newScore, oldScoreCodeList,
                        newScoreCodes, removedScoreCodes, changedScoreCodes);
               }
               if (!newScore.isSame(oldScore)) {
                  changedScores.add(newScore);
               }
            }
         }// for

         for (AssessmentScore newScore : as.getScores()) {
            AssessmentScore oldScore = findScore(newScore.getScoreName(),
                  oldScores);
            if (oldScore == null) {
               newScores.add(newScore);
            }
         }

         for (Iterator<AssessmentScore> iter = newScores.iterator(); iter
               .hasNext();) {
            AssessmentScore score = iter.next();
            if (score.getScoreCodes() == null) {
               continue;
            }
            for (AssessmentScoreCode asc : score.getScoreCodes()) {
               newScoreCodes.add(asc);
            }
         }

         for (Iterator<AssessmentScoreCode> iter = removedScoreCodes.iterator(); iter
               .hasNext();) {
            AssessmentScoreCode asc = iter.next();
            session
                  .createQuery(
                        "delete from AssessmentScoreCode sc where sc.assessmentID = :asID and sc.scoreName = :sn")
                  .setParameter("asID", asc.getAssessmentID()).setParameter(
                        "sn", asc.getScoreName()).executeUpdate();
         }

         for (Iterator<AssessmentScore> iter = removedScores.iterator(); iter
               .hasNext();) {
            AssessmentScore s = iter.next();
            session
                  .createQuery(
                        "delete from AssessmentScore s where s.assessmentID = :asID and s.scoreName = :sn")
                  .setParameter("asID", s.getAssessmentID()).setParameter("sn",
                        s.getScoreName()).executeUpdate();
         }

         for (Iterator<AssessmentScore> iter = changedScores.iterator(); iter
               .hasNext();) {
            AssessmentScore updatedScore = iter.next();
            updatedScore.setModUser(userID);
            updatedScore.setModTime(new java.util.Date());
            session.update(updatedScore);
         }

         for (Iterator<AssessmentScore> iter = newScores.iterator(); iter
               .hasNext();) {
            AssessmentScore score = iter.next();
            setScoreTableColumValues(score, userID, oc, session);
            session.save(score);
         }
         // new scorecodes for already existing scores and new scores
         for (Iterator<AssessmentScoreCode> iter = newScoreCodes.iterator(); iter
               .hasNext();) {
            AssessmentScoreCode asc = iter.next();
            session.save(asc);
         }

         // update the changed score codes
         for (Iterator<AssessmentScoreCode> iter = changedScoreCodes.iterator(); iter
               .hasNext();) {
            AssessmentScoreCode asc = iter.next();
            asc.setModUser(userID);
            asc.setModTime(new java.util.Date());
            session.update(asc);
         }

         // update assessment items
         updateAssessmentItems(session, as.getItems(), as, userID);

         trx.commit();
      } catch (Throwable t) {
         trx.rollback();
         throw new Exception(t);
      } finally {
         if (session != null)
            session.close();
      }
   }

   /**
    *
    * @param as
    * @throws Exception
    */
   public void saveAssessment(Assessment as) throws Exception {
      SessionFactory sessionFactory = SessionFactoryFactory
            .getSessionFactory(config);

      Session session = null;
      Transaction trx = null;

      Set<AssessmentScore> scores = as.getScores();
      try {
         session = sessionFactory.openSession();
         trx = session.beginTransaction();

         List<?> asList = session.createQuery(
               "from Assessment a where a.name = :name").setParameter("name",
               as.getName()).list();
         Long uniqueAsID = null;
         if (asList.isEmpty()) {
            Long userID = getDatabaseUser(session, config.getUser());
            if (userID == null) {
               throw new Exception("not a valid database user:"
                     + config.getUser());
            }
            as.setTableID(getTableID(session, "nc_assessment"));
            as.setOwner(userID);
            as.setModUser(userID);
            as.setModTime(new java.util.Date());
            uniqueAsID = SessionFactoryFactory.getNextSequenceNumber(config
                  .getDbType(), session);
            as.setID(uniqueAsID);
            log.info("saving assessment...");
            session.save(as);

            uniqueAsID = as.getID();
            log.info("added assessment " + as.getID());
            log.info("Checking and inserting ontology concept record...");
            OntologyConcept oc = checkAndInsertOntologyConcept(session, as,
                  userID);
            log.info("saving scores...");
            // save only the scores
            for (Iterator<AssessmentScore> iter = scores.iterator(); iter
                  .hasNext();) {
               AssessmentScore score = iter.next();
               score.setAssessmentID(uniqueAsID);
               /** TODO scoreLevel and score must be set before */
               if (score.getTableID() == null) {
                  setScoreTableColumValues(score, userID, oc, session);
               }
               // update the assessment ids (since the actual id not known
               // till this point
               score.setAssessmentID(as.getID());
               if (score.getParentScore() != null) {
                  score.setParentAsID(as.getID());
               }
               score.setDescription("");
               score.setMinAnswers(new Integer(1));
               score.setMaxAnswers(new Integer(1));
               score.setNullable(new Boolean(false));
               log.info("inserting score " + score.toString());
               session.save(score);
            }
            // save assessment items
            for (Iterator<AssessmentItem> iter = as.getItems().iterator(); iter
                  .hasNext();) {
               AssessmentItem item = iter.next();
               setItemTableColumnValues(item, userID, session);
               item.setAssessmentID(uniqueAsID);
               log.info(item.toString());
               session.save(item);
            }
            log.info("saving score codes...");
            // save the score codes also
            for (Iterator<AssessmentScore> iter = scores.iterator(); iter
                  .hasNext();) {
               AssessmentScore score = iter.next();
               if (score.getScoreCodes() == null) {
                  continue;
               }
               for (Iterator<AssessmentScoreCode> it = score.getScoreCodes()
                     .iterator(); it.hasNext();) {
                  AssessmentScoreCode asc = it.next();
                  // set the scorecode label from scorecode value
                  asc.setScoreCodeLabel(asc.getScoreCodeValue());
                  setScoreCodeTableColumnValues(asc, userID, session);
                  asc.setAssessmentID(uniqueAsID);
                  log.info("inserting " + asc.toString());
                  session.save(asc);
               }
            }
         }

         trx.commit();
      } catch (Throwable t) {
         trx.rollback();
         t.printStackTrace();
         throw new Exception(t);
      } finally {
         if (session != null)
            session.close();
      }
   }

   protected AssessmentScore findScore(String scoreName, List<?> scores) {
      for (Iterator<?> it = scores.iterator(); it.hasNext();) {
         AssessmentScore s = (AssessmentScore) it.next();
         if (s.getScoreName().equals(scoreName)) {
            return s;
         }
      }
      return null;
   }

   protected OntologyConcept checkAndInsertOntologyConcept(Session session,
         Assessment as, Long userID) throws Exception {
      OntologyConcept oc = null;

      List<?> ocList = session
            .createQuery(
                  "from OntologyConcept o where o.ontologySource = :os and o.concept = :c")
            .setParameter("os", "UMLS").setParameter("c", as.getName()).list();
      if (ocList.isEmpty()) {
         Long tableID = getTableID(session, "nc_ontologyconcept");

         Long uniqID = SessionFactoryFactory.getNextSequenceNumber(config
               .getDbType(), session);
         oc = new OntologyConcept();

         oc.setOntologySource("UMLS");
         oc.setConceptID(uniqID.toString());
         oc.setTableID(tableID);
         oc.setModUser(userID);
         oc.setOwner(userID);
         oc.setModTime(new java.util.Date());
         oc.setConcept(as.getName());
         log.info("inserting ontology concept " + oc.getConcept());
         session.save(oc);
         return oc;
      } else {
         return (OntologyConcept) ocList.get(0);
      }
   }

   protected void deleteOntologyConcept(Session session, String ontologySource,
         String ontologyConcept) {
      session
            .createQuery(
                  "delete from OntologyConcept o where o.ontologySource = :os and o.concept = :oc")
            .setParameter("os", ontologySource).setParameter("oc",
                  ontologyConcept).executeUpdate();
   }

   public List<?> loadItems(Assessment as, Session session) {
      List<?> items = session.createQuery(
            "from AssessmentItem i where i.assessmentID = :asID").setParameter(
            "asID", as.getID()).list();
      return items;
   }

   public List<?> loadScoreCodes(Assessment as, Session session) {
      List<?> scoreCodes = session.createQuery(
            "from AssessmentScoreCode sc where sc.assessmentID = :asID")
            .setParameter("asID", as.getID()).list();
      return scoreCodes;
   }

   protected boolean hasAssessmentData(Assessment as, Session session) {
      Object noRecs = session
            .createSQLQuery(
                  "select count(*) from nc_storedassessment where assessmentid = :asID")
            .setLong("asID", as.getID().longValue()).uniqueResult();
      return (noRecs != null && getIntValue(noRecs) > 0);
   }

   public static int getIntValue(Object o) {
      if (o instanceof BigDecimal) {
         return ((BigDecimal) o).intValue();
      } else {
         return ((BigInteger) o).intValue();
      }
   }

   protected void setItemTableColumnValues(AssessmentItem ai, Long userID,
         Session session) throws Exception {
      Long tableID = getTableID(session, "NC_ASSESSMENTITEM");
      ai.setTableID(tableID);
      ai.setOwner(userID);
      ai.setModUser(userID);
      ai.setModTime(new java.util.Date());
   }

   protected void setScoreTableColumValues(AssessmentScore score, Long userID,
         OntologyConcept oc, Session session) throws Exception {
      Long tableID = getTableID(session, "NC_ASSESSMENTSCORE");
      score.setTableID(tableID);

      score.setOwner(userID);
      score.setModUser(userID);
      score.setModTime(new java.util.Date());

      score.setAssessmentOntology(oc.getOntologySource());
      score.setAssessmentConcept(oc.getConceptID());
   }

   protected void setScoreCodeTableColumnValues(AssessmentScoreCode asc,
         Long userID, Session session) throws Exception {
      Long tableID = getTableID(session, "NC_ASSESSMENTSCORECODE");
      asc.setTableID(tableID);
      asc.setOwner(userID);
      asc.setModUser(userID);
      asc.setModTime(new java.util.Date());
   }

   protected void populateRemovedScoreCodes(AssessmentScore score,
         List<AssessmentScoreCode> removedScoreCodes,
         Map<String, List<AssessmentScoreCode>> oldScoreCodeMap) {
      List<AssessmentScoreCode> scoreCodeList = oldScoreCodeMap.get(score
            .getScoreName());
      if (scoreCodeList != null) {
         removedScoreCodes.addAll(scoreCodeList);
      }
   }

   protected void populateChangedScoreCodes(AssessmentScore score,
         List<AssessmentScoreCode> oldScoreCodes,
         List<AssessmentScoreCode> newScoreCodes,
         List<AssessmentScoreCode> removedScoreCodes,
         List<AssessmentScoreCode> changedScoreCodes) throws Exception {
      List<AssessmentScoreCode> scoreCodeList = score.getScoreCodes();
      if (scoreCodeList == null) {
         return;
      }
      for (Iterator<AssessmentScoreCode> iter = scoreCodeList.iterator(); iter
            .hasNext();) {
         AssessmentScoreCode asc = iter.next();
         AssessmentScoreCode oldAsc = findScoreCode(oldScoreCodes, asc
               .getScoreCode());
         if (oldAsc == null) {
            newScoreCodes.add(asc);
         } else if (!asc.isSame(oldAsc)) {
            changedScoreCodes.add(asc);
         }
      }
      for (Iterator<AssessmentScoreCode> iter = oldScoreCodes.iterator(); iter
            .hasNext();) {
         AssessmentScoreCode oldAsc = iter.next();
         if (findScoreCode(scoreCodeList, oldAsc.getScoreCode()) == null) {
            removedScoreCodes.add(oldAsc);
         }
      }
   }

   public AssessmentScoreCode findScoreCode(
         List<AssessmentScoreCode> scoreCodes, String scoreCode) {
      for (Iterator<AssessmentScoreCode> iter = scoreCodes.iterator(); iter
            .hasNext();) {
         AssessmentScoreCode asc = iter.next();
         if (asc.getScoreCode().equals(scoreCode)) {
            return asc;
         }
      }
      return null;
   }

   public static Long getTableID(Session session, String tableName) {
      Object o = session.createSQLQuery(
            "select tableid from nc_tableid where tablename = :tn")
            .setParameter("tn", tableName.toUpperCase()).uniqueResult();

      if (o instanceof BigDecimal) {
         return new Long(((BigDecimal) o).longValue());
      } else {
         return new Long(((BigInteger) o).longValue());
      }
   }

   public static Long getDatabaseUser(Session session, String userName) {
      List<?> userIDs = session.createSQLQuery(
            "select uniqueid from nc_databaseuser where name= :un")
            .setParameter("un", userName.toUpperCase()).list();
      System.out.println("UserName: " + userName.toUpperCase());
      if (userIDs.isEmpty()) {
         return null;
      }

      Object o = userIDs.get(0);
      if (o instanceof BigDecimal) {
         return new Long(((BigDecimal) o).longValue());
      } else {
         return new Long(((BigInteger) o).longValue());
      }
   }

   public static Map<String, AssessmentItem> toItemMap(List<?> items) {
      Map<String, AssessmentItem> itemsMap = new HashMap<String, AssessmentItem>();
      for (Iterator<?> iter = items.iterator(); iter.hasNext();) {
         AssessmentItem item = (AssessmentItem) iter.next();
         itemsMap.put(item.getScoreName(), item);
      }
      return itemsMap;
   }

   public static Map<String, List<AssessmentScoreCode>> groupByScoreName(
         List<?> scoreCodes) {
      Map<String, List<AssessmentScoreCode>> map = new HashMap<String, List<AssessmentScoreCode>>();
      for (Iterator<?> iter = scoreCodes.iterator(); iter.hasNext();) {
         AssessmentScoreCode asc = (AssessmentScoreCode) iter.next();
         List<AssessmentScoreCode> codeList = map.get(asc.getScoreName());
         if (codeList == null) {
            codeList = new LinkedList<AssessmentScoreCode>();
            map.put(asc.getScoreName(), codeList);
         }
         codeList.add(asc);
      }
      return map;
   }

   public String getSecurityClassification() {
      return securityClassification;
   }

   public void setSecurityClassification(String securityClassification) {
      this.securityClassification = securityClassification;
   }

   /**
    * Returns a list of security classification strings
    *
    * @return a list of security classification strings
    * @throws java.lang.Exception
    */
   public List<?> getSecurityClassifications() throws Exception {
      SessionFactory sessionFactory = SessionFactoryFactory
            .getSessionFactory(config);
      Session session = null;
      try {
         session = sessionFactory.openSession();
         Transaction trx = session.beginTransaction();
         List<?> secList = session.createSQLQuery(
               "select securityclassification from nc_securityclassification")
               .list();
         trx.commit();
         return secList;
      } finally {
         if (session != null)
            session.close();
      }
   }

   public Long getAssessmentID(String asName) throws Exception {
      assert (asName != null);
      SessionFactory sessionFactory = SessionFactoryFactory
            .getSessionFactory(config);
      Session session = null;
      try {
         session = sessionFactory.openSession();
         Transaction trx = session.beginTransaction();
         Assessment as = (Assessment) session.createQuery(
               "from Assessment a where a.name = :name").setParameter("name",
               asName).uniqueResult();
         if (as == null) {
            trx.commit();
            throw new Exception("No record for assessment '" + asName + "'!");
         }
         trx.commit();
         return as.getID();
      } finally {
         if (session != null)
            session.close();
      }
   }

}
