package caslayout.ui.model;

import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.jdom.Element;
import org.jdom.Namespace;

import caslayout.Constants;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentInfo.java,v 1.18 2008/10/13 23:58:08 bozyurt Exp $
 */
public class AssessmentInfo {
   protected String name;
   protected String description;
   protected List<FieldInfo> mandatoryFields = new ArrayList<FieldInfo>(2);
   protected List<ScoreInfo> scores = new LinkedList<ScoreInfo>();
   protected List<ItemInfo> items = new LinkedList<ItemInfo>();
   protected static List<SecurityClassificationInfo> securityClassifications = new LinkedList<SecurityClassificationInfo>();

   protected LinkedHashMap<ScoreInfo, Object> scoreLevelMap = new LinkedHashMap<ScoreInfo, Object>();
   protected static Logger log = Logger.getLogger(AssessmentInfo.class);

   public AssessmentInfo(String name) {
      this.name = name;
   }

   public static synchronized void populateSecurityClassifications(
         List<?> secClassList) {
      for (Object o : secClassList) {
         String sc = (String) o;
         securityClassifications.add(new SecurityClassificationInfo(sc));
      }
   }

   public static synchronized List<SecurityClassificationInfo> getSecurityClassifications() {
      return securityClassifications;
   }

   public void addScoreInfo(ScoreInfo si) {
      if (!scores.contains(si)) {
         scores.add(si);
      }
   }

   public void addMandatoryField(FieldInfo mandatoryField) {
      if (!mandatoryFields.contains(mandatoryField)) {
         // log.info("added mandatory Field "+ mandatoryField.getName());
         mandatoryFields.add(mandatoryField);
      }
   }

   /**
    * makes sure that the mandatory fields are consistent
    */
   public void checkMandatoryFields() {
      if (mandatoryFields.size() != Constants.NUM_MANDATORY_FIELDS) {
         if (findMandatoryField(Constants.INFORMANT_MAN_FIELD) == null)
            mandatoryFields.add(new FieldInfo(Constants.INFORMANT_MAN_FIELD,
                  "string"));
         if (findMandatoryField(Constants.INFORMANT_RELATION_MAN_FIELD) == null)
            mandatoryFields.add(new FieldInfo(
                  Constants.INFORMANT_RELATION_MAN_FIELD, "string"));
         if (findMandatoryField(Constants.CLINICAL_RATER_MAN_FIELD) == null)
            mandatoryFields.add(new FieldInfo(
                  Constants.CLINICAL_RATER_MAN_FIELD, "string"));

      }
   }

   protected FieldInfo findMandatoryField(String fieldName) {
      for (Iterator<FieldInfo> iter = mandatoryFields.iterator(); iter
            .hasNext();) {
         FieldInfo mf = iter.next();
         if (mf.getName().equalsIgnoreCase(fieldName))
            return mf;
      }
      return null;
   }

   public List<ScoreInfo> getScores() {
      return scores;
   }

   public LinkedHashMap<ScoreInfo, Object> getScoreLevelMap() {
      return scoreLevelMap;
   }

   public List<FieldInfo> getMandatoryFields() {
      return mandatoryFields;
   }

   public ScoreInfo findScore(String name) {
      for (Iterator<ScoreInfo> iter = scores.iterator(); iter.hasNext();) {
         ScoreInfo si = iter.next();
         if (si.getScoreName().equals(name))
            return si;
      }
      return null;
   }

   public void removeScore(ScoreInfo si) {
      scores.remove(si);
      // remove the corresponding Assessment Item also
      if (items != null) {
         for (Iterator<ItemInfo> iter = items.iterator(); iter.hasNext();) {
            ItemInfo ii = iter.next();
            if (ii.getScoreName().equals(si.getScoreName())) {
               iter.remove();
               break;
            }
         }

      }
   }

   public ItemInfo findAsItem(String scoreName) {
      for (Iterator<ItemInfo> iter = items.iterator(); iter.hasNext();) {
         ItemInfo item = iter.next();
         if (item.getScoreName().equals(scoreName)) {
            return item;
         }
      }
      return null;
   }

   public void addItem(ItemInfo itemInfo) {
      if (items.contains(itemInfo)) {
         throw new RuntimeException("The itemInfo already exists:" + itemInfo);
      }
      items.add(itemInfo);
   }

   public List<ItemInfo> getItems() {
      return items;
   }

   // ---------------------- setters --------------
   public void setName(String newName) {
      this.name = newName;
   }

   public void setDescription(String newDescription) {
      this.description = newDescription;
   }

   // ---------------------- getters --------------
   public String getName() {
      return this.name;
   }

   public String getDescription() {
      return this.description;
   }

   public int hashCode() {
      return name.hashCode();
   }

   public boolean equals(Object other) {
      if (other == this) {
         return true;
      }
      if (!(other instanceof AssessmentInfo)) {
         return false;
      }
      AssessmentInfo that = (AssessmentInfo) other;
      return name.equals(that.name);
   }

   public static AssessmentInfo initializeFromXML(Element e) {
      AssessmentInfo ai = null;
      Map<String, ScoreInfo> availableScoreInfoMap = new HashMap<String, ScoreInfo>();
      String name = e.getAttributeValue("name");
      ai = new AssessmentInfo(name);
      Element descElem = e.getChild("description");
      if (descElem != null) {
         ai.setDescription(descElem.getText());
      }
      List<?> scoreElems = e.getChild("scores").getChildren("score");
      for (Iterator<?> iter = scoreElems.iterator(); iter.hasNext();) {
         Element elem = (Element) iter.next();
         ScoreInfo si = ScoreInfo.initializeFromXML(elem, ai,
               availableScoreInfoMap);
         ai.addScoreInfo(si);
         availableScoreInfoMap.put(si.getScoreName(), si);

         // make sure that every score code has the correct score
         // which can happen when assessment/score info is generated by an
         // external program
         // and hand edited thereafter (i.e renaming the score)
         for (ScoreCodeInfo sci : si.getScoreCodes()) {
            sci.setScoreName(si.getScoreName());
         }
      }

      // initialize the mandatory fields also
      Element fe = e.getChild("mandatory-fields");
      if (fe != null) {
         List<?> fieldElems = fe.getChildren("field");
         for (Iterator<?> iter = fieldElems.iterator(); iter.hasNext();) {
            Element elem = (Element) iter.next();
            FieldInfo fi = FieldInfo.initializeFromXML(elem);
            ai.addMandatoryField(fi);
         }
         // make sure all manadatory fields are accounted for
         ai.checkMandatoryFields();
      }

      // intialize assessment items also
      Element itemsElem = e.getChild("items");
      if (itemsElem != null) {
         List<?> itemElems = itemsElem.getChildren("as-item");
         for (Iterator<?> iter = itemElems.iterator(); iter.hasNext();) {
            Element elem = (Element) iter.next();
            ItemInfo ii = ItemInfo.initializeFromXML(elem);
            ai.addItem(ii);
         }
      }
      return ai;
   }

   public Element toXML(Element root) {
      Element e = new Element("assessment");
      e.setAttribute("name", name);
      if (description != null) {
         Element descElem = new Element("description");
         descElem.setText(description);
         e.addContent(descElem);
      }
      Element scoresElem = new Element("scores");
      e.addContent(scoresElem);
      for (Iterator<ScoreInfo> iter = scores.iterator(); iter.hasNext();) {
         ScoreInfo si = iter.next();
         scoresElem.addContent(si.toXML(root));
      }

      // mandatory fields
      Element fieldsElem = new Element("mandatory-fields");
      e.addContent(fieldsElem);
      for (Iterator<FieldInfo> iter = mandatoryFields.iterator(); iter
            .hasNext();) {
         FieldInfo fi = iter.next();
         fieldsElem.addContent(fi.toXML(root));
      }
      // assessment items
      Element itemsElem = new Element("items");
      e.addContent(itemsElem);
      for (Iterator<ItemInfo> iter = items.iterator(); iter.hasNext();) {
         ItemInfo item = iter.next();
         itemsElem.addContent(item.toXML(root));
      }

      return e;
   }

   public Element toXFormsXML(Namespace ns) {
      AssociationHelper ah = AssociationHelper.getInstance();
      Element e = new Element("assessment", ns);
      e.setAttribute("name", name);
      if (description != null) {
         Element descElem = new Element("description", ns);
         descElem.setText(description);
         e.addContent(descElem);
      }
      Element scoresElem = new Element("scores", ns);
      e.addContent(scoresElem);
      for (Iterator<ScoreInfo> iter = scores.iterator(); iter.hasNext();) {
         ScoreInfo si = iter.next();
         ScoreAssociation sa = (ScoreAssociation) ah.getAssocMap().get(si);

         scoresElem.addContent(si.toXFormsXML(ns, sa));
      }

      // mandatory fields
      Element fieldsElem = new Element("mandatory-fields", ns);
      e.addContent(fieldsElem);
      for (Iterator<FieldInfo> iter = mandatoryFields.iterator(); iter
            .hasNext();) {
         FieldInfo fi = iter.next();
         String varName = fi.getName().toLowerCase();
         MandatoryFieldAssociation mfa = ah
               .findMandatoryFieldAssociation(varName);
         fieldsElem.addContent(fi.toXFormsXML(ns, mfa));
      }

      return e;
   }

   public Entry findScoreEntry(String scoreName) {
      for(Map.Entry<ScoreInfo, Object> entry : scoreLevelMap.entrySet()) {
         Entry parent = new Entry(entry.getKey(), entry.getValue());
         Entry foundEntry = findScoreEntry(scoreName, parent);
         if (foundEntry != null) {
            return foundEntry;
         }
      }
      return null;
   }

   @SuppressWarnings("unchecked")
   public Entry findScoreEntry(String scoreName, Entry parent) {
      if (parent == null)
         return null;
      ScoreInfo si = (ScoreInfo) parent.getKey();
      if (si.getScoreName().equals(scoreName)) {
         return parent;
      }
      Object branchObj = parent.getValue();
      if (branchObj instanceof ScoreInfo) {
         ScoreInfo child = (ScoreInfo) branchObj;
         if (child.getScoreName().equals(scoreName)) {
            Entry e = new Entry(child, child);
            return e;
         }
         return null;
      } else {
         Map children = (Map) branchObj;
         for (Iterator iter = children.entrySet().iterator(); iter.hasNext();) {
            Map.Entry child = (Map.Entry) iter.next();
            Entry e = new Entry(child.getKey(), child.getValue());
            Entry foundEntry = findScoreEntry(scoreName, e);
            if (foundEntry != null) {
               return foundEntry;
            }
         }
      }
      return null;
   }

   @SuppressWarnings("unchecked")
   public void prepareLevelMap() {
      Map<ScoreInfo, Object> remMap = new LinkedHashMap<ScoreInfo, Object>();
      Map<ScoreInfo, Object> curLevelMap = new LinkedHashMap<ScoreInfo, Object>();
      for (Iterator<ScoreInfo> iter = getScores().iterator(); iter.hasNext();) {
         ScoreInfo si = iter.next();
         if (si.getParent() == null) {
            curLevelMap.put(si, si);
         } else {
            remMap.put(si, si.getParent());
         }
      }
      for(Map.Entry<ScoreInfo, Object> entry : curLevelMap.entrySet()) {
         ScoreInfo si = entry.getKey();
         scoreLevelMap.put(si, si);
         entry.setValue(si);
      }

      do {
         Map<ScoreInfo, Object> nextLevelMap = new LinkedHashMap<ScoreInfo, Object>();
         remMap = populateScoreLevel(remMap, nextLevelMap);
         for(Map.Entry<ScoreInfo, Object> entry : nextLevelMap.entrySet()) {
            ScoreInfo si =  entry.getKey();
            Object parent = curLevelMap.get(si.getParent());
            if (parent instanceof ScoreInfo) {
               ScoreInfo leaf = (ScoreInfo) parent;
               if (leaf == si.getParent()) {
                  // not child is set yet, so set it now.
                  leaf = si;
                  entry.setValue(leaf);
               } else {
                  // there was a child, so replace the leaf with a branch
                  LinkedHashMap<ScoreInfo, ScoreInfo> branch = new LinkedHashMap<ScoreInfo, ScoreInfo>();
                  branch.put(leaf, leaf);
                  branch.put(si, si);
                  entry.setValue(branch);
               }
            } else {
               LinkedHashMap<ScoreInfo, Object> branch = (LinkedHashMap<ScoreInfo, Object>) parent;
               branch.put(si, si);
            }

         }
         curLevelMap = nextLevelMap;
      } while (!remMap.isEmpty());
   }

   protected Map<ScoreInfo, Object> populateScoreLevel(
         Map<ScoreInfo, Object> remMap, Map<ScoreInfo, Object> curLevelMap) {
      Map<ScoreInfo, Object> rMap = new LinkedHashMap<ScoreInfo, Object>();
      for (Iterator<ScoreInfo> iter = remMap.keySet().iterator(); iter
            .hasNext();) {
         ScoreInfo si = iter.next();
         if (remMap.get(si.getParent()) == null) {
            curLevelMap.put(si, si);
         } else {
            rMap.put(si, si.getParent());
         }
      }
      return rMap;
   }

   public void dumpScoreHierarchy(PrintWriter out) {
      for(Map.Entry<ScoreInfo, Object>  entry : scoreLevelMap.entrySet()) {
         ScoreInfo si = entry.getKey();
         out.println(si.getScoreName());
         if (entry.getValue() == null)
            continue;
         dumpBranch(entry.getValue(), 1, out);
      }
   }

   @SuppressWarnings("unchecked")
   protected void dumpBranch(Object branch, int level, PrintWriter out) {
      if (branch instanceof ScoreInfo) {
         ScoreInfo leaf = (ScoreInfo) branch;
         indent(level, out);
         out.println(leaf.getScoreName());
      } else {
         Map<Object,Object> children = (Map<Object, Object>) branch;
         for (Iterator iter = children.entrySet().iterator(); iter.hasNext();) {
            Map.Entry entry = (Map.Entry) iter.next();
            ScoreInfo si = (ScoreInfo) entry.getKey();
            indent(level, out);
            out.println(si.getScoreName());
            if (entry.getValue() == null)
               continue;
            dumpBranch(entry.getValue(), level + 1, out);

         }
      }
   }

   protected void indent(int level, PrintWriter out) {
      for (int i = 0; i < level; i++) {
         out.print("  ");
      }
   }

   public void setMandatoryFieldMetaData(String fieldName, String metaName,
         String metaValue) {
      FieldInfo fieldInfo = findMandatoryField(fieldName);
      fieldInfo.addMetadata(metaName, metaValue);
   }

   public static class Entry {
      Object key;
      Object value;

      public Entry(Object key, Object value) {
         this.key = key;
         this.value = value;
      }

      public Object getKey() {
         return key;
      }

      public Object getValue() {
         return value;
      }
   }

}
