package caslayout.ui.model;

import guilib.common.BaseDialog;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.swing.JOptionPane;

import org.apache.log4j.Logger;

import caslayout.Constants;
import caslayout.ui.AssessmentSelectionDialog;
import caslayout.ui.CALMConfig;
import caslayout.ui.CALMHelper;
import caslayout.ui.db.Assessment;
import caslayout.ui.db.AssessmentItem;
import caslayout.ui.db.AssessmentScore;
import caslayout.ui.db.AssessmentScoreCode;
import caslayout.ui.db.DBParamsDialog;
import caslayout.ui.db.SessionFactoryFactory;
import caslayout.ui.db.services.AssessmentService;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentMaintenanceHelper.java,v 1.19 2005/12/01 01:06:23
 *          bozyurt Exp $
 */
public class AssessmentMaintenanceHelper {
   protected static Logger log = Logger
         .getLogger(AssessmentMaintenanceHelper.class);
   protected static boolean connectToDB = true;

   public AssessmentMaintenanceHelper() {}

   /**
    * initializes database connection and caches static info like security
    * classifications etc
    *
    * @param config
    */
   public static void initialize(CALMConfig config) {
      AssessmentService service;
      if (config.getPwd() == null || config.getPwd().trim().length() == 0) {
         // get the database password and other db params from the user
         DBParamsDialog dlg = new DBParamsDialog(null, "CALM Login", config);
         int rc = dlg.showDialog();

         if (rc != BaseDialog.OK_PRESSED) {
            JOptionPane
                  .showMessageDialog(
                        null,
                        "You have declined to connect to the HID database.\n"
                              + "Beware the fact you can only be able to work with form layout!",
                        "Warning", JOptionPane.WARNING_MESSAGE);
            connectToDB = false;
         }
      }

      try {
         if (connectToDB) {
            service = AssessmentService.getInstance(config);

            if (AssessmentInfo.getSecurityClassifications().isEmpty()) {
               List<?> scList = service.getSecurityClassifications();
               AssessmentInfo.populateSecurityClassifications(scList);
            }
         }
      } catch (Exception x) {
         x.printStackTrace();
         CALMHelper.showError(null, x.getMessage(), "An Error occured");
         config.setPwd(null);
         connectToDB = false;
      }
   }

   /**
    *
    * creates and returns an empty assessment
    *
    * @return
    * @throws java.lang.Exception
    */
   public static AssessmentInfo createEmptyAssessment() throws Exception {
      AssessmentInfo asi = new AssessmentInfo("New Assessment");
      // add the mandatory fields also
      prepareMandatoryFields(asi);

      return asi;
   }

   private static void prepareMandatoryFields(AssessmentInfo asi) {
      asi.addMandatoryField(new FieldInfo("date", "string"));
      asi.addMandatoryField(new FieldInfo("time", "string"));
      asi.addMandatoryField(new FieldInfo(Constants.INFORMANT_MAN_FIELD,
            "string"));
      asi.addMandatoryField(new FieldInfo(
            Constants.INFORMANT_RELATION_MAN_FIELD, "string"));
      asi.addMandatoryField(new FieldInfo(Constants.CLINICAL_RATER_MAN_FIELD,
            "string"));
   }

   public static AssessmentInfo loadAssessment(CALMConfig config)
         throws Exception {
      return loadAssessment(config, null);
   }

   public static void populateFromAssessmentScore(ScoreInfo si,
         AssessmentScore score) {
      si.setScoreType(score.getScoreType());
      si.setScoreSequence(score.getScoreSequence().intValue());
      si.setScoreLevel(score.getScoreLevel().intValue());
      si.setSecurityClassification(score.getSecurityClassification());
   }

   /**
    *
    * @param config
    *           CALMConfig
    * @param assessmentName
    *           the name of the assessment
    * @return <code>AssessmentInfo</code> object or null if no assessment is
    *         selected
    * @throws java.lang.Exception
    */
   public static AssessmentInfo loadAssessment(CALMConfig config,
         String assessmentName) throws Exception {
      AssessmentService service = null;
      AssessmentInfo asi = null;

      if (config.getPwd() == null || config.getPwd().trim().length() == 0
            && !config.isNoPassword()) {
         // get the database password and other db params from the user
         DBParamsDialog dlg = new DBParamsDialog(null, "CALM Login", config);
         dlg.showDialog();
      }

      try {
         service = AssessmentService.getInstance(config);

         boolean newAssessment = false;
         // ask the user for the assessment name

         if (assessmentName == null) {
            List<?> asList = service.getAllAssessments();
            List<String> asNames = new ArrayList<String>(asList.size());
            for (Iterator<?> it = asList.iterator(); it.hasNext();) {
               Assessment as = (Assessment) it.next();
               asNames.add(as.getName());
            }

            AssessmentSelectionDialog asDlg = new AssessmentSelectionDialog(
                  null, "Assessment Selection", asNames);

            int rc = -1;
            rc = asDlg.showDialog();

            if (rc == BaseDialog.OK_PRESSED) {
               assessmentName = asDlg.getSelectedAssessmentName();
            } else {
               do {
                  assessmentName = JOptionPane.showInputDialog(null,
                        "Enter the name of the Assessment to create:",
                        "New Assessment");
               } while (assessmentName == null
                     || assessmentName.trim().length() == 0);
               newAssessment = true;
            }
         }

         if (newAssessment) {
            asi = new AssessmentInfo(assessmentName);
            // add the mandatory fields also
            prepareMandatoryFields(asi);
            return asi;
         }

         Assessment as = service.getAssessment(assessmentName);
         if (as != null) {
            asi = new AssessmentInfo(as.getName());

            // add the mandatory fields also
            prepareMandatoryFields(asi);

            Map<ScoreInfo, ScoreInfo> siMap = new LinkedHashMap<ScoreInfo, ScoreInfo>();
            for (Iterator<AssessmentScore> iter = as.getScores().iterator(); iter
                  .hasNext();) {
               AssessmentScore score = iter.next();
               ScoreInfo si = new ScoreInfo(score.getScoreName());
               si.setAssessment(asi);
               populateFromAssessmentScore(si, score);

               siMap.put(si, si);
               if (score.getParent() != null) {
                  AssessmentScore parentScore = (AssessmentScore) score
                        .getParent();
                  ScoreInfo parentSi = new ScoreInfo(parentScore.getScoreName());
                  if (siMap.get(parentSi) != null) {
                     parentSi = siMap.get(parentSi);
                  } else {
                     parentSi.setAssessment(asi);
                     populateFromAssessmentScore(parentSi, parentScore);
                     siMap.put(parentSi, parentSi);
                  }
                  si.setParent(parentSi);
               }
               // add subscores also
               if (score.getScoreCodes() != null
                     && !score.getScoreCodes().isEmpty()) {
                  for (Iterator<AssessmentScoreCode> it = score.getScoreCodes()
                        .iterator(); it.hasNext();) {
                     AssessmentScoreCode asc = it.next();

                     ScoreCodeInfo sci = new ScoreCodeInfo(asc.getScoreName(),
                           asc.getScoreCode(), asc.getScoreCodeValue(), asc
                                 .getScoreCodeType(), asc
                                 .getScoreCodeTypeFormat());
                     si.addScoreCode(sci);
                  }
               }

            }// iter
            List<ScoreInfo> scores = new ArrayList<ScoreInfo>(siMap.values());
            Collections.sort(scores, new Comparator<ScoreInfo>() {
               public int compare(ScoreInfo o1, ScoreInfo o2) {
                  return o1.getScoreSequence() - o2.getScoreSequence();
               }
            });
            for (Iterator<ScoreInfo> iter = scores.iterator(); iter.hasNext();) {
               ScoreInfo si = iter.next();
               asi.addScoreInfo(si);
            }

            // add assessment items also
            for (Iterator<AssessmentItem> iter = as.getItems().iterator(); iter
                  .hasNext();) {
               AssessmentItem item = iter.next();
               ItemInfo ii = new ItemInfo(item.getAssessmentID(), item
                     .getScoreName());
               ii.setItemLeadingText(item.getItemLeadingText());
               ii.setItemTrailingText(item.getItemTrailingText());
               asi.addItem(ii);
            }
         } // as != null

      } catch (Exception x) {
         x.printStackTrace();
         CALMHelper.showError(null, x.getMessage(), "An Error occured");
         config.setPwd(null);
      }
      return asi;
   }

   public static boolean hasOnlyAssessmentItemChanges(List<AsDiffInfo> diffList) {
      for (Object element : diffList) {
         AsDiffInfo adi = (AsDiffInfo) element;
         if (adi.getType() != AsDiffInfo.AS_ITEM_ADDED
               && adi.getType() != AsDiffInfo.AS_ITEM_REMOVED
               && adi.getType() != AsDiffInfo.AS_ITEM_DIFFER) {
            return false;
         }
      }
      return true;
   }

   public static List<AsDiffInfo> findAssessmentDifference(AssessmentInfo asi,
         AssessmentInfo refAsi) throws Exception {
      List<AsDiffInfo> diffList = new ArrayList<AsDiffInfo>();
      Map<String, ScoreInfo> refScoresMap = new LinkedHashMap<String, ScoreInfo>();
      for (Iterator<ScoreInfo> iter = refAsi.getScores().iterator(); iter.hasNext();) {
         ScoreInfo si = (ScoreInfo) iter.next();
         refScoresMap.put(si.getScoreName(), si);
      }

      for (Iterator<ScoreInfo> iter = asi.getScores().iterator(); iter.hasNext();) {
         ScoreInfo si = (ScoreInfo) iter.next();
         ScoreInfo refSi = refScoresMap.get(si.getScoreName());
         if (refSi == null) {
            AsDiffInfo adi = new AsDiffInfo(si.getScoreName(),
                  AsDiffInfo.AS_SCORE_ADDED, null, si.getScoreName());
            diffList.add(adi);
         } else {
            if (!si.getScoreType().equals(refSi.getScoreType())) {
               AsDiffInfo adi = new AsDiffInfo(refSi.getScoreName(),
                     AsDiffInfo.AS_SCORE_DIFFER, refSi.getScoreName(),
                     "reference type " + refSi.getScoreType()
                           + " differs from " + si.getScoreType());
               diffList.add(adi);
               findScoreCodeDiffs(refSi, si, diffList);
            }
         }
      }

      Map<String, ScoreInfo> scoresMap = new LinkedHashMap<String, ScoreInfo>();
      for (Iterator<ScoreInfo> iter = asi.getScores().iterator(); iter.hasNext();) {
         ScoreInfo si = (ScoreInfo) iter.next();
         scoresMap.put(si.getScoreName(), si);
      }
      for (Iterator<ScoreInfo> iter = refAsi.getScores().iterator(); iter.hasNext();) {
         ScoreInfo refSi = (ScoreInfo) iter.next();
         ScoreInfo si = scoresMap.get(refSi.getScoreName());
         if (si == null) {
            AsDiffInfo adi = new AsDiffInfo(refSi.getScoreName(),
                  AsDiffInfo.AS_SCORE_REMOVED, refSi.getScoreName(), null);
            diffList.add(adi);
         }
      }

      Map<String, ItemInfo> refAiMap = new LinkedHashMap<String, ItemInfo>();
      for (Iterator<ItemInfo> iter = refAsi.getItems().iterator(); iter.hasNext();) {
         ItemInfo refAi = iter.next();
         refAiMap.put(refAi.getScoreName(), refAi);
      }

      for (Iterator<ItemInfo> iter = asi.getItems().iterator(); iter.hasNext();) {
         ItemInfo ai = iter.next();
         ItemInfo refAi = refAiMap.get(ai.getScoreName());
         if (refAi == null) {
            AsDiffInfo adi = new AsDiffInfo(ai.getScoreName(),
                  AsDiffInfo.AS_ITEM_ADDED, null, ai.getItemLeadingText());
            diffList.add(adi);
         } else {
            if (!refAi.getItemLeadingText().equals(ai.getItemLeadingText())) {
               AsDiffInfo adi = new AsDiffInfo(ai.getScoreName(),
                     AsDiffInfo.AS_ITEM_DIFFER, refAi.getItemLeadingText(), ai
                           .getItemLeadingText());
               diffList.add(adi);
            }
         }
      }
      return diffList;
   }

   protected static void findScoreCodeDiffs(ScoreInfo refSi, ScoreInfo si,
         List<AsDiffInfo> diffList) {
      Map<ScoreCodeInfo, ScoreCodeInfo> refScoreCodesMap = new LinkedHashMap<ScoreCodeInfo, ScoreCodeInfo>(
            11);
      Map<ScoreCodeInfo, ScoreCodeInfo> scoreCodesMap = new LinkedHashMap<ScoreCodeInfo, ScoreCodeInfo>(
            11);
      for (Iterator<ScoreCodeInfo> iter = refSi.getScoreCodes().iterator(); iter.hasNext();) {
         ScoreCodeInfo sci = (ScoreCodeInfo) iter.next();
         refScoreCodesMap.put(sci, sci);
      }
      for (Object element : si.getScoreCodes()) {
         ScoreCodeInfo sci = (ScoreCodeInfo) element;
         scoreCodesMap.put(sci, sci);
      }

      for (Object element : si.getScoreCodes()) {
         ScoreCodeInfo sci = (ScoreCodeInfo) element;
         if (refScoreCodesMap.get(sci) == null) {
            AsDiffInfo adi = new AsDiffInfo(refSi.getScoreName(),
                  AsDiffInfo.AS_SCORECODE_ADDED, null, sci.getScoreCodeValue());
            diffList.add(adi);
         }
      }

      for (Object element : refSi.getScoreCodes()) {
         ScoreCodeInfo sci = (ScoreCodeInfo) element;
         if (scoreCodesMap.get(sci) == null) {
            AsDiffInfo adi = new AsDiffInfo(refSi.getScoreName(),
                  AsDiffInfo.AS_SCORECODE_REMOVED, sci.getScoreCodeValue(),
                  null);
            diffList.add(adi);
         }
      }
   }

   public static void updateAssessmentItems(CALMConfig config,
         AssessmentInfo asi) throws Exception {
      AssessmentService service = null;
      Assessment as = null;
      try {
         service = AssessmentService.getInstance(config);
         Long asID = service.getAssessmentID(asi.getName());
         as = new Assessment(asID, asi.getName());
         // add the assessment items also
         for (Object element : asi.getItems()) {
            ItemInfo ii = (ItemInfo) element;
            AssessmentItem asItem = new AssessmentItem(ii.getAssessmentID(), ii
                  .getScoreName());
            asItem.setItemLeadingText(ii.getItemLeadingText());
            asItem.setItemTrailingText(ii.getItemTrailingText());
            as.addItem(asItem);
         }

         service.updateAssessmentItems(as);
      } catch (Exception x) {
         CALMHelper.showError(null, x.getMessage(), "An Error occured");
         log.error("", x);
         config.setPwd(null);
         throw x;
      }
   }

   /**
    * saves the entire assessment tree to the database.
    *
    * @param config
    * @param asi
    */
   public static void saveAssessment(CALMConfig config, AssessmentInfo asi,
         boolean deleteAsData) throws Exception {
      AssessmentService service = null;

      Assessment as = null;
      try {
         service = AssessmentService.getInstance(config);

         as = new Assessment(null, asi.getName());
         for (Object element : asi.getScores()) {
            ScoreInfo si = (ScoreInfo) element;
            AssessmentScore score = new AssessmentScore(null, si.getScoreName());
            score.setScoreType(si.getScoreType());
            /** @todo ScoreLevel needs to be set for ScoreInfo */
            score.setScoreLevel(new Integer(si.getScoreLevel()));
            score.setScoreSequence(new Integer(si.getScoreSequence()));
            score.setSecurityClassification(si.getSecurityClassification());
            as.addScore(score);
         }
         // second pass for setting the parent scores
         Map<String, AssessmentScore> scoreMap = toScoreMap(as.getScores());
         for (Object element : asi.getScores()) {
            ScoreInfo si = (ScoreInfo) element;
            AssessmentScore score = scoreMap.get(si.getScoreName());

            if (si.getParent() != null) {
               ScoreInfo parentSi = (ScoreInfo) si.getParent();
               AssessmentScore parentScore = scoreMap.get(parentSi
                     .getScoreName());

               score.setParentScore(parentSi.getScoreName());
               score.setParent(parentScore);
            }

            // set the score codes also
            prepareScoreCodes(score, si);
         }

         // add the assessment items also
         for (Object element : asi.getItems()) {
            ItemInfo ii = (ItemInfo) element;
            AssessmentItem asItem = new AssessmentItem(ii.getAssessmentID(), ii
                  .getScoreName());
            asItem.setItemLeadingText(ii.getItemLeadingText());
            asItem.setItemTrailingText(ii.getItemTrailingText());
            as.addItem(asItem);
         }

         // now assessment object is ready and can be persisted

         // first delete the old assessment (if any)
         // this operation will fail if the assessment is populated with data
         Assessment oldAs = service.getAssessment(as.getName());
         if (oldAs != null) {
            service.deleteAssessment(oldAs, deleteAsData);
         }
         service.saveAssessment(as);

      } catch (Exception x) {
         CALMHelper.showError(null, x.getMessage(), "An Error occured");
         log.error("", x);
         config.setPwd(null);
         throw x;
      }
   }

   protected static void prepareScoreCodes(AssessmentScore score, ScoreInfo si) {
      if (si.getScoreCodes().isEmpty())
         return;
      for (Object element : si.getScoreCodes()) {
         ScoreCodeInfo sci = (ScoreCodeInfo) element;
         AssessmentScoreCode asc = new AssessmentScoreCode(null, sci
               .getScoreName(), sci.getScoreCode(), sci.getScoreCodeValue(),
               sci.getScoreCodeType(), sci.getScoreCodeTypeFormat());
         score.addScoreCode(asc);
      }
   }

   protected static Map<String, AssessmentScore> toScoreMap(Set<AssessmentScore> scores) {
      Map<String, AssessmentScore> map = new HashMap<String, AssessmentScore>();
      for (Object element : scores) {
         AssessmentScore score = (AssessmentScore) element;
         map.put(score.getScoreName(), score);
      }
      return map;
   }

   protected static Map<String, List<AssessmentScoreCode>> groupByScoreName(List<AssessmentScoreCode> scoreCodes) {
      Map<String, List<AssessmentScoreCode>> map = new HashMap<String, List<AssessmentScoreCode>>();
      for (Object element : scoreCodes) {
         AssessmentScoreCode asc = (AssessmentScoreCode) element;
         List<AssessmentScoreCode> codeList = map.get(asc.getScoreName());
         if (codeList == null) {
            codeList = new LinkedList<AssessmentScoreCode>();
            map.put(asc.getScoreName(), codeList);
         }
         codeList.add(asc);
      }
      return map;
   }

   public static boolean isConnectToDB() {
      return connectToDB;
   }

   public static synchronized void shutdown() {
      if (isConnectToDB()) {
         SessionFactoryFactory.shutdown();
      }
   }
}
