package caslayout.ui.model;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Observable;

import org.apache.log4j.Logger;
import org.jdom.Element;

import caslayout.ui.AbstractDisplayComponent;
import caslayout.ui.Document;
import caslayout.ui.IDisplayComponent;
import caslayout.ui.LogicalGroup;
import caslayout.ui.LogicalGroupRepository;
import caslayout.ui.TextDisplayComponent;
import caslayout.util.GenUtils;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: ScoreAssociation.java,v 1.11 2008/10/13 23:58:08 bozyurt Exp $
 */
public class ScoreAssociation extends Association {
   protected ScoreInfo left;
   protected Object right;
   protected TextDisplayComponent scoreName;
   protected Object scoreValue;
   protected String scoreValueFieldName;
   protected Map<String, ScoreCodeInfo> scoreCode2CompIDMap = new LinkedHashMap<String, ScoreCodeInfo>(
         7);
   /* to add optional meta data as name value pairs to the score association */
   protected Map<String, String> metaDataMap = new LinkedHashMap<String, String>(
         3);

   protected static Logger log = Logger.getLogger(ScoreAssociation.class);

   public ScoreAssociation(String name, int type, ScoreInfo left,
         IDisplayComponent right) {
      super(name, type);
      this.left = left;
      this.right = right;
   }

   public ScoreAssociation(String name, int type, ScoreInfo left,
         LogicalGroup right) {
      super(name, type);
      this.left = left;
      this.right = right;
      // associate score value with the logical group id

   }

   public void associateScoreName(TextDisplayComponent tdc) {
      scoreName = tdc;
      left.setScoreName(tdc.getLabel().getText());
   }

   public void associateScoreValue(AbstractDisplayComponent adc) {
      scoreValue = adc;
   }

   public String getRHSID() {
      if (right instanceof AbstractDisplayComponent) {
         return ((AbstractDisplayComponent) right).getId();
      } else {
         return ((LogicalGroup) right).getId();
      }
   }

   public void update(Observable o, Object arg) {
      if (!o.hasChanged())
         return;
      if (scoreName == null && scoreValue == null)
         return;
      if (scoreName != null && ((Object) o) instanceof TextDisplayComponent) {
         left.setScoreName(scoreName.getLabel().getText());
      }
   }

   public void addScoreCodeInfo2CompIDPair(ScoreCodeInfo sci, String compID) {
      scoreCode2CompIDMap.put(compID, sci);
   }

   public Map<String, ScoreCodeInfo> getScoreCode2CompIDMap() {
      return scoreCode2CompIDMap;
   }

   public ScoreCodeInfo findScoreCodeInfo(String compID) {
      return scoreCode2CompIDMap.get(compID);
   }

   // ---------------------- setters --------------
   public void setLeft(ScoreInfo newLeft) {
      this.left = newLeft;
   }

   public void setRight(Object newRight) {
      this.right = newRight;
   }

   public void setScoreName(TextDisplayComponent newScoreName) {
      this.scoreName = newScoreName;
   }

   public void setScoreValue(Object newScoreValue) {
      this.scoreValue = newScoreValue;
   }

   public void setScoreValueFieldName(String newScoreValueFieldName) {
      this.scoreValueFieldName = newScoreValueFieldName;
   }

   // ---------------------- getters --------------
   public ScoreInfo getLeft() {
      return this.left;
   }

   public Object getRight() {
      return this.right;
   }

   public TextDisplayComponent getScoreName() {
      return this.scoreName;
   }

   public Object getScoreValue() {
      return this.scoreValue;
   }

   public String getScoreValueFieldName() {
      return this.scoreValueFieldName;
   }

   public void addMetadata(String name, String value) {
      metaDataMap.put(name, value);
   }

   public String getMetaDataValue(String key) {
      return metaDataMap.get(key);
   }

   public List<String> getMetaDataVarNames() {
      return new ArrayList<String>(metaDataMap.keySet());
   }

   public boolean hasMetaData() {
      return !metaDataMap.isEmpty();
   }

   public static ScoreAssociation initializeFromXML(Element e,
         AssessmentInfo ai, Document doc) {
      String name = e.getAttributeValue("name");
      Element leftElem = e.getChild("left");
      String scoreName = leftElem.getAttributeValue("score-name");
      String asName = leftElem.getAttributeValue("assessment");
      if (!ai.getName().equals(asName))
         throw new RuntimeException("current assessment '" + ai.getName()
               + "' is not same as the one read from the XML!");
      ScoreInfo left = ai.findScore(scoreName);
      Element rightElem = e.getChild("right");
      String id = rightElem.getAttributeValue("id");
      AbstractDisplayComponent right = GenUtils.findDisplayComponent(id, doc);
      ScoreAssociation sa = null;
      if (right != null) {
         sa = new ScoreAssociation(name, Association.BIDIRECTIONAL, left, right);
      } else {
         // must be a LogicalGroup
         LogicalGroup lg = LogicalGroupRepository.getInstance().get(id);
         if (lg == null) {
            // throw new RuntimeException("Cannot find a logical group for id "+
            // id);
            log.error("Cannot find a logical group for id " + id);
            doc.handleMissingCompInAssociation(id);
            return null;
         }
         sa = new ScoreAssociation(name, Association.BIDIRECTIONAL, left, lg);
      }

      Element svbElem = e.getChild("score-value-binding");
      if (svbElem != null) {
         id = svbElem.getAttributeValue("id");
         if (id != null) {
            AbstractDisplayComponent sv = GenUtils
                  .findDisplayComponent(id, doc);
            if (sv == null) {
               doc.handleMissingCompInAssociation(id);
               return null;
            }
            sa.setScoreValue(sv);
         } else {
            String lid = svbElem.getAttributeValue("lid");
            LogicalGroup lg = LogicalGroupRepository.getInstance().get(lid);
            if (lg == null) {
               doc.handleMissingCompInAssociation(lid);
               return null;
            }
            sa.setScoreValue(lg);
         }
      }

      Element sc2cidElem = e.getChild("score-code-bindings");
      if (sc2cidElem != null) {
         List<?> elems = sc2cidElem.getChildren("sc-binding");
         for (Iterator<?> iter = elems.iterator(); iter.hasNext();) {
            Element elem = (Element) iter.next();
            String scoreCode = elem.getAttributeValue("score-code");
            String compID = elem.getAttributeValue("id");
            ScoreCodeInfo sci = left.findScoreCodeInfo(scoreCode);
            sa.addScoreCodeInfo2CompIDPair(sci, compID);
         }
      }

      Element svnElem = e.getChild("score-name-binding");
      if (svnElem != null) {
         id = svbElem.getAttributeValue("id");
         TextDisplayComponent sn = (TextDisplayComponent) GenUtils
               .findDisplayComponent(id, doc);
         sa.setScoreName(sn);
      }
      Element svfnElem = e.getChild("score-value-field-name");
      if (svfnElem != null) {
         sa.setScoreValueFieldName(svnElem.getText());
      }

      // add the meta data also
      List<?> children = e.getChildren("meta");
      for (Iterator<?> iter = children.iterator(); iter.hasNext();) {
         Element child = (Element) iter.next();
         String vname = child.getAttributeValue("name");
         String value = child.getAttributeValue("value");
         sa.addMetadata(vname, value);
      }
      return sa;
   }

   public Element toXML(Element root) {
      if (left == null || right == null)
         return null;
      Element e = super.toXML(root);
      e.setName("score-binding");

      Element leftElem = new Element("left");
      leftElem.setAttribute("score-name", left.getScoreName());
      leftElem.setAttribute("assessment", left.getAssessment().getName());
      e.addContent(leftElem);

      if (right != null) {
         Element rightElem = new Element("right");
         if (right instanceof AbstractDisplayComponent) {
            rightElem.setAttribute("id", ((AbstractDisplayComponent) right)
                  .getId());
         } else if (right instanceof LogicalGroup) {
            rightElem.setAttribute("id", ((LogicalGroup) right).getId());
         }

         e.addContent(rightElem);

         Element scoreValueElem = new Element("score-value-binding");
         if (scoreValue instanceof AbstractDisplayComponent) {
            scoreValueElem.setAttribute("id",
                  ((AbstractDisplayComponent) scoreValue).getId());
         } else {
            scoreValueElem.setAttribute("lid", ((LogicalGroup) scoreValue)
                  .getId());
         }
         e.addContent(scoreValueElem);
         if (scoreValueFieldName != null) {
            Element scoreValueFieldNameElem = new Element(
                  "score-value-field-name");
            scoreValueFieldNameElem.setText(scoreValueFieldName);
            e.addContent(scoreValueFieldNameElem);
         }

         if (!scoreCode2CompIDMap.isEmpty()) {
            Element sc2cidElem = new Element("score-code-bindings");
            for(Map.Entry<String,ScoreCodeInfo> entry : scoreCode2CompIDMap.entrySet()) {
               Element elem = new Element("sc-binding");
               ScoreCodeInfo sci =  entry.getValue();
               String compID =  entry.getKey();
               elem.setAttribute("score-code", sci.getScoreCode());
               elem.setAttribute("id", compID);
               sc2cidElem.addContent(elem);
            }
            e.addContent(sc2cidElem);
         }

         if (scoreName != null) {
            Element scoreNameElem = new Element("score-name-binding");
            scoreNameElem.setAttribute("id", scoreName.getId());
            e.addContent(scoreNameElem);
         }
      } else {
         log.warn("Score " + left.getScoreName()
               + " is not associated with any component!");
      }

      // add meta data (if any)
      for(Map.Entry<String, String> entry : metaDataMap.entrySet()) {
         Element me = new Element("meta");
         me.setAttribute("name",  entry.getKey());
         me.setAttribute("value",  entry.getValue());
         e.addContent(me);
      }

      return e;
   }

   public String toString() {
      StringBuffer sb = new StringBuffer(128);
      sb.append("ScoreAssociation::[");
      sb.append("left=").append(left);
      sb.append(", right=").append(right);
      sb.append(", scoreName=").append(scoreName);
      sb.append(']');
      return sb.toString();
   }
}