package caslayout.util;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import caslayout.codegen.QuestionInfo;
import caslayout.codegen.TDColumn;
import caslayout.ui.AbstractDisplayComponent;
import caslayout.ui.CAContainer;
import caslayout.ui.CAGridLayout;
import caslayout.ui.CellConstraint;
import caslayout.ui.IDisplayComponent;
import caslayout.ui.MultipleCellSpanConstraint;
import caslayout.ui.PercentCellConstraint;
import caslayout.ui.QuestionGroup;
import caslayout.ui.QuestionGroupRepository;
import caslayout.ui.TextFieldDisplayComponent;
import caslayout.ui.model.Association;
import caslayout.ui.model.AssociationHelper;
import caslayout.ui.model.ScoreAssociation;
import caslayout.ui.model.ScoreInfo;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: CodegenUtils.java,v 1.18 2008/10/14 23:21:16 bozyurt Exp $
 */
public class CodegenUtils {

   protected CodegenUtils() {}

   public static StringBuilder loadCodeSnippet(String filename, String snippetID)
         throws IOException {
      StringBuilder buf = new StringBuilder(4096);
      BufferedReader in = null;
      String eol = System.getProperty("line.separator");
      try {
         in = new BufferedReader(new FileReader(filename));
         String line = null;
         boolean inSnippet = false;
         while ((line = in.readLine()) != null) {
            if (line.startsWith("/*++")) {
               StringTokenizer stok = new StringTokenizer(line);
               stok.nextToken(); // skip the first token
               if (stok.nextToken().equalsIgnoreCase(snippetID)) {
                  inSnippet = true;
                  continue;
               }
            }

            if (inSnippet) {
               if (line.startsWith("/*++")) {
                  // end of snippet (start of next snippet
                  break;
               }
               buf.append(line).append(eol);
            }
         }

      } finally {
         if (in != null) {
            try {
               in.close();
            } catch (Exception x) {}
         }
      }
      if (buf.length() == 0) {
         return null;
      }
      return buf;
   }

   public static String findAlignment(IDisplayComponent ic) {
      if (ic == null) {
         return null;
      }
      String value = null;

      value = ic.getJustification().getValue();
      // System.out.println(">>> " + ic.getClass() + " align=" + value);
      if (value != null) {
         value = value.toLowerCase();
         return value;
      }
      return null;
   }

   /**
    *
    * @param layout
    * @param row
    * @param colCount
    * @return
    */
   public static boolean areAllColumnWidthsEqual(CAGridLayout layout, int row,
         int colCount) {
      if (colCount == 0) {
         return false;
      }
      CellConstraint cc = layout.getCellConstraint(row, 0);
      if (cc instanceof PercentCellConstraint) {
         PercentCellConstraint pcc = (PercentCellConstraint) cc;
         double colPercent = pcc.getColPercent();
         for (int i = 1; i < colCount; i++) {
            pcc = (PercentCellConstraint) layout.getCellConstraint(row, i);
            if (Math.abs(pcc.getColPercent() - colPercent) > 0.01) {
               return false;
            }
         }

      } else {
         MultipleCellSpanConstraint mc = (MultipleCellSpanConstraint) cc;
         int theColSpan = mc.getColSpan();
         for (int i = 1; i < colCount; i++) {
            mc = (MultipleCellSpanConstraint) layout.getCellConstraint(row, i);
            if (mc.getColSpan() != theColSpan) {
               return false;
            }
         }
      }
      return true;
   }

   /**
    *
    * @param row
    *           the row index (zero based) of the container
    * @param layout
    * @param noCols
    *           number of columns in the row
    * @param container
    *           the container containing the display components
    * @param offset
    *           the offset for the container element list at which the display
    *           components for the specified row starts.
    *
    * @return
    */
   public static TDColumn[] prepareTableColumns(int row, CAGridLayout layout,
         int noCols, CAContainer container, int offset) {
      TDColumn[] tdCols = null;
      CellConstraint[] ccArr = new CellConstraint[noCols];
      // int maxCols = layout.getCols();

      for (int i = 0; i < noCols; i++) {
         CellConstraint cc = (CellConstraint) layout.getCellConstraint(row, i);
         ccArr[i] = cc;
      }

      boolean equalColWidths = areAllColumnWidthsEqual(layout, row, noCols);
      List<TDColumn> tdColList = new ArrayList<TDColumn>();

      if (ccArr[0] instanceof MultipleCellSpanConstraint) {
         if (equalColWidths) {
            for (int i = 0; i < noCols; i++) {
               MultipleCellSpanConstraint mc = (MultipleCellSpanConstraint) ccArr[i];
               TDColumn tdc = new TDColumn();
               IDisplayComponent ic = getDisplayComponent(container, offset + i);

               tdc.setAlign(CodegenUtils.findAlignment(ic));
               if (i != (noCols - 1)) {
                  tdc.setWidth(String.valueOf(100.0 / noCols) + "%");
               }
               if (mc.getColSpan() > 1) {
                  tdc.setColSpan(mc.getColSpan());

               }
               tdc.setCol(mc.getColIdx());
               tdColList.add(tdc);
            }
         } else {
            int totColSpan = 0;
            for (int i = 0; i < noCols; i++) {
               MultipleCellSpanConstraint mc = (MultipleCellSpanConstraint) ccArr[i];
               TDColumn tdc = new TDColumn();
               IDisplayComponent ic = getDisplayComponent(container, offset + i);
               tdc.setAlign(CodegenUtils.findAlignment(ic));
               tdc.setColSpan(mc.getColSpan());
               totColSpan += tdc.getColSpan();
               tdc.setCol(mc.getColIdx());
               tdColList.add(tdc);
               if (totColSpan == noCols) {
                  break;
               }
            }
         }
      } else {
         if (equalColWidths) {
            for (int i = 0; i < noCols; i++) {
               PercentCellConstraint pcc = (PercentCellConstraint) ccArr[i];
               TDColumn tdc = new TDColumn();
               IDisplayComponent ic = getDisplayComponent(container, offset + i);
               tdc.setAlign(CodegenUtils.findAlignment(ic));
               if (i != (noCols - 1)) {
                  tdc.setWidth(String.valueOf(100.0 / noCols) + "%");
               }
               tdc.setCol(pcc.getColIdx());
               tdColList.add(tdc);
            }
         } else {
            int noZeroWidthColumns = 0;
            for (int i = 0; i < noCols; i++) {
               PercentCellConstraint pcc = (PercentCellConstraint) ccArr[i];
               if (pcc.getColPercent() > 0) {
                  TDColumn tdc = new TDColumn();
                  IDisplayComponent ic = getDisplayComponent(container, offset
                        + i);
                  tdc.setAlign(CodegenUtils.findAlignment(ic));
                  if (i != (noCols - 1)) {
                     tdc.setWidth(String.valueOf(pcc.getColPercent()) + "%");
                  }
                  tdc.setCol(pcc.getColIdx());
                  tdColList.add(tdc);
               } else {
                  ++noZeroWidthColumns;
               }
            }
            // if there are any zero width columns, handle them by making
            // the last column to span the remaining number of columns
            if (noZeroWidthColumns > 0) {
               TDColumn tdc = tdColList.get(tdColList.size() - 1);
               tdc.setColSpan(noZeroWidthColumns + 1);
            }
            // last column does not need width
            TDColumn tdc = tdColList.get(tdColList.size() - 1);
            tdc.setWidth(null);
         }
      }

      tdCols = new TDColumn[tdColList.size()];
      tdColList.toArray(tdCols);

      return tdCols;
   }

   public static IDisplayComponent getDisplayComponent(CAContainer container,
         int idx) {
      if (idx >= container.getComponents().size()) {
         return null;
      }
      return (IDisplayComponent) container.getComponents().get(idx);
   }

   public static void prepareCALMTagForCSS(Writer out,
         AbstractDisplayComponent adc, int level, int indent)
         throws IOException {
      if (adc.getCSSClass() == null || adc.getCSSClass().trim().length() == 0) {
         return;
      }
      doIndent(out, level, indent);
      out.write("<calm:css class=\"");
      out.write(adc.getCSSClass());
      out.write("\"/>\n");
   }

   public static void doIndent(Writer out, int level, int indent)
         throws IOException {
      for (int i = 0; i < level * indent; i++) {
         out.write(' ');
      }
   }

   public static QuestionGroup findQuestionGroup(IDisplayComponent comp) {
      QuestionGroup qg = QuestionGroupRepository.getInstance().findMatching(
            comp);
      return qg;
   }

   /**
    * multiple answer question are assumed to have notes variable names encoded
    * ALWAYS as q<question#>_ans<questionIdx>_<scoreVarID>_notes
    *
    * e.g. q3_ans1_0_notes
    *
    * @param qg
    * @param idx
    *           question index (question index starts from 1)
    * @param idx2
    * @return
    */
   public static String createHiddenNotesVariable(QuestionGroup qg, int idx,
         int idx2) {
      return createHiddenNotesVariable(qg, idx, idx2, "_notes");
   }

   public static String createHiddenNotesVariable(QuestionGroup qg, int idx,
         int idx2, String suffix) {
      StringBuilder buf = new StringBuilder();
      buf.append("q").append(qg.getId());
      if (idx >= 0) {
         buf.append("_ans").append(idx);
      }
      if (idx2 >= 0) {
         buf.append("_").append(idx2);
      }
      buf.append(suffix);
      return buf.toString();
   }

   // unused
   public static String createNotesVariable(QuestionGroup qg, int idx,
         int idx2, String suffix) {
      StringBuilder buf = new StringBuilder();
      buf.append("q").append(qg.getId());
      if (idx >= 0) {
         buf.append("_ans").append(idx);
      }
      if (idx2 >= 0) {
         buf.append("_").append(idx2);
      }
      buf.append("_notes");
      return buf.toString();
   }

   public static void createScoreIDMap(QuestionGroup qg) {
      int count = 0;
      Set<Association> uniqueScores = new HashSet<Association>();
      for (Iterator<IDisplayComponent> iter = qg.getElements().iterator(); iter
            .hasNext();) {
         IDisplayComponent ic = iter.next();
         Association asoc = AssociationHelper.getInstance()
               .findScoreAssociationForDisplayComponent(ic);
         if (asoc == null) {
            continue;
         }
         if (uniqueScores.contains(asoc)) {
            continue;
         } else {
            uniqueScores.add(asoc);
         }
         if (asoc != null && asoc instanceof ScoreAssociation) {
            ScoreAssociation sa = (ScoreAssociation) asoc;
            qg.setScoreID(sa.getLeft().getScoreName(), count);
            count = count + 1;
         }
      }
   }

   public static String createMultiAnswerIteratorVariableName(QuestionGroup qg) {
      StringBuilder buf = new StringBuilder();
      buf.append("question").append(qg.getId()).append("Counter");
      return buf.toString();
   }

   public static String createAccessorName(String varName, String prefix) {
      StringBuilder buf = new StringBuilder();
      buf.append(prefix);
      buf.append(Character.toUpperCase(varName.charAt(0)));
      buf.append(varName.substring(1));
      return buf.toString();
   }

   public static QuestionInfo prepareQuestionInfo(QuestionGroup qg) {
      AbstractDisplayComponent adc = (AbstractDisplayComponent) qg
            .getElements().get(0);
      return new QuestionInfo(qg, adc);
   }

   public static Map<TextFieldDisplayComponent, Association> prepareCalculatedFieldMap(
         CAContainer container, AssociationHelper ah) {
      Map<TextFieldDisplayComponent, Association> cfMap = new HashMap<TextFieldDisplayComponent, Association>(
            3);
      findCalcFields(container, cfMap, ah);
      return cfMap;
   }

   public static void findCalcFields(IDisplayComponent ic,
         Map<TextFieldDisplayComponent, Association> cfMap, AssociationHelper ah) {
      if (ic instanceof TextFieldDisplayComponent) {
         TextFieldDisplayComponent tdc = (TextFieldDisplayComponent) ic;
         if (tdc.getExpression() != null
               && tdc.getExpression().getExprString() != null
               && tdc.getExpression().getExprString().length() > 0) {
            Association asoc = ah.findScoreAssociationForDisplayComponent(ic);
            cfMap.put(tdc, asoc);

         }
      } else if (ic instanceof CAContainer) {
         CAContainer cont = (CAContainer) ic;
         for (IDisplayComponent child : cont.getComponents()) {
            if (child != null) {
               findCalcFields(child, cfMap, ah);
            }
         }
      }
   }

   public static Map<String, ScoreInfo> prepJavaVar2ScoreInfoMap(
         List<ScoreInfo> scoreList) {
      Map<String, ScoreInfo> jv2ScoreInfoMap = new HashMap<String, ScoreInfo>();
      for (ScoreInfo si : scoreList) {
         String propertyName = GenUtils.convertToJavaVariableName(si
               .getScoreName());
         jv2ScoreInfoMap.put(propertyName, si);
      }
      return jv2ScoreInfoMap;
   }

   public static String prepareCalcFieldJSMethod(ScoreAssociation sa,
         TextFieldDisplayComponent tdc, Map<String, ScoreInfo> jv2ScoreInfoMap) {
      StringBuilder buf = new StringBuilder(1024);
      String expr = tdc.getExpression().getExprString();
      if (expr.endsWith(";")) {
         expr = expr.substring(0, expr.length() - 1);
      }

      String propertyName = GenUtils.convertToJavaVariableName(sa.getName());
      String funcName = "calc" + GenUtils.toTitleCase(propertyName);
      buf.append("function ").append(funcName).append("(elem,calcFieldID) {\n");
      buf.append("  var form = elem.form;\n");
      Map<String, String> varMap = findVariables(expr, jv2ScoreInfoMap);
      String section = prepareJSValidateSection(varMap);
      buf.append(section);
      String jsExpr = prepareJSExpression(expr, varMap);
      buf.append(jsExpr);
      buf.append("  var cfElem = document.getElementById(calcFieldID);\n");
      buf.append("  if ( !isNaN(value) && cfElem) {\n");
      buf.append("    cfElem.value = value;\n");
      buf.append("    return true;\n");
      buf.append("  }\n");
      buf.append("  return false;\n");
      buf.append("}\n");
      return buf.toString();
   }

   static Map<String, String> findVariables(String expr,
         Map<String, ScoreInfo> jv2ScoreInfoMap) {
      Map<String, String> varMap = new HashMap<String, String>(3);
      for (String propertyName : jv2ScoreInfoMap.keySet()) {
         int idx = expr.indexOf(propertyName);
         if (idx != -1) {
            varMap.put(propertyName, propertyName);
         }
      }
      return varMap;
   }

   static String prepareJSExpression(String expr, Map<String, String> varMap) {
      StringBuilder buf = new StringBuilder();
      String jsExpr = expr.replaceAll("log\\s*\\(", "Math.log(");
      jsExpr = jsExpr.replaceAll("sin\\s*\\(", "Math.sin(");
      jsExpr = jsExpr.replaceAll("cos\\s*\\(", "Math.cos(");
      jsExpr = jsExpr.replaceAll("exp\\s*\\(", "Math.exp(");
      jsExpr = jsExpr.replaceAll("sqrt\\s*\\(", "Math.sqrt(");
      for (Iterator<String> iter = varMap.keySet().iterator(); iter.hasNext();) {
         String propertyName = iter.next();
         jsExpr = jsExpr.replaceAll(propertyName, "form." + propertyName
               + ".value");
      }
      buf.append("  var value = ").append(jsExpr).append(";\n");
      return buf.toString();
   }

   static String prepareJSValidateSection(Map<String, String> varMap) {
      StringBuilder buf = new StringBuilder(200);
      if (!varMap.isEmpty()) {
         buf.append("  var badOperands = new Array();\n");
      }
      for (Iterator<String> iter = varMap.keySet().iterator(); iter.hasNext();) {
         String propertyName = iter.next();
         String jsValue = "form." + propertyName + ".value";
         buf.append("  if (isEmpty(").append(jsValue).append(") || isNaN(")
               .append(jsValue);
         buf.append(" ) ) {\n");
         buf.append("    badOperands.push(\"").append(propertyName).append(
               "\");\n");
         buf.append("  }\n");
      }
      if (!varMap.isEmpty()) {
         buf.append("  if (badOperands.length > 0) {\n");
         buf
               .append("    var str = \"Proper numeric values are required for the following field(s):\\n\";\n");
         buf.append("    for(var i = 0; i <  badOperands.length; i++) {\n");
         buf.append("      str += badOperands[i] + \"\\n\";\n");
         buf.append("    }\n");
         buf.append("    alert(str);\n");
         buf.append("    return false;\n");
         buf.append("  }\n");
      }
      return buf.toString();
   }

   /**
    * for each multi-answer question determine their outermost container and
    * return them in a hashtable containing container - question group pairs.
    *
    * @param container
    *           CAContainer
    * @return Map
    */
   public static Map<CAContainer, QuestionGroup> prepareQuestionGroupContainerMap(
         CAContainer container) {
      // for each multi-answer question determine their outermost container
      // and
      // return them in a hashtable containing container - question group
      // pairs
      Map<CAContainer, QuestionGroup> qgContainerMap = new HashMap<CAContainer, QuestionGroup>(
            17);
      QuestionGroupRepository repository = QuestionGroupRepository
            .getInstance();
      for (String qgID : repository.getQuestionIDs()) {
         QuestionGroup qg = repository.get(qgID);
         if (qg.isMultiAnswer()) {
            QuestionElemContainerInfo qeci = new QuestionElemContainerInfo();
            for (IDisplayComponent idc : qg.getElements()) {
               qeci.qgElementMap.put(idc, idc);
            }
            List<CAContainer> containerList = new ArrayList<CAContainer>();
            ReturnStatus rs = new ReturnStatus();
            findEnclosingContainers4Question(container, qg, containerList,
                  qeci, rs);
            if (rs.isSuccess()) {
               CAContainer outermost = findOuterMostContainers4Question(
                     containerList, qeci);
               qgContainerMap.put(outermost, qg);
            }
         }
      }

      return qgContainerMap;
   }

   static class QuestionElemContainerInfo {
      Map<IDisplayComponent, IDisplayComponent> qgElementMap = new HashMap<IDisplayComponent, IDisplayComponent>(
            17);
      /**
       * holds immediate containers for each display component (display
       * component - container it is found in
       */
      Map<IDisplayComponent, CAContainer> containerMap = new HashMap<IDisplayComponent, CAContainer>(
            7);
   }

   static class ReturnStatus {
      boolean success = false;

      public void setSuccess(boolean value) {
         this.success = value;
      }

      public boolean isSuccess() {
         return success;
      }
   }

   /**
    * search the whole containment hierarchy of <code>parentCon</code> for
    * <code>con</con>
    * @param parentCon
    * @param con
    * @param foundList
    */
   protected static void findIfContains(CAContainer parentCon, CAContainer con,
         List<CAContainer> foundList) {
      if (parentCon == con) {
         foundList.add(parentCon);
         return;
      }
      for (Iterator<IDisplayComponent> iter = parentCon.getComponents()
            .iterator(); iter.hasNext();) {
         AbstractDisplayComponent adc = (AbstractDisplayComponent) iter.next();
         if (adc instanceof CAContainer) {
            CAContainer childCon = (CAContainer) adc;
            if (childCon == con) {
               foundList.add(parentCon);
               return;
            } else {
               findIfContains(childCon, con, foundList);
            }
         }
      }
   }

   protected static CAContainer findLargestContainer(
         QuestionElemContainerInfo qeci) {
      Set<CAContainer> containerSet = new LinkedHashSet<CAContainer>();
      for (Iterator<CAContainer> iter = qeci.containerMap.values().iterator(); iter
            .hasNext();) {
         CAContainer con = iter.next();
         containerSet.add(con);
      }
      List<CAContainer> containerList = new ArrayList<CAContainer>(containerSet);
      // order the container list by descending container area
      Collections.sort(containerList, new Comparator<CAContainer>() {
         public int compare(CAContainer c1, CAContainer c2) {
            int diff = (c2.getWidth() * c2.getHeight())
                  - (c1.getWidth() * c1.getHeight());
            return diff;
         }
      });

      // since component Ids are always given in increasing order and
      // no container can be added before its enclosing container
      // the container with the smallest container id within a set
      // is the oldest one
      // TODO 6/1/2007 check this approach with regular CALM forms
      Collections.sort(containerList, new Comparator<CAContainer>() {
         public int compare(CAContainer c1, CAContainer c2) {
            int diff = getCompIDNum(c1.getId()) - getCompIDNum(c2.getId());
            return diff;
         }
      });
      // since all of these containers belong to the same multi-answer
      // question
      // the largest container must contain all the others otherwise there is
      // no
      return containerList.get(0);
   }

   public static int getCompIDNum(String id) {
      return GenUtils.toInt(id.substring(1), Integer.MAX_VALUE);
   }

   protected static CAContainer findOuterMostContainers4Question(
         List<CAContainer> containerList, QuestionElemContainerInfo qeci) {
      Map<Object, Object> finishedICMap = new HashMap<Object, Object>(17);
      Map<CAContainer, List<IDisplayComponent>> contIDCListMap = new HashMap<CAContainer, List<IDisplayComponent>>(
            7);

      CAContainer largestCon = findLargestContainer(qeci);

      for (Map.Entry<IDisplayComponent, CAContainer> entry : qeci.containerMap
            .entrySet()) {
         IDisplayComponent idc = entry.getKey();
         List<IDisplayComponent> idcList = contIDCListMap.get(largestCon);
         if (idcList == null) {
            idcList = new LinkedList<IDisplayComponent>();
            contIDCListMap.put(largestCon, idcList);
         }
         // assign display components of the question to the largest
         // component
         idcList.add(idc);
         // dumpList("idcList for " + largestCon.toString(), idcList);
      }
      // in case the largest container within a question is not included
      // in the container list
      if (!containerList.contains(largestCon)) {
         containerList.add(largestCon);
      }

      // dumpList("containerList before sort ", containerList);

      // order the container list by ascending container area
      Collections.sort(containerList, new Comparator<CAContainer>() {
         public int compare(CAContainer c1, CAContainer c2) {
            int diff = (c1.getWidth() * c1.getHeight())
                  - (c2.getWidth() * c2.getHeight());
            return diff;
         }
      });

      dumpList("containerList after area sort ", containerList);

      Collections.sort(containerList, new Comparator<CAContainer>() {
         public int compare(CAContainer c1, CAContainer c2) {
            int diff = getCompIDNum(c2.getId()) - getCompIDNum(c1.getId());
            return diff;
         }
      });

      dumpList("containerList after sort ", containerList);

      // remove containers till the one containing all components for the
      // question is seen
      CAContainer con = null;
      while (!containerList.isEmpty()) {
         con = containerList.remove(0);
         List<IDisplayComponent> idcList = contIDCListMap.get(con);
         if (idcList != null) {
            for (Iterator<IDisplayComponent> iter = idcList.iterator(); iter
                  .hasNext();) {
               IDisplayComponent idc = iter.next();
               finishedICMap.put(idc, idc);
            }
            if (finishedICMap.size() == qeci.containerMap.size()) {
               break;
            }
         }
      }

      // continue popping till you find the first container that contains
      // all of the components of the multiple answer question.

      Set<CAContainer> containerSet = new HashSet<CAContainer>();
      for (Iterator<CAContainer> iter = qeci.containerMap.values().iterator(); iter
            .hasNext();) {
         CAContainer c = iter.next();
         containerSet.add(c);
      }
      do {
         if (containsAll(containerSet, con)) {
            // found the outermost container
            return con;
         }
         if (!containerList.isEmpty()) {
            con = containerList.remove(0);
         }
      } while (!containerList.isEmpty());

      return null;
   }

   protected static boolean containsAll(Set<CAContainer> containerSet,
         CAContainer cont2Check) {
      List<CAContainer> foundList = new ArrayList<CAContainer>(1);
      for (Iterator<CAContainer> iter = containerSet.iterator(); iter.hasNext();) {
         CAContainer con = iter.next();
         foundList.clear();
         findIfContains(cont2Check, con, foundList);
         if (foundList.isEmpty()) {
            return false;
         }
      }
      return true;
   }

   public static void dumpList(String msg, List<CAContainer> aList) {
      if (msg != null) {
         System.out.println(msg);
      }
      for (Iterator<CAContainer> iter = aList.iterator(); iter.hasNext();) {
         System.out.println(iter.next().toString());
      }
      System.out.println();
   }

   protected static void findEnclosingContainers4Question(
         CAContainer container, QuestionGroup qg,
         List<CAContainer> containerList, QuestionElemContainerInfo ceci,
         ReturnStatus rs) {
      if (ceci.qgElementMap.isEmpty()) {
         return;
      }
      for (IDisplayComponent ic : container.getComponents()) {
         if (ic == null) {
            continue;
         }
         if (ic instanceof CAContainer) {
            containerList.add(container);
            findEnclosingContainers4Question((CAContainer) ic, qg,
                  containerList, ceci, rs);

         } else {
            QuestionGroup aqg = QuestionGroupRepository.getInstance()
                  .findMatching(ic);
            if (aqg == qg) {
               rs.setSuccess(true);
               // remove the found element
               ceci.qgElementMap.remove(ic);
               ceci.containerMap.put(ic, container);
               if (ceci.qgElementMap.isEmpty()) {
                  return;
               }
            }
         }
      }
   }

}
