package caslayout.util;

import java.beans.Introspector;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.Writer;
import java.text.NumberFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;
import java.util.Stack;
import java.util.StringTokenizer;

import caslayout.ui.AbstractDisplayComponent;
import caslayout.ui.CAContainer;
import caslayout.ui.CAPanel;
import caslayout.ui.Document;
import caslayout.ui.IDisplayComponent;
import caslayout.ui.Page;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: GenUtils.java,v 1.18 2008/10/13 23:58:08 bozyurt Exp $
 */
public class GenUtils {
   private static SimpleDateFormat daf = new SimpleDateFormat("MM/dd/yyyy");
   private static SimpleDateFormat tsf = new SimpleDateFormat(
         "MM/dd/yyyy HH:mm:ss");
   private static SimpleDateFormat tmf = new SimpleDateFormat(
         "MM/dd/yyyy HH:mm");
   private static SimpleDateFormat thf = new SimpleDateFormat("MM/dd/yyyy HH");

   static {
      daf.setLenient(false);
      tsf.setLenient(false);
      tmf.setLenient(false);
      thf.setLenient(false);
   }

   protected GenUtils() {}

   /**
    *
    * @param propsFilename
    *           properties file to load the properties from. The file must be
    *           accessible via the CLASSPATH
    * @return a <code>java.util.Properties</code> object holding the
    *         properties loaded from the file
    * @throws IOException
    *            if there is an error finding or reading from the properties
    *            file
    */
   public static Properties loadProperties(String propsFilename)
         throws IOException {
      InputStream is = GenUtils.class.getClassLoader().getResourceAsStream(
            propsFilename);
      Properties props = new Properties();
      props.load(is);

      return props;
   }

   /**
    *
    * @param path
    * @return
    */
   public static String getBasename(String path) {
      File f = new File(path);
      String name = f.getName();
      int idx = name.lastIndexOf(".");
      if (idx != -1)
         return name.substring(0, idx);
      else
         return name;
   }

   /**
    *
    * @param dir
    * @param fileExtension
    * @return
    */
   public static String[] getFiles(String dir, String fileExtension) {
      File f = new File(dir);
      String[] files = f.list(new FileExtensionFilter(fileExtension));

      return files;
   }

   /**
    *
    * @param adc
    */
   public static void setUIDs(AbstractDisplayComponent adc) {
      adc.setId(UIDManager.getInstance().getNextUID());
      if (adc instanceof CAContainer) {
         CAContainer c = (CAContainer) adc;
         for (Iterator<IDisplayComponent> iter = c.getComponents().iterator(); iter
               .hasNext();) {
            AbstractDisplayComponent child = (AbstractDisplayComponent) iter
                  .next();
            if (child != null) {
               setUIDs(child);
            }
         }
      }
   }

   /**
    *
    * @param str
    * @return
    */
   public static String convertToJavaVariableName(String str) {
      str = str.trim();
      StringBuffer buf = new StringBuffer();
      if (str.indexOf(' ') != -1 || str.indexOf('\t') != -1
            || str.indexOf('-') != -1 || str.indexOf(',') != -1
            || str.indexOf('\'') != -1 || str.indexOf('"') != -1) {
         StringTokenizer stok = new StringTokenizer(str, " \t-,\'\"");
         buf.append(Introspector.decapitalize(stok.nextToken()));
         while (stok.hasMoreTokens()) {
            buf.append(toTitleCase(stok.nextToken()));
         }
         String s = buf.toString();
         if (s.indexOf('.') != -1) {
            s = s.replaceAll("\\.", "");
         }
         return s;
      } else {
         return Introspector.decapitalize(str);
      }
   }

   public static String toTitleCase(String str) {
      StringBuffer buf = new StringBuffer(str.length());
      char[] carr = str.toCharArray();
      buf.append(Character.toUpperCase(carr[0]));
      for (int i = 1; i < carr.length; i++) {
         if (Character.isJavaIdentifierPart(carr[i])) {
            buf.append(carr[i]);
         }
      }
      return buf.toString();
   }

   public static String toFilename(String str) {
      if (str.indexOf(' ') == -1)
         return str;
      return str.replaceAll("\\s+", "_");
   }

   /**
    *
    * @param javaVarName
    * @param prefix
    * @return
    */
   public static String prepareAccessorName(String javaVarName, String prefix) {
      StringBuffer buf = new StringBuffer();

      buf.append(prefix).append(toTitleCase(javaVarName));
      return buf.toString();
   }

   public static class FileExtensionFilter implements FilenameFilter {
      String extension;

      public FileExtensionFilter(String extension) {
         this.extension = extension;
      }

      public boolean accept(File dir, String name) {
         return (name.toLowerCase().endsWith(extension));
      }
   }

   public static void copyFile(File srcFile, File destFile) throws IOException {
      copyFile(srcFile.getAbsolutePath(), destFile.getAbsolutePath());
   }

   public static void copyFile(String srcFile, String destFile)
         throws IOException {
      BufferedOutputStream out = null;
      BufferedInputStream in = null;
      byte[] buffer = new byte[4096];
      try {
         in = new BufferedInputStream(new FileInputStream(srcFile), 4096);
         out = new BufferedOutputStream(new FileOutputStream(destFile));
         int bytesRead = -1;

         while ((bytesRead = in.read(buffer, 0, buffer.length)) != -1) {
            out.write(buffer, 0, bytesRead);
         }
      } finally {
         if (in != null)
            try {
               in.close();
            } catch (Exception x) {}
         if (out != null)
            try {
               out.close();
            } catch (Exception x) {}
      }
   }

   public static AbstractDisplayComponent findDisplayComponent(String id,
         Document doc) {
      for (Iterator<Page> iter = doc.getPagesIterator(); iter.hasNext();) {
         Page page = iter.next();
         CAPanel panel = page.getViewPanel();
         Stack<IDisplayComponent> stack = new Stack<IDisplayComponent>();
         findComponentById(panel, id, stack);

         AbstractDisplayComponent adc = null;
         if (!stack.isEmpty())
            adc = (AbstractDisplayComponent) stack.pop();
         if (adc != null)
            return adc;
      }
      return null;
   }

   /**
    * Do a depth-first search to find the component with the given id
    *
    * @param adc
    * @param id
    * @param stack
    */
   protected static void findComponentById(AbstractDisplayComponent adc,
         String id, Stack<IDisplayComponent> stack) {
      if (adc.getId().equals(id)) {
         stack.push(adc);
         return;
      }
      if (adc instanceof CAContainer) {
         CAContainer c = (CAContainer) adc;
         List<IDisplayComponent> nextLevel = new LinkedList<IDisplayComponent>();
         for (Iterator<IDisplayComponent> iter = c.getComponents().iterator(); iter
               .hasNext();) {
            AbstractDisplayComponent child = (AbstractDisplayComponent) iter
                  .next();
            if (child != null) {
               if (!(child instanceof CAContainer)) {
                  if (child.getId().equals(id)) {
                     stack.push(child);
                     return;
                  }
               } else {
                  nextLevel.add(child);
               }
            }
         }

         for (Iterator<IDisplayComponent> iter = nextLevel.iterator(); iter
               .hasNext();) {
            CAContainer con = (CAContainer) iter.next();
            findComponentById(con, id, stack);
         }
      }
   }

   public static String formatDouble(double num) {
      NumberFormat nf = NumberFormat.getNumberInstance();
      nf.setMaximumFractionDigits(2);
      return nf.format(num);
   }

   public static String cleanHTMLTags(String text) {
      char[] carr = text.toCharArray();
      StringBuffer buf = new StringBuffer();
      boolean inTag = false;
      for (int i = 0; i < carr.length; i++) {
         if (carr[i] == '<') {
            if ((i + 1) < carr.length
                  && (Character.isLetter(carr[i + 1]) || carr[i + 1] == '/')) {
               inTag = true;
            }
         } else if (carr[i] == '>') {
            inTag = false;
            continue;
         }

         if (!inTag) {
            buf.append(carr[i]);
         }
      }
      return buf.toString();
   }

   public static void close(Reader in) {
      if (in != null) {
         try {
            in.close();
         } catch (Exception x) {
            // ignore
         }
      }
   }

   public static void close(Writer out) {
      if (out != null) {
         try {
            out.close();
         } catch (Exception x) {
            // ignore
         }
      }
   }

   public static void close(InputStream in) {
      if (in != null) {
         try {
            in.close();
         } catch (Exception x) {
            // ignore
         }
      }
   }

   public static int toInt(String numberStr, int defaultValue) {
      int num = defaultValue;
      try {
         num = Integer.parseInt(numberStr);
      } catch (NumberFormatException nfe) {
         num = defaultValue;
      }
      return num;
   }

   public static int numberOfMatch(String s, char c) {
      int idx = s.indexOf(c);
      int count = 0;
      while (idx != -1) {
         ++count;
         idx = s.indexOf(c, idx + 1);
      }
      return count;
   }

   public static Date toDate(String ts) throws ParseException {
      Date date = null;
      if (ts == null || ts.trim().length() == 0) {
         throw new ParseException("Missing date!", 0);
      }
      try {
         if (ts.indexOf(':') != -1) {
            int numMatch = GenUtils.numberOfMatch(ts, ':');
            switch (numMatch) {
            case 1:
               date = tmf.parse(ts);
               break;
            case 2:
               date = tsf.parse(ts);
               break;
            default:
               throw new ParseException("Not a valid date: " + ts, 0);
            }
         } else {
            date = thf.parse(ts);
         }
      } catch (ParseException pe) {
         date = daf.parse(ts);
      }
      return date;
   }
}
