package caslayout.util;

import java.awt.Component;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.jdom.Attribute;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

import caslayout.ui.AbstractDisplayComponent;
import caslayout.ui.CAContainer;
import caslayout.ui.CAPanel;
import caslayout.ui.IDisplayComponent;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: XMLUtils.java,v 1.13 2008/10/13 23:58:08 bozyurt Exp $
 */
public class XMLUtils {
	protected static Map<String, Object> refMap = Collections.synchronizedMap(new HashMap<String, Object>());

	public XMLUtils() {
	}

	public static void registerObject(String refID, Object object) {
		refMap.put(refID, object);
	}

	public synchronized static void clearCache() {
		refMap.clear();
	}

	public static Object findObject(String refID) {
		return refMap.get(refID);
	}

	public static Element prepareProperty(String name, String value) {
		Element e = new Element("property");
		e.setAttribute("name", name);
		e.setAttribute("value", value);
		return e;
	}

	public static Element prepareRefID(String name, String refID) {
		Element e = new Element("property");
		e.setAttribute("name", name);
		e.setAttribute("ref-id", refID);
		return e;
	}

	public static String getStringPropertyValue(String propertyName, Element e,
			String defaultValue) {
		List<?> children = e.getChildren("property");
		if (children == null)
			return defaultValue;
		for (Iterator<?> iter = children.iterator(); iter.hasNext();) {
			Element ce = (Element) iter.next();
			if (ce.getAttributeValue("name").equals(propertyName)) {
				return ce.getAttributeValue("value");
			}
		}
		return defaultValue;
	}

	public static int getPropertyValue(String propertyName, Element e,
			int defaultValue) {
		List<?> children = e.getChildren("property");
		if (children == null)
			return defaultValue;
		for (Iterator<?> iter = children.iterator(); iter.hasNext();) {
			Element ce = (Element) iter.next();
			if (ce.getAttributeValue("name").equals(propertyName)) {
				return Integer.parseInt(ce.getAttributeValue("value"));
			}
		}
		return defaultValue;
	}

	public static double getPropertyValueDouble(String propertyName, Element e,
			double defaultValue) {
		List<?> children = e.getChildren("property");
		if (children == null)
			return defaultValue;
		for (Iterator<?> iter = children.iterator(); iter.hasNext();) {
			Element ce = (Element) iter.next();
			if (ce.getAttributeValue("name").equals(propertyName)) {
				return Double.parseDouble(ce.getAttributeValue("value"));
			}
		}
		return defaultValue;
	}

	public static String getReferencePropertyValue(String propertyName,
			Element e, String defaultValue) {
		List<?> children = e.getChildren("property");
		if (children == null)
			return defaultValue;
		for (Iterator<?> iter = children.iterator(); iter.hasNext();) {
			Element ce = (Element) iter.next();
			if (ce.getAttributeValue("name").equals(propertyName)) {
				return ce.getAttributeValue("ref-id");
			}
		}
		return defaultValue;
	}

	public static Element getObjectElement(Element e, String className) {
		List<?> children = e.getChildren("object");
		for (Iterator<?> iter = children.iterator(); iter.hasNext();) {
			Element item = (Element) iter.next();
			if (className.equals(item.getAttributeValue("class")))
				return item;

		}
		return null;
	}

	public static Element getObjectElementByName(Element e, String name) {
		List<?> children = e.getChildren("object");
		for (Iterator<?> iter = children.iterator(); iter.hasNext();) {
			Element item = (Element) iter.next();
			Attribute attr = item.getAttribute("name");
			if (attr == null)
				continue;
			if (name.equals(attr.getValue()))
				return item;
		}
		return null;
	}

	public static Object getReferencedObject(String refID, Element e) {
		Object object = findObject(refID);
		if (object != null)
			return object;
		/** @todo create and register the referenced object */
		return null;
	}

	public static ListInfo prepareList(String collectionName, Element e) {
		List<?> colElems = e.getChildren("collection");
		if (colElems == null)
			return null;
		for (Iterator<?> iter = colElems.iterator(); iter.hasNext();) {
			Element item = (Element) iter.next();
			if (item.getAttributeValue("name").equals(collectionName)) {
				if (item.getAttributeValue("class").equals(
						"java.util.ArrayList")) {
					ListInfo listInfo = new ListInfo();
					int size = Integer.parseInt(item.getAttributeValue("size"));
					listInfo.list = new ArrayList<Object>(size);
					List<Object> entries = listInfo.list;
					List<?> entryElems = item.getChildren("entry");
					listInfo.entryElems = entryElems;
					for (int i = 0; i < size; ++i) {
						entries.add(null);
					}
					for (Iterator<?> it = entryElems.iterator(); it.hasNext();) {
						Element entryElem = (Element) it.next();
						int index = Integer.parseInt(entryElem
								.getAttributeValue("index"));
						if (entryElem.getChild("array") != null) {
							Element arrElem = entryElem.getChild("array");
							Object[] arr = prepareArray(arrElem);
							entries.set(index, arr);
						} else if (entryElem.getChild("object") != null) {
							Element objElem = entryElem.getChild("object");
							Object o = prepareObject(objElem);
							entries.set(index, o);
						} else if (entryElem.getChild("container") != null) {
							Element objElem = entryElem.getChild("container");
							Object o = prepareObject(objElem);
							entries.set(index, o);
						}
					}

					return listInfo;
				}

			}

		}
		return null;
	}

	public static Object[] prepareArray(Element e) {
		Element arrElem = e;
		if (arrElem == null)
			return null;
		String clazzName = arrElem.getAttributeValue("class");
		int length = Integer.parseInt(arrElem.getAttributeValue("length"));
		try {
			Class<?> clazz = Class.forName(clazzName);
			Object[] arr = (Object[]) Array.newInstance(clazz, length);
			List<?> arrElems = e.getChildren("object");
			int idx = 0;
			for (Iterator<?> iter = arrElems.iterator(); iter.hasNext();) {
				Element item = (Element) iter.next();
				Method m = clazz.getMethod("initializeFromXML",
						new Class[] { Element.class });
				arr[idx] = m.invoke(null, new Object[] { item });
				++idx;
			}
			return arr;
		} catch (Exception x) {
			x.printStackTrace();
		}
		return null;
	}

	public static void savePanel(String filename, CAPanel panel)
			throws IOException {
		XMLOutputter xmlOut = null;
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(filename));
			xmlOut = new XMLOutputter(Format.getPrettyFormat()); // " ",
			// true);
			Element rootElem = new Element("caslayout");
			rootElem.addContent(panel.toXML(rootElem));
			xmlOut.output(rootElem, out);
		} finally {
			if (out != null)
				try {
					out.close();
				} catch (Exception x) {
				}
		}
	}

	public static CAPanel loadPanel(String filename, Component peer)
			throws JDOMException, IOException {
		SAXBuilder builder = null;

		builder = new SAXBuilder(false);
		org.jdom.Document doc = builder.build(filename);
		Element element = doc.getRootElement().getChild("container");

		CAPanel newPanel = CAPanel.initializeFromXML(element);
		setPeerForChildren(newPanel, peer);
		XMLUtils.clearCache();
		return newPanel;
	}

	public static void setPeerForChildren(AbstractDisplayComponent adc,
			Component peer) {
		if (adc == null)
			return;
		if (adc instanceof CAContainer) {
			CAContainer con = (CAContainer) adc;
			for (IDisplayComponent element : con.getComponents()) {
				AbstractDisplayComponent child = (AbstractDisplayComponent) element;
				if (child != null)
					setPeerForChildren(child, peer);
			}
		} else {
			adc.setPeer(peer);
		}

	}

	public static Object prepareObject(Element e) {
		String className = e.getAttributeValue("class");
		try {
			Class<?> clazz = Class.forName(className);
			Method m = clazz.getMethod("initializeFromXML",
					new Class[] { Element.class });
			Object o = m.invoke(null, new Object[] { e });
			return o;
		} catch (Exception x) {
			x.printStackTrace();
		}
		return null;
	}

	public static String convertEOL2BR(String str) {
		if (str.indexOf('\n') != -1) {
			StringBuffer buf = new StringBuffer(str.length() + 10);
			char[] carr = str.toCharArray();
			for (int i = 0; i < carr.length; i++) {
				if (carr[i] == '\n') {
					buf.append("<br>");
				} else {
					buf.append(carr[i]);
				}
			}
			return buf.toString();
		} else {
			return str;
		}
	}

	public static String convertBR2EOL(String str) {
		if (str.indexOf("<br>") == -1) {
			return str;
		}
		str = str.replaceAll("<br>", String.valueOf('\n'));
		return str;
	}

	/**
	 * Replaces symbols like &,>,<,\n,'," with corresponding character entities
	 *
	 * @param str
	 * @return
	 */
	public static String toXML(String str) {
		StringBuffer buf = new StringBuffer(str.length() + 10);
		char[] carr = str.toCharArray();
		for (int i = 0; i < carr.length; i++) {
			switch (carr[i]) {
			case '&':
				buf.append("&amp;");
				break;
			// case '\n':
			// buf.append("&#013;");
			// break;
			case '>':
				buf.append("&gt;");
				break;
			case '<':
				buf.append("&lt;");
				break;
			case '\'':
				buf.append("&apos;");
				break;
			case '"':
				buf.append("&quot;");
				break;
			default:
				buf.append(carr[i]);
			}
		}// i
		return buf.toString();
	}

	static boolean matches(char[] carr, int offset, String matchStr) {
		if (offset >= carr.length) {
			return false;
		}
		if (offset + matchStr.length() > carr.length) {
			return false;
		}
		int msLength = matchStr.length();
		for (int i = offset; i < offset + msLength; ++i) {
			if (carr[i] != matchStr.charAt(i - offset)) {
				return false;
			}
		}
		return true;
	}

	public static String decodeXML(String encodedStr) {
		StringBuffer buf = new StringBuffer(encodedStr.length());
		char[] carr = encodedStr.toCharArray();
		int i = 0;
		while (i < carr.length) {
			if (carr[i] == '&') {
				if (matches(carr, i + 1, "amp;")) {
					buf.append('&');
					i += 5;
				} else if (matches(carr, i + 1, "apos;")) {
					buf.append('\'');
					i += 6;
				} else if (matches(carr, i + 1, "quot;")) {
					buf.append('"');
					i += 6;
				} else if (matches(carr, i + 1, "gt;")) {
					buf.append('>');
					i += 4;
				} else if (matches(carr, i + 1, "lt;")) {
					buf.append('<');
					i += 4;
				} else {
					buf.append(carr[i++]);
				}
			} else {
				buf.append(carr[i]);
				++i;
			}
		}

		return buf.toString();
	}

	public static class ListInfo {
		List<Object> list;

		List<?> entryElems;

		public List<Object> getList() {
			return list;
		}

		public List<?> getEntryElems() {
			return entryElems;
		}
	}

	public static void verifyByXMLSchema(String xmlDocFile, String schemaFile)
			throws Exception {
		SAXBuilder builder = null;

		File f = new File(schemaFile);

		builder = new SAXBuilder("org.apache.xerces.parsers.SAXParser", true); // validation
		builder.setFeature("http://apache.org/xml/features/validation/schema",
				true);

		builder.setProperty(
						"http://apache.org/xml/properties/schema/external-noNamespaceSchemaLocation",
						f.toURI().toURL().toString());
		System.out.println("Using XML Schema " + schemaFile
				+ " for XML document validation...");
		builder.build(new File(xmlDocFile));
	}

	public static void saveXML(Element rootElem, String xmlFile)
			throws IOException {
		XMLOutputter xout = new XMLOutputter(Format.getPrettyFormat());
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(xmlFile));
			xout.output(rootElem, out);
		} finally {
			GenUtils.close(out);
		}
	}
}
