package cbfbirn

import groovy.transform.ToString;
import groovy.xml.MarkupBuilder;

import java.io.File;
import java.text.SimpleDateFormat

class CaffeineProjUploadXmlBuilder {
	def local2SubjectIDMap = [:]
	
	@ToString
	static class SubjectInfo {
		def localId
		def subjectId
		def birthDate
		def gender
		def day1
		def day2
		def day1Caffeine
		def day2Caffeine
		def viList = []

		def isDay1(VisitInfo vi) {
			def df = new SimpleDateFormat("MM/dd/yyyy")
			def firstDay = df.parse(day1) < df.parse(day2) ? day1 : day2
			return firstDay == vi.day
		}
		
		def sortVisits() {
			if (viList.empty) return
			def df = new SimpleDateFormat("MM/dd/yyyy")
			viList.sort { a, b ->
				def val = df.parse(a.day).compareTo(df.parse(b.day))
				if (val == 0) {
					if (a.session == b.session) {
						return a.eyeCondition == 'Open' ? -1 : 1;
					} else {
					   return a.session == 'Pre' ? -1 : 1;
					}
				}
				return val
			}
		}
	}

	@ToString
	static class VisitInfo {
		def location
		def drug
		def session
		def day
		def eyeCondition
	}

	def loadCSV(File subjectFile) {
		def lines = []
		subjectFile.text.eachLine { line, lineNo ->
			if (lineNo > 0 && lineNo < 11) lines << line
		}
		def siList = []
		lines.each {
			def toks = it.split(/,/)
			def age = toks[1].toInteger()
			def birthYear = 2010 - age;
			def birthDate = "01/01/" + birthYear;
			SubjectInfo si = new SubjectInfo(localId:toks[0], gender: toks[2].toUpperCase(),
			day1: toDate(toks[3]), day2: toDate(toks[4]), day1Caffeine: toks[5] == "1",
			day2Caffeine: toks[6] == "1", birthDate: birthDate)
			siList << si
		}
		siList.each { println it }
		return siList
	}
	
	def loadLocalId2SubjectIDMap(File cvsLookupFile) {
		cvsLookupFile.text.eachLine { line ->
			def toks = line.split(/,/)
			assert toks.size() == 2
			local2SubjectIDMap[toks[0]] = toks[1]
		}
	}

	def static boolean hasFiles(File dir) {
		def count = 0;
		dir.eachFile { if (it.file) count++ }
		return count > 0
	}

	def associateWithImageData(siList, File imageRoot) {
		def expName = 'Tom_EEG_Caffeine'
		def visitDirs = []
		def map = [:]
		siList.each { map[it.localId] = it }
		imageRoot.eachFileRecurse {
			if (it.directory && hasFiles(it) &&
			(it.name == 'ec' || it.name == 'eo') )
			visitDirs << it
		}
		visitDirs.each { vd ->
			println vd
			def eyeCond = vd.name
			def session = vd.parentFile.name
			def visitDate = vd.parentFile.parentFile.name
			def localId = vd.parentFile.parentFile.parentFile.name
			def formattedVisitDate = toDate(visitDate)
			SubjectInfo si = map[localId]
			def drug = "Placebo";

			if (session.startsWith('post') && si.day1 == formattedVisitDate && si.day1Caffeine) {
				drug = "Caffeine"
			} else if (session.startsWith('post') && si.day2 == formattedVisitDate && si.day2Caffeine) {
				drug = "Caffeine"
			}
			VisitInfo vi = new VisitInfo(location: vd.canonicalPath, eyeCondition: eyeCond == "ec" ? "Closed" : "Open",
			session: session.startsWith("pre") ? "Pre" : "Post", day: formattedVisitDate, drug : drug)

			si.viList << vi
		}
	}

	def renameAnatBriks(siList) {
		for(si in siList) {
			if (!si.viList.empty) {
				si.sortVisits()
				for(vi in si.viList) {
				     File location = new File(vi.location)
					 def anatFiles = []
					 location.eachFile {  
						 if (it.name =~ /anat\+nuc\+orig/) anatFiles << it }
					 println vi.toString() + " anatFiles:" + anatFiles
					 anatFiles.each { file ->  
						 def suffix = (file.name =~ /(\.\w+)$/)[0][1]
						 File newFile = new File(file.parent, "anat+orig" + suffix)
						 println "newFile: $newFile" 
						 file.renameTo(newFile)
					 }
				}
			}
		}
	}
	
	def writeXml(siList, File outFile) {
		assert local2SubjectIDMap.size() > 0
		def writer = new StringWriter()
		def xml = new MarkupBuilder(writer)
		xml.cbfImport() {
			for(si in siList) {
				if (!si.viList.empty) {
					si.sortVisits()
					for(vi in si.viList) {
						def subjectID = local2SubjectIDMap[si.localId]
						ds(subjectID:subjectID,  diagnosis:'control', expName:'Tom_EEG_Caffeine',
						visitDate: vi.day, birthDate: si.birthDate, gender: si.gender,
						scannerInfo:'GE MEDICAL SYSTEMS DISCOVERY MR750 fmri3tw',
						location: vi.location) {
							'as'(name:'Caffeine Assessment')	 {
								score(name:'Session', value:vi.session)
								score(name:'Eye Condition', value:vi.eyeCondition)
								score(name:'Drug', value:vi.drug)
								if (si.isDay1(vi)) {
									score(name:'Day', value:'D1')
								} else {
									score(name:'Day', value:'D2')
								}
							}
						}
					}
				}
			}
		}
		println writer.toString()
		if (outFile != null) {
			writeFile(outFile, writer.toString())
		}
	}
	
	def static writeFile(File outFile, String content) {
		outFile.withWriter { out -> out.write(content); out.append("\n") }
		println "wrote file:$outFile"
	}

	static String toDate(String oddDateFormat) {
		def m = oddDateFormat =~ /(\d\d)(\d\d)(\d\d)/
		return m[0][2] + "/" + m[0][3] + "/20" + m[0][1]
	}

	static void main(args) {
		CaffeineProjUploadXmlBuilder builder = new CaffeineProjUploadXmlBuilder()

		def siList = builder.loadCSV(new File('/home/bozyurt/dev/java/clinical/scripts/caffeine_project/caffeine_data.csv'));

		builder.associateWithImageData(siList, new File('/data/share/tom_caffeine_processed/out'))
		// builder.associateWithImageData(siList, new File('/data/share/tomcaffeine'))
		
		builder.loadLocalId2SubjectIDMap(new File('/home/bozyurt/dev/java/clinical/scripts/caffeine_project/caffeine_subject_map.txt'))
		
		
		builder.writeXml(siList, new File("/home/bozyurt/dev/java/clinical/scripts/caffeine_project/caffeine_data_import.xml"))
		
		// builder.renameAnatBriks(siList)
	}
}
