package clinical.test.framework;

import java.sql.SQLException;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;

import org.dbunit.DatabaseUnitException;
import org.dbunit.database.DatabaseConfig;
import org.dbunit.database.IDatabaseConnection;
import org.dbunit.dataset.Column;
import org.dbunit.dataset.DataSetException;
import org.dbunit.dataset.DataSetUtils;
import org.dbunit.dataset.DefaultTableMetaData;
import org.dbunit.dataset.IDataSet;
import org.dbunit.dataset.ITable;
import org.dbunit.dataset.ITableMetaData;
import org.dbunit.dataset.NoSuchColumnException;
import org.dbunit.dataset.RowOutOfBoundsException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: DBUnitUtils.java 365 2011-05-05 20:04:18Z bozyurt $
 */
public class DBUnitUtils {
	private static final Logger logger = LoggerFactory
			.getLogger(DBUnitUtils.class);

	public static String getQualifiedName(String prefix, String name,
			IDatabaseConnection connection) {
		logger.debug("getQualifiedName(prefix=" + prefix + ", name=" + name
				+ ", connection=" + connection + ") - start");

		String escapePattern = (String) connection.getConfig().getProperty(
				DatabaseConfig.PROPERTY_ESCAPE_PATTERN);

		return DataSetUtils.getQualifiedName(prefix, name, escapePattern);
	}

	public static boolean isEmpty(ITable table) throws DataSetException {
		if (logger.isDebugEnabled())
			logger.debug("isEmpty(table=" + table + ") - start");

		Column[] columns = table.getTableMetaData().getColumns();

		// No columns = empty
		if (columns.length == 0) {
			return true;
		}
		// Try to fetch first table value
		try {
			table.getValue(0, columns[0].getColumnName());
			return false;
		} catch (RowOutOfBoundsException e) {
			// Not able to access first row thus empty
			return true;
		}
	}

	public static boolean equalsIgnoreMapping(BitSet ignoreMapping,
			ITable table, int row) throws DataSetException {

		logger.debug("equalsIgnoreMapping(ignoreMapping=" + ignoreMapping
				+ ", table=" + table + ", row=" + row + ") - start");

		Column[] columns = table.getTableMetaData().getColumns();

		for (int i = 0; i < columns.length; i++) {
			boolean bit = ignoreMapping.get(i);
			Object value = table.getValue(row, columns[i].getColumnName());
			if ((bit && value != ITable.NO_VALUE)
					|| (!bit && value == ITable.NO_VALUE)) {
				return false;
			}
		}
		return true;
	}

	public static BitSet getIgnoreMapping(ITable table, int row)
			throws DataSetException {
		logger.debug("getIgnoreMapping(table=" + table + ", row=" + row
				+ ") - start");

		Column[] columns = table.getTableMetaData().getColumns();

		BitSet ignoreMapping = new BitSet();
		for (int i = 0; i < columns.length; i++) {
			Object value = table.getValue(row, columns[i].getColumnName());
			if (value == ITable.NO_VALUE) {
				ignoreMapping.set(i);
			}
		}
		return ignoreMapping;
	}

	public static ITableMetaData getOperationMetaData(
			IDatabaseConnection connection, ITableMetaData metaData)
			throws DatabaseUnitException, SQLException {
		logger.debug("getOperationMetaData(connection=" + connection
				+ ", metaData=" + metaData + ") - start");

		IDataSet databaseDataSet = connection.createDataSet();
		String tableName = metaData.getTableName();

		ITableMetaData databaseMetaData = databaseDataSet
				.getTableMetaData(tableName);
		Column[] databaseColumns = databaseMetaData.getColumns();
		Column[] columns = metaData.getColumns();

		List<Column> columnList = new ArrayList<Column>();
		for (int j = 0; j < columns.length; j++) {
			String columnName = columns[j].getColumnName();
			Column column = DataSetUtils.getColumn(columnName, databaseColumns);
			if (column == null) {
				throw new NoSuchColumnException(tableName + "." + columnName);
			}
			columnList.add(column);
		}

		return new DefaultTableMetaData(databaseMetaData.getTableName(),
				(Column[]) columnList.toArray(new Column[0]), databaseMetaData
						.getPrimaryKeys());
	}
}
