package clinical.tools.dbadmin;

import java.io.File;
import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jdom.Element;

import clinical.server.dao.AssessmentDAO;
import clinical.server.dao.AssessmentitemDAO;
import clinical.server.dao.AssessmentscoreDAO;
import clinical.server.dao.AssessmentscorecodeDAO;
import clinical.server.dao.ExperimentDAO;
import clinical.server.dao.StoredassessmentDAO;
import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentitem;
import clinical.server.vo.Assessmentscore;
import clinical.server.vo.Assessmentscorecode;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Experiment;
import clinical.server.vo.Storedassessment;
import clinical.server.vo.Subjexperiment;
import clinical.utils.Assertion;
import clinical.utils.DateTimeUtils;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.IDBUserManService;
import clinical.web.ISequenceHelper;
import clinical.web.ISubjectAssessmentManagement;
import clinical.web.ISubjectVisitManagement;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.common.query.SearchCriteria;
import clinical.web.common.query.SearchPredicate;
import clinical.web.common.query.SearchPredicateList;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.services.SecurityService;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.AssessmentInfo.ScoreInfo;
import clinical.web.vo.AssessmentScoreValues;
import clinical.web.vo.Visit;
import clinical.web.vo.upload.AssessmentType;
import clinical.web.vo.upload.ScoreType;
import clinical.web.vo.upload.ScoreValue;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class AssessmentManAdmin extends AbstractAdmin {

	public AssessmentManAdmin(String usersXmlFile) throws Exception {
		csm = new ConnectionSupportMixin(usersXmlFile);
		csm.startup();

		SecurityService ss = SecurityService.getInstance(
				csm.getDbPoolService(), csm.getDbID(), csm.getDbType());
		ss.startup();
	}

	public List<Assessment> findAssessmentsWithNoData() throws Exception {
		StoredassessmentDAO saDAO = DAOFactory.createStoredassessmentDAO(csm
				.getDbID());
		AssessmentDAO asDAO = DAOFactory.createAssessmentDAO(csm.getDbID());

		List<Assessment> emptyAsList = new ArrayList<Assessment>(10);
		Connection con = null;
		try {
			con = csm.getConnection();
			List<Assessment> asList = asDAO.find(con, new Assessment());
			for (Assessment as : asList) {
				Storedassessment cr = new Storedassessment();
				cr.setAssessmentid(as.getUniqueid());
				List<Storedassessment> list = saDAO.find(con, cr);
				if (list.isEmpty()) {
					emptyAsList.add(as);
				}
			}
		} finally {
			csm.releaseConnection(con);
		}

		return emptyAsList;
	}

	public void deleteAssessmentMetaData(String asName) throws Exception {
		AssessmentDAO asDAO = DAOFactory.createAssessmentDAO(csm.getDbID());
		AssessmentscoreDAO scoreDAO = DAOFactory.createAssessmentscoreDAO(csm
				.getDbID());
		AssessmentscorecodeDAO scDAO = DAOFactory
				.createAssessmentscorecodeDAO(csm.getDbID());
		AssessmentitemDAO aiDAO = DAOFactory.createAssessmentitemDAO(csm
				.getDbID());

		Connection con = null;
		try {
			con = csm.getConnection();
			con.setAutoCommit(false);
			Assessment asCR = new Assessment();
			asCR.setName(asName);
			List<Assessment> asList = asDAO.find(con, asCR);
			if (asList.isEmpty()) {
				System.out.println("No matching assessment with name:" + asName
						+ "!");
				return;
			}
			Assertion.assertTrue(asList.size() == 1);
			System.out.println("deleting assessment with name:" + asName);
			Assessment as = asList.get(0);
			Assessmentitem aiCR = new Assessmentitem();
			aiCR.setAssessmentid(as.getUniqueid());
			aiDAO.delete(con, aiCR);
			Assessmentscorecode scCR = new Assessmentscorecode();
			scCR.setAssessmentid(as.getUniqueid());
			scDAO.delete(con, scCR);
			Assessmentscore scoreCR = new Assessmentscore();
			scoreCR.setAssessmentid(as.getUniqueid());
			scoreDAO.delete(con, scoreCR);
			asCR.setName(null);
			asCR.setUniqueid(as.getUniqueid());
			asDAO.delete(con, asCR);
			con.commit();
		} catch (Throwable t) {
			con.rollback();
			t.printStackTrace();
		} finally {
			csm.releaseConnection(con);
		}
	}

	public void deleteAssessmentData(String asName, String expName,
			String subjectID) throws Exception {
		Connection con = null;
		try {
			con = csm.getConnection();
			con.setAutoCommit(false);
			Storedassessment saCR = new Storedassessment();
			StoredassessmentDAO saDAO = DAOFactory
					.createStoredassessmentDAO(csm.getDbID());
			Assessment theAs = null;
			if (asName != null) {
				IDBCache dbCache = ServiceFactory.getDBCache(csm.getDbID());
				List<Assessment> asList = dbCache.getAssessments(csm.getUi(),
						false);
				for (Assessment as : asList) {
					if (as.getName().equals(asName)) {
						theAs = as;
						break;
					}
				}
				Assertion.assertNotNull(theAs);
			}

			TSQLProcessor tsp = new TSQLProcessor(csm.getSqlDialect());

			StringBuilder qb = new StringBuilder(200);
			qb.append("select a.uniqueid from Storedassessment as a, Experiment as b ");
			qb.append("where a.ncExperimentUniqueid = b.uniqueid and ");
			qb.append("b.name = '").append(expName).append("' ");
			if (asName != null) {
				qb.append("and a.assessmentid = ").append(theAs.getUniqueid())
						.append(' ');
			}
			if (subjectID != null) {
				qb.append("and a.subjectid = '").append(subjectID).append("' ");
			}

			List<?> results = tsp.executeQuery(con, qb.toString());
			List<BigDecimal> saIdList = new ArrayList<BigDecimal>(
					results.size());
			for (Iterator<?> iter = results.iterator(); iter.hasNext();) {
				Storedassessment sa = (Storedassessment) iter.next();
				saIdList.add(sa.getUniqueid());
			}

			for (BigDecimal saId : saIdList) {
				AsManHelper.delAssessmentValues(con, saId);
				saCR.setUniqueid(saId);
				saDAO.delete(con, saCR);
			}

			con.commit();
		} catch (Throwable t) {
			con.rollback();
			t.printStackTrace();
		} finally {
			csm.releaseConnection(con);
		}

	}

	public void addAssessment(AssessmentWrapper aw) throws Exception {
		if (aw.as == null || aw.scores == null) {
			return;
		}
		Connection con = null;
		try {
			con = csm.getConnection();
			con.setAutoCommit(false);
			Assessment assessment = addAssessment(con, aw.as);
			if (assessment != null) {
				for (Assessmentscore score : aw.scores) {
					addScore(con, score, assessment);
				}
			}
			con.commit();
		} catch (Throwable t) {
			con.rollback();
			t.printStackTrace();
		} finally {
			csm.releaseConnection(con);
		}
	}

	private Assessment addAssessment(Connection con, Assessment as)
			throws Exception {
		AssessmentDAO asDAO = DAOFactory.createAssessmentDAO(csm.getDbID());
		Assessment cr = new Assessment();
		cr.setName(as.getName());
		List<Assessment> asList = asDAO.find(con, cr);
		if (!asList.isEmpty()) {
			System.out.println("Assessment '" + as.getName()
					+ "' already exists in the db:" + asList.get(0)
					+ "\nSkipping...");
			return null;
		}
		Databaseuser adminDU = getDatabaseUser(csm.getUi(),
				Constants.USERCLASS_ADMIN);
		as.setOwner(adminDU.getUniqueid());
		as.setModuser(adminDU.getUniqueid());
		as.setModtime(new Date());

		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(csm.getDbID());
		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.ASSESSMENT_TABLE, "uniqueid");
		as.setUniqueid(uniqueid);
		as.setAssessmentid(uniqueid);
		int tableID = DBUtils.getTableID(csm.getDbID(), con,
				Constants.ASSESSMENT_TABLE);
		as.setTableid(GenUtils.toBigDecimal(tableID));

		asDAO.insert(con, as);
		return as;
	}

	public void addScore(Connection con, Assessmentscore score, Assessment as)
			throws Exception {
		AssessmentscoreDAO scoreDAO = DAOFactory.createAssessmentscoreDAO(csm
				.getDbID());
		score.setAssessmentid(as.getUniqueid());
		Databaseuser adminDU = getDatabaseUser(csm.getUi(),
				Constants.USERCLASS_ADMIN);
		score.setOwner(adminDU.getUniqueid());
		score.setModuser(adminDU.getUniqueid());
		score.setModtime(new Date());
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(csm.getDbID());
		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.ASSESSMENTSCORE_TABLE, "uniqueid");
		score.setUniqueid(uniqueid);
		int tableID = DBUtils.getTableID(csm.getDbID(), con,
				Constants.ASSESSMENTSCORE_TABLE);
		score.setTableid(GenUtils.toBigDecimal(tableID));

		scoreDAO.insert(con, score);
	}

	public List<AssessmentWrapper> loadFromXml(String xmlFile) throws Exception {
		Element rootEl = FileUtils.loadXML(xmlFile);
		List<?> asEls = rootEl.getChild("assessments")
				.getChildren("assessment");
		List<AssessmentWrapper> wrapperList = new ArrayList<AssessmentWrapper>(
				asEls.size());
		for (Object o : asEls) {
			Element asEl = (Element) o;
			String name = asEl.getAttributeValue("name");
			String description = asEl.getChildTextTrim("description");
			Assessment as = new Assessment();
			as.setName(name);
			as.setDescription(description);
			AssessmentWrapper wrapper = new AssessmentWrapper();
			wrapper.as = as;
			wrapperList.add(wrapper);
			List<?> scoreEls = asEl.getChild("scores").getChildren("score");
			wrapper.scores = new ArrayList<Assessmentscore>(scoreEls.size());
			for (Object o2 : scoreEls) {
				Element scoreEl = (Element) o2;
				wrapper.scores.add(fromXml(scoreEl));
			}
		}
		return wrapperList;
	}

	Assessmentscore fromXml(Element scoreEl) {
		Assessmentscore score = new Assessmentscore();
		score.setScorename(scoreEl.getAttributeValue("scorename"));
		score.setScoresequence(toBigDecimal(scoreEl, "scoresequence"));
		score.setScoretype(scoreEl.getAttributeValue("scoretype"));
		score.setScorelevel(toBigDecimal(scoreEl, "scorelevel"));
		score.setParentasid(null);
		score.setDefaultvalue(toStrValue(scoreEl, "defaultvalue"));
		score.setParentscore(null);
		score.setNullable(toBool(scoreEl, "nullable"));
		score.setAssessmentontology(toStrValue(scoreEl, "assessmentontology"));
		score.setAssessmentconcept(toStrValue(scoreEl, "assessmentconcept"));
		score.setSecurityclassification(toStrValue(scoreEl,
				"securityclassification"));
		score.setMinanswers(toBigDecimal(scoreEl, "minanswers"));
		score.setMaxanswers(toBigDecimal(scoreEl, "maxanswers"));
		score.setIsrequired(toBool(scoreEl, "isrequired"));
		score.setIsexcluded(toBool(scoreEl, "isexcluded"));
		score.setIsnew(toBool(scoreEl, "isnew"));
		score.setIsmodified(toBool(scoreEl, "ismodified"));
		return score;
	}

	public void updateAssessmentData(Connection con, AssessmentDataWrapper adw,
			BigDecimal expID, List<String> scoresToBeUpdated) throws Exception {
		UserInfo ui = csm.getUi();
		ISubjectAssessmentManagement isam = ServiceFactory
				.getSubjectAssessmentManagement(csm.getDbID());
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(csm.getDbID());
		SearchPredicateList spList = new SearchPredicateList();
		SearchPredicate sp = new SearchPredicate("subjectid", adw.subjectID,
				SearchPredicate.EQUAL, SearchPredicate.STRING);
		spList.addSearchPredicate(sp, SearchPredicateList.NONE);

		SearchCriteria sc = new SearchCriteria(spList);
		List<Subjexperiment> matchingSubjects = isvm
				.getMatchingExperimentSubjects(ui, sc);
		if (matchingSubjects.isEmpty()) {
			System.out
					.println("No matching subject in the database! subjectID:"
							+ adw.subjectID + "\nSkipping...");
			return;
		}
		boolean foundExpSubject = false;
		for (Subjexperiment se : matchingSubjects) {
			if (se.getNcExperimentUniqueid().equals(expID)) {
				foundExpSubject = true;
				break;
			}
		}
		if (!foundExpSubject) {
			System.out
					.println("No matching subject for the experiment! subjectID:"
							+ adw.subjectID + "\nSkipping...");
			return;
		}

		List<Visit> visitsForSubject = isvm.getAllVisitsForSubject(csm.getUi(),
				adw.subjectID);
		boolean visitFound = false;
		for (Visit v : visitsForSubject) {
			if (v.getExperimentID() == expID.intValue()
					&& v.getTs().equals(adw.timeStamp)
					// && v.getComponentID() == adw.visitID.intValue()
					) {
				visitFound = true;
				adw.visitID = new BigDecimal( v.getComponentID());
			}
		}
		if (!visitFound) {
			System.out.println("No matching visit found for subject "
					+ adw.subjectID + " visitID:" + adw.visitID);
			return;
		}

		// FIXME assumption the first segment is where clinical data resides
		adw.segmentID = new BigDecimal(1);
		String asName = adw.at.getName();
		Assessment assessment = isam.getAssessment(asName, con);
		Assertion.assertNotNull(assessment);
		AssessmentInfo asInfo =  prepAssessmentInfo(adw.at, assessment);
		AssessmentScoreValues asv = new AssessmentScoreValues(asInfo);
		asv.setSubjectID(adw.subjectID);
		asv.setExperimentID(expID.intValue());
		asv.setVisitID(adw.visitID.intValue());
		asv.setSegmentID(adw.segmentID.intValue());
		asv.setTimeStamp(adw.timeStamp);
		asv.setInformantID(adw.subjectID);
		asv.setInformantRelation(Constants.SELF);
		String clinicalRater = ui.getName();
		asv.setClinicalRater(clinicalRater);

		List<AssessmentInfo> aiList = isam.getAssessmentsForSubject(con, ui,
				adw.subjectID, expID.intValue());

		int visitId = adw.visitID.intValue();
		int segmentId = adw.segmentID.intValue();
		boolean foundAsEntry = false;
		for (AssessmentInfo ai : aiList) {
			if (ai.getAssessmentID() == asInfo.getAssessmentID()) {
				if (visitId == ai.getVisitID()
						&& segmentId == ai.getSegmentID()) {
					foundAsEntry = true;
					break;
				}
			}
		}

		if (!foundAsEntry) {
			throw new Exception("An entry of the assessment '"
					+ asInfo.getName() + "' for visit with ID " + visitId
					+ " and segment(episode) ID " + segmentId
					+ " does not exists!");
		}
		Map<String, Map<String, String>> dbVarMetaDataMap = new HashMap<String, Map<String, String>>(
				3);
		Map<String, String> metaMap = new HashMap<String, String>(3);
		metaMap.put("format", "MM/dd/yyyy");
		for (ScoreType st : adw.at.getSiList()) {
			if (st.getType().equals(ScoreType.TIMESTAMP)) {
				dbVarMetaDataMap.put(st.getName(), metaMap);
			}
		}
		Set<String> scoreNameSet = new HashSet<String>(scoresToBeUpdated);
		for (ScoreValue sv : adw.at.getSvList()) {
			if (scoreNameSet.contains(sv.getName())) {
				String value = sv.getValue();
				clinical.web.vo.ScoreValue scoreValue = new clinical.web.vo.ScoreValue(
						sv.getName(), value, 1);
				asv.addScoreValue(scoreValue);
			}
		}

		isam.updateAssessmentValues(con, ui, csm.getDbID(), asv,
				scoresToBeUpdated, 1, dbVarMetaDataMap);
	}

	public void addAssessmentData(Connection con, AssessmentDataWrapper adw,
			BigDecimal expID) throws Exception {
		UserInfo ui = csm.getUi();
		ISubjectAssessmentManagement isam = ServiceFactory
				.getSubjectAssessmentManagement(csm.getDbID());
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(csm.getDbID());
		SearchPredicateList spList = new SearchPredicateList();
		SearchPredicate sp = new SearchPredicate("subjectid", adw.subjectID,
				SearchPredicate.EQUAL, SearchPredicate.STRING);
		spList.addSearchPredicate(sp, SearchPredicateList.NONE);

		SearchCriteria sc = new SearchCriteria(spList);
		// List<Humansubject> matchingSubjects = isvm.getMatchingSubjects(ui,
		// sc);
		List<Subjexperiment> matchingSubjects = isvm
				.getMatchingExperimentSubjects(ui, sc);
		if (matchingSubjects.isEmpty()) {
			System.out
					.println("No matching subject in the database! subjectID:"
							+ adw.subjectID + "\nSkipping...");
			return;
		}
		boolean foundExpSubject = false;
		for (Subjexperiment se : matchingSubjects) {
			if (se.getNcExperimentUniqueid().equals(expID)) {
				foundExpSubject = true;
				break;
			}
		}

		if (!foundExpSubject) {
			System.out
					.println("No matching subject for the experiment! subjectID:"
							+ adw.subjectID + "\nSkipping...");
			return;
		}

		List<Visit> visitsForSubject = isvm.getAllVisitsForSubject(csm.getUi(),
				adw.subjectID);
		boolean visitFound = false;
		for (Visit v : visitsForSubject) {
			if (v.getExperimentID() == expID.intValue()
					&& v.getComponentID() == adw.visitID.intValue()) {
				visitFound = true;
			}
		}
		if (!visitFound) {
			System.out.println("No matching visit found for subject "
					+ adw.subjectID + " visitID:" + adw.visitID);
			return;
		}

		String asName = adw.at.getName();
		Assessment assessment = isam.getAssessment(asName, con);
		Assertion.assertNotNull(assessment);
		AssessmentInfo asInfo = prepAssessmentInfo(adw.at, assessment);
		AssessmentScoreValues asv = new AssessmentScoreValues(asInfo);
		asv.setSubjectID(adw.subjectID);
		asv.setExperimentID(expID.intValue());
		asv.setVisitID(adw.visitID.intValue());
		asv.setSegmentID(adw.segmentID.intValue());
		asv.setTimeStamp(adw.timeStamp);
		asv.setInformantID(adw.subjectID);
		asv.setInformantRelation(Constants.SELF);
		String clinicalRater = ui.getName();
		asv.setClinicalRater(clinicalRater);

		List<AssessmentInfo> aiList = isam.getAssessmentsForSubject(con, ui,
				adw.subjectID, expID.intValue());

		int visitId = adw.visitID.intValue();
		int segmentId = adw.segmentID.intValue();
		boolean foundAsEntry = false;
		for (AssessmentInfo ai : aiList) {
			if (ai.getAssessmentID() == asInfo.getAssessmentID()) {
				if (visitId == ai.getVisitID()
						&& segmentId == ai.getSegmentID()) {
					foundAsEntry = true;
					break;
				}
			}
		}

		if (foundAsEntry) {
			throw new Exception("An entry of the assessment '"
					+ asInfo.getName() + "' for visit with ID " + visitId
					+ " and segment(episode) ID " + segmentId
					+ " already exists!");
		}
		IDBUserManService dbusm = ServiceFactory.getDBUserManService(csm
				.getDbID());
		dbusm.ensureClinicalRater(con, ui, clinicalRater);
		Map<String, Map<String, String>> dbVarMetaDataMap = new HashMap<String, Map<String, String>>(
				3);
		Map<String, String> metaMap = new HashMap<String, String>(3);
		metaMap.put("format", "MM/dd/yyyy");
		for (ScoreType st : adw.at.getSiList()) {
			if (st.getType().equals(ScoreType.TIMESTAMP)) {
				dbVarMetaDataMap.put(st.getName(), metaMap);
			}
		}
		for (ScoreValue sv : adw.at.getSvList()) {
			String value = sv.getValue();
			if (sv.getName().endsWith("Annotation")) {
				System.out.println(">>>>>> Skipping annotation score: "
						+ sv.getName());
				continue;
			}

			clinical.web.vo.ScoreValue scoreValue = new clinical.web.vo.ScoreValue(
					sv.getName(), value, 1);
			asv.addScoreValue(scoreValue);
		}
		BigDecimal entryID = new BigDecimal(1);
		isam.insertAssessmentValues(con, ui, csm.getDbID(), asv, entryID, true,
				dbVarMetaDataMap);
	}

	AssessmentInfo prepAssessmentInfo(AssessmentType asType,
			Assessment assessment) {
		AssessmentInfo asInfo = new AssessmentInfo(assessment.getUniqueid()
				.intValue(), assessment.getName(), null, -1, -1);
		int i = 1;
		for (ScoreType st : asType.getSiList()) {
			ScoreInfo si = new ScoreInfo(st.getName(),
					getScoreType(st.getType()), 1, -1, null, i);
			asInfo.addScore(si);
			i++;
		}
		return asInfo;
	}

	public static String getScoreType(String stType) {
		if (stType.equals(ScoreType.INT)) {
			return "integer";
		} else if (stType.equals(ScoreType.FLOAT)) {
			return stType;
		} else if (stType.equals(ScoreType.STRING)) {
			return "varchar";
		} else if (stType.equals(ScoreType.TIMESTAMP)) {
			return "timestamp";
		}
		throw new RuntimeException("Unrecognized score type:" + stType);
	}

	public static String getDBScoreType(String scoreType) {
		if (scoreType.equals("integer")) {
			return ScoreType.INT;
		} else if (scoreType.equals("float")) {
			return ScoreType.FLOAT;
		} else if (scoreType.equals("varchar")) {
			return ScoreType.STRING;
		} else if (scoreType.equals("timestamp")) {
			return ScoreType.TIMESTAMP;
		}
		throw new RuntimeException("Unrecognized DB score type:" + scoreType);
	}

	public List<AssessmentDataWrapper> loadAsDataFromUploadXml(
			String uploadXmlFile) throws Exception {
		Element rootEl = FileUtils.loadXML(uploadXmlFile);

		List<?> dsList = rootEl.getChildren("ds");
		List<AssessmentDataWrapper> adwList = new ArrayList<AssessmentDataWrapper>(
				dsList.size());
		for (Object o : dsList) {
			Element dsEl = (Element) o;
			AssessmentDataWrapper adw = new AssessmentDataWrapper();
			adwList.add(adw);
			adw.subjectID = dsEl.getAttributeValue("subjectID");
			adw.informantRelation = "self";
			String ts = dsEl.getAttributeValue("visitDate");
			adw.timeStamp = DateTimeUtils.toTimeStamp(ts);
			adw.at = prepSubjectInfoAssesmentData(dsEl);

			List<?> asList = dsEl.getChildren("as");
			for (Object o2 : asList) {
				Element asEl = (Element) o2;
				adw = new AssessmentDataWrapper();
				adwList.add(adw);
				adw.subjectID = dsEl.getAttributeValue("subjectID");
				adw.informantRelation = "self";
				adw.timeStamp = DateTimeUtils.toTimeStamp(ts);
				adw.at = prepUploadXmlAssessmentData(asEl);
			}
		}
		return adwList;
	}

	AssessmentType prepUploadXmlAssessmentData(Element asEl) {
		String asName = asEl.getAttributeValue("name");
		AssessmentType at = new AssessmentType(asName, "");
		List<?> avEls = asEl.getChildren("score");
		for (Object o : avEls) {
			Element avEl = (Element) o;
			String scoreName = avEl.getAttributeValue("name");
			String value = avEl.getAttributeValue("value");
			// FIXME all scores are assumed to be of type varchar
			at.addScore(new ScoreType(scoreName, ScoreType.STRING, null, null));
			at.addValue(new ScoreValue(scoreName, value));
		}
		return at;
	}

	AssessmentType prepSubjectInfoAssesmentData(Element dsEl) {
		AssessmentType at = new AssessmentType("Subject Information", "");
		addScoreAndValue(dsEl, "diagnosis", at);
		addScoreAndValue(dsEl, "gender", at);
		addScoreAndValue(dsEl, "age", at, ScoreType.INT);
		return at;
	}

	void addScoreAndValue(Element dsEl, String scoreName, AssessmentType at) {
		addScoreAndValue(dsEl, scoreName, at, ScoreType.STRING);
	}

	void addScoreAndValue(Element dsEl, String scoreName, AssessmentType at,
			String scoreType) {
		String value = dsEl.getAttributeValue(scoreName);
		at.addScore(new ScoreType(scoreName, scoreType, null, null));
		at.addValue(new ScoreValue(scoreName, value));
	}

	public List<AssessmentDataWrapper> loadAsDataFromXml(String asDataXmlFile)
			throws Exception {
		Element rootEl = FileUtils.loadXML(asDataXmlFile);
		List<?> saList = rootEl.getChild("values").getChildren("storedas");
		List<AssessmentDataWrapper> adwList = new ArrayList<AssessmentDataWrapper>();
		for (Object o : saList) {
			Element saEl = (Element) o;
			AssessmentDataWrapper adw = new AssessmentDataWrapper();
			adwList.add(adw);
			adw.subjectID = saEl.getAttributeValue("subjectid");
			String asName = saEl.getAttributeValue("asName");
			adw.informantRelation = saEl.getAttributeValue("informantrelation");
			adw.visitID = toBigDecimal(saEl, "componentid");
			adw.segmentID = toBigDecimal(saEl, "segmentid");
			String ts = saEl.getAttributeValue("timeStamp");
			adw.timeStamp = DateTimeUtils.convertTimestampPostgres(ts);
			adw.at = prepAssessmentData(saEl, asName);
		}
		return adwList;
	}

	AssessmentType prepAssessmentData(Element saEl, String asName) {
		AssessmentType at = new AssessmentType(asName, "");
		List<?> avEls = saEl.getChildren("asval");
		for (Object o : avEls) {
			Element avEl = (Element) o;
			String scoreName = avEl.getAttributeValue("scorename");
			String scoreType = getDBScoreType(avEl
					.getAttributeValue("scoretype"));
			String value = avEl.getAttributeValue("textvalue");
			// int scoreOrder = toInt(avEl, "scoreorder");
			at.addScore(new ScoreType(scoreName, scoreType, null, null));
			at.addValue(new ScoreValue(scoreName, value));
		}
		return at;
	}

	public static int toInt(Element el, String attrName) {
		int value = GenUtils.toInt(el.getAttributeValue(attrName), -1);
		Assertion.assertTrue(value != -1);
		return value;
	}

	public static BigDecimal toBigDecimal(Element el, String attrName) {
		int value = GenUtils.toInt(el.getAttributeValue(attrName), -1);
		Assertion.assertTrue(value != -1);
		return GenUtils.toBigDecimal(value);
	}

	public static Boolean toBool(Element el, String attrName) {
		String value = el.getAttributeValue(attrName);
		return value.equalsIgnoreCase("true") ? Boolean.TRUE : Boolean.FALSE;
	}

	public static String toStrValue(Element el, String attrName) {
		String value = el.getAttributeValue(attrName);
		if (value == null || value.trim().length() == 0) {
			return null;
		}
		return value;
	}

	public static class AssessmentWrapper {
		Assessment as;
		List<Assessmentscore> scores;

		@Override
		public String toString() {
			StringBuilder sb = new StringBuilder(1024);
			sb.append("AsWrapper::[");
			sb.append(as.toString()).append("\n");
			sb.append("Scores:\n");
			for (Assessmentscore score : scores) {
				sb.append("\t").append(score).append("\n");
			}
			sb.append(']');
			return sb.toString();
		}
	}

	public static AssessmentWrapper findByAssessmentName(
			List<AssessmentWrapper> aswList, String asName) {
		for (AssessmentWrapper aw : aswList) {
			if (aw.as.getName().equalsIgnoreCase(asName)) {
				return aw;
			}
		}
		return null;
	}

	public static class AssessmentDataWrapper {
		String subjectID;
		BigDecimal visitID;
		BigDecimal segmentID;
		String informantRelation;
		Date timeStamp;
		AssessmentType at;

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result
					+ ((segmentID == null) ? 0 : segmentID.hashCode());
			result = prime * result
					+ ((subjectID == null) ? 0 : subjectID.hashCode());
			result = prime * result
					+ ((timeStamp == null) ? 0 : timeStamp.hashCode());
			result = prime * result
					+ ((visitID == null) ? 0 : visitID.hashCode());
			result = prime * result
					+ ((at.getName() == null) ? 0 : at.getName().hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			AssessmentDataWrapper other = (AssessmentDataWrapper) obj;
			if (segmentID == null) {
				if (other.segmentID != null)
					return false;
			} else if (!segmentID.equals(other.segmentID))
				return false;
			if (subjectID == null) {
				if (other.subjectID != null)
					return false;
			} else if (!subjectID.equals(other.subjectID))
				return false;
			if (timeStamp == null) {
				if (other.timeStamp != null)
					return false;
			} else if (!timeStamp.equals(other.timeStamp))
				return false;
			if (visitID == null) {
				if (other.visitID != null)
					return false;
			} else if (!visitID.equals(other.visitID))
				return false;
			if (at.getName() == null) {
				if (other.at.getName() != null)
					return false;
			} else if (!at.getName().equals(other.at.getName()))
				return false;
			return true;
		}

		@Override
		public String toString() {
			return "AssessmentDataWrapper [subjectID=" + subjectID
					+ ", visitID=" + visitID + ", segmentID=" + segmentID
					+ ", informantRelation=" + informantRelation
					+ ", timeStamp=" + timeStamp + ", at=" + at + "]";
		}

	}// ;

	public static class Phase3ProjectInfo {
		static String[] sites = { "uci", "ucsf", "umn", "mrn", "uiowa", "duke",
				"ucsd" };
		static String rootDir = "/home/bozyurt/dev/java/clinical/scripts/results";
		static String[] metaDataXmls = { "as_meta_fBIRNPhaseIII__0090_uci.xml",
				"as_meta_fBIRNPhaseIII__0090_ucsf.xml",
				"as_meta_fBIRNPhaseIII__0090_umn.xml",
				"as_meta_fBIRNPhaseIII__0090_mrn.xml",
				"as_meta_fBIRNPhaseIII__0090_uiowa.xml",
				"as_meta_fBIRNPhaseIII__0090_duke.xml",
				"as_meta_fBIRNPhaseIII__0090_ucsd.xml" };
		static String[] asDataXmls = { "as_data_fBIRNPhaseIII__0090_uci.xml",
				"as_data_fBIRNPhaseIII__0090_ucsf.xml",
				"as_data_fBIRNPhaseIII__0090_umn.xml",
				"as_data_fBIRNPhaseIII__0090_mrn.xml",
				"as_data_fBIRNPhaseIII__0090_uiowa.xml",
				"as_data_fBIRNPhaseIII__0090_duke.xml",
				"as_data_fBIRNPhaseIII__0090_ucsd.xml" };
		static Map<String, String> asMetaMap = new HashMap<String, String>();
		static Map<String, String> asDataMap = new HashMap<String, String>();

		static {
			for (int i = 0; i < sites.length; i++) {
				String site = sites[i];
				Assertion.assertTrue(metaDataXmls[i].indexOf(site) != -1);
				Assertion.assertTrue(asDataXmls[i].indexOf(site) != -1);
				asMetaMap.put(site,
						new File(rootDir, metaDataXmls[i]).getAbsolutePath());
				asDataMap.put(site,
						new File(rootDir, asDataXmls[i]).getAbsolutePath());
			}
		}

		public static boolean isValidSite(String site) {
			for (String s : sites) {
				if (s.equals(site)) {
					return true;
				}
			}
			return false;
		}

		public static String getExpName(String site) {
			return "Phase3_" + site;
		}
	}// ;

	public static void test3() throws Exception {
		AssessmentManAdmin admin = null;
		try {
			admin = new AssessmentManAdmin("users.xml");
			String asDataXmlFile = "/home/bozyurt/dev/java/clinical/scripts/results/as_data_fBIRNPhaseIII__0090_uci.xml";
			admin.loadAsDataFromXml(asDataXmlFile);
		} finally {
			if (admin != null) {
				admin.shutdown();
			}
		}
	}

	public static void test1() throws Exception {
		AssessmentManAdmin admin = null;
		try {
			admin = new AssessmentManAdmin("users.xml");

			List<AssessmentWrapper> wrapperList = admin
					.loadFromXml("/home/bozyurt/dev/java/clinical/scripts/"
							+ "results/as_meta_fBIRNPhaseIII__0090_uci.xml");
			int numAssessments = wrapperList.size();
			int numScores = 0;
			for (AssessmentWrapper aw : wrapperList) {
				System.out.println(aw);
				numScores += aw.scores.size();
			}

			System.out.println("numAssessments:" + numAssessments
					+ " numScores:" + numScores);
			AssessmentWrapper aw = findByAssessmentName(wrapperList,
					"Demographics");
			if (aw != null) {
				admin.addAssessment(aw);
			}

		} finally {
			if (admin != null) {
				admin.shutdown();
			}
		}
	}

	void addAssessmentData(List<AssessmentDataWrapper> adwList, String site)
			throws Exception {
		Connection con = null;
		try {
			con = csm.getConnection();
			con.setAutoCommit(false);
			String expName = Phase3ProjectInfo.getExpName(site);
			ExperimentDAO expDAO = DAOFactory
					.createExperimentDAO(csm.getDbID());
			Experiment cr = new Experiment();
			cr.setName(expName);
			List<Experiment> expList = expDAO.find(con, cr);
			Assertion.assertTrue(expList.size() == 1);
			BigDecimal expID = expList.get(0).getUniqueid();
			List<AssessmentDataWrapper> uniqueADWList = new ArrayList<AssessmentDataWrapper>();
			for (AssessmentDataWrapper adw : adwList) {
				boolean found = false;
				for (AssessmentDataWrapper adw2 : uniqueADWList) {
					if (adw.equals(adw2)) {
						found = true;
						break;
					}
				}
				if (!found) {
					uniqueADWList.add(adw);
				} else {
					System.out.println("Skipping duplicate adw:" + adw);
				}
			}

			for (AssessmentDataWrapper adw : uniqueADWList) {
				addAssessmentData(con, adw, expID);
			}

			con.commit();
		} catch (Throwable t) {
			t.printStackTrace();
			con.rollback();
			throw new Exception(t);
		} finally {
			csm.releaseConnection(con);
		}
	}

	public void updateAssessmentData(List<AssessmentDataWrapper> adwList,
			String expName, String assessmentName, List<String> scores2Update)
			throws Exception {
		Connection con = null;
		try {
			con = csm.getConnection();
			con.setAutoCommit(false);
			ExperimentDAO expDAO = DAOFactory
					.createExperimentDAO(csm.getDbID());
			Experiment cr = new Experiment();
			cr.setName(expName);
			List<Experiment> expList = expDAO.find(con, cr);
			Assertion.assertTrue(expList.size() == 1);
			BigDecimal expID = expList.get(0).getUniqueid();
			List<AssessmentDataWrapper> filteredList = new ArrayList<AssessmentDataWrapper>();
			for (AssessmentDataWrapper adw : adwList) {
				if (adw.at.getName().equals(assessmentName)) {
					filteredList.add(adw);
				}
			}

			for (AssessmentDataWrapper adw : filteredList) {
				updateAssessmentData(con, adw, expID, scores2Update);
			}

			con.commit();
		} catch (Throwable t) {
			t.printStackTrace();
			con.rollback();
			throw new Exception(t);
		} finally {
			csm.releaseConnection(con);
		}
	}

	public static void updateAgeForPhFMRI() throws Exception {
		AssessmentManAdmin admin = null;
		String homeDir = System.getProperty("user.home");
		String uploadXmlFile = homeDir
				+ "/dev/java/clinical/scripts/phFMRI_project/phFMRI_data_import.xml";
		try {
			admin = new AssessmentManAdmin("users.xml");

			List<AssessmentDataWrapper> adwList = admin.loadAsDataFromUploadXml(uploadXmlFile);
			
			for(AssessmentDataWrapper adw : adwList) {
				System.out.println(adw);
			}
			
			List<String> scores2Update = new ArrayList<String>(1);
			scores2Update.add("age");
			admin.updateAssessmentData(adwList,"phFMRI","Subject Information", scores2Update);
		} finally {
			if (admin != null) {
				admin.shutdown();
			}
		}

	}

	public static void createPhase3AsData() throws Exception {
		AssessmentManAdmin admin = null;
		try {
			admin = new AssessmentManAdmin("users.xml");
			for (String site : Phase3ProjectInfo.sites) {
				System.out.println("Processing assessments from site:" + site);
				System.out
						.println("==========================================");
				List<AssessmentDataWrapper> adwList = admin
						.loadAsDataFromXml(Phase3ProjectInfo.asDataMap
								.get(site));

				admin.addAssessmentData(adwList, site);

				// for test
				break;
			}
		} finally {
			if (admin != null) {
				admin.shutdown();
			}
		}
	}

	public static void createPhase3AsData(String site) throws Exception {
		Assertion.assertTrue(Phase3ProjectInfo.isValidSite(site));
		AssessmentManAdmin admin = null;
		try {
			admin = new AssessmentManAdmin("users.xml");
			System.out.println("Processing assessments from site:" + site);
			System.out.println("==========================================");
			List<AssessmentDataWrapper> adwList = admin
					.loadAsDataFromXml(Phase3ProjectInfo.asDataMap.get(site));

			admin.addAssessmentData(adwList, site);

		} finally {
			if (admin != null) {
				admin.shutdown();
			}
		}
	}

	public static void checkPhase3AsData(String site) throws Exception {
		Assertion.assertTrue(Phase3ProjectInfo.isValidSite(site));
		AssessmentManAdmin admin = null;
		try {
			admin = new AssessmentManAdmin("users.xml");
			System.out.println("Processing assessments from site:" + site);
			System.out.println("==========================================");
			List<AssessmentDataWrapper> adwList = admin
					.loadAsDataFromXml(Phase3ProjectInfo.asDataMap.get(site));
			AsManHelper.checkForDuplicateAssessments(adwList);
		} finally {
			if (admin != null) {
				admin.shutdown();
			}
		}
	}

	public static void cleanupAllAsData(String site) throws Exception {
		AssessmentManAdmin admin = null;
		try {
			admin = new AssessmentManAdmin("users.xml");
			String expName = Phase3ProjectInfo.getExpName(site);

			admin.deleteAssessmentData(null, expName, null);
		} finally {
			if (admin != null) {
				admin.shutdown();
			}
		}
	}

	public static void createPhase3AssessmentMetaData() throws Exception {
		AssessmentManAdmin admin = null;
		try {
			admin = new AssessmentManAdmin("users.xml");
			for (String site : Phase3ProjectInfo.sites) {
				System.out.println("Processing assessments from site:" + site);
				System.out
						.println("==========================================");
				List<AssessmentWrapper> wrapperList = admin
						.loadFromXml(Phase3ProjectInfo.asMetaMap.get(site));
				for (AssessmentWrapper aw : wrapperList) {
					admin.addAssessment(aw);
				}

				System.out
						.println("==========================================");
			}

		} finally {
			if (admin != null) {
				admin.shutdown();
			}
		}

	}

	public static void test2() throws Exception {
		AssessmentManAdmin admin = null;
		try {
			admin = new AssessmentManAdmin("users.xml");

			List<Assessment> emptyAsList = admin.findAssessmentsWithNoData();
			System.out.println("Empty assessments");
			System.out.println("-----------------");
			for (Assessment as : emptyAsList) {
				System.out.println(as);
			}
			// admin.deleteAssessmentMetaData("quick mood scale");
			String[] emptyAssessments = new String[] {
					"Barnes Akathisia Rating Scale",
					"Simpson Angus Rating Scale", "Premorbid Adjustment Scale",
					"Calgary Depression Scale", "Socio-Economic Status (SES)",
					"North American Adult Reading Test (NAART)" };
			for (String asName : emptyAssessments) {
				admin.deleteAssessmentMetaData(asName);
			}
		} finally {
			if (admin != null) {
				admin.shutdown();
			}
		}
	}

	public static void main(String[] args) throws Exception {
		// test2();
		// createPhase3AssessmentMetaData();
		// finished uci, ucsf, uiowa,ucsd, mrn, umn
		// "uci", "ucsf", "umn", "mrn", "uiowa", "duke", "ucsd"
		// createPhase3AsData("duke");
		// checkPhase3AsData("duke");

		// cleanupAllAsData("uci");
		
		updateAgeForPhFMRI();
	}
}
