package clinical.tools.dbadmin.migration;

import java.io.File;
import java.sql.Connection;
import java.util.List;

import org.jdom.Element;

import clinical.server.dao.ExperimentDAO;
import clinical.server.vo.ConfParams;
import clinical.server.vo.Experiment;
import clinical.tools.dbadmin.AbstractAdmin;
import clinical.tools.dbadmin.migration.VOMigrationManager.VOXMLPersister;
import clinical.utils.Assertion;
import clinical.utils.FileUtils;
import clinical.web.ConnectionSupportMixin;
import clinical.web.DAOFactory;
import clinical.web.services.SecurityService;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class MigrationAdmin extends AbstractAdmin {

	public MigrationAdmin(String usersXmlFile) throws Exception {
		if ((new File(usersXmlFile)).isFile()) {
			csm = ConnectionSupportMixin.createFromFullPath(usersXmlFile);
		} else {
			csm = new ConnectionSupportMixin(usersXmlFile);
		}
		csm.startup();

		SecurityService ss = SecurityService.getInstance(
				csm.getDbPoolService(), csm.getDbID(), csm.getDbType());
		ss.startup();
	}

	public void testVOPersisting(String xmlFile) throws Exception {
		Element rootEl = new Element("migration");

		VOXMLPersister<Experiment> expPersister = new VOXMLPersister<Experiment>(
				Experiment.class);

		ExperimentDAO expDAO = DAOFactory.createExperimentDAO(csm.getDbID());

		Connection con = null;
		try {
			con = csm.getConnection();

			List<Experiment> expList = expDAO.find(con, new Experiment());
			for (Experiment exp : expList) {
				rootEl.addContent(expPersister.toXml(exp));
			}

			FileUtils.saveXML(rootEl, xmlFile);
			System.out.println("persisted as xml to file:" + xmlFile);

		} finally {
			csm.releaseConnection(con);
		}
	}

	public void extractExpData(String expName, String migrationXmlFile)
			throws Exception {
		SourceExtractor se = new SourceExtractor(migrationXmlFile, csm);

		se.collectExperimentData(expName);
	}

	public void putExpData(String migrationXmlFile, String expName)
			throws Exception {
		DestSaver ds = new DestSaver(migrationXmlFile, csm);

		Connection con = null;
		try {
			con = csm.getConnection();

			List<Experiment> expList = ds.findMatching(con, Experiment.class,
					"name", expName);
			for (Experiment exp : expList) {
				System.out.println(exp);
			}

			List<ConfParams> cpList = ds.findMatching(con, ConfParams.class,
					"name", "cbfbirn.data.root");
			for (ConfParams cp : cpList) {
				System.out.println(cp);
			}

			ds.putExpData();

			ds.fixJobContexts();
			
			ds.putDataObjects();
		} finally {
			csm.releaseConnection(con);
		}
	}
	
	
	public void putDataObjects(String expName, String migrationXmlFile) throws Exception {
		DestSaver ds = new DestSaver(migrationXmlFile, csm);

		Connection con = null;
		try {
			con = csm.getConnection();

			List<Experiment> expList = ds.findMatching(con, Experiment.class,
					"name", expName);
			for (Experiment exp : expList) {
				System.out.println(exp);
			}
			
			List<ConfParams> cpList = ds.findMatching(con, ConfParams.class,
					"name", "cbfbirn.data.root");
			for (ConfParams cp : cpList) {
				System.out.println(cp);
			}
			
			ds.putDataObjects();
			
		} finally {
			csm.releaseConnection(con);
		}
	}
	
	public void fixJobContexts(String migrationXmlFile, String expName)
			throws Exception {
		DestSaver ds = new DestSaver(migrationXmlFile, csm);

		Connection con = null;
		try {
			con = csm.getConnection();

			List<Experiment> expList = ds.findMatching(con, Experiment.class,
					"name", expName);
			for (Experiment exp : expList) {
				System.out.println(exp);
			}

			List<ConfParams> cpList = ds.findMatching(con, ConfParams.class,
					"name", "cbfbirn.data.root");
			for (ConfParams cp : cpList) {
				System.out.println(cp);
			}

			ds.fixJobContexts();
		} finally {
			csm.releaseConnection(con);
		}
	}

	public static void testMigrateToDest() throws Exception {
		MigrationAdmin admin = null;
		try {
			admin = new MigrationAdmin(
					"/home/bozyurt/dev/java/cbfbirn/clinical/conf/users.xml");

			admin.putExpData("/tmp/phFMRI_migration.xml", "phFMRI");
		} finally {
			if (admin != null)
				admin.shutdown();
		}
	}

	public static void importExperiment(String usersXmlFile, String expName,
			String dumpFile) throws Exception {
		MigrationAdmin admin = null;
		try {
			admin = new MigrationAdmin(usersXmlFile);
			admin.putExpData(dumpFile, expName);
		} finally {
			if (admin != null)
				admin.shutdown();
		}
	}
	
	public static void fix(String usersXmlFile, String expName,
			String dumpFile) throws Exception {
		MigrationAdmin admin = null;
		try {
			admin = new MigrationAdmin(usersXmlFile);
			admin.fixJobContexts(dumpFile, expName);
		} finally {
			if (admin != null)
				admin.shutdown();
		}

	}

	public static void testSourceExtraction() throws Exception {
		MigrationAdmin admin = null;
		try {
			admin = new MigrationAdmin("users.xml");
			admin.extractExpData("phFMRI", "/tmp/phFMRI_migration.xml");
		} finally {
			if (admin != null)
				admin.shutdown();
		}
	}

	public static void exportExperiment(String usersXmlFile, String expName,
			String dumpFile) throws Exception {
		MigrationAdmin admin = null;
		try {
			admin = new MigrationAdmin(usersXmlFile);
			admin.extractExpData(expName, dumpFile);
		} finally {
			if (admin != null)
				admin.shutdown();
		}
	}
	
	
	public static void putDataObjects(String usersXmlFile, String expName,
			String dumpFile) throws Exception {
		MigrationAdmin admin = null;
		try {
			admin = new MigrationAdmin(usersXmlFile);
			admin.putDataObjects(expName, dumpFile);
		} finally {
			if (admin != null)
				admin.shutdown();
		}
	}


	public static enum ActionType {
		EXPORT, IMPORT, FIX, ADD_DO;
	}

	public static void usage() {
		System.err
				.println("Usage:MigrationAdmin [-h] -action <export|import|fix|addDO> "
						+ "-e <exp-name> -uxpath <users-xml-file-path> -df <dump-file>\n");
		System.exit(1);
	}

	public static String getArgValue(String[] args, int idx) {
		if (args.length <= idx) {
			usage();
		}
		return args[idx];
	}

	public static void main(String[] args) throws Exception {
		if (args.length < 1) {
			usage();
		}
		if (args.length == 1 || args[0].equals("-h")) {
			usage();
		}
		ActionType action = null;
		String expName = null;
		String usersXmlFile = null;
		String dumpFile = null;
		int i = 0;
		while (i < args.length) {
			String argName = args[i];
			if (argName.equals("-action")) {
				String value = getArgValue(args, i + 1);
				System.out.println("action value:" + value);
				if (value.equalsIgnoreCase("import")) {
					action = ActionType.IMPORT;
				} else if (value.equalsIgnoreCase("export")) {
					action = ActionType.EXPORT;
				} else if (value.equals("fix")) {
					action = ActionType.FIX;
				} else if (value.equalsIgnoreCase("addDO")) {
					action = ActionType.ADD_DO;
				}
				i += 2;
			} else if (argName.equals("-e")) {
				expName = getArgValue(args, i + 1);
				i += 2;
			} else if (argName.equals("-uxpath")) {
				usersXmlFile = getArgValue(args, i + 1);
				Assertion.assertTrue(new File(usersXmlFile).isFile());
				i += 2;
			} else if (argName.equals("-df")) {
				dumpFile = getArgValue(args, i + 1);
				i += 2;
			} else {
				i++;
			}
		}
		if (action == null) {
			usage();
		}
		if (dumpFile == null || expName == null || usersXmlFile == null) {
			usage();
		}

		if (action == ActionType.EXPORT) {
			exportExperiment(usersXmlFile, expName, dumpFile);
		} else if (action == ActionType.IMPORT ){
			// import
			importExperiment(usersXmlFile, expName, dumpFile);
		} else if (action == ActionType.FIX) {
			fix(usersXmlFile, expName, dumpFile);
		} else if (action == ActionType.ADD_DO) {
			putDataObjects(usersXmlFile, expName, dumpFile);
		}

		// testMigrateToDest();
		// testSourceExtraction();
	}

}
