package clinical.utils;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.jdom.Element;

import clinical.web.vo.HeaderFieldType;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AFNIHeaderReader.java 178 2010-02-17 00:28:56Z bozyurt $
 */
public class AFNIHeaderReader {
	public final static SimpleDateFormat tsFormat = new SimpleDateFormat(
			"EEE MMM dd hh:mm:ss yyyy");
	String headerFile;
	Map<String, Field<Integer>> integerFieldMap = new HashMap<String, Field<Integer>>();
	Map<String, Field<String>> stringFieldMap = new HashMap<String, Field<String>>();
	Map<String, Field<Float>> realFieldMap = new HashMap<String, Field<Float>>();

	public AFNIHeaderReader(String headerFile) {
		this.headerFile = headerFile;
	}

	public void readHeader() throws IOException {
		BufferedReader in = null;
		try {
			in = new BufferedReader(new FileReader(headerFile));
			String line = null;

			while ((line = in.readLine()) != null) {
				line = line.trim();
				if (line.length() == 0)
					continue;
				String type = getFieldValue(line, "type");
				line = nextLine(in);
				String name = getFieldValue(line, "name");
				line = nextLine(in);
				String countStr = getFieldValue(line, "count");
				int count = GenUtils.toInt(countStr, -1);
				Assertion.assertTrue(count != -1);
				if (type.startsWith("integer-attribute")) {
					Field<Integer> field = new Field<Integer>(name);
					integerFieldMap.put(name, field);
					line = nextLine(in);
					if (count == 1) {
						field.value = GenUtils.toInt(line.trim(), -1);
					} else {
						field.initArray(count);
						int valueCount = 0;
						do {
							line = line.trim();
							String[] toks = line.split("\\s+");
							for (String tok : toks) {
								field.add(GenUtils.toInt(tok, -1));
								valueCount++;
							}
							line = nextLine(in);
							if (line.trim().length() == 0)
								break;
						} while (true);
						Assertion.assertTrue(count == valueCount);
					}
				} else if (type.startsWith("float-attribute")) {
					Field<Float> field = new Field<Float>(name);
					realFieldMap.put(name, field);
					line = nextLine(in).trim();
					if (count == 1) {
						field.value = new Float(GenUtils.toDouble(line, -1));
					} else {
						field.initArray(count);
						int valueCount = 0;
						do {
							line = line.trim();
							String[] toks = line.split("\\s+");
							for (String tok : toks) {
								field.add(new Float(GenUtils.toDouble(tok, -1)));
								valueCount++;
							}
							line = nextLine(in);
							if (line.trim().length() == 0)
								break;
						} while (true);
						Assertion.assertTrue(count == valueCount);
					}

				} else if (type.startsWith("string-attribute")) {
					Field<String> field = new Field<String>(name);
					stringFieldMap.put(name, field);
					StringBuilder sb = new StringBuilder(count);
					boolean eos = false;
					boolean first = true;
					do {
						line = nextLine(in);
						line = line.trim();
						if (line.length() == 0 || line.endsWith("~")) {
							if (line.endsWith("~") ) {
								if (first) {
									line = line.substring(1);
									first = false;
								}
								sb.append(line);
							}
							eos= true;
						} else {
							if (first) {
								line = line.substring(1);
								first = false;
							}
							sb.append(line);
						}
					} while(!eos);
					String str = sb.toString();
					str = str.substring(0, str.length() - 1);
					field.value = str;
				}
			}
		} finally {
			FileUtils.close(in);
		}
	}

	public Field<Integer> getIntField(String name) {
		return integerFieldMap.get(name);
	}

	public List<String> getStringFieldNames() {
		List<String> names = new ArrayList<String>(stringFieldMap.keySet());
		Collections.sort(names);
		return names;
	}

	public List<String> getIntFieldNames() {
		List<String> names = new ArrayList<String>(integerFieldMap.keySet());
		Collections.sort(names);
		return names;
	}

	public List<String> getFloatFieldNames() {
		List<String> names = new ArrayList<String>(realFieldMap.keySet());
		Collections.sort(names);
		return names;
	}

	public Field<String> getStringField(String name) {
		return stringFieldMap.get(name);
	}

	public Field<Float> getFloatField(String name) {
		return realFieldMap.get(name);
	}

	static String nextLine(BufferedReader in) throws IOException {
		String line = in.readLine();
		if (line == null)
			throw new RuntimeException("Unexpected end of header");
		return line;
	}

	public static String getFieldValue(String line, String name) {
		String[] toks = line.split("\\s*=\\s*");
		if (!toks[0].equals(name)) {
			System.err.println("Offending header line:" + line);
		}
		Assertion.assertTrue(toks[0].equals(name));
		return toks[1];
	}

	public HeaderFields extractFields(List<HeaderFieldType> hftList) {
		HeaderFields hf = new HeaderFields();
		Set<String> seenSet = new HashSet<String>();
		// first one-to-one matching fields
		for (HeaderFieldType hft : hftList) {
			if (hft.getType().equals(HeaderFieldType.STRING)) {
				Field<String> field = getStringField(hft.getExtractionName());
				if (field != null) {
					hf.addStringField(field, hft);
					seenSet.add(hft.getExtractionName());
				}
			} else if (hft.getType().equals(HeaderFieldType.INT)) {
				Field<Integer> field = getIntField(hft.getExtractionName());
				if (field != null) {
					hf.addIntField(field, hft);
					seenSet.add(hft.getExtractionName());
				}
			} else if (hft.getType().equals(HeaderFieldType.FLOAT)) {
				Field<Float> field = getFloatField(hft.getExtractionName());
				if (field != null) {
					hf.addFloatField(field, hft);
					seenSet.add(hft.getExtractionName());
				}
			}
		}

		// search all NOTE header fields for params to extract
		List<String> stringFieldNames = getStringFieldNames();
		for (String strFieldName : stringFieldNames) {
			if (!strFieldName.startsWith("NOTE_NUMBER")) {
				continue;
			}
			Field<String> field = getStringField(strFieldName);
			String fieldValue = field.getValue();
			for (HeaderFieldType hft : hftList) {
				if (seenSet.contains(hft.getExtractionName())) {
					continue;
				}
				Pattern p = Pattern.compile(hft.getExtractionName()
						+ " ([\\d\\-\\.]+)");
				Matcher m = p.matcher(fieldValue);
				if (m.find()) {
					String value = m.group(1);
					if (hft.getType().equals(HeaderFieldType.STRING)) {
						Field<String> nf = new Field<String>(hft.getName());
						nf.value = value;
						hf.addStringField(nf, hft);
						seenSet.add(hft.getExtractionName());
					} else if (hft.getType().equals(HeaderFieldType.INT)) {
						Field<Integer> nf = new Field<Integer>(hft.getName());
						nf.value = Integer.parseInt(value);
						hf.addIntField(nf, hft);
						seenSet.add(hft.getExtractionName());
					} else if (hft.getType().equals(HeaderFieldType.FLOAT)) {
						Field<Float> nf = new Field<Float>(hft.getName());
						nf.value = Float.parseFloat(value);
						hf.addFloatField(nf, hft);
						seenSet.add(hft.getExtractionName());
					}
				}
			}
		}
		return hf;
	}

	public List<HeaderFieldType> readHeaderFieldTypes(String fieldTypesXmlFile)
			throws Exception {
		Element root = FileUtils.loadXML(fieldTypesXmlFile);
		Element afniEl = root.getChild("afni");
		List<?> ftElems = afniEl.getChildren("field");
		List<HeaderFieldType> hftList = new ArrayList<HeaderFieldType>(ftElems
				.size());
		for (Object o : ftElems) {
			Element ftEl = (Element) o;
			HeaderFieldType hft = HeaderFieldType.fromXml(ftEl);
			hftList.add(hft);
		}
		return hftList;
	}

	public static void main(String[] args) throws Exception {
		String rootDir = "/Users/bozyurt/cbfbirn_data/sampledata/090819_vg_visual";
		String headerFile = rootDir + "/P17408.7brik_e01+orig.HEAD";
		headerFile = rootDir + "/P18944.7brik_e02+orig.HEAD";

		AFNIHeaderReader ahr = new AFNIHeaderReader(headerFile);
		ahr.readHeader();
		for (String fieldName : ahr.getStringFieldNames()) {
			System.out.println(ahr.getStringField(fieldName));
		}
		System.out.println("===========================");
		for (String fieldName : ahr.getIntFieldNames()) {
			System.out.println(ahr.getIntField(fieldName));
		}

		String xmlFile = "/Users/bozyurt/dev/java/cbf_birn/hid/clinical/branches/BRANCH_cbfbirn/clinical/conf/header_field_types.xml";

		List<HeaderFieldType> hftList = ahr.readHeaderFieldTypes(xmlFile);
		HeaderFields hf = ahr.extractFields(hftList);
		System.out.println("=================================");

		System.out.println(hf.getIntField("tag"));
		System.out.println(hf.getIntField("q"));
	}
}
