package clinical.utils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;


import com.pixelmed.dicom.Attribute;
import com.pixelmed.dicom.AttributeFactory;
import com.pixelmed.dicom.AttributeList;
import com.pixelmed.dicom.AttributeTag;
import com.pixelmed.dicom.DicomException;
import com.pixelmed.dicom.DicomFileUtilities;
import com.pixelmed.dicom.TagFromName;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class DICOMHeaderAnonymizer {
	public static int ADD = 1;
	public static int DELETE = 2;
	public static int REPLACE = 3;
	List<Rule> anonRules;
	private boolean verbose = false;

	public DICOMHeaderAnonymizer() throws Exception {
		InputStream is = FileUtils.class.getClassLoader().getResourceAsStream(
				"anondcm.rules");
		anonRules = loadRules(is);
		if (verbose) {
			System.out.println("read " + anonRules.size() + " rules.");
		}
	}

	public List<File> anonymizeDICOMSeries(File dir, File anonDir) throws IOException,
			DicomException {
		File[] inFiles = dir.listFiles();
		List<File> dicomFiles = new ArrayList<File>();
		for (int i = 0; i < inFiles.length; i++) {
			if (inFiles[i].isFile()
					&& DicomFileUtilities.isDicomOrAcrNemaFile(inFiles[i])) {
				dicomFiles.add(inFiles[i]);
			}
		}
		inFiles = null;
		List<File> anonFiles = new ArrayList<File>(dicomFiles.size());
		for (File dcmFile : dicomFiles) {
			File anonFile = new File(anonDir, dcmFile.getName());
			anonymizeDICOM(dcmFile.getAbsolutePath(), anonFile.getAbsolutePath());
			anonFiles.add(anonFile);
		}

		return anonFiles;
	}

	public void anonymizeDICOM(String dicomFile, String anonFile)
			throws IOException, DicomException {
		AttributeList atList = new AttributeList();
		atList.read(dicomFile);

		String transferSyntaxUID = Attribute.getSingleStringValueOrNull(atList,
				TagFromName.TransferSyntaxUID);
		// System.out.println("transferSyntaxUID:" + transferSyntaxUID);
		for (Rule rule : anonRules) {
			AttributeTag at = AttributeList.getDictionary().getTagFromName(
					rule.tag);
			try {
				if (at != null) {
					Attribute attribute = atList.get(at);
					if (attribute != null) {
						String s = Attribute.getSingleStringValueOrNull(atList,
								at);
						if (verbose) {
							System.out.println(rule.tag + " :" + s);
						}
						if (rule.cmd == DELETE) {
							atList.replaceWithZeroLengthIfPresent(at);
						} else if (rule.cmd == REPLACE) {
							Attribute na = AttributeFactory.newAttribute(at);
							na.setValue(rule.value);
							atList.put(at, na);
						}
					}
				}
			} catch (Throwable t) {
				System.out.println("No attribute for " + rule.tag);
			}
		}
		if (verbose) {
			System.out.println("\nAfter Anonymization");
			System.out.println("-------------------");
			showAfterAnon(atList);
		}
		atList.write(anonFile, transferSyntaxUID, true, true);
		if (verbose) {
			System.out.println("wrote anonymized DICOM file to:" + anonFile);
		}
	}

	void showAfterAnon(AttributeList atList) {
		for (Rule rule : anonRules) {
			AttributeTag at = AttributeList.getDictionary().getTagFromName(
					rule.tag);
			if (at == null)
				continue;
			Attribute attribute = atList.get(at);
			if (attribute == null) {
				continue;
			}
			String s = Attribute.getSingleStringValueOrNull(atList, at);
			System.out.println(rule.tag + " :" + s);
		}
	}
	
	public List<Rule> loadRules(InputStream is) throws Exception {
		List<Rule> rules = new ArrayList<Rule>();
		BufferedReader in = null;
		try {
			in = new BufferedReader(new InputStreamReader(is));
			readRules(rules, in);
		} finally {
			FileUtils.close(in);
		}
		return rules;
	}

	public List<Rule> loadRules(String ruleFile) throws Exception {
		List<Rule> rules = new ArrayList<Rule>();
		BufferedReader in = null;
		try {
			in = new BufferedReader(new FileReader(ruleFile));
			readRules(rules, in);
		} finally {
			FileUtils.close(in);
		}
		return rules;
	}

	private void readRules(List<Rule> rules, BufferedReader in)
			throws IOException, Exception {
		String line;
		while ((line = in.readLine()) != null) {
			line = line.trim();
			if (line.length() == 0 || line.startsWith("#")) {
				continue;
			}
			String[] toks = line.split("\\s+");
			String cmd = toks[0];
			if (!cmd.equals("add") && !cmd.equals("replace")
					&& !cmd.equals("delete")) {
				throw new Exception("Unrecognized command in rule:" + cmd);
			}
			if (cmd.equals("replace") && toks.length < 3) {
				throw new Exception(
						"replace command requires two arguments: tag name and value");
			}
			if (cmd.equals("add") && toks.length < 3) {
				throw new Exception(
						"add command requires two arguments: tag name and value");
			}
			if (cmd.equals("delete") && toks.length < 2) {
				throw new Exception(
						"add command requires one argument: tag name");
			}
			Rule rule = new Rule();
			if (cmd.equals("add")) {
				rule.cmd = ADD;
			} else if (cmd.equals("delete")) {
				rule.cmd = DELETE;
			} else {
				rule.cmd = REPLACE;
			}
			rule.tag = toks[1];
			if (toks.length == 3) {
				rule.value = toks[2];
			}
			rules.add(rule);
		}
	}

	public static class Rule {
		int cmd;
		String tag;
		String value;
	}

	public static void main(String[] args) throws Exception {
		DICOMHeaderAnonymizer anonymizer = new DICOMHeaderAnonymizer();

		String dicomDir = "/data/cbfbirn_data/sampledata/090713_ds_stern/s2277706";
		// String dicomFile = "/data/cbfbirn_data/sampledata/090713_ds_stern/s2277706/i00019.CFMRI.19";

		// anonymizer.anonymizeDICOM(dicomFile, dicomFile + ".anon");

		long start = System.currentTimeMillis();
		File dcmDir = new File(dicomDir);
		String anonDirname = dcmDir.getName() + "_anon";
		File anonDir = new File( dcmDir.getParent(), anonDirname);
		List<File> anonFiles = anonymizer
				.anonymizeDICOMSeries(dcmDir, anonDir);
		System.out.println("# of anonymized files:" + anonFiles.size());
		long diff = System.currentTimeMillis() - start;
		System.out.println("Elapsed time (msecs) :" + diff);
	}
}
