package clinical.web.common.query;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.web.ISQLDialect;
import clinical.web.common.vo.TupleQueryInfo;

/**
 * @author I. Burak Ozyurt
 * @version $Id: AnalysisDDQueryBuilder.java,v 1.1.2.2 2008/06/06 23:32:06
 *          bozyurt Exp $
 */
public class AnalysisDDQueryBuilder extends AbstractQueryBuilder implements
		ISearchCriteriaVisitor {
	protected StringBuilder query;
	protected List<Integer> experimentIDs;
	protected String dbID;
	protected ISQLDialect sqlDialect;
	protected List<MeasurementGroupColInfo> mgciList = new ArrayList<MeasurementGroupColInfo>(
			10);
	protected Log log = LogFactory.getLog(AnalysisDDQueryBuilder.class);

	public AnalysisDDQueryBuilder(ISQLDialect sqlDialect, List<Integer> experimentIDs) {
		super();
		this.sqlDialect = sqlDialect;
		this.experimentIDs = experimentIDs;
	}

	/**
	 * Returns the generated SQL query.
	 *
	 * @return the generated SQL query
	 */
	public String getQuery() {
		return query.toString();
	}

	@Override
	public void visit(Operator op) throws Exception {
		this.query = new StringBuilder(2048);
		prepareQuery(query, op);
	}

	protected void prepareQuery(StringBuilder inq, Operator op)
			throws Exception {
		if (op instanceof UnaryOperator) {
			UnaryOperator uop = (UnaryOperator) op;
			SearchPredicate sp = uop.getPredicate();
			prepareQueryPart(sp, inq);
		} else if (op instanceof BinaryOperator) {
			BinaryOperator bop = (BinaryOperator) op;
			Operator lhs = bop.getLhs();
			prepareQuery(inq, lhs);
			// get the widest matching set to be filtered second time on the app
			// server
			inq.append("\nUNION\n");
			Operator rhs = bop.getRhs();
			prepareQuery(inq, rhs);
		}
	}

	protected void prepareQueryPart(SearchPredicate sp, StringBuilder inq)
			throws Exception {
		Object o = sp.getAttribute();
		if (!(o instanceof TupleQueryInfo))
			throw new Exception("Attribute needs to be of type TupleQueryInfo!");
		TupleQueryInfo tqi = (TupleQueryInfo) o;

		inq.append(getInnerStaticPart(tqi.getCtdi().getColumnType()));
		inq.append("and (d.columnname =");
		inq.append(sqlDialect.prepareString(tqi.getCtdi().getColumnName()));
		inq.append(" and ");
		if (sp.getValue() instanceof SearchPredicate.Range) {
			SearchPredicate.Range range = (SearchPredicate.Range) sp.getValue();
			inq.append("d.datavalue between ").append(range.getLowBound());
			inq.append(" and ").append(range.getUppBound());
		} else if (sp.getValue() instanceof SearchPredicate.DateRange) {
			SearchPredicate.DateRange dateRange = (SearchPredicate.DateRange) sp
					.getValue();
			inq.append("a.DATAVALUE BETWEEN ").append(
					toTimeStamp(dateRange.getLowBound(), sqlDialect));
			inq.append(" AND ").append(
					toTimeStamp(dateRange.getUppBound(), sqlDialect));
		} else {
			inq.append(createPredicate("d.datavalue", sp.getValue(), sp
					.getOperator(), sp.getType(), CastingOption.NONE));
		}
		inq.append(')');
		inq.append(" intersect ");
		inq.append(getInnerStaticPart("varchar"));
		inq.append("and (d.columnname =");
		inq.append(sqlDialect.prepareString(tqi.getMeasGroupFieldName()));
		inq.append(" and d.datavalue = ");
		inq.append(sqlDialect.prepareString(tqi.getMeasGroupName()));
		inq.append(')');

		MeasurementGroupColInfo mgci = new MeasurementGroupColInfo(tqi
				.getMeasGroupFieldName(), tqi.getMeasGroupName(), tqi.getCtdi()
				.getColumnName());
		mgciList.add(mgci);
	}

	/**
	 * Prepares the static (the first part) of the inner query ( an uncorrelated
	 * sub select).
	 *
	 * @param type
	 *            the type of the extended tuple data (can be 'integer',
	 *            'varchar','float' or 'boolean'.
	 * @return the (more and less) fixed part of the inner query
	 */
	protected String getInnerStaticPart(String type) throws Exception {
		StringBuilder buf = new StringBuilder(256);
		buf.append("SELECT r.subjectid, d.storedtupleid, e.uniqueid,");
		buf.append("r.experimentid, r.componentid, r.segmentid,");
		buf.append("r.analysisid, r.analysisname from ");
		if (type.equalsIgnoreCase("integer")) {
			buf.append(" nc_tupleinteger d, ");
		} else if (type.equalsIgnoreCase("varchar")) {
			buf.append(" nc_tuplevarchar d, ");
		} else if (type.equalsIgnoreCase("float")) {
			buf.append(" nc_tuplefloat d, ");
		} else if (type.equalsIgnoreCase("boolean")) {
			buf.append(" nc_tupleboolean d, ");
		} else if (type.equalsIgnoreCase("timestamp")) {
			buf.append(" nc_tupletimestamp d, ");
		} else {
			throw new Exception("Not a supported type: " + type);
		}
		buf.append("nc_analysisresult r, nc_storedtuple st,");
		buf.append("nc_extendedtuple e where  ");
		buf
				.append("d.storedtupleid = st.uniqueid and st.basetupleid = r.uniqueid");
		buf.append(" and st.extendedtupleid = e.uniqueid ");
		buf.append(prepareExperimentQueryPart());
		return buf.toString();
	}

	protected String prepareExperimentQueryPart() {
		if (experimentIDs == null || experimentIDs.isEmpty()) {
			return "";
		}
		StringBuilder buf = new StringBuilder();
		buf.append(" and ");
		if (experimentIDs.size() == 1) {
			buf.append("r.experimentid = ").append(
					(Integer) experimentIDs.get(0));
			buf.append(' ');
		} else {
			buf.append("r.experimentid in (");
			for (Iterator<Integer> iter = experimentIDs.iterator(); iter
					.hasNext();) {
				Integer experimentID = iter.next();
				buf.append(experimentID);
				if (iter.hasNext())
					buf.append(',');
			}
			buf.append(") ");
		}
		return buf.toString();
	}

	public static class MeasurementGroupColInfo {
		private String measGroupFieldName; // tupleCol
		private String measGroupName; // anatomicalEntity
		private String measName;

		public MeasurementGroupColInfo(String measGroupFieldName,
				String measGroupName, String measName) {
			this.measGroupFieldName = measGroupFieldName;
			this.measGroupName = measGroupName;
			this.measName = measName;
		}

		public String getMeasGroupFieldName() {
			return measGroupFieldName;
		}

		public String getMeasGroupName() {
			return measGroupName;
		}

		public String getMeasName() {
			return measName;
		}

	}// ;

	public List<MeasurementGroupColInfo> getMgciList() {
		return mgciList;
	}
}
