package clinical.web.common.query;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.web.Constants;
import clinical.web.common.AssessmentMapping;
import clinical.web.common.MediatedQueryHelper;
import clinical.web.common.vo.AsScoreInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: MediatedAssessmentQueryBuilder.java,v 1.7 2005/05/17 22:06:00
 *          kpease Exp $
 */

public class MediatedAssessmentQueryBuilder implements ISearchCriteriaVisitor {
	protected StringBuffer queryBuffer;
	protected Map<AsScoreInfo, AssessmentSelectionInfo> scoreMap = 
		new HashMap<AsScoreInfo, AssessmentSelectionInfo>();
	protected Map<String,ViewInfo> viewMap = new HashMap<String,ViewInfo>(17);
	protected String[] asDataColumns = { "TABLEID", "UNIQUEID", "SCOREORDER",
			"OWNER", "MODTIME", "MODUSER", "ASSESSMENTID", "SCORENAME",
			"SCORETYPE", "TEXTVALUE", "TEXTNORMVALUE", "ENCRYPTED", "COMMENTS",
			"NC_STOREDASSESSMENT_UNIQUEID" };
	protected String[] asTypeColumns = { "TABLEID",
			"NC_ASSESSMENTDATA_UNIQUEID", "SCOREORDER", "OWNER", "MODTIME",
			"MODUSER", "TEXTVALUE", "TEXTNORMVALUE", "COMMENTS", "DATAVALUE",
			"DATANORMVALUE", "STOREDASSESSMENTID", "1263", "MMSE Score",
			"SCORETYPE" };
	protected String[] saColumns = { "UNIQUEID", "ASSESSMENTID", "TABLEID",
			"OWNER", "MODTIME", "MODUSER", "COMPONENTID",
			"NC_EXPERIMENT_UNIQUEID", "SEGMENTID", "SUBJECTID", "TIME_STAMP" };
	protected Map<Integer, List<AssessmentMapping>> siteAsMap;
	private Log log = LogFactory.getLog(MediatedAssessmentQueryBuilder.class);

	public MediatedAssessmentQueryBuilder(
			List<AssessmentSelectionInfo> assessments,
			Map<Integer, List<AssessmentMapping>> siteAsMap) {
		this.siteAsMap = siteAsMap;
		for (Iterator<AssessmentSelectionInfo> it = assessments.iterator(); it
				.hasNext();) {
			AssessmentSelectionInfo asi = it.next();
			for (Object element : asi.getScores()) {
				AsScoreInfo si = (AsScoreInfo) element;
				scoreMap.put(si, asi);
			}
		}

		ViewInfo[] vis = new ViewInfo[10];
		vis[0] = new ViewInfo("ucsd_assessmentdata_view",
				Constants.UCSD_SOURCE, "data");
		vis[1] = new ViewInfo("ucsd_assessmentinteger_view",
				Constants.UCSD_SOURCE, "integer");
		vis[2] = new ViewInfo("ucsd_assessmentvarchar_view",
				Constants.UCSD_SOURCE, "varchar");
		vis[3] = new ViewInfo("ucsd_assessmentfloat_view",
				Constants.UCSD_SOURCE, "float");
		vis[3] = new ViewInfo("ucsd_assessmentboolean_view",
				Constants.UCSD_SOURCE, "boolean");
		vis[3] = new ViewInfo("ucsd_assessmenttimestamp_view",
				Constants.UCSD_SOURCE, "timestamp");
		vis[4] = new ViewInfo("ucsd_storedassessment_view",
				Constants.UCSD_SOURCE, "stored");

		vis[5] = new ViewInfo("mgh_assessmentdata_view", Constants.MGH_SOURCE,
				"data");
		vis[6] = new ViewInfo("mgh_assessmentinteger_view",
				Constants.MGH_SOURCE, "integer");
		vis[7] = new ViewInfo("mgh_assessmentvarchar_view",
				Constants.MGH_SOURCE, "varchar");
		vis[8] = new ViewInfo("mgh_assessmentfloat_view", Constants.MGH_SOURCE,
				"float");
		vis[8] = new ViewInfo("mgh_assessmentboolean_view",
				Constants.MGH_SOURCE, "boolean");
		vis[8] = new ViewInfo("mgh_assessmenttimestamp_view",
				Constants.MGH_SOURCE, "timestamp");
		vis[9] = new ViewInfo("mgh_storedassessment_view",
				Constants.MGH_SOURCE, "stored");

		for (int i = 0; i < vis.length; i++) {
			viewMap.put(vis[i].sourceName + vis[i].type, vis[i]);
		}
	}

	protected String getMatchingScoreName(AsScoreInfo si) {
		AssessmentSelectionInfo asi = scoreMap.get(si);
		AssessmentMapping am = (AssessmentMapping) siteAsMap.get(new Integer(
				asi.getAssessmentID().intValue()));
		AssessmentMapping.ScoreMap sm = (AssessmentMapping.ScoreMap) am
				.getMappedScoreName(si.getName());
		return sm.getMappedName();
	}

	protected int getMatchingScoreAsID(AsScoreInfo si) {
		AssessmentSelectionInfo asi = scoreMap.get(si);
		AssessmentMapping am = (AssessmentMapping) siteAsMap.get(new Integer(
				asi.getAssessmentID().intValue()));
		AssessmentMapping.ScoreMap sm = (AssessmentMapping.ScoreMap) am
				.getMappedScoreName(si.getName());
		return sm.getMappedAsID();
	}

	public void visit(Operator op) throws Exception {
		Map<String, List<SearchPredicate>> searchPredicateMap = new LinkedHashMap<String, List<SearchPredicate>>(
				5);
		populateSearchPredicateMap(op, searchPredicateMap);
		try {
			String q = prepareQuery(Constants.UCSD_SOURCE, searchPredicateMap);

			queryBuffer = new StringBuffer(q.length() * 2 + 50);
			queryBuffer
					.append("query.q4v2(SUBJECTID11, TEXTVALUE, SCORENAME, ASSESSMENTID, COMPONENTID11, SEGMENTID11, NC_EXPERIMENT_UNIQUEID11) :- ");

			queryBuffer.append(q);

			boolean allowMultiSiteQuery = true;
			for (Iterator<List<SearchPredicate>> iter = searchPredicateMap.values().iterator(); iter
					.hasNext();) {
				List<SearchPredicate> spList = iter.next();
				for (SearchPredicate sp : spList) {
					AsScoreInfo si = (AsScoreInfo) sp.getAttribute();
					AssessmentSelectionInfo asi = scoreMap.get(si);
					AssessmentMapping am = (AssessmentMapping) siteAsMap
							.get(new Integer(asi.getAssessmentID().intValue()));
					if (am == null) {
						allowMultiSiteQuery = false;
						break;
					}
					AssessmentMapping.ScoreMap sm = (AssessmentMapping.ScoreMap) am
							.getMappedScoreName(si.getName());
					if (sm == null) {
						allowMultiSiteQuery = false;
						break;
					}
				}
			}
			if (allowMultiSiteQuery) {
				queryBuffer.append(" ##  ");
				q = prepareQuery(Constants.MGH_SOURCE, searchPredicateMap);
				queryBuffer.append(q);
			}

		} catch (Exception x) {
			log.error("MediatedAssessmentQuery", x);
			queryBuffer = null;
			throw x;
		}

	}

	/**
	 * This method walks down the operator tree in a depth first session and
	 * flattens the tree to an association table keyed by the data type of the
	 * search predicates Assumption: no variable is repeated in the search
	 * predicate tree.
	 * 
	 * @param op
	 * @param searchPredicateMap
	 */
	protected void populateSearchPredicateMap(Operator op,
			Map<String, List<SearchPredicate>> searchPredicateMap) {
		if (op instanceof UnaryOperator) {
			UnaryOperator uop = (UnaryOperator) op;
			SearchPredicate sp = uop.getPredicate();
			Object o = sp.getAttribute();
			if (!(o instanceof AsScoreInfo))
				throw new RuntimeException(
						"Attribute needs to be of type AsScoreInfo!");

			AsScoreInfo si = (AsScoreInfo) o;
			List<SearchPredicate> spList = searchPredicateMap.get(si.getType());
			if (spList == null) {
				spList = new LinkedList<SearchPredicate>();
				searchPredicateMap.put(si.getType(), spList);
			}
			spList.add(sp);

		} else if (op instanceof BinaryOperator) {
			BinaryOperator bop = (BinaryOperator) op;
			Operator lhs = bop.getLhs();
			populateSearchPredicateMap(lhs, searchPredicateMap);
			Operator rhs = bop.getRhs();
			populateSearchPredicateMap(rhs, searchPredicateMap);
		}
	}

	protected Object conditionValue(AsScoreInfo si, String sourceName,
			Object value) {
		if (!sourceName.equals(Constants.MGH_SOURCE))
			return value;

		if (si.getName().equalsIgnoreCase("diagnosis")) {
			String s = (String) value;
			if (s.equals("Mem Imp")) {
				return "questionable";
			} else if (s.equals("Control")) {
				return "control";
			} else {
				if (s.startsWith("M")) {
					return "q";
				} else if (s.startsWith("C")) {
					return "c";
				}
			}
		}
		return value;
	}

	protected String prepareQuery(String sourceName,
			Map<String, List<SearchPredicate>> searchPredicateMap) throws Exception {
		StringBuffer queryBuf = new StringBuffer(2048);
		StringBuffer buf = new StringBuffer(1024);
		Map<SearchPredicate, VariableInfo> varJoinMap = new HashMap<SearchPredicate, VariableInfo>();
		prepareJoins(sourceName, searchPredicateMap, varJoinMap, buf);

		queryBuf.append(buf.toString());

		// only and logical operator between search predicates for mediated
		// queries
		String[] types = { "integer", "varchar", "float", "boolean",
				"timestamp" };
		int spCount = 0;
		for (int i = 0; i < types.length; ++i) {
			List<SearchPredicate> spList = searchPredicateMap.get(types[i]);
			if (spList != null) {
				for (SearchPredicate sp : spList) {
					AsScoreInfo si = (AsScoreInfo) sp.getAttribute();
					VariableInfo varInfo = varJoinMap.get(sp);
					queryBuf.append(sourceName).append(
							".eval_boolean(eq, STOREDASSESSMENTID").append(
							varInfo.idx);
					queryBuf.append(",UNIQUEID").append(varInfo.idx + 10)
							.append(") && ");

					if (sp.getValue() instanceof SearchPredicate.Range) {
						throw new Exception(
								"Range search predicates are not supported in mediated queries!");
					}
					// check if user tried wildcard for numeric search predicate
					if (sp.getType() != SearchPredicate.STRING
							&& sp.getValue().toString().trim().equals("*")) {
						throw new Exception(
								"Wildcard is not supported for numerical variables in mediated queries!");
					}
					String predStr = QueryUtils.createPredicate(sourceName,
							varInfo.variableName, conditionValue(si,
									sourceName, sp.getValue()), sp
									.getOperator());
					queryBuf.append(predStr).append(" && ");
					++spCount;
				}
			}
		}

		queryBuf
				.append(sourceName)
				.append(
						".eval_boolean(eq, NC_STOREDASSESSMENT_UNIQUEID, UNIQUEID11) && ");
		queryBuf.append("(");
		int spIdx = 0;
		for (int i = 0; i < types.length; ++i) {
			List<SearchPredicate> spList = searchPredicateMap.get(types[i]);
			if (spList != null) {
				for (SearchPredicate sp : spList) {
					AsScoreInfo si = (AsScoreInfo) sp.getAttribute();
					// VariableInfo varInfo = varJoinMap.get(sp);
					AssessmentSelectionInfo asi = scoreMap.get(si);
					int asID = -1;
					String scoreName = null;
					/** @todo too specific */
					if (sourceName == Constants.UCSD_SOURCE) {
						scoreName = si.getName();
						asID = asi.getAssessmentID().intValue();
					} else {
						scoreName = getMatchingScoreName(si);
						asID = getMatchingScoreAsID(si);
					}

					queryBuf.append('(');
					queryBuf.append(sourceName).append(
							".eval_boolean(eq, ASSESSMENTID, ");
					queryBuf.append(asID).append(") && ");

					queryBuf.append(sourceName).append(
							".eval_boolean(eq, SCORENAME, '");
					queryBuf.append(scoreName).append("')");

					queryBuf.append(')');
					if ((spIdx + 1) < spCount) {
						queryBuf.append(" ## ");
					}
					++spIdx;

				}
			} // if
		}// i
		queryBuf.append(")");

		queryBuf.append(")");
		return queryBuf.toString();
	}

	protected void prepareJoins(String sourceName,
			Map<String, List<SearchPredicate>> searchPredicateMap,
			Map<SearchPredicate, VariableInfo> varJoinMap, StringBuffer queryBuf) {
		ViewInfo vi = viewMap.get(sourceName + "data");
		queryBuf.append("( view.").append(vi.viewName).append("(");

		boolean[] suffixStripList = new boolean[asDataColumns.length];
		for (int i = 0; i < suffixStripList.length; i++) {
			suffixStripList[i] = false;
		}
		queryBuf.append(prepareColumnList(asDataColumns, -1, suffixStripList))
				.append(") && ");
		int idx = 2;

		suffixStripList = new boolean[asTypeColumns.length];
		for (int i = 0; i < suffixStripList.length; i++) {
			suffixStripList[i] = false;
		}
		suffixStripList[12] = true; // assessmentid
		suffixStripList[13] = true; // score name

		// first integers
		String[] types = { "integer", "varchar", "float", "boolean",
				"timestamp" };
		for (int i = 0; i < types.length; ++i) {

			List<SearchPredicate> spList = searchPredicateMap.get(types[i]);

			if (spList != null) {
				vi = viewMap.get(sourceName + types[i]);
				for (SearchPredicate sp : spList) {
					AsScoreInfo si = (AsScoreInfo) sp.getAttribute();
					queryBuf.append(" view.").append(vi.viewName).append("(");
					AssessmentSelectionInfo asi = scoreMap.get(si);

					int asID = -1;
					String scoreName = null;
					/** @todo too specific */
					if (sourceName == Constants.UCSD_SOURCE) {
						scoreName = si.getName();
						asID = asi.getAssessmentID().intValue();
					} else {
						scoreName = getMatchingScoreName(si);
						asID = getMatchingScoreAsID(si);
					}

					asTypeColumns[12] = "" + asID;
					asTypeColumns[13] = "'" + scoreName + "'";
					queryBuf.append(
							prepareColumnList(asTypeColumns, idx,
									suffixStripList)).append(") && ");
					varJoinMap
							.put(sp, new VariableInfo("DATAVALUE" + idx, idx));
					++idx;
				}
			}
		}// i

		// now the storedassessment self joins
		int noJoins = idx - 1;
		int idx2 = 11;
		vi = viewMap.get(sourceName + "stored");
		suffixStripList = new boolean[saColumns.length];
		for (int i = 0; i < suffixStripList.length; i++) {
			suffixStripList[i] = false;
		}
		suffixStripList[9] = true; // Subjectid
		saColumns[9] = "SUBJECTID" + idx2;
		for (int i = 0; i < noJoins; i++) {
			queryBuf.append(" view.").append(vi.viewName).append("(");
			queryBuf
					.append(prepareColumnList(saColumns, idx2, suffixStripList))
					.append(") && ");
			idx2++;
		}
	}

	protected String prepareColumnList(String[] columns, int suffix,
			boolean[] suffixSkipList) {
		StringBuffer buf = new StringBuffer(256);
		for (int i = 0; i < columns.length; i++) {
			buf.append(columns[i]);
			if (suffixSkipList[i] == false && suffix > 0)
				buf.append(suffix);
			if ((i + 1) < columns.length)
				buf.append(',');
		}
		return buf.toString();
	}

	public String getQuery() {
		if (queryBuffer == null)
			return null;
		return queryBuffer.toString();
	}

	class VariableInfo {
		String variableName;
		int idx;

		public VariableInfo(String variableName, int idx) {
			this.variableName = variableName;
			this.idx = idx;
		}
	}

	class ViewInfo {
		String viewName;
		String sourceName;
		/** integer, varchar, boolean, float, timestamp, stored, data */
		String type;

		public ViewInfo(String viewName, String sourceName, String type) {
			this.viewName = viewName;
			this.sourceName = sourceName;
			this.type = type;
		}

		public int hashCode() {
			StringBuffer buf = new StringBuffer();
			buf.append(sourceName).append('_').append(type);
			return buf.toString().hashCode();
		}

		public boolean equals(Object other) {
			if (other == null || !(other instanceof ViewInfo))
				return false;
			ViewInfo vi = (ViewInfo) other;
			return viewName.equals(vi.viewName)
					&& sourceName.equals(vi.sourceName) && type.equals(vi.type);

		}
	}

	public static void main(String[] args) {
		List<AssessmentSelectionInfo> assessments = new LinkedList<AssessmentSelectionInfo>();
		AssessmentSelectionInfo asi = new AssessmentSelectionInfo();
		// will change every time data is reuploaded
		String MMSEAssessmentID = "1263";
		String DemographicsAssessmentID = "1264";

		asi.setAssessmentID(new BigDecimal(MMSEAssessmentID));
		asi.setName("MMSE");
		AsScoreInfo si1 = new AsScoreInfo(asi.getAssessmentID());
		si1.setType("integer");
		si1.setName("MMSE Score");
		asi.addScore(si1);
		assessments.add(asi);

		asi = new AssessmentSelectionInfo();
		asi.setAssessmentID(new BigDecimal("1265"));
		asi.setName("Diagnosis");
		AsScoreInfo si3 = new AsScoreInfo(asi.getAssessmentID());
		si3.setType("varchar");
		si3.setName("Diagnosis");
		asi.addScore(si3);
		assessments.add(asi);

		asi = new AssessmentSelectionInfo();
		asi.setAssessmentID(new BigDecimal(DemographicsAssessmentID));
		asi.setName("Demographics");
		AsScoreInfo si2 = new AsScoreInfo(asi.getAssessmentID());
		si2.setType("varchar");
		si2.setName("Gender");
		asi.addScore(si2);
		assessments.add(asi);

		asi = new AssessmentSelectionInfo();
		asi.setAssessmentID(new BigDecimal("2908"));
		asi.setName("Diagnosis");
		AsScoreInfo si4 = new AsScoreInfo(asi.getAssessmentID());
		si4.setType("float");
		si4.setName("Discriminability raw score");
		asi.addScore(si4);
		assessments.add(asi);

		@SuppressWarnings("unused")
      SearchPredicate.Range range = new SearchPredicate.Range(
				new Integer(15), new Integer(25));

		// prepare Operator tree
		/** @todo no range search predicates for mediated queries */
		@SuppressWarnings("unused")
      UnaryOperator lhs = new UnaryOperator(
				new SearchPredicate(si1, new Integer(21), SearchPredicate.LESS,
						SearchPredicate.INTEGER), Operator.NONE);
		/*
		 * UnaryOperator rhs = new UnaryOperator( new SearchPredicate(si2, new
		 * Integer(50), SearchPredicate.GREATER, SearchPredicate.INTEGER),
		 * Operator.NONE);
		 */
		@SuppressWarnings("unused")
      UnaryOperator rhs = new UnaryOperator(new SearchPredicate(si2, "M",
				SearchPredicate.EQUAL, SearchPredicate.STRING), Operator.NONE);

		UnaryOperator lhs1 = new UnaryOperator(new SearchPredicate(si3, "M",
				SearchPredicate.STARTS_WITH, SearchPredicate.STRING),
				Operator.NONE);
		UnaryOperator rhs1 = new UnaryOperator(
				new SearchPredicate(si4, new Integer(0),
						SearchPredicate.GREATER, SearchPredicate.FLOAT),
				Operator.NONE);

		// BinaryOperator bop = new BinaryOperator(lhs, rhs, Operator.AND);

		BinaryOperator bop = new BinaryOperator(lhs1, rhs1, Operator.AND);

		try {
			String mapFile = "/home/bozyurt/dev/java/clinical/conf/as_var_map.xml";
			Map<Integer, List<AssessmentMapping>> siteAsMap = MediatedQueryHelper
					.prepareAssesmentMapping(mapFile);

			MediatedAssessmentQueryBuilder mqb = new MediatedAssessmentQueryBuilder(
					assessments, siteAsMap);

			mqb.visit(bop);

			String query = mqb.getQuery();

			System.out.println("query=" + query);
		} catch (Exception x) {
			x.printStackTrace();
		}

	}

}