package clinical.web.common.query;

import java.util.*;

import org.apache.commons.logging.*;
import clinical.web.*;
import clinical.web.helpers.*;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: MediatedDerivedDataQueryBuilder.java,v 1.2 2008/08/26 00:07:58
 *          bozyurt Exp $
 */

public class MediatedDerivedDataQueryBuilder implements ISearchCriteriaVisitor {
   protected StringBuffer queryBuffer;
   protected String ucsdSource = "ucsdv3";
   protected String mghSource = "mghv3";
   protected String ucsdView = "ucsd_brainsegmentdata_view";
   protected String mghView = "mgh_brainsegmentdata_view";
   private Log log = LogFactory.getLog(MediatedDerivedDataQueryBuilder.class);

   public MediatedDerivedDataQueryBuilder() {}

   public void visit(Operator op) throws Exception {
      List<SearchPredicate> searchPredicateList = new LinkedList<SearchPredicate>();
      populateSearchPredicateList(op, searchPredicateList);
      try {
         String q = prepareQuery(Constants.UCSD_SOURCE, ucsdView,
               searchPredicateList);

         queryBuffer = new StringBuffer(q.length() * 2 + 50);
         queryBuffer
               .append("query.q5c(SUBJECTID, BRAINREGIONNAME, LATERALITY, MEASUREMENT, COMPONENTID, SEGMENTID, NC_EXPERIMENTID) :- ");
         queryBuffer.append(q);

         q = prepareQuery(Constants.MGH_SOURCE, mghView, searchPredicateList);
         queryBuffer.append(" ## ");
         queryBuffer.append(q);

      } catch (Exception x) {
         log.error("MediatedDerivedDataQueryBuilder", x);
         queryBuffer = null;
         throw x;
      }
   }

   public String getQuery() {
      if (queryBuffer == null)
         return null;
      return queryBuffer.toString();
   }

   /**
    * This method walks down the operator tree in a depth first session and
    * flattens the tree to a list Assumption: no variable is repeated in the
    * search predicate tree.
    * 
    * @param op
    * @param searchPredicateList
    */
   protected void populateSearchPredicateList(Operator op,
         List<SearchPredicate> searchPredicateList) {
      if (op instanceof UnaryOperator) {
         UnaryOperator uop = (UnaryOperator) op;
         SearchPredicate sp = uop.getPredicate();
         Object o = sp.getAttribute();
         if (!(o instanceof SubCorticalVarInfo))
            throw new RuntimeException(
                  "Attribute needs to be of type SubCorticalVarInfo!");
         searchPredicateList.add(sp);
      } else if (op instanceof BinaryOperator) {
         BinaryOperator bop = (BinaryOperator) op;
         Operator lhs = bop.getLhs();
         populateSearchPredicateList(lhs, searchPredicateList);
         Operator rhs = bop.getRhs();
         populateSearchPredicateList(rhs, searchPredicateList);
      }
   }

   protected void prepareJoins(String viewName, List<SearchPredicate> searchPredicateList,
         Map<SubCorticalVarInfo,String> varJoinMap, StringBuffer queryBuf) {
      queryBuf.append("( view.").append(viewName).append("(");
      String[] columns = { "DATAID", "TABLEID", "SEGMENTID", "COMPONENTID",
            "NC_EXPERIMENTID", "SUBJECTID", "BRAINREGIONNAME", "LATERALITY",
            "MEASUREMENT", "MEASUREMENTTYPE", "UNIT", "OWNER", "MODTIME",
            "MODUSER", "ONTOLOGYSOURCE", "CONCEPTID" };

      boolean[] suffixStripList = new boolean[columns.length];
      for (int i = 0; i < suffixStripList.length; i++) {
         suffixStripList[i] = false;
      }
      queryBuf.append(prepareColumnList(columns, -1, suffixStripList)).append(
            ") && ");
      suffixStripList[5] = true; // subjectid
      suffixStripList[6] = true; // brainregionname
      suffixStripList[7] = true; // laterality
      suffixStripList[9] = true; // measurement type

      int idx = 2;
      for (Iterator<SearchPredicate> iter = searchPredicateList.iterator(); iter.hasNext();) {
         SearchPredicate sp = iter.next();
         queryBuf.append(" view.").append(viewName).append("(");
         SubCorticalVarInfo sv = (SubCorticalVarInfo) sp.getAttribute();
         columns[6] = "'" + sv.getBrainRegionName() + "'";
         columns[7] = "'" + sv.getLaterality() + "'";
         columns[9] = "'" + sv.getMeasurementType() + "'";
         queryBuf.append(prepareColumnList(columns, idx, suffixStripList))
               .append(") && ");
         varJoinMap.put(sv, "MEASUREMENT" + idx);

         ++idx;
      }
   }

   protected String prepareColumnList(String[] columns, int suffix,
         boolean[] suffixSkipList) {
      StringBuffer buf = new StringBuffer(256);
      for (int i = 0; i < columns.length; i++) {

         buf.append(columns[i]);
         if (suffixSkipList[i] == false && suffix > 0)
            buf.append(suffix);
         if ((i + 1) < columns.length)
            buf.append(',');
      }
      return buf.toString();
   }

   protected String prepareQuery(String sourceName, String viewName,
         List<SearchPredicate> searchPredicateList) throws Exception {
      StringBuffer queryBuf = new StringBuffer(2048);
      StringBuffer buf = new StringBuffer(1024);
      Map<SubCorticalVarInfo, String> varJoinMap = new HashMap<SubCorticalVarInfo, String>();
      prepareJoins(viewName, searchPredicateList, varJoinMap, buf);

      queryBuf.append(buf.toString());
      queryBuf.append(sourceName).append(
            ".eval_boolean(eq, MEASUREMENTTYPE, 'volume') && ");

      // only and logical operator between search predicates for mediated
      // queries
      //
      for (Iterator<SearchPredicate> iter = searchPredicateList.iterator(); iter.hasNext();) {
         SearchPredicate sp = iter.next();
         SubCorticalVarInfo sv = (SubCorticalVarInfo) sp.getAttribute();
         String variableName = (String) varJoinMap.get(sv);
         if (sp.getValue() instanceof SearchPredicate.Range) {
            throw new Exception(
                  "Range search predicates are not supported in mediated queries!");
         }
         // check if user tried wildcard for numeric search predicate
         if (sp.getType() != SearchPredicate.STRING
               && sp.getValue().toString().trim().equals("*")) {
            throw new Exception(
                  "Wildcard is not supported for numerical variables in mediated queries!");
         }

         String predStr = QueryUtils.createPredicate(sourceName, variableName,
               sp.getValue(), sp.getOperator());
         queryBuf.append(predStr).append(" && ");
      }

      queryBuf.append("(");
      for (Iterator<SearchPredicate> iter = searchPredicateList.iterator(); iter.hasNext();) {
         SearchPredicate sp = iter.next();
         SubCorticalVarInfo sv = (SubCorticalVarInfo) sp.getAttribute();
         // String variableName = (String) varJoinMap.get(sv);
         queryBuf.append('(');
         queryBuf.append(sourceName).append(
               ".eval_boolean(eq, BRAINREGIONNAME, '");
         queryBuf.append(sv.getBrainRegionName()).append("') && ");
         queryBuf.append(sourceName).append(".eval_boolean(eq, LATERALITY, '");
         queryBuf.append(sv.getLaterality()).append("')");
         queryBuf.append(')');
         if (iter.hasNext()) {
            queryBuf.append(" ## ");
         }
      }
      queryBuf.append(")");

      queryBuf.append(")");
      return queryBuf.toString();
   }

   public static void main(String[] args) {

      SubCorticalVarInfo sv1 = new SubCorticalVarInfo("Caudate", "left",
            "volume", "mm^3");
      @SuppressWarnings("unused")
      SubCorticalVarInfo sv2 = new SubCorticalVarInfo("Brain-Stem", "n/a",
            "volume", "mm^3");
      SubCorticalVarInfo sv3 = new SubCorticalVarInfo("Amygdala", "left",
            "volume", "mm^3");
      SubCorticalVarInfo sv4 = new SubCorticalVarInfo("Caudate", "right",
            "volume", "mm^3");

      // prepare Operator tree
      UnaryOperator lhs = new UnaryOperator(new SearchPredicate(sv3, new Float(
            0), SearchPredicate.GREATER, SearchPredicate.FLOAT), Operator.NONE);
      /*
       * UnaryOperator rhs = new UnaryOperator( new SearchPredicate(sv2, new
       * Float(0), SearchPredicate.GREATER, SearchPredicate.FLOAT),
       * Operator.NONE);
       */
      @SuppressWarnings("unused")
      UnaryOperator lhs1 = new UnaryOperator(new SearchPredicate(sv1,
            new Float(0), SearchPredicate.GREATER, SearchPredicate.FLOAT),
            Operator.NONE);

      @SuppressWarnings("unused")
      UnaryOperator rhs1 = new UnaryOperator(new SearchPredicate(sv4,
            new Float(0), SearchPredicate.GREATER, SearchPredicate.FLOAT),
            Operator.NONE);

      // BinaryOperator bop1 = new BinaryOperator(lhs1, rhs1, Operator.AND);

      // BinaryOperator bop = new BinaryOperator(lhs, bop1, Operator.AND);
      // BinaryOperator bop = new BinaryOperator(lhs, rhs1, Operator.AND);

      MediatedDerivedDataQueryBuilder mqb = new MediatedDerivedDataQueryBuilder();
      // mqb.visit(bop);
      try {
         mqb.visit(lhs);
         System.out.println("query=" + mqb.getQuery());
      } catch (Exception x) {
         x.printStackTrace();
      }
   }

}