package clinical.web.common.security;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import clinical.web.helpers.security.UserPrivilegeSet;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: User.java 366 2011-05-05 20:06:27Z bozyurt $
 */

public class User {
	protected String name;
	protected String pwd;
	protected boolean useGSI = false;
	protected String email;
	protected boolean invited = false;

	protected User dbUser;
	protected Map<String, Privilege> fullPrivilegesMap = new LinkedHashMap<String, Privilege>(
			7);
	protected Map<String, String> availableTables = new HashMap<String, String>();
	protected Map<String, Privilege> privilegeMap = Collections
			.synchronizedMap(new LinkedHashMap<String, Privilege>(7));
	protected UserPrivilegeSet userPrivSet;
	protected List<PrivilegeExperiments> privExperiments;

	public User(String name, String pwd, UserPrivilegeSet userPrivSet) {
		this.name = name;
		this.pwd = pwd;
		this.userPrivSet = userPrivSet;
	}

	public User(String name, String pwd, UserPrivilegeSet userPrivSet,
			boolean useGSI) {
		this(name, pwd, userPrivSet);
		this.useGSI = useGSI;
	}

	public String getName() {
		return this.name;
	}

	public String getPwd() {
		return this.pwd;
	}

	public void setPwd(String pwd) {
		this.pwd = pwd;
	}

	public User getDbUser() {
		return this.dbUser;
	}

	public void setDbUser(User dbUser) {
		this.dbUser = dbUser;
	}

	public int hashCode() {
		return name.hashCode();
	}

	public boolean equals(Object other) {
		if (other == null || !(other instanceof User))
			return false;
		return name.equals(((User) other).name);
	}

	public void addAvailableTable(String tableName) {
		availableTables.put(tableName, tableName);
	}

	public Map<String, String> getAvailableTables() {
		return availableTables;
	}

	public synchronized void addPrivilege(Privilege privilege) {
		privilegeMap.put(privilege.getName(), privilege);
	}

	public synchronized void removePrivilege(String privName) {
		privilegeMap.remove(privName);
	}

	public List<Privilege> listPrivileges() {
		return new ArrayList<Privilege>(privilegeMap.values());
	}

	public List<Privilege> getPrivileges() {
		return new ArrayList<Privilege>(privilegeMap.values());
	}

	public boolean hasPrivilege(String privName) {
		return privilegeMap.get(privName) != null;
	}

	public boolean hasPrivilege(String privName, Integer experimentID) {
		Privilege privilege = privilegeMap.get(privName);
		if (privilege != null) {
			return true;
		} else {
			// check if user has admin privilege
			if (privilegeMap.containsKey("admin"))  {
				return true;
			}
		}
		
		if (experimentID == null) {
			// FIXME project specific and no project ID specified
			return this.userPrivSet.hasPrivilege(privName);
		}
		return this.userPrivSet.hasPrivilege(experimentID, privName);
	}

	public int getNumOfPrivileges() {
		return privilegeMap.size();
	}

	public Map<String, Privilege> getFullPrivilegesMap() {
		return fullPrivilegesMap;
	}

	public void setFullPrivilegesMap(Map<String, Privilege> fullPrivilegesMap) {
		this.fullPrivilegesMap = fullPrivilegesMap;
	}

	public List<Privilege> getMissingPrivileges() {
		if (privilegeMap.size() == fullPrivilegesMap.size()) {
			return new ArrayList<Privilege>(0);
		}
		List<Privilege> list = new ArrayList<Privilege>(fullPrivilegesMap.size());
		for (String privName : fullPrivilegesMap.keySet()) {
			if (!privilegeMap.containsKey(privName)) {
				list.add(fullPrivilegesMap.get(privName));
			}
		}
		return list;
	}

	public List<Privilege> getNonProjectMissingPrivileges() {
		if (privilegeMap.size() == fullPrivilegesMap.size()) {
			return new ArrayList<Privilege>(0);
		}
		List<Privilege> list = new ArrayList<Privilege>(fullPrivilegesMap.size());
		for (String privName : fullPrivilegesMap.keySet()) {
			if (privName.equals("admin"))
				continue;
			if (!privilegeMap.containsKey(privName)) {
				Privilege priv = fullPrivilegesMap.get(privName);
				if (!priv.isProjSpec()) {
					list.add(fullPrivilegesMap.get(privName));
				}
			}
		}
		return list;
	}

	public boolean isSame(User other) {
		if (other == null)
			return false;
		boolean ok = true;
		ok = name.equals(other.name) && pwd.equals(other.pwd);
		if (!ok)
			return false;
		if (dbUser != null) {
			return dbUser.isSame(other.dbUser);
		}
		return ok;
	}

	public boolean isUseGSI() {
		return useGSI;
	}

	public void setUseGSI(boolean useGSI) {
		this.useGSI = useGSI;
	}

	public String toString() {
		StringBuffer buf = new StringBuffer(104);
		buf.append("User::[");
		buf.append("name=").append(name);
		buf.append(", dbUser=").append(dbUser);
		buf.append(']');
		return buf.toString();
	}

	public static class PrivilegeExperiments {
		String privilege;
		List<ProjectPrivilegeStatus> ppStatusList;

		public PrivilegeExperiments(String privilege,
				List<ProjectPrivilegeStatus> ppStatusList) {
			super();
			this.privilege = privilege;
			this.ppStatusList = ppStatusList;
		}

		public String getPrivilege() {
			return privilege;
		}

		public List<ProjectPrivilegeStatus> getPpStatusList() {
			return ppStatusList;
		}

		public boolean hasGranted() {
			for (ProjectPrivilegeStatus pps : ppStatusList) {
				if (pps.isGranted())
					return true;
			}
			return false;
		}
		
		public void addProjectPrivilegeStatus(ProjectPrivilegeStatus pps) {
			if (!ppStatusList.contains(pps)) {
				ppStatusList.add(pps);
			}
		}
	}

	public static class ProjectPrivilegeStatus {
		int expId;
		String expName;
		boolean granted;

		public ProjectPrivilegeStatus(int expId, boolean granted, String expName) {
			super();
			this.expId = expId;
			this.granted = granted;
			this.expName = expName;
		}

		public int getExpId() {
			return expId;
		}

		public boolean isGranted() {
			return granted;
		}

		public void setGranted(boolean granted) {
			this.granted = granted;
		}

		public String getExpName() {
			return expName;
		}

		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + expId;
			return result;
		}

		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			ProjectPrivilegeStatus other = (ProjectPrivilegeStatus) obj;
			if (expId != other.expId)
				return false;
			return true;
		}
	}// ;

	public List<PrivilegeExperiments> getPrivExperiments() {
		return privExperiments;
	}

	public void setPrivExperiments(List<PrivilegeExperiments> privExperiments) {
		this.privExperiments = privExperiments;
	}

	public String getEmail() {
		return email;
	}

	public void setEmail(String email) {
		this.email = email;
	}

	public boolean isInvited() {
		return invited;
	}

	public void setInvited(boolean invited) {
		this.invited = invited;
	}

	public UserPrivilegeSet getUserPrivSet() {
		return userPrivSet;
	}

	public void setUserPrivSet(UserPrivilegeSet userPrivSet) {
		this.userPrivSet = userPrivSet;
	}

}
