package clinical.web.common.vo;

import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.utils.Assertion;
import clinical.utils.DateTimeUtils;
import clinical.utils.GenUtils;
import clinical.web.workflow.common.JSONUtils;

/**
 * This class represents a record for the process status and summary results for
 * a CBF processable ASL image series.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: CBFProcessReportRec.java 695 2012-10-08 23:37:45Z bozyurt $
 */
public class CBFProcessReportRec {
	private String projectName;
	private String subjectID;
	private boolean processed;
	private String failureReason;
	private String tag;
	/** research group */
	private String condition;
	private String cbfImagePath;
	private String histogramImagePath;

	private String csfMasksImagePath;
	private String csfSlicesImagePath;
	private String motionImagePath;
	private String reportContent;
	
	private String csfMapPVImagePath;
	private String csfMapTop5ImagePath;
	private String csfRoiPVImagePath;
	private String csfRoiTop5ImagePath;
	
	/** */
	private Double gmCBF;
	private String visitId;
	private Date visitDate;
	private String localID;
	private String jobID;
	private int jobUniqueID = -1;
	/** name describing the visit */
	private String visitName;
	private QualityMeasureType qmType = QualityMeasureType.NONE;
	private List<Param> provenanceParams;
	private Map<String, List<Param>> otherRunsParamMap = null;
	public final static Pattern orderPattern = Pattern.compile("(\\d+)\\)");

	public static enum QualityMeasureType {
		NONE(0), CBF(1), SEGMENTATION(2), CBF_AND_SEG(3);

		private final int code;

		private QualityMeasureType(int code) {
			this.code = code;
		}

		public int getCode() {
			return code;
		}
	}

	public CBFProcessReportRec() {
	}

	public void addProvenanceParam(Param param) {
		if (provenanceParams == null) {
			provenanceParams = new ArrayList<Param>(5);
		}
		provenanceParams.add(param);
	}

	public void putParam(String runKey, Param param) {
		if (otherRunsParamMap == null) {
			otherRunsParamMap = new LinkedHashMap<String, List<Param>>(7);
		}
		List<Param> list = otherRunsParamMap.get(runKey);
		if (list == null) {
			list = new ArrayList<Param>(5);
			otherRunsParamMap.put(runKey, list);
		}
		list.add(param);
	}

	public String getKey() {
		StringBuilder sb = new StringBuilder();
		sb.append(projectName).append(":").append(subjectID);
		sb.append(":").append(visitId);
		return sb.toString();
	}

	public int getOrder() {
		Matcher matcher = orderPattern.matcher(getTag());
		if (matcher.find()) {
			String numStr = matcher.group(1);
			int order = GenUtils.toInt(numStr, -1);
			Assertion.assertTrue(order >= 0);
			return order;
		} else {
			return 0;
		}

	}

	public String getProjectName() {
		return projectName;
	}

	public void setProjectName(String projectName) {
		this.projectName = projectName;
	}

	public String getSubjectID() {
		return subjectID;
	}

	public void setSubjectID(String subjectID) {
		this.subjectID = subjectID;
	}

	public boolean isProcessed() {
		return processed;
	}

	public void setProcessed(boolean processed) {
		this.processed = processed;
	}

	public String getFailureReason() {
		return failureReason;
	}

	public void setFailureReason(String failureReason) {
		this.failureReason = failureReason;
	}

	public String getTag() {
		return tag;
	}

	public void setTag(String tag) {
		this.tag = tag;
	}

	public String getCondition() {
		return condition;
	}

	public void setCondition(String condition) {
		this.condition = condition;
	}

	public String getCbfImagePath() {
		return cbfImagePath;
	}

	public void setCbfImagePath(String cbfImagePath) {
		this.cbfImagePath = cbfImagePath;
	}

	public String getHistogramImagePath() {
		return histogramImagePath;
	}

	public void setHistogramImagePath(String histogramImagePath) {
		this.histogramImagePath = histogramImagePath;
	}

	public Double getGmCBF() {
		return gmCBF;
	}

	public void setGmCBF(Double gmCBF) {
		this.gmCBF = gmCBF;
	}

	public String getVisitId() {
		return visitId;
	}

	public void setVisitId(String visitId) {
		this.visitId = visitId;
	}

	public Date getVisitDate() {
		return visitDate;
	}

	public void setVisitDate(Date visitDate) {
		this.visitDate = visitDate;
	}

	public String getLocalID() {
		return localID;
	}

	public void setLocalID(String localID) {
		this.localID = localID;
	}

	public JSONObject toJSON() throws JSONException {
		JSONObject js = new JSONObject();
		js.put("project", projectName);
		js.put("subjectID", subjectID);
		js.put("processed", processed);
		js.put("tag", tag);
		js.put("condition", condition);
		
		JSONUtils.addStringField(js, "cbfImg", cbfImagePath, "NA");
		JSONUtils.addStringField(js, "histImg", histogramImagePath, "NA");
		
		JSONUtils.addStringField(js, "csfMasks", csfMasksImagePath, "");
		JSONUtils.addStringField(js, "csfSlices", csfSlicesImagePath, "");
		JSONUtils.addStringField(js, "motionImg", motionImagePath, "");
		JSONUtils.addStringField(js, "csfMapPV", csfMapPVImagePath, "");
		JSONUtils.addStringField(js, "csfMapTop5", csfMapTop5ImagePath, "");
		JSONUtils.addStringField(js, "csfRoiPV", csfRoiPVImagePath, "");
		JSONUtils.addStringField(js, "csfRoiTop5", csfRoiTop5ImagePath, "");
		
		if (gmCBF != null) {
			js.put("gmCBF", gmCBF.toString());
		} else {
			js.put("gmCBF", "NA");
		}
		if (visitDate != null) {
			js.put("visitDate", DateTimeUtils.formatDate(visitDate));
		} else {
			js.put("visitDate", "NA");
		}
		JSONUtils.addStringField(js, "localID", localID, "NA");
	
		if (jobID != null) {
			js.put("jobID", jobID);
		}
		if (jobUniqueID >= 0) {
			js.put("juid", jobUniqueID);
		}

		if (visitName != null) {
			js.put("visitName", visitName);
		}
		if (qmType != null) {
			js.put("qmt", qmType.getCode());
		}
		JSONArray jsarr = new JSONArray();
		js.put("provArr", jsarr);
		if (provenanceParams != null) {
			for (Param p : provenanceParams) {
				jsarr.put(p.toJSON());
			}
		}
		if (otherRunsParamMap != null) {
			JSONArray tmpArr = new JSONArray();
			js.put("others", tmpArr);
			for (Map.Entry<String, List<Param>> entry : otherRunsParamMap
					.entrySet()) {
				String key = entry.getKey();
				List<Param> list = entry.getValue();
				tmpArr.put(paramsToJSON(key, list));
			}
			int noRuns = otherRunsParamMap.size() + 1;
			js.put("noRuns", noRuns);
		} else {
			JSONArray tmpArr = new JSONArray();
			js.put("others", tmpArr);
			js.put("noRuns", 1);
		}

		if (reportContent != null) {
			js.put("report", reportContent);
		}
		return js;
	}

	private JSONObject paramsToJSON(String key, List<Param> params)
			throws JSONException {
		JSONObject js = new JSONObject();
		js.put("key", key);
		JSONArray jsArr = new JSONArray();
		js.put("pr", jsArr);
		for (Param p : params) {
			jsArr.put(p.toJSON());
		}
		return js;
	}

	public String toString() {
		StringBuilder sb = new StringBuilder(256);
		sb.append("CBFProcessReportRec::[");
		sb.append("project:").append(projectName);
		sb.append(",subjectId:").append(subjectID);
		sb.append(",processed:").append(processed);
		sb.append(",tag:").append(tag);
		sb.append(",condition:").append(condition);
		if (this.histogramImagePath != null) {
			sb.append("\n\t").append(histogramImagePath);
		}
		if (this.cbfImagePath != null) {
			sb.append("\n\t").append(cbfImagePath).append("\n");
		}
		if (csfMasksImagePath != null) {
			sb.append("\n\t").append(csfMasksImagePath).append("\n");
		}
		if (csfSlicesImagePath != null) {
			sb.append("\n\t").append(csfSlicesImagePath).append("\n");
		}
		if (motionImagePath != null) {
			sb.append("\n\t").append(motionImagePath).append("\n");
		}
		sb.append(",localid:").append(localID);
		if (visitName != null) {
			sb.append(",visitName:").append(visitName);
		}
		if (qmType != null) {
			sb.append(",qmType:").append(qmType);
		}
		if (provenanceParams != null) {
			for (Param p : provenanceParams) {
				sb.append("\n\t").append(p);
			}
		}

		if (otherRunsParamMap != null) {
			for (Map.Entry<String, List<Param>> entry : otherRunsParamMap
					.entrySet()) {
				String key = entry.getKey();
				sb.append("\n\t").append(key).append(":");
				List<Param> list = entry.getValue();
				for (Param p : list) {
					sb.append("\n\t\t").append(p);
				}
			}
		}
		sb.append(']');
		return sb.toString();
	}

	public static enum ParamType {
		STRING, BOOL, NUMBER, DATE
	};

	public static class Param {
		private String name;
		private String value;
		private ParamType type;

		public Param(String name, String value, ParamType type) {
			super();
			this.name = name;
			this.value = value;
			this.type = type;
		}

		public String getName() {
			return name;
		}

		public String getValue() {
			return value;
		}

		public ParamType getType() {
			return type;
		}

		public JSONObject toJSON() throws JSONException {
			JSONObject js = new JSONObject();
			js.put("name", name);
			if (type == ParamType.STRING || type == ParamType.DATE) {
				js.put("val", value);
			} else if (type == ParamType.BOOL) {
				js.put("val", value.equalsIgnoreCase("yes"));
			} else {
				js.put("val", GenUtils.toDouble(value, -1.0));
			}
			return js;
		}

		@Override
		public String toString() {
			StringBuilder sb = new StringBuilder();
			sb.append("Param::[");
			sb.append("name:").append(name);
			sb.append(",value:").append(value);
			sb.append(']');
			return sb.toString();
		}
	}

	public List<Param> getProvenanceParams() {
		return provenanceParams;
	}

	public Map<String, List<Param>> getOtherRunsParamMap() {
		return otherRunsParamMap;
	}

	public String getCsfMasksImagePath() {
		return csfMasksImagePath;
	}

	public void setCsfMasksImagePath(String csfMasksImagePath) {
		this.csfMasksImagePath = csfMasksImagePath;
	}

	public String getCsfSlicesImagePath() {
		return csfSlicesImagePath;
	}

	public void setCsfSlicesImagePath(String csfSlicesImagePath) {
		this.csfSlicesImagePath = csfSlicesImagePath;
	}

	public String getJobID() {
		return jobID;
	}

	public void setJobID(String jobID) {
		this.jobID = jobID;
	}

	public int getJobUniqueID() {
		return jobUniqueID;
	}

	public void setJobUniqueID(int jobUniqueID) {
		this.jobUniqueID = jobUniqueID;
	}

	public String getVisitName() {
		return visitName;
	}

	public void setVisitName(String visitName) {
		this.visitName = visitName;
	}

	public QualityMeasureType getQmType() {
		return qmType;
	}

	public void setQmType(QualityMeasureType qmType) {
		this.qmType = qmType;
	}

	public String getMotionImagePath() {
		return motionImagePath;
	}

	public void setMotionImagePath(String motionImagePath) {
		this.motionImagePath = motionImagePath;
	}

	public String getReportContent() {
		return reportContent;
	}

	public void setReportContent(String reportContent) {
		this.reportContent = reportContent;
	}

	public String getCsfMapPVImagePath() {
		return csfMapPVImagePath;
	}

	public void setCsfMapPVImagePath(String csfMapPVImagePath) {
		this.csfMapPVImagePath = csfMapPVImagePath;
	}

	public String getCsfMapTop5ImagePath() {
		return csfMapTop5ImagePath;
	}

	public void setCsfMapTop5ImagePath(String csfMapTop5ImagePath) {
		this.csfMapTop5ImagePath = csfMapTop5ImagePath;
	}

	public String getCsfRoiPVImagePath() {
		return csfRoiPVImagePath;
	}

	public void setCsfRoiPVImagePath(String csfRoiPVImagePath) {
		this.csfRoiPVImagePath = csfRoiPVImagePath;
	}

	public String getCsfRoiTop5ImagePath() {
		return csfRoiTop5ImagePath;
	}

	public void setCsfRoiTop5ImagePath(String csfRoiTop5ImagePath) {
		this.csfRoiTop5ImagePath = csfRoiTop5ImagePath;
	}

}
