package clinical.web.download;

import java.io.File;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import clinical.utils.BIRNURIException;
import clinical.utils.GenUtils;

/**
 * @author I. Burak Ozyurt
 * @version $Id: ProjectImageDataBundle.java 366 2011-05-05 20:06:27Z bozyurt $
 */
public class ProjectImageDataBundle implements Serializable {
	private static final long serialVersionUID = -5890774343546226295L;
	protected String experimentName;
	protected String treeId;
	protected String description;
	protected DataNode root;
	protected Map<String,DataNode> pathMap = new HashMap<String,DataNode>(7);
	/** everything under a ProjectImageDataBundle is assumed to have the same root directory*/
	protected String rootDir;
	
	public ProjectImageDataBundle(String experimentName, String treeId, String description) {
		super();
		this.experimentName = experimentName;
		this.treeId = treeId;
		this.description = description;
		this.root = new DataNode(experimentName, experimentName, "experiment");
	}
	
	public void addSubjectVisitFile(String dataURI, long dataSize, String description) throws BIRNURIException {
		File f = new File(dataURI);
		String title = f.getName();
		FileInfo fi = null;
		String suffix = "";
		String pathKey = dataURI;
		int idx = title.lastIndexOf('.');
		if ( idx != -1) {
			suffix = title.substring(idx);
		}
		if ( title.endsWith(".tgz") ) {
			title = title.replaceFirst("\\.tgz$", "");
			pathKey = pathKey.replaceFirst("\\.tgz$", "");
			fi = new FileInfo(dataURI, FileInfo.TGZ, FileInfo.LOCAL, null, false);
		} else if ( title.endsWith(".tar")) {
			title = title.replaceFirst("\\.tar$", "");
			pathKey = pathKey.replaceFirst("\\.tar$", "");
			fi = new FileInfo(dataURI, FileInfo.TAR, FileInfo.LOCAL, null, false);
		} else if (title.endsWith("tar.gz")) {
			title = title.replaceFirst("\\.tar\\.gz$", "");
			pathKey = pathKey.replaceFirst("\\.tar\\.gz$", "");			
			fi = new FileInfo(dataURI, FileInfo.TAR_GZ, FileInfo.LOCAL, null, false);
	    } else {
	    	throw new RuntimeException("Not a support project image download type:" + suffix);
	    }
		if ( description != null) {
			fi.addMetaData("description", description);
		}
		fi.setFileSize(dataSize);
		String rd = f.getParent();
		if ( rootDir == null) {
			rootDir = rd;
		} 
		
		String formattedDataSize = GenUtils.formatFileSize(dataSize);
		title += " (" + formattedDataSize + ")";
		DataNode child = new DataNode(title, fi, "file");
		
		root.addChild(child);
		pathMap.put(pathKey, child);
	}
	
	public DataNode findByPath(String path) {
		if (!path.startsWith("/")) {
			path = rootDir + "/" + path;
		}
		return (DataNode) pathMap.get(path);
	}
	
	public String getJavascriptTree() {
		return buildJSTree();
	}
	
	public String buildJSTree() {
		StringBuffer buf = new StringBuffer(2048);
		buf.append("var ").append(treeId).append(" = new dTree('").append(
				treeId).append("'); ").append("\n");
		buf.append(treeId).append(".config.useIcons=false; ");
		buf.append(treeId).append(".config.useCookies=false; ").append('\n');
		IDGenerator idGen = new IDGenerator();
		int labelNodeId = idGen.nextID();
		buf.append(treeId).append(".add(").append(labelNodeId).append(',').append(
				-1);
		buf.append(",'Project',null,null,null);\n");
		
		build(root, idGen, labelNodeId, buf);
		buf.append("document.write(").append(treeId).append("); ");
		buf.append(treeId).append(".registerCBObservers();\n");
		buf.append(treeId).append(".registerCBEventListener(totalSizeHandler);\n");

		return buf.toString();
	}
	
	protected void build(DataNode node, IDGenerator idGen, int parentId,
			StringBuffer buf) {
		int nodeId = idGen.nextID();
		buf.append(treeId).append(".add(").append(nodeId).append(',').append(
				parentId);
		buf.append(",'").append(node.getTitle()).append("'");			
		
		if ( node.getUserObject() instanceof FileInfo) {	
			FileInfo fi = (FileInfo) node.getUserObject(); 
			
			buf.append(",null,").append(fi.getFileSize());
			buf.append(",null,null,null,null,null,null,null,null,");
			if ( fi.hasMetaData()) {
				buf.append("'").append(fi.getMetaData("description")).append("'");			
			} else {
				buf.append("null");
			}
			buf.append(");\n");
		} else {
			buf.append(");\n");
		}
		if (node.hasChildren()) {
			for (DataNode child : node.getChildren()) {
				build(child, idGen, nodeId, buf);
			}
		}
	}

	public String getExperimentName() {
		return experimentName;
	}

	public String getTreeId() {
		return treeId;
	}

	public DataNode getRoot() {
		return root;
	}

	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((experimentName == null) ? 0 : experimentName.hashCode());
		return result;
	}

	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final ProjectImageDataBundle other = (ProjectImageDataBundle) obj;
		if (experimentName == null) {
			if (other.experimentName != null)
				return false;
		} else if (!experimentName.equals(other.experimentName))
			return false;
		return true;
	}

	public String getDescription() {
		return description;
	}

}
