package clinical.web.scheduler;

import java.io.File;
import java.util.List;

import clinical.web.common.UserInfo;
import clinical.web.scheduler.remote.WFESLocation;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: IJob.java 384 2011-06-23 00:15:36Z bozyurt $
 */
public interface IJob {

	/** constant for a short duration process (less than a minute) */
	public final static int SHORT = 1;

	/**
	 * constant for a medium duration process (more than a minute and less than
	 * an hour)
	 */
	public final static int MEDIUM = 2;

	/** constant for a long duration process (more than an hour) */
	public final static int LONG = 3;

	/**
	 * returns the unique id of the job
	 * 
	 * @return
	 */
	public String getID();

	/**
	 * 
	 * @return
	 */
	public String getUser();

	/**
	 * returns the job type
	 * 
	 * @return
	 */
	public String getType();

	/**
	 * 
	 * @return
	 */
	public String getDescription();

	/**
	 * returns the number of processing stages. Each stage is atomic unit of
	 * work not needing any user interaction.
	 */
	public int getNumberOfStages();

	/**
	 * main work is done here.
	 * 
	 * @throws JobException
	 */
	public void execute(int stageId) throws JobException;

	/**
	 * 
	 * @param file
	 * @throws JobException
	 */
	public void saveResults(File file) throws JobException;

	/**
	 * the full paths for the files where the results will be saved
	 * 
	 * @return
	 */
	public String[] getResultsFiles();

	/**
	 * in order a job to be cancelable this method needs to set a flag when
	 * called and the logic in <code>execute()</code> should check for this flag
	 * periodically.
	 * 
	 */
	public void cancel();

	public boolean isCanceled();

	public String getDbID();

	public UserInfo getUserInfo();

	/**
	 * 
	 * @return either <code>IJob.SHORT</code>, <code>IJob.MEDIUM</code> or
	 *         <code>IJob.LONG</code>
	 */
	public int getDurationType();

	public void addJobEventListener(IJobEventListener listener);

	/**
	 * called before permanent removal of the job from job queue.
	 */
	public void shutdown();

	/**
	 * returns zero or more {@link JobVisitContext} objects.
	 * 
	 * @return zero or more {@link JobVisitContext} objects.
	 */
	public List<JobVisitContext> getJobVisitContextList();

	/**
	 * If a context is attached to the job (used for user-interaction and/or
	 * persistent jobs surviving server restart) returns it as string in JSON
	 * representation, or null otherwise.
	 * 
	 * @return
	 */
	public String getContextAsJSON();

	/**
	 * 
	 * @return
	 */
	public IJobFactory getJobFactory();

	public void setStatus(String status);

	public String getStatus();

	/**
	 * allows the job to cleanup its temporary files etc. Called by the job
	 * scheduler at the end of the job.
	 */
	public void cleanup();

	/**
	 * sets remote location information (address) for an available slave worker.
	 * This signals the particular job to use the remote server for load
	 * sharing.
	 * 
	 * @param location
	 * @return true if the job can ship all or part of it workload to a remote
	 *         slave worker
	 */
	public boolean setWFESLocation(WFESLocation location);
}
