package clinical.web.scheduler;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.utils.FileUtils;
import clinical.web.IJobManagementService;
import clinical.web.ServiceFactory;
import clinical.web.common.ICachePolicy;
import clinical.web.common.UserInfo;
import clinical.web.exception.BaseException;

/**
 * This class is responsible for scheduling of asynchronous jobs submitted via
 * the web interface. The asynchronous jobs are mainly used for long running
 * OLAP like queries and other long running tasks.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: JobScheduler.java 757 2012-12-12 01:12:45Z bozyurt $
 */
public class JobScheduler implements Runnable, IJobEventListener {
	private List<JobInfo> jobQueue = Collections
			.synchronizedList(new ArrayList<JobInfo>());
	private List<JobInfo> batchJobQueue = Collections
			.synchronizedList(new ArrayList<JobInfo>());
	private Map<String, ThreadInfo> threadMap = Collections
			.synchronizedMap(new HashMap<String, ThreadInfo>(7));
	private Map<String, IJob> jobIDMap = Collections
			.synchronizedMap(new HashMap<String, IJob>());
	private Map<String, JobRecord> jobRecordMap = Collections
			.synchronizedMap(new HashMap<String, JobRecord>());
	private ICachePolicy policy;
	private Map<String, IJobFactory> factoryMap = Collections
			.synchronizedMap(new HashMap<String, IJobFactory>(17));

	/**
	 * the root directory where the finished processes will write their
	 * results/logs.
	 */
	private String outputRootDir;
	private final int maxConcurrentJobs = 20;
	private final int maxConcurrentLongJobs = 10;
	private final int maxConcurrentBatchJobs = 2;

	private Log log = LogFactory.getLog(JobScheduler.class);

	private static JobScheduler instance = null;

	protected JobScheduler(String outputRootDir, ICachePolicy policy)
			throws JobSchedulerException {
		super();
		this.outputRootDir = outputRootDir;
		this.policy = policy;
	}

	public static synchronized JobScheduler getInstance(String outputRootDir,
			ICachePolicy policy) throws JobSchedulerException {
		if (instance == null) {
			instance = new JobScheduler(outputRootDir, policy);
		}
		return instance;
	}

	public static synchronized JobScheduler getInstance()
			throws JobSchedulerException {
		if (instance == null) {
			throw new JobSchedulerException("No proper initialization!");
		}
		return instance;
	}

	public void shutdown() {
		// no op
	}

	public synchronized void registerJobFactory(String jobType,
			IJobFactory jobFactory) {
		if (jobType != null && jobFactory != null) {
			factoryMap.put(jobType, jobFactory);
		}
	}

	public synchronized IJobFactory getJobFactory(String jobType) {
		return factoryMap.get(jobType);
	}

	public synchronized void recoverJob(IJob job) throws BaseException {
		if (!jobIDMap.containsKey(job.getID())) {
			JobInfo jobInfo = null;
			try {
				jobIDMap.put(job.getID(), job);
				jobInfo = new JobInfo(job);
				jobInfo.setStatus(JobInfo.WAITING);
				job.setStatus(JobInfo.WAITING);
				log.info("recovering job with id:" + job.getID());
				String jobID = job.getID();
				JobRunner runner = new JobRunner(this, jobInfo);
				log.info("*** starting recovered job:"
						+ jobInfo.getJob().getID());

				// register to listen to job event messages
				jobInfo.getJob().addJobEventListener(this);

				// add to the job queue
				jobQueue.add(jobInfo);

				Thread thread = new Thread(runner);
				thread.setDaemon(true);
				thread.setPriority(Thread.NORM_PRIORITY - 1);
				ThreadInfo ti = new ThreadInfo(jobInfo.job.getDurationType(),
						runner, JobSubmissionType.INDIVIDUAL);
				threadMap.put(jobID, ti);
				thread.start();
				log.info("# of concurrent job threads:" + threadMap.size());
				showCurrentJobs();

			} catch (Throwable e) {
				e.printStackTrace();
				if (jobInfo != null) {
					jobInfo.setStatus(JobInfo.FINISHED_WITH_ERR);
					job.setStatus(JobInfo.FINISHED_WITH_ERR);
					jobInfo.setErrorMsg(e.getMessage());
					try {
						updateJobDB(jobInfo, job.getID());
					} catch (Exception x) {
					}
				}
			}
			// this.notify();
		}
	}

	public synchronized void addJob(IJob job, JobSubmissionType submissionType)
			throws BaseException {
		if (!jobIDMap.containsKey(job.getID())) {
			try {
				IJobFactory jobFactory = factoryMap.get(job.getType());
				if (jobFactory == null) {
					try {
						jobFactory = job.getJobFactory();
						factoryMap.put(job.getType(), jobFactory);
					} catch (UnsupportedOperationException e) {
						factoryMap.put(job.getType(), new NullJobFactory());
					}
				}

				jobIDMap.put(job.getID(), job);
				JobInfo jobInfo = new JobInfo(job);

				if (submissionType == JobSubmissionType.INDIVIDUAL) {
					jobQueue.add(jobInfo);
				} else {
					batchJobQueue.add(jobInfo);
				}
				IJobManagementService jms = ServiceFactory
						.getJobManagementService(job.getDbID());
				JobRecord jr = new JobRecord(job.getUser(), job.getType(), job
						.getID(), new Date(), JobInfo.NOT_STARTED, job
						.getDescription(), job.getResultsFiles(), true);

				List<JobVisitContext> jvcList = job.getJobVisitContextList();
				if (!jvcList.isEmpty()) {
					for (JobVisitContext jvc : jvcList) {
						jr.addVisitContext(jvc);
					}
				}

				// if there is a job context store it with the JobRecord
				String contextAsJSON = job.getContextAsJSON();
				if (contextAsJSON != null) {
					jr.setContext(contextAsJSON);
				}
				jms.addJob(job.getUserInfo(), jr);

				this.notify();
			} catch (Throwable t) {
				if (t instanceof BaseException) {
					throw (BaseException) t;
				} else {
					throw new BaseException(t);
				}
			}
		}
	}

	protected String prepKey(JobRecord jr) {
		StringBuilder buf = new StringBuilder();
		buf.append(jr.jobID).append('_').append(jr.user);
		return buf.toString();
	}

	protected String prepKey(String jobUser, String jobID) {
		StringBuilder buf = new StringBuilder();
		buf.append(jobID).append('_').append(jobUser);
		return buf.toString();
	}

	public JobRecord getJobRecord(UserInfo ui, String dbID, String user,
			String jobID) throws BaseException {
		IJobManagementService jms = ServiceFactory
				.getJobManagementService(dbID);
		JobRecord jr = jms.findJob(ui, user, jobID);
		return jr;
	}

	public File getResultsFileForJob(String user, String jobID) {
		return new File(getOutputRootDir(), jobID + ".csv");
	}

	public List<JobRecord> getJobsForUser(UserInfo ui, String dbID, String user)
			throws BaseException {
		IJobManagementService jms = ServiceFactory
				.getJobManagementService(dbID);

		List<JobRecord> jrList = jms.getJobsForUser(ui, user);
		for (Iterator<JobRecord> it = jrList.iterator(); it.hasNext();) {
			JobRecord jr = it.next();
			if (jr.getStatus().endsWith("_expired")) {
				it.remove();
			}
		}
		Collections.sort(jrList, new Comparator<JobRecord>() {
			public int compare(JobRecord jr1, JobRecord jr2) {
				return jr2.date.compareTo(jr1.date);
			}
		});

		return jrList;
	}

	public void run() {
		long checkInterval = 5000L;
		while (true) {
			synchronized (this) {
				try {
					this.wait(checkInterval);
				} catch (InterruptedException e) {
				}
			}
			if (jobQueue.isEmpty() && batchJobQueue.isEmpty()) {
				continue;
			}

			synchronized (this) {
				showIfLeaks();
				// first cleanup any finished or canceled jobs
				cleanupFinishedCanceledJobsInQueue(jobQueue, "job queue");
				cleanupFinishedCanceledJobsInQueue(batchJobQueue,
						"batch job queue");

				boolean hasEnoughSpace = hasEnoughSpace();
				processJobsFromQueue(hasEnoughSpace, jobQueue, false);
				processJobsFromQueue(hasEnoughSpace, batchJobQueue, true);
			} // synchronized
		} // while
	}

	private final void processJobsFromQueue(boolean hasEnoughSpace,
			List<JobInfo> queue, boolean batchMode) {
		@SuppressWarnings("unused")
		int numOfStarted = 0;
		for (Iterator<JobInfo> iter = queue.iterator(); iter.hasNext();) {
			JobInfo jobInfo = iter.next();
			String jobID = jobInfo.getJob().getID();
			if (jobInfo.getStatus().equals(JobInfo.NOT_STARTED)) {
				if (threadMap.size() < maxConcurrentJobs && hasEnoughSpace) {
					if (jobInfo.getJob().getDurationType() == IJob.LONG
							&& !canRunMoreLongJobs(batchMode)) {
						// no space for an another long running job, so
						// skip the job
						continue;
					}
					JobRunner runner = new JobRunner(this, jobInfo);
					if (batchMode) {
						log.info("*** starting batch job "
								+ jobInfo.getJob().getID());
					} else {
						log.info("*** starting job "
										+ jobInfo.getJob().getID());
					}
					jobInfo.setStatus(JobInfo.RUNNING);
					try {
						// register to listen to job event messages
						jobInfo.getJob().addJobEventListener(this);

						updateJobDB(jobInfo, jobID);
						Thread thread = new Thread(runner);
						thread.setDaemon(true);
						thread.setPriority(Thread.NORM_PRIORITY - 1);
						ThreadInfo ti = new ThreadInfo(jobInfo.job
								.getDurationType(), runner,
								batchMode ? JobSubmissionType.BATCH
										: JobSubmissionType.INDIVIDUAL);
						threadMap.put(jobID, ti);
						thread.start();
						jobInfo.setStatus(JobInfo.RUNNING);
						numOfStarted++;
						updateJobDB(jobInfo, jobID);
						System.out.println("# of concurrent job threads:"
								+ threadMap.size());
						showCurrentJobs();
					} catch (BaseException e) {
						e.printStackTrace();
						jobInfo.setStatus(JobInfo.FINISHED_WITH_ERR);
						jobInfo.setErrorMsg(e.getMessage());
						try {
							updateJobDB(jobInfo, jobID);
							jobInfo.getJob().cleanup();
						} catch (Exception x) {
						}
					}
				}
			} else if (jobInfo.getStatus().equals(JobInfo.FINISHED)
					|| jobInfo.getStatus().equals(JobInfo.FINISHED_WITH_ERR)
					|| jobInfo.getStatus().equals(JobInfo.CANCELED)
					|| jobInfo.getStatus().equals(JobInfo.SHELVED)) {
				try {
					jobIDMap.remove(jobID);
					updateJobDB(jobInfo, jobID);
					if (!jobInfo.getStatus().equals(JobInfo.SHELVED)) {
						jobInfo.getJob().cleanup();
					}
				} catch (BaseException e) {
					e.printStackTrace();
				} finally {
					iter.remove();
					threadMap.remove(jobID);
					log.info("# of concurrent job threads (3) :"
							+ threadMap.size());
					showCurrentJobs();
				}
			}
		}
	}

	private final void cleanupFinishedCanceledJobsInQueue(List<JobInfo> queue,
			String queueName) {
		for (Iterator<JobInfo> iter = queue.iterator(); iter.hasNext();) {
			JobInfo jobInfo = iter.next();
			String jobID = jobInfo.getJob().getID();

			if (jobInfo.getStatus().equals(JobInfo.FINISHED)
					|| jobInfo.getStatus().equals(JobInfo.FINISHED_WITH_ERR)
					|| jobInfo.getStatus().equals(JobInfo.CANCELED)
					|| jobInfo.getStatus().equals(JobInfo.REMOVED)
					|| jobInfo.getStatus().equals(JobInfo.SHELVED)) {
				try {
					jobIDMap.remove(jobID);
					updateJobDB(jobInfo, jobID);
					if (!jobInfo.getStatus().equals(JobInfo.SHELVED)) {
						jobInfo.getJob().cleanup();
					}
				} catch (BaseException e) {
					log.error("", e);
				} finally {
					iter.remove();
					log.info("removed job " + jobInfo.getJob().getID()
							+ " for user:" + jobInfo.getJob().getUser()
							+ " from " + queueName + ". (status:"
							+ jobInfo.getStatus() + ")");
					threadMap.remove(jobID);
					log.info("# of concurrent job threads (2) :"
							+ threadMap.size());
					showCurrentJobs();
				}
			}
		}
	}

	private void showIfLeaks() {
		boolean needsDisplay = false;
		for (ThreadInfo ti : threadMap.values()) {
			String status = ti.jr.jobInfo.getStatus();
			if (!status.equals(JobInfo.RUNNING)
					&& !status.equals(JobInfo.WAITING)) {
				needsDisplay = true;
				break;
			}
		}
		if (needsDisplay) {
			showCurrentJobs();
		}
	}

	private void showCurrentJobs() {
		log.info("Job IDs currently in threadMap");
		log.info("======================");
		for (String jobID : threadMap.keySet()) {
			log.info(jobID);
		}
		log.info("======================");
	}

	protected boolean canRunMoreLongJobs(boolean batchMode) {
		int longJobCount = 0;
		int batchJobCount = 0;
		for (ThreadInfo ti : threadMap.values()) {
			if (ti.durationType == IJob.LONG) {
				longJobCount++;
			}
			if (ti.type == JobSubmissionType.BATCH) {
				batchJobCount++;
			}
		}
		boolean ok = longJobCount < maxConcurrentLongJobs;
		if (ok) {
			if (batchMode) {
				return batchJobCount < maxConcurrentBatchJobs;
			}
		}
		return ok;
	}

	public void updateJob(UserInfo ui, String dbID, JobRecord jr)
			throws BaseException {
		IJobManagementService jms = ServiceFactory
				.getJobManagementService(dbID);
		jms.updateJob(ui, jr);
	}

	public void updateJobDB(JobInfo jobInfo, String jobID) throws BaseException {
		try {
			String user = jobInfo.getJob().getUser();
			String jobType = jobInfo.getJob().getType();
			String key = jobID + "_" + user;

			UserInfo ui = jobInfo.getJob().getUserInfo();
			IJobManagementService jms = ServiceFactory
					.getJobManagementService(jobInfo.getJob().getDbID());
			JobRecord jr = jms.findJob(ui, user, jobID);

			if (jr == null) {
				jr = new JobRecord(user, jobType, jobID, new Date(), jobInfo
						.getStatus(), jobInfo.getJob().getDescription(),
						jobInfo.getJob().getResultsFiles(), true);
				if (jobInfo.getErrorMsg() != null) {
					jr.setErrorMsg(jobInfo.getErrorMsg());
				}
				String contextAsJSON = jobInfo.getJob().getContextAsJSON();
				if (contextAsJSON != null) {
					jr.setContext(contextAsJSON);
				}
				jms.addJob(ui, jr);
			} else {
				jr.setStatus(jobInfo.getStatus());
				jr.setDate(new Date());
				if (jobInfo.getErrorMsg() != null) {
					jr.setErrorMsg(jobInfo.getErrorMsg());
				}
				if (JobInfo.FINISHED.equals(jr.getStatus())
						&& jr.getSavedResultFiles() != null) {
					long jobSize = 0;
					// FIXME total job size
					for (int i = 0; i < jr.getSavedResultFiles().length; i++) {
						File f = new File(jr.getSavedResultFiles()[i]);
						if (f.isFile()) {
							jobSize += f.length();
						}
					}

					jr.setJobSize(jobSize);
				}
				if (JobInfo.WAITING.equals(jr.getStatus())) {
					// for recovery
					String contextAsJSON = jobInfo.getJob().getContextAsJSON();
					if (contextAsJSON != null) {
						jr.setContext(contextAsJSON);
					}
				}
				jr.setDateFinished(new Date());

				jms.updateJob(ui, jr);
			}

			jobRecordMap.put(key, jr);
		} catch (Throwable t) {
			t.printStackTrace();
			throw new BaseException(t);
		}
	}

	String getOutputRootDir() {
		return outputRootDir;
	}

	// 8/2/07
	public void removeJob(UserInfo ui, String dbID, String user, String jobID)
			throws BaseException {
		log.info("**** removeJob");
		String key = jobID + "_" + user;
		IJobManagementService jms = ServiceFactory
				.getJobManagementService(dbID);

		synchronized (this) {
			JobRecord jrLocal = jobRecordMap.get(key);
			if (jrLocal != null) {
				jobRecordMap.remove(key);
			}
		}
		JobRecord jr = jms.findJob(ui, user, jobID);
		if (jr != null) {
			log.info("Removing job:" + jr.toString());
			jr.setStatus(JobInfo.REMOVED);
			jms.updateJob(ui, jr);
			cleanupCache(jobID);
		}
	}

	public synchronized void cleanupCache(String jobID) {
		File dir = new File(outputRootDir, jobID);
		FileUtils.deleteSubdirs(outputRootDir, dir.getAbsolutePath());
		FileUtils.deleteRecursively(dir);
	}

	public void resumeJob(UserInfo ui, String dbID, String user, String jobID)
			throws BaseException {
		ThreadInfo threadInfo = threadMap.get(jobID);
		log.info("threadInfo:" + threadInfo);
		if (threadInfo != null) {
			synchronized (threadInfo.jr) {
				log.info("notifying runner");
				threadInfo.jr.resume();
				threadInfo.jr.notify();
			}
		} else {
			IJobManagementService jms = ServiceFactory
					.getJobManagementService(dbID);
			JobRecord jr = jms.findJob(ui, user, jobID);
			if (jr == null) {
				log.warn("No valid JobRecord for job with ID:" + jobID
						+ ". Cannot resume!");
				return;
			}
			if (!jr.getStatus().equals(JobInfo.SHELVED)) {
				log.warn("Not a shelved job with ID:" + jobID
						+ ". Cannot resume!");
				return;
			}
			IJobFactory jobFactory = factoryMap.get(jr.getType());
			if (jobFactory == null || jobFactory instanceof NullJobFactory) {
				log.warn("No valid jobFactory for job with ID:" + jobID
						+ ". Cannot resume!");
				return;
			}
			try {
				IJob job = jobFactory.create(jr);
				recoverJob(job);
				threadInfo = threadMap.get(jobID);
				log.info("threadInfo:" + threadInfo);
				if (threadInfo != null) {
					synchronized (threadInfo.jr) {
						log.info("notifying runner for the resurrected job");
						threadInfo.jr.resume();
						threadInfo.jr.notify();
					}
				}
			} catch (JobException je) {
				log.error(je);
				throw new BaseException(je);
			}
		}
	}

	public void cancelJob(UserInfo ui, String dbID, String user, String jobID)
			throws BaseException {
		String key = jobID + "_" + user;
		JobRecord jrLocal = jobRecordMap.get(key);
		if (jrLocal == null) {
			return;
		}
		IJobManagementService jms = ServiceFactory
				.getJobManagementService(dbID);

		synchronized (this) {
			boolean found = cancelJobInQueue(user, jobID, jobQueue);
			if (!found) {
				cancelJobInQueue(user, jobID, batchJobQueue);
			}
			jobRecordMap.remove(key);
		}

		JobRecord jr = jms.findJob(ui, user, jobID);
		if (jr != null) {
			jr.setStatus(JobInfo.CANCELED);
			jms.updateJob(ui, jr);
		}
	}

	private final boolean cancelJobInQueue(String user, String jobID,
			List<JobInfo> queue) {
		boolean found = false;
		for (Iterator<JobInfo> iter = queue.iterator(); iter.hasNext();) {
			JobInfo jobInfo = iter.next();
			if (jobInfo.getJob().getID().equals(jobID)
					&& jobInfo.getJob().getUser().equals(user)) {
				jobInfo.setStatus(JobInfo.CANCELED);
				if (jobInfo.getStatus().equals(JobInfo.WAITING)) {
					jobInfo.getJob().cancel();
					ThreadInfo threadInfo = threadMap.get(jobID);
					if (threadInfo != null) {
						synchronized (threadInfo.jr) {
							log.info("************ Waking up canceled job");
							threadInfo.jr.notify();
						}
					}
				} else {
					jobInfo.getJob().cancel();
					log.info(">>>>> Canceled job notification to "
							+ "the scheduler...");
					this.notify();
				}
				found = true;
				break;
			}
		}
		return found;
	}

	protected boolean hasEnoughSpace() {
		return !policy.isBelowLowMark(new File(outputRootDir).getUsableSpace());
	}

	public static enum JobSubmissionType {
		INDIVIDUAL, BATCH
	};

	static class ThreadInfo {
		private final JobRunner jr;
		private final int durationType;
		private final JobSubmissionType type;

		public ThreadInfo(int durationType, JobRunner jr, JobSubmissionType type) {
			this.durationType = durationType;
			this.jr = jr;
			this.type = type;
		}

	}// ;

	@Override
	public void statusChanged(JobEvent event) {
		if (event.getStatusDetail() == null)
			return;
		IJob job = event.getSource();
		String key = job.getID() + "_" + job.getUser();
		synchronized (this) {
			JobRecord jrLocal = jobRecordMap.get(key);
			if (jrLocal != null) {
				jrLocal.setStatusDetail(event.getStatusDetail());
				try {
					IJobManagementService jms = ServiceFactory
							.getJobManagementService(job.getDbID());
					jms.updateJob(job.getUserInfo(), jrLocal);
				} catch (Exception x) {
					log.error("jobStatusChanged", x);
				}
			}
		}
	}

	public static class NullJobFactory implements IJobFactory {
		@Override
		public IJob create(JobRecord jr) throws JobException {
			return null;
		}
	}
}
