package clinical.web.services;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import oracle.jdbc.OracleCallableStatement;
import oracle.jdbc.OracleTypes;
import oracle.sql.ARRAY;
import oracle.sql.ArrayDescriptor;
import clinical.server.utils.OracleUtils;
import clinical.web.DBUtils;
import clinical.web.common.UserInfo;
import clinical.web.exception.BaseException;
import clinical.web.exception.SubjectAssessmentManagementException;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.AssessmentScoreValues;
import clinical.web.vo.ScoreValue;

/**
 * Provides subject assessment management service for Oracle databases.
 *
 * @author I. Burak Ozyurt
 * @version $Id: SubjectAssessmentManagementImpl.java,v 1.21 2006/03/30 01:58:25
 *          bozyurt Exp $
 */
public class SubjectAssessmentManagementImpl extends
		AbstractSubjectAssessmentManImpl {

	public SubjectAssessmentManagementImpl(String dbID) throws BaseException{
		super(dbID);
	}

	public List<AssessmentInfo> getAssessmentsForSubjectWithStoredProc(
			UserInfo ui, String subjectID, int experimentID)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		CallableStatement cst = null;
		List<AssessmentInfo> asList = new LinkedList<AssessmentInfo>();
		try {
			con = pool.getConnection(ui.getName());
			cst = con
					.prepareCall("{ ? = call assessment_man.get_assessments(?,?)}");
			cst.registerOutParameter(1, OracleTypes.CURSOR);
			cst.setString(2, subjectID);
			cst.setInt(3, experimentID);
			cst.execute();
			ResultSet rs = (ResultSet) cst.getObject(1);
			while (rs.next()) {
				int assessmentID = rs.getInt(1);
				java.util.Date modTime = rs.getDate(2);
				String name = rs.getString(3);
				int visitID = rs.getInt(4);
				int segmentID = rs.getInt(5);
				AssessmentInfo asi = new AssessmentInfo(assessmentID, name,
						modTime, visitID, segmentID);
				asList.add(asi);
			}

			// get the scores also
			if (!asList.isEmpty()) {
				log.info("getting scores for assessments");
				SubjectAssessmentManHelper.getScores(con, asList);
			}

			return asList;
		} catch (Exception x) {
			log.error("Error in getAssessmentValuesForSubject", x);
			throw new SubjectAssessmentManagementException(x);
		} finally {
			DBUtils.close(cst);
			releaseConnection(con, ui);
		}
	}

	protected void getScoresWithStoredProc(Connection con,
			List<AssessmentInfo> assessmentInfos) throws Exception {
		OracleCallableStatement cst = null;
		try {
			cst = (OracleCallableStatement) con
					.prepareCall("{? = call assessment_man.get_assessment_scores_for(?)}");
			cst.registerOutParameter(1, OracleTypes.CURSOR);
			ArrayDescriptor ad = ArrayDescriptor.createDescriptor(
					"AS_ID_LIST_TYPE", con);
			int[] as_ids = new int[assessmentInfos.size()];
			int idx = 0;
			Map<Integer, AssessmentInfo> asiMap = new HashMap<Integer, AssessmentInfo>();
			for (Object element : assessmentInfos) {
				AssessmentInfo asi = (AssessmentInfo) element;
				as_ids[idx++] = asi.getAssessmentID();
				asiMap.put(new Integer(asi.getAssessmentID()), asi);
			}
			ARRAY oarr = new ARRAY(ad, con, as_ids);
			// set Oracle VARRAY
			cst.setARRAY(2, oarr);
			cst.execute();
			ResultSet rs = (ResultSet) cst.getObject(1);

			int currentAsID = -1;
			AssessmentInfo currentAsi = null;
			while (rs.next()) {
				int asID = rs.getInt(1);
				String scoreName = rs.getString(2);
				String scoreType = rs.getString(3);
				int scoreLevel = rs.getInt(4);
				int parentAsID = rs.getInt(5);
				String parentScore = rs.getString(6);
				int scoreOrder = rs.getInt(7);
				AssessmentInfo.ScoreInfo si = new AssessmentInfo.ScoreInfo(
						scoreName, scoreType, scoreLevel, parentAsID,
						parentScore, scoreOrder);
				if (currentAsID != asID) {
					currentAsi = asiMap.get(new Integer(asID));
					currentAsID = asID;
				}
				currentAsi.addScore(si);
				// log.info("adding score to " + currentAsi.getName() + " : "+
				// si.toString());
			}
			rs.close();

			/** @todo set the parents for each ScoreInfo object */
		} finally {
			DBUtils.close(cst);
		}
	}

	/**
	 * @deprecated
	 */
	public List<AssessmentScoreValues> getAssessmentValuesForSubjectWithStoredProc(
			UserInfo ui, String subjectID, int experimentID,
			List<AssessmentInfo> assessments)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		OracleCallableStatement cst = null;
		try {
			con = pool.getConnection(ui.getName());
			cst = (OracleCallableStatement) con
					.prepareCall("{? = call assessment_man.get_assessment_values(?,?,?)}");
			cst.registerOutParameter(1, OracleTypes.CURSOR);
			cst.setString(2, subjectID);
			cst.setInt(3, experimentID);
			ArrayDescriptor ad = ArrayDescriptor.createDescriptor(
					"AS_ID_LIST_TYPE", con);

			int[] as_ids = new int[assessments.size()];

			int idx = 0;
			for (Object element : assessments) {
				AssessmentInfo asi = (AssessmentInfo) element;
				as_ids[idx++] = asi.getAssessmentID();
			}

			ARRAY oarr = new ARRAY(ad, con, as_ids);
			// set Oracle VARRAY
			cst.setARRAY(4, oarr);
			cst.execute();

			Map<Integer, AssessmentInfo> asiMap = convertToMap(assessments);
			ResultSet rs = (ResultSet) cst.getObject(1);

			Map<Integer, AssessmentScoreValues> asvMap = new LinkedHashMap<Integer, AssessmentScoreValues>(
					17);
			while (rs.next()) {
				int asID = rs.getInt(1);
				String scoreName = rs.getString(2);
				// String scoreType = rs.getString(3); // scoreType
				String textValue = rs.getString(4);
				// int scoreOrder = rs.getInt(5); // scoreOrder

				Integer asIDObject = new Integer(asID);
				AssessmentScoreValues asv = asvMap.get(asIDObject);
				if (asv == null) {
					AssessmentInfo asi = asiMap.get(asIDObject);

					asv = new AssessmentScoreValues(asi);
					asvMap.put(asIDObject, asv);
				}
				/** @todo add score hierarchy */
				/**
				 * @todo need to implement logic for multiple score values per
				 *       score name (multiple answers)
				 */
				asv.addScoreValue(new ScoreValue(scoreName, textValue, 1));
				if (log.isDebugEnabled())
					log.debug(rs.getString(2) + ", " + rs.getString(3) + ", "
							+ rs.getString(4));
			}

			return new ArrayList<AssessmentScoreValues>(asvMap.values());
		} catch (Exception x) {
			log.error("Error in getAssessmentValuesForSubject", x);
			throw new SubjectAssessmentManagementException(x);
		} finally {
			if (cst != null)
				try {
					cst.close();
				} catch (Exception x) {
				}
			releaseConnection(con, ui);
		}
	}

	public List<AssessmentScoreValues> getAssessmentValuesForSubjectWithStoredProc(
			UserInfo ui, String subjectID, int experimentID, int visitID,
			int segmentID, List<AssessmentInfo> assessments)
			throws SubjectAssessmentManagementException {
		Connection con = null;
		OracleCallableStatement cst = null;
		try {
			con = pool.getConnection(ui.getName());
			cst = (OracleCallableStatement) con
					.prepareCall("{? = call assessment_man.get_assessment_values(?,?,?,?,?)}");
			cst.registerOutParameter(1, OracleTypes.CURSOR);
			cst.setString(2, subjectID);
			cst.setInt(3, experimentID);

			ArrayDescriptor ad = ArrayDescriptor.createDescriptor(
					"AS_ID_LIST_TYPE", con);

			int[] as_ids = new int[assessments.size()];

			int idx = 0;
			for (Object element : assessments) {
				AssessmentInfo asi = (AssessmentInfo) element;
				as_ids[idx++] = asi.getAssessmentID();
			}

			ARRAY oarr = new ARRAY(ad, con, as_ids);
			// set Oracle VARRAY
			cst.setARRAY(4, oarr);
			cst.setInt(5, visitID);
			cst.setInt(6, segmentID);

			cst.execute();

			Map<Integer, AssessmentInfo> asiMap = convertToMap(assessments);
			ResultSet rs = (ResultSet) cst.getObject(1);

			Map<Integer, AssessmentScoreValues> asvMap = new LinkedHashMap<Integer, AssessmentScoreValues>(
					17);
			while (rs.next()) {
				int asID = rs.getInt(1);
				String scoreName = rs.getString(2);
				// String scoreType = rs.getString(3);
				String textValue = rs.getString(4);
				// int scoreOrder = rs.getInt(5);

				Integer asIDObject = new Integer(asID);
				AssessmentScoreValues asv = asvMap.get(asIDObject);
				if (asv == null) {
					AssessmentInfo asi = (AssessmentInfo) asiMap
							.get(asIDObject);

					asv = new AssessmentScoreValues(asi);
					asv.setSubjectID(subjectID);
					asv.setExperimentID(experimentID);
					asv.setVisitID(visitID);
					asv.setSegmentID(segmentID);

					java.util.Date timeStamp = getTimeStampForAssessment(ui,
							subjectID, experimentID, visitID, segmentID, asi
									.getAssessmentID());
					asv.setTimeStamp(timeStamp);

					asvMap.put(asIDObject, asv);
				}
				/** @todo add score hierarchy */
				/**
				 * @todo need to implement logic for multiple score values per
				 *       score name (multiple answers)
				 */
				asv.addScoreValue(new ScoreValue(scoreName, textValue, 1));

				System.out.println(rs.getString(2) + ", " + rs.getString(3)
						+ ", " + rs.getString(4));
			}

			return new ArrayList<AssessmentScoreValues>(asvMap.values());
		} catch (Exception x) {
			log.error("Error in getAssessmentValuesForSubject", x);
			throw new SubjectAssessmentManagementException(x);
		} finally {
			DBUtils.close(cst);
			releaseConnection(con, ui);
		}
	}

	public List<AssessmentScoreValues> getAssessmentValuesForSubject(
			UserInfo ui, String subjectID, int experimentID,
			List<AssessmentInfo> assessments, int entryID, boolean validated)
			throws SubjectAssessmentManagementException {
		/** @todo CHECK entryID */
		return super.getAssessmentValuesForSubject(ui, subjectID, experimentID,
				-1, -1, assessments, entryID, validated);
	}

	protected String prepareSO(String schemaObjName) {
		return schemaObjName;
	}

	protected String prepareString(String value) {
		StringBuffer buf = new StringBuffer();
		value = OracleUtils.escapeQuotes(value);
		buf.append('\'').append(value).append('\'');
		return buf.toString();
	}

	protected String prepareBoolean(String prefix, boolean value) {
		StringBuffer buf = new StringBuffer();
		buf.append(prefix);
		buf.append(' ').append(value ? "1" : "0");
		return buf.toString();
	}

}
