package clinical.web.services;

import static clinical.web.CBFBIRNConstants.CASL;
import static clinical.web.CBFBIRNConstants.CASL_CSF;
import static clinical.web.CBFBIRNConstants.CSF;
import static clinical.web.CBFBIRNConstants.DCM_SERIES_DESC_FIELD;
import static clinical.web.CBFBIRNConstants.DTI;
import static clinical.web.CBFBIRNConstants.EPISTAR;
import static clinical.web.CBFBIRNConstants.EPISTAR_CSF;
import static clinical.web.CBFBIRNConstants.FAIR;
import static clinical.web.CBFBIRNConstants.FAIR_CSF;
import static clinical.web.CBFBIRNConstants.LOCALIZER;
import static clinical.web.CBFBIRNConstants.MINCON;
import static clinical.web.CBFBIRNConstants.Opt_PCASL;
import static clinical.web.CBFBIRNConstants.Opt_PCASL_CSF;
import static clinical.web.CBFBIRNConstants.Opt_PCASL_Calib;
import static clinical.web.CBFBIRNConstants.Opt_PCASL_Calib_CSF;
import static clinical.web.CBFBIRNConstants.PCASL;
import static clinical.web.CBFBIRNConstants.PCASL_4MP;
import static clinical.web.CBFBIRNConstants.PCASL_4MP_CSF;
import static clinical.web.CBFBIRNConstants.PCASL_4MP_or_Opt_PCASL_Calib;
import static clinical.web.CBFBIRNConstants.PCASL_4MP_or_Opt_PCASL_Calib_CSF;
import static clinical.web.CBFBIRNConstants.PCASL_8MP;
import static clinical.web.CBFBIRNConstants.PCASL_8MP_CSF;
import static clinical.web.CBFBIRNConstants.PCASL_CSF;
import static clinical.web.CBFBIRNConstants.PCASL_VARIANT;
import static clinical.web.CBFBIRNConstants.PCASL_VARIANT_CSF;
import static clinical.web.CBFBIRNConstants.PICORE;
import static clinical.web.CBFBIRNConstants.PICORE_CSF;
import static clinical.web.CBFBIRNConstants.VSASL;
import static clinical.web.CBFBIRNConstants.VSASL_CSF;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.cache.CacheUtils;
import clinical.server.dao.DataobjectDAO;
import clinical.server.dao.DataobjecttypeDAO;
import clinical.server.dao.RawdataDAO;
import clinical.server.dao.UploadStatusDAO;
import clinical.server.vo.Assessment;
import clinical.server.vo.Collectionequipment;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Dataobject;
import clinical.server.vo.Dataobjecttype;
import clinical.server.vo.Experiment;
import clinical.server.vo.Humansubject;
import clinical.server.vo.Protocol;
import clinical.server.vo.Rawdata;
import clinical.server.vo.Researchgroup;
import clinical.server.vo.Subjexperiment;
import clinical.server.vo.UploadStatus;
import clinical.utils.AFNIHeaderReader;
import clinical.utils.AnalyzeHeaderReader;
import clinical.utils.Assertion;
import clinical.utils.CBFImageTypeDecider;
import clinical.utils.DICOMHeaderAnonymizer;
import clinical.utils.DICOMHeaderReader;
import clinical.utils.DateTimeUtils;
import clinical.utils.Executor;
import clinical.utils.Field;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.utils.HeaderFields;
import clinical.utils.PFileHeaderUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.IDBUserManService;
import clinical.web.IExperimentManagement;
import clinical.web.IRemoteDBServices;
import clinical.web.ISequenceHelper;
import clinical.web.ISubjectAssessmentManagement;
import clinical.web.ISubjectVisitManagement;
import clinical.web.IUploadManagementService;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.IDBPoolService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.SearchCriteria;
import clinical.web.common.query.SearchPredicate;
import clinical.web.common.query.SearchPredicateList;
import clinical.web.common.vo.AssessmentSelectionInfo;
import clinical.web.download.Unpacker;
import clinical.web.download.Unzipper;
import clinical.web.exception.BaseException;
import clinical.web.exception.SubjectAssessmentManagementException;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.AssessmentInfo.ScoreInfo;
import clinical.web.vo.AssessmentScoreValues;
import clinical.web.vo.CollectionEquipment;
import clinical.web.vo.HeaderFieldType;
import clinical.web.vo.ScoreValue;
import clinical.web.vo.Subject;
import clinical.web.vo.Visit;
import clinical.web.vo.VisitSegment;
import clinical.web.vo.upload.AssessmentType;
import clinical.web.vo.upload.ScoreType;
import clinical.web.vo.upload.SegmentInfo;
import clinical.web.vo.upload.SeriesHeaderInfo;
import clinical.web.vo.upload.UploadStatusInfo;
import clinical.web.vo.upload.VisitInfo;

import com.pixelmed.dicom.DicomFileUtilities;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: UploadManagementService.java 808 2013-05-13 23:33:15Z bozyurt $
 */
public class UploadManagementService extends AbstractServiceImpl implements
		IUploadManagementService {
	protected IDBCache dbCache;
	static SimpleDateFormat visitDF = new SimpleDateFormat("MM_dd_yyyy");
	protected Log log = LogFactory.getLog(UploadManagementService.class);

	public UploadManagementService(String dbID) throws BaseException {
		super(dbID);
		log = LogFactory.getLog(UploadManagementService.class);
		dbCache = ServiceFactory.getDBCache(dbID);
	}

	public UploadStatusInfo doUploadStaging(UserInfo ui, String stagingDirRoot,
			String tarBallName, String experimentName, String subjectID)
			throws Exception {
		File archiveFile = new File(stagingDirRoot, tarBallName);

		String destDirName = tarBallName;
		String visitName = tarBallName.replaceFirst("(\\.[^\\.])+$", "");
		destDirName = visitName + "_"
				+ String.valueOf(System.currentTimeMillis());
		File destDir = new File(stagingDirRoot, destDirName);
		if (!destDir.mkdirs()) {
			log.error("doUploadStaging: Cannot create staging directory!:"
					+ destDir);
			throw new Exception("Cannot create staging directory!");
		}

		if (archiveFile.getName().endsWith(".zip")) {
			Unzipper unzipper = new Unzipper(archiveFile.getAbsolutePath(),
					destDir);

			unzipper.unpack();
		} else {
			Unpacker unpacker = new Unpacker(archiveFile.getAbsolutePath(),
					destDir.getAbsolutePath());

			unpacker.unpack();
		}
		List<SeriesHeaderInfo> dicomSHIList = getDICOMSeriesHeaderInfo(destDir);
		String theScannerInfo = null;
		// handle anonymization
		for (SeriesHeaderInfo shi : dicomSHIList) {

			File dcmFile = new File(shi.getImages().get(0));
			File srcDcmDir = dcmFile.getParentFile();
			Assertion.assertTrue(dcmFile.isFile());
			String scannerInfo = DICOMHeaderReader.getScannerInfo(dcmFile);
			if (scannerInfo != null && theScannerInfo == null) {
				theScannerInfo = scannerInfo;
			}

			if (shi.getName().endsWith("_anon")) {
				// skip anonymization
				continue;
			}

			if (scannerInfo != null
					&& scannerInfo.toLowerCase().indexOf("siemens") != -1) {
				// specific treatment for Siemens data (no anonymization
				// to not loose Siemens ASCII header)

			} else {
				File anonDcmDir = new File(srcDcmDir.getAbsolutePath()
						+ "_anon");
				if (anonDcmDir.isDirectory()) {
					FileUtils.deleteRecursively(anonDcmDir);
				}

				if (!anonDcmDir.mkdir()) {
					log.error("doUploadStaging: cannot create anonDicomDir:"
							+ anonDcmDir);
					throw new Exception(
							"An error occurred during DICOM header anonymization!");
				}
				DICOMHeaderAnonymizer anonymizer = new DICOMHeaderAnonymizer();
				anonymizer.anonymizeDICOMSeries(srcDcmDir, anonDcmDir);
				FileUtils.deleteRecursively(srcDcmDir);
			}
		}

		List<SeriesHeaderInfo> shiList = prepareSeriesHeaderInfo(destDir);
		List<String> extraFiles = prepareExtraFilesList(destDir);

		Connection con = null;
		try {
			con = getConnection(ui);
			con.setAutoCommit(false);

			UploadStatusDAO dao = DAOFactory
					.createUploadStatusDAO(this.theDBID);
			UploadStatus us = new UploadStatus();
			us.setStagingPath(destDir.getAbsolutePath());
			us.setWebUser(ui.getName());
			us.setStatusCode(STAGED);
			us.setSubjectid(subjectID);
			us.setName(visitName);
			if (experimentName != null) {
				Experiment exp = findExperiment(ui, experimentName);
				if (exp != null) {
					us.setExperimentId(exp.getUniqueid());
				}
			}
			us.setModTime(new Date());
			Databaseuser adminDU = getDatabaseUser(ui,
					Constants.USERCLASS_ADMIN);
			us.setModUser(adminDU.getUniqueid());
			us.setOwner(adminDU.getUniqueid());

			ISequenceHelper sequenceHelper = MinimalServiceFactory
					.getSequenceHelper(theDBID);
			BigDecimal uniqueid = sequenceHelper.getNextUID(con,
					Constants.UPLOAD_STATUS, "uniqueid");
			us.setUniqueid(uniqueid);

			dao.insert(con, us);

			con.commit();

			return new UploadStatusInfo(us, shiList, extraFiles, theScannerInfo);
		} catch (Exception x) {
			handleErrorAndRollBack(con, "doUploadStaging", x, true);
			throw x;
		} finally {
			super.releaseConnection(con, ui);
		}
	}

	protected void updateStatus(Connection con, int stagingId,
			String subjectID, int expID, String newStatusCode) throws Exception {
		UploadStatusDAO dao = DAOFactory.createUploadStatusDAO(this.theDBID);
		UploadStatus cr = new UploadStatus();
		UploadStatus us = new UploadStatus();

		cr.setUniqueid(GenUtils.toBigDecimal(stagingId));

		us.setStatusCode(newStatusCode);
		us.setSubjectid(subjectID);
		us.setExperimentId(GenUtils.toBigDecimal(expID));

		dao.update(con, us, cr);
	}

	@SuppressWarnings("unused")
	public void uploadVisit(UserInfo ui, VisitInfo visitInfo, int stagingId)
			throws Exception {
		Connection con = null;
		IDBPoolService dbPoolService = null;
		int visitId = 1;
		int startSegId = 1;
		try {
			dbPoolService = ServiceFactory.getPoolService(this.theDBID);
			con = dbPoolService.getConnection(ui.getName());
			con.setAutoCommit(false);
			IExperimentManagement em = ServiceFactory
					.getExperimentManagement(this.theDBID);
			List<Subjexperiment> subjExpList = em.getEnrolledSubjects(ui,
					visitInfo.getExpId());
			Subjexperiment theSE = findSubjExperiment(visitInfo.getSubjectID(),
					subjExpList);
			if (theSE == null) {
				throw new Exception("The subject '" + visitInfo.getSubjectID()
						+ "' is not enrolled with project!");
			}
			if (theSE == null) {
				theSE = enrollSubject(ui, con, visitInfo, em);
				addVisitForSubject(con, ui, visitInfo, visitId, startSegId);
			} else {
				ISubjectVisitManagement isvm = ServiceFactory
						.getSubjectVisitManagement(this.theDBID);
				List<Visit> visitList = isvm.getAllVisitsForSubject(ui,
						visitInfo.getSubjectID());

				// filter out visits not belonging to the visitInfo's experiment
				for (Iterator<Visit> it = visitList.iterator(); it.hasNext();) {
					Visit v = it.next();
					if (v.getExperimentID() != visitInfo.getExpId()) {
						it.remove();
					}
				}

				if (visitList.isEmpty()) {
					addVisitForSubject(con, ui, visitInfo, visitId, startSegId);
				} else {
					Visit theVisit = findVisit(visitInfo, visitList);
					if (theVisit == null) {
						visitId = getNextVisitID(visitList);
						addVisitForSubject(con, ui, visitInfo, visitId,
								startSegId);
					} else {
						log.warn("A visit with same time stamp already exists "
								+ "for subject " + visitInfo.getSubjectID());
						visitId = getNextVisitID(visitList);
						addVisitForSubject(con, ui, visitInfo, visitId,
								startSegId);
						/*
						 * throw new Exception(
						 * "A visit with same time stamp already exists for subject "
						 * + visitInfo.getSubjectID());
						 */
					}
				}
			}
			if (!visitInfo.getAssessments().isEmpty()) {
				for (AssessmentType at : visitInfo.getAssessments()) {
					handleAssessment(con, ui, at, visitInfo, visitId,
							startSegId);
				}
			}
			List<Collectionequipment> equipments = dbCache
					.getCollectionEquipments(ui, true);

			// add uris to data for this visit after moving raw image data to
			// official location
			addPointers(ui, con, visitInfo, visitId, startSegId, equipments);

			// update staging status
			updateStatus(con, stagingId, visitInfo.getSubjectID(),
					visitInfo.getExpId(), UPLOADED);
			// cleanup staging
			cleanupStaging4Upload(visitInfo);

			con.commit();

		} catch (Throwable t) {
			t.printStackTrace();
			try {
				con.rollback();
			} catch (Exception e) {
				log.error("rollback error", e);
			}
			throw new Exception(t.getMessage());
		} finally {
			if (dbPoolService != null) {
				dbPoolService.releaseConnection(ui.getName(), con);
			}
		}
	}

	protected void handleAssessment(Connection con, UserInfo ui,
			AssessmentType at, VisitInfo visitInfo, int visitId, int segmentId)
			throws Exception {
		ISubjectAssessmentManagement isam = ServiceFactory
				.getSubjectAssessmentManagement(theDBID);

		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(theDBID);
		SearchPredicateList spList = new SearchPredicateList();
		SearchPredicate sp = new SearchPredicate("subjectid",
				visitInfo.getSubjectID(), SearchPredicate.EQUAL,
				SearchPredicate.STRING);
		spList.addSearchPredicate(sp, SearchPredicateList.NONE);

		SearchCriteria sc = new SearchCriteria(spList);
		List<Humansubject> matchingSubjects = isvm.getMatchingSubjects(ui, sc);
		Humansubject theHS = matchingSubjects.get(0);

		String asName = at.getName();

		Assessment assessment = isam.getAssessment(asName, con);

		if (assessment == null) {
			assessment = addAssessment(ui, at, isam);
		}
		AssessmentInfo asInfo = prepAssessmentInfo(at, assessment);
		AssessmentScoreValues asv = new AssessmentScoreValues(asInfo);
		asv.setSubjectID(visitInfo.getSubjectID());
		asv.setExperimentID(visitInfo.getExpId());
		asv.setVisitID(visitId);
		asv.setSegmentID(segmentId);
		asv.setTimeStamp(visitInfo.getVisitDate());
		// informant and clinical rater information
		asv.setInformantID(visitInfo.getSubjectID());
		asv.setInformantRelation(Constants.SELF);
		String clinicalRater = ui.getName();
		asv.setClinicalRater(clinicalRater);

		List<AssessmentInfo> aiList = isam.getAssessmentsForSubject(con, ui,
				visitInfo.getSubjectID(), visitInfo.getExpId());
		boolean foundAsEntry = false;
		for (AssessmentInfo ai : aiList) {
			if (ai.getAssessmentID() == asInfo.getAssessmentID()) {
				if (visitId == ai.getVisitID()
						&& segmentId == ai.getSegmentID()) {
					foundAsEntry = true;
					break;
				}
			}
		}

		if (foundAsEntry) {
			throw new Exception("An entry of the assessment '"
					+ asInfo.getName() + "' for visit with ID " + visitId
					+ " and segment(episode) ID " + segmentId
					+ " already exists!");
		}
		IDBUserManService dbusm = ServiceFactory.getDBUserManService(theDBID);
		dbusm.ensureClinicalRater(con, ui, clinicalRater);
		Map<String, Map<String, String>> dbVarMetaDataMap = new HashMap<String, Map<String, String>>(
				3);
		for (clinical.web.vo.upload.ScoreValue sv : at.getSvList()) {
			String value = sv.getValue();
			// special handling for age score
			if (sv.getName().equalsIgnoreCase("age")) {
				int age = DateTimeUtils.calculateAge(theHS.getBirthdate(),
						visitInfo.getVisitDate());
				if (age <= 0) {
					throw new Exception(
							"Age should be greater than 0! Please check subject's birthdate!");
				}
				value = String.valueOf(age);
			}

			// FIXME no date type is supported for upload assessments
			ScoreValue scoreValue = new ScoreValue(sv.getName(), value, 1);
			asv.addScoreValue(scoreValue);
		}
		BigDecimal entryID = new BigDecimal(1);
		isam.insertAssessmentValues(con, ui, theDBID, asv, entryID, true,
				dbVarMetaDataMap);
	}

	protected Subjexperiment enrollSubject(UserInfo ui, Connection con,
			VisitInfo visitInfo, IExperimentManagement em) throws Exception {
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(this.theDBID);
		Humansubject hs = isvm.getSubjectByID(ui, visitInfo.getSubjectID());
		if (hs == null) {
			Subject subject = new Subject(visitInfo.getSubjectID());
			isvm.addSubject(ui, subject);
		}
		List<Researchgroup> rgList = dbCache.getResearchGroups(ui, false);
		Researchgroup theRG = null;

		String rgName = visitInfo.getRgInfo().getName();
		// System.out.println("visit Exp ID:" + visitInfo.getExpId() +
		// " rgName:"
		// + rgName + "+++++");
		for (Researchgroup rg : rgList) {
			// System.out.println(rg);
			if (rg.getName().equals(rgName)
					&& rg.getNcExperimentUniqueid().intValue() == visitInfo
							.getExpId()) {
				theRG = rg;
				break;
			}
		}
		if (theRG == null) {
			throw new Exception("Unknown subject group ID:"
					+ visitInfo.getRgInfo().getName());
		}
		Subjexperiment se = em.enrollSubject(ui, visitInfo.getExpId(),
				visitInfo.getSubjectID(), theRG.getUniqueid().intValue());
		return se;
	}

	public static Visit findVisit(VisitInfo vi, List<Visit> visitList) {
		Timestamp ts = new Timestamp(vi.getVisitDate().getTime());
		for (Visit v : visitList) {
			if (v.getTs().equals(ts)) {
				return v;
			}
		}
		return null;
	}

	public static int getNextVisitID(List<Visit> visitList) {
		int maxVisitId = -1;
		for (Visit v : visitList) {
			if (v.getComponentID() > maxVisitId) {
				maxVisitId = v.getComponentID();
			}
		}
		Assertion.assertTrue(maxVisitId > 0);
		return maxVisitId + 1;
	}

	protected void addVisitForSubject(Connection con, UserInfo ui,
			VisitInfo vi, int visitId, int startSegId) throws BaseException {
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(this.theDBID);

		List<Collectionequipment> equipments = dbCache.getCollectionEquipments(
				ui, true);
		Visit v = toVisit(ui, vi, visitId, startSegId, equipments);

		isvm.addVisitForSubjectNonContiguous(con, ui, v);
	}

	protected Visit toVisit(UserInfo ui, VisitInfo vi, int visitId,
			int startSegId, List<Collectionequipment> equipments)
			throws BaseException {
		Map<Integer, Collectionequipment> eqMap = new HashMap<Integer, Collectionequipment>(
				17);
		for (Collectionequipment colEq : equipments) {
			eqMap.put(colEq.getUniqueid().intValue(), colEq);
		}
		Visit v = new Visit();
		v.setComponentID(visitId);
		v.setName(vi.getName());
		v.setDescription(vi.getDescr());
		v.setSubjectID(vi.getSubjectID());
		v.setExperimentID(vi.getExpId());
		v.setTimeStamp(new Timestamp(vi.getVisitDate().getTime()));
		v.setVisitType(Constants.VISIT_TYPE_MRI_SCAN);

		int i = 0;
		if (!vi.getAssessments().isEmpty()) {
			IDBCache dbCache = ServiceFactory.getDBCache(this.theDBID);
			List<Protocol> protocols = dbCache.getProtocols(ui, false);
			Protocol clinicalProtocol = null;
			for (Protocol pr : protocols) {
				if (pr.getProtocolid().equals(Constants.CLINICAL_PROTOCOL)) {
					clinicalProtocol = pr;
					break;
				}
			}
			Assertion.assertNotNull(clinicalProtocol, "No protocol named '"
					+ Constants.CLINICAL_PROTOCOL
					+ "' is available in the database!");
			VisitSegment vs = new VisitSegment();
			vs.setSegmentID(startSegId + i);
			vs.setName(Constants.ASSESSMENT_SEGMENT);

			vs.setVisitID(visitId);
			vs.setExperimentID(vi.getExpId());
			vs.setSubjectID(vi.getSubjectID());
			vs.setTimeStamp(new Timestamp(vi.getVisitDate().getTime()));
			// protocol and coleq info
			vs.setProtocolID(clinicalProtocol.getProtocolid());
			vs.setProtocolVersion(clinicalProtocol.getProtocolversion()
					.intValue());
			v.addVisitSegment(vs);
			i = 1;
		}
		for (SegmentInfo si : vi.getSiList()) {
			VisitSegment vs = new VisitSegment();
			vs.setSegmentID(startSegId + i);
			vs.setName(si.getName());
			vs.setDescription(si.getDescr());
			vs.setVisitID(visitId);
			vs.setExperimentID(vi.getExpId());
			vs.setSubjectID(vi.getSubjectID());
			vs.setTimeStamp(new Timestamp(si.getTimestamp().getTime()));
			// protocol and coleq info
			vs.setProtocolID(si.getProtocolId());
			vs.setProtocolVersion(si.getProtocolVersion());

			Collectionequipment colEq = eqMap.get(si.getColEqId());
			Assertion.assertNotNull(colEq);
			CollectionEquipment ce = new CollectionEquipment(colEq.getMake(),
					colEq.getModel());
			vs.setEquipment(ce);

			v.addVisitSegment(vs);
			i++;
		}
		return v;
	}

	protected void addPointers(UserInfo ui, Connection con,
			VisitInfo visitInfo, int visitId, int startSegId,
			List<Collectionequipment> equipments) throws Exception {
		Map<Integer, Collectionequipment> eqMap = new HashMap<Integer, Collectionequipment>(
				17);
		for (Collectionequipment eq : equipments) {
			eqMap.put(eq.getUniqueid().intValue(), eq);
		}

		IDBCache dbCache = ServiceFactory.getDBCache(this.theDBID);

		List<Experiment> experiments = dbCache.getExperiments(ui, false);
		Experiment theExp = null;
		for (Experiment exp : experiments) {
			if (visitInfo.getExpId() == exp.getUniqueid().intValue()) {
				theExp = exp;
				break;
			}
		}
		Assertion.assertNotNull(theExp, "No experiment in "
				+ "the database with id:" + visitInfo.getExpId());
		String baseURI = theExp.getBaseuri();
		Assertion.assertNotNull(baseURI,
				"No baseURI for experiment:" + theExp.getName());

		int i = 0;
		if (!visitInfo.getAssessments().isEmpty()) {
			i = 1;
		}

		File visitDir = prepareVisitDir(baseURI, visitInfo, visitId);

		for (SegmentInfo si : visitInfo.getSiList()) {
			int segmentId = startSegId + i;
			Collectionequipment colEq = eqMap.get(si.getColEqId());
			Assertion.assertNotNull(colEq);
			if (si.getShi() != null) {
				if (si.getShi().getImageType().equals(SeriesHeaderInfo.DICOM)) {
					File dest = move2UploadLoc4DICOM(si.getShi().getImages(),
							baseURI, visitInfo, visitDir);
					Assertion.assertNotNull(dest);
					addRawData(ui, con, si, visitId, segmentId, visitInfo,
							dest.getAbsolutePath(), colEq);
				} else {
					for (String dataURI : si.getShi().getImages()) {
						File dest = move2UploadLoc(dataURI, visitInfo, visitDir);
						Assertion.assertNotNull(dest);
						addRawData(ui, con, si, visitId, segmentId, visitInfo,
								dest.getAbsolutePath(), colEq);
					}
				}
			}
			i++;
		}

		for (String extraFile : visitInfo.getExtraFiles()) {
			File dest = move2UploadLoc(extraFile, visitInfo, visitDir);
			Assertion.assertNotNull(dest);
		}

		// create also a contents file
		prepareContentsFile(visitDir, visitInfo);

		// register data objects for download
		registerDataObjects(con, ui, visitInfo, visitId);
	}

	protected File move2UploadLoc4DICOM(List<String> dcmFiles, String baseURI,
			VisitInfo vi, File visitDir) throws Exception {
		String aDcmFile = dcmFiles.get(0);
		File dcmFile = new File(aDcmFile);
		String dirName = dcmFile.getParentFile().getName();
		if (!visitDir.exists()) {
			if (!visitDir.mkdirs()) {
				throw new Exception("Cannot create directory:" + visitDir);
			}
		}
		File destDir = new File(visitDir, dirName);
		if (!destDir.exists()) {
			if (!destDir.mkdir()) {
				throw new Exception("Cannot create directory:" + destDir);
			}
		}
		for (String df : dcmFiles) {
			File srcFile = new File(df);
			File dest = new File(destDir, srcFile.getName());
			boolean ok = clinical.utils.FileUtils.moveTo(srcFile, dest);
			if (!ok) {
				throw new Exception("Cannot move file to:"
						+ dest.getAbsolutePath());
			}
		}
		return destDir;
	}

	protected File move2UploadLoc(String stagingDataUri, VisitInfo vi,
			File visitDir) throws Exception {
		if (!visitDir.exists()) {
			if (!visitDir.mkdirs()) {
				throw new Exception("Cannot create directory:" + visitDir);
			}
		}

		File sf = new File(stagingDataUri);

		if (sf.isDirectory()) {
			// FIXME only one level is supported (i.e. directories contained in
			// the image series directory will be ignored)
			File destDir = new File(visitDir, sf.getName());
			if (!destDir.exists()) {
				if (!destDir.mkdir()) {
					throw new Exception("Cannot create directory:" + destDir);
				}
			}
			File[] srcFiles = sf.listFiles();
			for (File srcFile : srcFiles) {
				if (srcFile.isFile()) {
					File dest = new File(destDir, srcFile.getName());
					boolean ok = clinical.utils.FileUtils.moveTo(srcFile, dest);
					if (!ok) {
						throw new Exception("Cannot move file:" + srcFile
								+ " to:" + dest.getAbsolutePath());
					}
				}
			}
			return destDir;
		} else if (sf.isFile()) {
			File dest = new File(visitDir, sf.getName());
			boolean ok = clinical.utils.FileUtils.moveTo(sf, dest);
			if (!ok) {
				throw new Exception("Cannot move file:" + sf + " to:"
						+ dest.getAbsolutePath());
			}
			return dest;
		}
		return null;
	}

	protected File prepareVisitDir(String baseURI, VisitInfo vi, int visitId) {
		File visitDir = new File(baseURI, vi.getSubjectID());
		String visitDirName = "Visit_" + visitDF.format(vi.getVisitDate());
		visitDir = new File(visitDir, visitDirName);
		if (visitDir.exists()) {
			visitDirName += "_" + visitId;
			visitDir = new File(new File(baseURI, vi.getSubjectID()),
					visitDirName);
		}
		return visitDir;
	}

	protected void prepareContentsFile(File visitDir, VisitInfo vi)
			throws Exception {
		File contentsFile = new File(visitDir, "CONTENTS");
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(contentsFile));

			for (SegmentInfo si : vi.getSiList()) {
				if (si.getShi() != null) {
					SeriesHeaderInfo shi = si.getShi();
					if (shi.getImageType().equals(SeriesHeaderInfo.DICOM)) {
						String dicomDirName = new File(shi.getImages().get(0))
								.getParentFile().getName();
						StringBuilder sb = new StringBuilder();
						sb.append(dicomDirName).append("\t")
								.append(si.getProtocolId());
						sb.append("\t").append(shi.getImageType());
						out.write(sb.toString());
						out.newLine();

					} else {
						for (String dataURI : shi.getImages()) {
							String filename = new File(dataURI).getName();
							StringBuilder sb = new StringBuilder();
							sb.append(filename).append("\t")
									.append(si.getProtocolId());
							sb.append("\t").append(shi.getImageType());

							out.write(sb.toString());
							out.newLine();
						}
					}
				}
			}
			for (String ef : vi.getExtraFiles()) {
				String filename = new File(ef).getName();
				out.write(filename);
				out.newLine();
			}
		} finally {
			clinical.utils.FileUtils.close(out);
		}
	}

	protected void cleanupStaging4Upload(VisitInfo vi) {
		String commonRoot = findCommonRoot(vi);
		if (commonRoot != null) {
			File dir = new File(commonRoot);
			if (dir.exists() && dir.isDirectory()) {
				clinical.utils.FileUtils.deleteRecursively(dir);
			}
		} else {
			log.warn("Cannot find common root dir for staging area cleanup:"
					+ vi.getSubjectID() + " " + vi.getVisitDate());
		}
	}

	public static String findCommonRoot(VisitInfo vi) {
		for (SegmentInfo si : vi.getSiList()) {
			if (si.getShi() != null) {
				return findCommonRoot(si.getShi().getImages());
			}
		}
		return null;
	}

	public static String findCommonRoot(List<String> uris) {
		List<String> commonFragments = new ArrayList<String>(10);
		boolean first = true;
		for (String uri : uris) {
			uri = uri.replaceFirst("^\\/+", "");
			String[] toks = uri.split("\\/");
			if (first) {
				for (String tok : toks) {
					commonFragments.add(tok);
				}
				first = false;
			} else {
				for (int i = 0; i < toks.length; i++) {
					if (i >= commonFragments.size()) {
						break;
					}
					String tok = toks[i];
					String cf = commonFragments.get(i);
					if (!cf.equals(tok)) {
						commonFragments.remove(i);
						int count = commonFragments.size() - i;
						for (int j = 0; j < count; j++) {
							commonFragments.remove(i);
						}
					}
				}
			}
		}
		StringBuilder sb = new StringBuilder();
		for (Iterator<String> it = commonFragments.iterator(); it.hasNext();) {
			sb.append('/').append(it.next());
		}

		return sb.toString();
	}

	protected void addRawData(UserInfo ui, Connection con, SegmentInfo si,
			int visitId, int segmentId, VisitInfo vi, String dataURI,
			Collectionequipment eq) throws Exception {
		Databaseuser dbUser = DBUtils.getDatabaseUser(ui,
				Constants.USERCLASS_ADMIN, this.theDBID);
		Assertion.assertNotNull(dbUser);

		RawdataDAO dao = DAOFactory.createRawdataDAO(this.theDBID);
		Rawdata rd = new Rawdata();
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(this.theDBID);
		rd.setComponentid(GenUtils.toBigDecimal(visitId));
		rd.setNcExperimentUniqueid(GenUtils.toBigDecimal(vi.getExpId()));
		rd.setSubjectid(vi.getSubjectID());
		rd.setSegmentid(GenUtils.toBigDecimal(segmentId));
		rd.setDatauri(dataURI);
		rd.setProtocolid(si.getProtocolId());
		rd.setProtocolversion(GenUtils.toBigDecimal(si.getProtocolVersion()));
		rd.setNcColequipmentUniqueid(eq.getUniqueid());

		rd.setOwner(dbUser.getUniqueid());
		rd.setModuser(dbUser.getUniqueid());
		rd.setModtime(new Date());
		rd.setOntologysource(Constants.DEFAULT_ONT_SRC);
		rd.setConceptid(Constants.DEFAULT_ONT_CONCEPT);
		BigDecimal uniqueid = sequenceHelper.getNextUID(con, Constants.RAWDATA,
				"uniqueid");
		rd.setExtensionname("nc_rawdata");
		rd.setIsraw(Boolean.TRUE);
		rd.setUniqueid(uniqueid);
		rd.setIsbad(Boolean.FALSE);
		rd.setTableid(DBUtils.getTableID(ui, Constants.RAWDATA, this.theDBID));

		dao.insert(con, rd);
	}

	public static Subjexperiment findSubjExperiment(String subjectID,
			List<Subjexperiment> subjExpList) {
		for (Subjexperiment se : subjExpList) {
			if (se.getSubjectid().equals(subjectID)) {
				return se;
			}
		}
		return null;
	}

	public static Experiment findExperiment(String projectID,
			List<Experiment> expList) {
		for (Experiment exp : expList) {
			if (exp.getName().equals(projectID)) {
				return exp;
			}
		}
		return null;
	}

	/**
	 * any file with extension .txt and/or .xml found in the tar file is assumed
	 * to be and extra file associated with the visit to be uploaded
	 * 
	 * @param destDir
	 * @return
	 */
	List<String> prepareExtraFilesList(File destDir) {
		File[] files = destDir.listFiles();
		List<String> extraFiles = new ArrayList<String>(1);
		for (File file : files) {
			if (file.isFile()) {
				if (file.getName().endsWith(".txt")
						|| file.getName().endsWith(".xml")) {
					extraFiles.add(file.getAbsolutePath());
				}
			}
		}
		return extraFiles;
	}

	private List<SeriesHeaderInfo> getDICOMSeriesHeaderInfo(File destDir)
			throws Exception {
		File[] files = destDir.listFiles();
		Map<String, SeriesHeaderInfo> shiMap = new HashMap<String, SeriesHeaderInfo>(
				37);
		for (File file : files) {
			if (file.isDirectory()) {
				List<File> dicomDirs = getDicomDirs(file);
				for (File dicomDir : dicomDirs) {
					String name = dicomDir.getName();
					SeriesHeaderInfo shi = shiMap.get(name);
					if (shi == null) {
						shi = new SeriesHeaderInfo(name, SeriesHeaderInfo.DICOM);
						shiMap.put(name, shi);
					}
					File[] dicomFiles = doImageSequencing(dicomDir);

					for (File dcmFile : dicomFiles) {
						if (DicomFileUtilities.isDicomOrAcrNemaFile(dcmFile)) {
							shi.addImage(dcmFile.getAbsolutePath());
							if (shi.getHeaderFile() == null) {
								shi.setHeaderFile(dcmFile.getAbsolutePath());
							}
						}
					}
				}
			}
		}
		List<SeriesHeaderInfo> dicomSHIList = new ArrayList<SeriesHeaderInfo>(
				10);
		for (SeriesHeaderInfo shi : shiMap.values()) {
			if (shi.getImageType().equals(SeriesHeaderInfo.DICOM)) {
				prepDICOMSeriesHeaderInfo(shi);
				dicomSHIList.add(shi);
			}
		}
		return dicomSHIList;
	}

	List<SeriesHeaderInfo> prepareSeriesHeaderInfo(File destDir)
			throws Exception {
		File[] files = destDir.listFiles();
		Map<String, SeriesHeaderInfo> shiMap = new HashMap<String, SeriesHeaderInfo>(
				37);
		boolean hasAnalyzeFiles = false;
		for (File file : files) {
			if (file.isFile()) {
				if (FileUtils.isPFile(file.getAbsolutePath())) {
					String name = extractName(file);
					SeriesHeaderInfo shi = new SeriesHeaderInfo(name,
							SeriesHeaderInfo.PFILE);
					shiMap.put(name, shi);
					shi.addImage(file.getAbsolutePath());
					shi.setHeaderFile(file.getAbsolutePath());

				} else if (FileUtils.isAFNIHeader(file.getAbsolutePath())) {
					String name = extractName(file);
					SeriesHeaderInfo shi = shiMap.get(name);
					if (shi == null) {
						shi = new SeriesHeaderInfo(name, SeriesHeaderInfo.AFNI);
						shiMap.put(name, shi);
					}
					shi.addImage(file.getAbsolutePath());
					shi.setHeaderFile(file.getAbsolutePath());
				} else if (FileUtils.isAFNIBrick(file.getAbsolutePath())) {
					String name = extractName(file);
					SeriesHeaderInfo shi = shiMap.get(name);
					if (shi == null) {
						shi = new SeriesHeaderInfo(name, SeriesHeaderInfo.AFNI);
						shiMap.put(name, shi);
					}
					shi.addImage(file.getAbsolutePath());
				} else if (FileUtils.isAnalyzeHeader(file)
						|| FileUtils.isAnalyzeImage(file)) {
					String name = extractName(file);
					SeriesHeaderInfo shi = shiMap.get(name);
					if (shi == null) {
						shi = new SeriesHeaderInfo(name,
								SeriesHeaderInfo.ANALYZE);
						shiMap.put(name, shi);
					}
					if (FileUtils.isAnalyzeHeader(file)) {
						shi.setHeaderFile(file.getAbsolutePath());
					}
					shi.addImage(file.getAbsolutePath());
					hasAnalyzeFiles = true;
				}

			} else if (file.isDirectory()) {
				// TODO handle DICOM directories
				List<File> dicomDirs = getDicomDirs(file);
				for (File dicomDir : dicomDirs) {
					String name = dicomDir.getName();
					SeriesHeaderInfo shi = shiMap.get(name);
					if (shi == null) {
						shi = new SeriesHeaderInfo(name, SeriesHeaderInfo.DICOM);
						shiMap.put(name, shi);
					}
					File[] dicomFiles = doImageSequencing(dicomDir);

					for (File dcmFile : dicomFiles) {
						if (DicomFileUtilities.isDicomOrAcrNemaFile(dcmFile)) {
							shi.addImage(dcmFile.getAbsolutePath());
							if (shi.getHeaderFile() == null) {
								shi.setHeaderFile(dcmFile.getAbsolutePath());
							}
						}
					}
				}
			}
		}

		List<HeaderFieldType> hftList = new ArrayList<HeaderFieldType>(5);
		hftList.add(new HeaderFieldType("tag", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("timestamp", HeaderFieldType.STRING));
		hftList.add(new HeaderFieldType("reps", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("dda", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("nIntlv", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("speprev", HeaderFieldType.STRING));
		hftList.add(new HeaderFieldType("mpphase", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("mppcasl", HeaderFieldType.INT));
		// Mon Jan 9 11:46:35 PST 2012
		hftList.add(new HeaderFieldType("pcaslopt", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("docsf", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("optpcasl", HeaderFieldType.INT));

		List<SeriesHeaderInfo> dicomSHIList = new ArrayList<SeriesHeaderInfo>(
				10);
		boolean foundLocalizer = false;
		for (SeriesHeaderInfo shi : shiMap.values()) {
			if (shi.getImageType().equals(SeriesHeaderInfo.PFILE)) {
				PFileHeaderUtils pfhu = new PFileHeaderUtils(
						shi.getHeaderFile());
				pfhu.readHeader();
				Field<String> tsField = pfhu.getStringField("timestamp");
				if (tsField != null && tsField.getValue() != null
						&& tsField.getValue().trim().length() > 0) {
					Date tsDate = PFileHeaderUtils.tsFormat.parse(tsField
							.getValue());
					shi.setScanDate(tsDate);
				}
				Field<Integer> tagField = pfhu.getIntField("tag");
				Field<Integer> nprepField = pfhu.getIntField("nprep");
				Field<Integer> repsField = pfhu.getIntField("reps");
				Field<Integer> ddaField = pfhu.getIntField("dda");
				Field<Integer> nIntlvField = pfhu.getIntField("nIntlv");
				Field<String> speprevField = pfhu.getStringField("speprev");
				Field<Integer> mpphaseField = pfhu.getIntField("mpphase");
				Field<Integer> mppcaslField = pfhu.getIntField("mppcasl");
				Field<Integer> pcasloptField = pfhu.getIntField("pcaslopt");
				Field<Integer> docsfField = pfhu.getIntField("docsf");
				Field<Integer> optpcaslField = pfhu.getIntField("optpcasl");

				String seriesType = findSeriesType(tagField, nprepField, null,
						repsField, ddaField, nIntlvField, speprevField,
						mpphaseField, mppcaslField, pcasloptField, docsfField,
						optpcaslField);
				if (seriesType == null)
					seriesType = "";
				shi.setSeriesType(seriesType);

			} else if (shi.getImageType().equals(SeriesHeaderInfo.AFNI)) {
				prepAFNISeriesHeaderInfo(shi);
			} else if (shi.getImageType().equals(SeriesHeaderInfo.DICOM)) {
				prepDICOMSeriesHeaderInfo(shi);
				dicomSHIList.add(shi);
				if (shi.getSeriesType().equals(LOCALIZER)) {
					foundLocalizer = true;
				}
			} else if (shi.getImageType().equals(SeriesHeaderInfo.ANALYZE)) {
				prepAnalyzeSeriesHeaderInfo(shi);
			}
		}

		if (hasAnalyzeFiles) {
			// no scan date information besides anatomical series
			// just assign all the same scan date
			Date theScanDate = null;
			for (SeriesHeaderInfo shi : shiMap.values()) {
				if (shi.getScanDate() != null) {
					theScanDate = shi.getScanDate();
					break;
				}
			}
			Assertion.assertNotNull(theScanDate);
			for (SeriesHeaderInfo shi : shiMap.values()) {
				if (shi.getScanDate() == null) {
					shi.setScanDate(theScanDate);
				}
			}
		}

		// if there is a DICOM series with no series type and it has the
		// earliest scan time, set it as the localizer
		if (!foundLocalizer && !dicomSHIList.isEmpty()) {

			Collections.sort(dicomSHIList, new Comparator<SeriesHeaderInfo>() {
				public int compare(SeriesHeaderInfo o1, SeriesHeaderInfo o2) {
					return o1.getScanDate().compareTo(o2.getScanDate());
				}
			});
			SeriesHeaderInfo locSHI = dicomSHIList.get(0);
			if (locSHI.getSeriesType() == null
					|| locSHI.getSeriesType().length() == 0) {
				locSHI.setSeriesType(LOCALIZER);
			}
		}

		List<SeriesHeaderInfo> shiList = new ArrayList<SeriesHeaderInfo>(
				shiMap.values());

		return shiList;
	}

	public List<AssessmentSelectionInfo> getAssessmentMetaData(UserInfo ui,
			String primarySiteID) throws Exception {
		List<AssessmentSelectionInfo> asiList = null;
		IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();
		asiList = rds.getAllAssessmentsWithScores(primarySiteID);
		return asiList;
	}

	private File[] doImageSequencing(File dicomDir) throws Exception {
		File[] dicomFiles = dicomDir.listFiles();
		// run imseq if necessary
		// image sequencing
		Assertion.assertTrue(dicomFiles.length > 0);
		File dicomFile = null;
		for (File f : dicomFiles) {
			if (DicomFileUtilities.isDicomOrAcrNemaFile(f)) {
				dicomFile = f;
				break;
			}
		}
		String dicomFileName = dicomFile.getName();
		if (dicomFileName.indexOf(".CFMRI.") == -1) {
			Executor executor = new Executor("imseq", true);
			executor.execute(dicomDir.getAbsolutePath());
		}

		dicomFiles = dicomDir.listFiles();
		return dicomFiles;
	}

	public static List<File> getDicomDirs(File dir) {
		List<File> dicomDirs = new ArrayList<File>(10);
		if (hasDICOMFiles(dir)) {
			dicomDirs.add(dir);
		}

		List<File> subDirs = getSubDirs(dir);
		while (!subDirs.isEmpty()) {
			List<File> subsubDirList = new ArrayList<File>();
			for (File subDir : subDirs) {
				if (hasDICOMFiles(subDir)) {
					dicomDirs.add(subDir);
				}
				subsubDirList.addAll(getSubDirs(subDir));
			}
			subDirs = subsubDirList;
		}
		return dicomDirs;
	}

	public static boolean hasDICOMFiles(File dir) {
		File[] files = dir.listFiles();
		for (File f : files) {
			if (f.isFile() && DicomFileUtilities.isDicomOrAcrNemaFile(f)) {
				return true;
			}
		}
		return false;
	}

	public static List<File> getSubDirs(File dir) {
		List<File> subdirList = new ArrayList<File>(5);
		File[] files = dir.listFiles();
		for (File f : files) {
			if (f.isDirectory()) {
				subdirList.add(f);
			}
		}
		return subdirList;
	}

	protected void prepDICOMSeriesHeaderInfo(SeriesHeaderInfo shi)
			throws Exception {
		File dicomDir = new File(shi.getImages().get(0)).getParentFile();
		DICOMHeaderReader dhr = new DICOMHeaderReader(
				dicomDir.getAbsolutePath());
		HeaderFields hf = dhr.extractFields();
		Field<String> tsField = hf.getStringField("timestamp");
		if (tsField != null && tsField.getValue() != null
				&& tsField.getValue().trim().length() > 0) {
			Date tsDate = DICOMHeaderReader.tsFormat.parse(tsField.getValue());
			shi.setScanDate(tsDate);
		}

		Field<String> sdField = hf.getStringField(DCM_SERIES_DESC_FIELD);
		String seriesType = "";
		if (sdField != null && sdField.getValue() != null
				&& sdField.getValue().trim().length() > 0) {
			String value = sdField.getValue().trim();

			CBFImageTypeLookupService service = ServiceFactory
					.getCBFImageTypeLookupService();
			CBFImageTypeDecider decider = new CBFImageTypeDecider(service);
			seriesType = decider.findDicomSeriesType(value, dicomDir);
			if (seriesType == null) {
				seriesType = "";
			}
		}
		if (seriesType.length() == 0) {
			if (DICOMHeaderReader.isDTI(hf)) {
				seriesType = DTI;
			}
		}
		shi.setSeriesType(seriesType);
	}

	protected void prepAnalyzeSeriesHeaderInfo(SeriesHeaderInfo shi)
			throws IOException, ParseException {
		List<HeaderFieldType> hftList = new ArrayList<HeaderFieldType>(5);
		hftList.add(new HeaderFieldType("timestamp", HeaderFieldType.STRING));
		String headerFile = shi.getHeaderFile();
		System.out.println("prepAnalyzeSeriesHeaderInfo:: headerFile:"
				+ headerFile);
		AnalyzeHeaderReader ahr = new AnalyzeHeaderReader(headerFile);
		ahr.readHeader();
		HeaderFields hf = ahr.extractFields(hftList);

		Field<String> tsField = hf.getStringField("timestamp");
		if (tsField != null && tsField.getValue() != null
				&& tsField.getValue().trim().length() > 0) {
			Date tsDate = AnalyzeHeaderReader.tsFormat
					.parse(tsField.getValue());
			shi.setScanDate(tsDate);
		}
		shi.setSeriesType("");
	}

	protected void prepAFNISeriesHeaderInfo(SeriesHeaderInfo shi)
			throws ParseException, IOException {
		List<HeaderFieldType> hftList = new ArrayList<HeaderFieldType>(5);
		hftList.add(new HeaderFieldType("tag", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("timestamp", HeaderFieldType.STRING,
				"NOTE_DATE_001"));

		hftList.add(new HeaderFieldType("reps", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("dda", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("nIntlv", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("speprev", HeaderFieldType.STRING));
		hftList.add(new HeaderFieldType("mpphase", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("mppcasl", HeaderFieldType.INT));
		// Mon Jan 9 11:46:35 PST 2012
		hftList.add(new HeaderFieldType("pcaslopt", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("docsf", HeaderFieldType.INT));
		hftList.add(new HeaderFieldType("optpcasl", HeaderFieldType.INT));

		String headerFile = shi.getHeaderFile();
		System.out.println("prepAFNISeriesHeaderInfo:: headerFile:"
				+ headerFile);
		AFNIHeaderReader ahr = new AFNIHeaderReader(headerFile);
		ahr.readHeader();
		HeaderFields hf = ahr.extractFields(hftList);
		System.out.println(hf.getIntField("tag"));
		System.out.println(hf.getStringField("timestamp"));

		Field<String> tsField = hf.getStringField("timestamp");
		if (tsField != null && tsField.getValue() != null
				&& tsField.getValue().trim().length() > 0) {
			Date tsDate = AFNIHeaderReader.tsFormat.parse(tsField.getValue());
			shi.setScanDate(tsDate);
		}

		Field<Integer> tagField = hf.getIntField("tag");
		Field<Integer> repsField = hf.getIntField("reps");
		Field<Integer> ddaField = hf.getIntField("dda");
		Field<Integer> nIntlvField = hf.getIntField("nIntlv");
		Field<String> speprevField = hf.getStringField("speprev");
		Field<Integer> mpphaseField = hf.getIntField("mpphase");
		Field<Integer> mppcaslField = hf.getIntField("mppcasl");
		Field<Integer> pcasloptField = hf.getIntField("pcaslopt");
		Field<Integer> docsfField = hf.getIntField("docsf");
		Field<Integer> optpcaslField = hf.getIntField("optpcasl");

		if (mppcaslField != null && mpphaseField != null) {
			if (speprevField == null) {
				// to trigger new rule set for AFNI briks
				speprevField = new Field<String>("speprev", "50.4");
			}
		}

		String seriesType = findSeriesType(tagField, null, null, repsField,
				ddaField, nIntlvField, speprevField, mpphaseField,
				mppcaslField, pcasloptField, docsfField, optpcaslField);
		if (seriesType == null)
			seriesType = "";
		shi.setSeriesType(seriesType);

	}

	public static boolean hasValue(Field<Integer> field) {
		return (field != null && field.getValue() != null);
	}

	public String findSeriesType(Field<Integer> tagField,
			Field<Integer> nprepField, Field<Integer> pcaslcalField,
			Field<Integer> repsField, Field<Integer> ddaField,
			Field<Integer> nIntlvField, Field<String> speprevField,
			Field<Integer> mpphaseField, Field<Integer> mppcaslField,
			Field<Integer> pcasloptField, Field<Integer> docsfField,
			Field<Integer> optpcaslField) {
		if (!hasValue(tagField)) {
			return null;
		}
		if (hasValue(ddaField) && hasValue(repsField) && hasValue(nIntlvField)) {
			int tagValue = tagField.getValue().intValue();
			if (tagValue == 0) {
				if (repsField.getValue().intValue() == 1
						&& ddaField.getValue().intValue() == 8) {
					return CSF;
				} else if (repsField.getValue().intValue() == 2
						&& (ddaField.getValue().intValue() == 1 || ddaField
								.getValue().intValue() == 0)
						&& nIntlvField.getValue().intValue() == 8) {
					return MINCON;
				}
			}
		}

		boolean hasEmbeddedCSF = docsfField != null
				&& docsfField.getValue() == 1;

		switch (tagField.getValue().intValue()) {
		case 1:
			return (hasEmbeddedCSF) ? EPISTAR_CSF : EPISTAR;
		case 2:
			return (hasEmbeddedCSF) ? FAIR_CSF : FAIR;
		case 3:
			return (hasEmbeddedCSF) ? PICORE_CSF : PICORE;
		case 4:
			return (hasEmbeddedCSF) ? VSASL_CSF : VSASL;
		case 5:
			return (hasEmbeddedCSF) ? CASL_CSF : CASL;
		case 9:
			if (needsNewRules(speprevField)) {
				if (mppcaslField == null || mpphaseField == null
						|| optpcaslField == null) {
					return null;
				}
				int mpphase = mpphaseField.getValue().intValue();
				int mppcasl = mppcaslField.getValue().intValue();
				int optpcasl = optpcaslField.getValue().intValue();
				if (mppcasl == 1) {
					if (mpphase == 4) {
						return (hasEmbeddedCSF) ? PCASL_4MP_CSF : PCASL_4MP;
					} else if (mpphase == 8) {
						return (hasEmbeddedCSF) ? PCASL_8MP_CSF : PCASL_8MP;
					} else {
						// UNKNOWN
						return null;
					}
				} else {
					if (nprepField != null) {
						int nprep = nprepField.getValue().intValue();
						if (nprep == 2 && optpcasl == 0) {
							return (hasEmbeddedCSF) ? PCASL_CSF : PCASL;
						} else if (nprep == 2 && optpcasl == 1) {
							return (hasEmbeddedCSF) ? Opt_PCASL_CSF : Opt_PCASL;
						} else {
							return null;
						}
					} else {
						return null;
					}
				}

			} else {
				if (nprepField != null) {
					int nprep = nprepField.getValue().intValue();
					if (nprep == 0) {
						return (hasEmbeddedCSF) ? PCASL_CSF : PCASL;
					} else if (nprep == 2) {
						return (hasEmbeddedCSF) ? Opt_PCASL_CSF : Opt_PCASL;
					} else if (nprep == 8) {
						return (hasEmbeddedCSF) ? PCASL_8MP_CSF : PCASL_8MP;
					} else if (nprep == 4) {
						if (pcaslcalField != null) {
							int pcaslcal = pcaslcalField.getValue().intValue();
							if (pcaslcal == 0) {
								return (hasEmbeddedCSF) ? PCASL_4MP_CSF
										: PCASL_4MP;
							} else if (pcaslcal == 1) {
								return (hasEmbeddedCSF) ? Opt_PCASL_Calib_CSF
										: Opt_PCASL_Calib;
							}
						} else {
							return (hasEmbeddedCSF) ? PCASL_4MP_or_Opt_PCASL_Calib_CSF
									: PCASL_4MP_or_Opt_PCASL_Calib;
						}
					}
				} else {
					return (hasEmbeddedCSF) ? PCASL_VARIANT_CSF : PCASL_VARIANT;
				}
			}
		}
		return null;
	}

	private boolean needsNewRules(Field<String> speprevField) {
		if (speprevField == null) {
			return false;
		}
		try {
			double speprev = Double.parseDouble(speprevField.getValue());
			return (speprev > 50.3);
		} catch (NumberFormatException nfe) {
			log.error("speprevField not a float:" + speprevField.getValue());
			return false;
		}
	}

	public static String extractName(File f) {
		String name = f.getName();
		return name.replaceFirst("\\.[^\\.]+$", "");
	}

	protected Experiment findExperiment(UserInfo ui, String experimentName)
			throws Exception {
		Map<String, Experiment> experimentMap = dbCache.getExperimentMap(ui,
				false);
		return experimentMap.get(experimentName);
	}

	protected AssessmentInfo prepAssessmentInfo(AssessmentType asType,
			Assessment assessment) {
		AssessmentInfo asInfo = new AssessmentInfo(assessment.getUniqueid()
				.intValue(), assessment.getName(), null, -1, -1);
		int i = 1;
		for (ScoreType st : asType.getSiList()) {
			ScoreInfo si = new ScoreInfo(st.getName(),
					getScoreType(st.getType()), 1, -1, null, i);
			asInfo.addScore(si);
			i++;
		}
		return asInfo;
	}

	public static String getScoreType(String stType) {
		if (stType.equals(ScoreType.INT)) {
			return "integer";
		} else if (stType.equals(ScoreType.FLOAT)) {
			return stType; 
		} else if (stType.equals(ScoreType.STRING)) {
			return "varchar";
		}
		return null;
	}

	protected Assessment addAssessment(UserInfo ui, AssessmentType asInfo,
			ISubjectAssessmentManagement isam)
			throws SubjectAssessmentManagementException {
		AssessmentInfo ai = new AssessmentInfo(-1, asInfo.getName(), null, -1,
				-1);
		if (asInfo.getDescr() != null) {
			ai.setDescription(asInfo.getDescr());
		}
		int i = 1;
		for (ScoreType st : asInfo.getSiList()) {
			ScoreInfo si = new ScoreInfo(st.getName(),
					getScoreType(st.getType()), 1, -1, null, i);
			ai.addScore(si);
			i++;
		}

		// make sure that any assessment cache is dropped.
		String siteID = CacheUtils.getDBID2SiteIDMap().get(this.theDBID);
		Assertion.assertNotNull(siteID);
		String key = siteID + ":asiScores";

		CacheManager man = CacheManager.getInstance();
		Cache cache = man.getCache("remote");
		if (cache != null)
			cache.remove(key);

		return isam.addAssessment(ui, ai);
	}

	protected void registerDataObjects(Connection con, UserInfo ui,
			VisitInfo vi, int visitId) throws Exception {
		RawdataDAO rdDAO = DAOFactory.createRawdataDAO(this.theDBID);
		DataobjectDAO doDAO = DAOFactory.createDataobjectDAO(this.theDBID);

		Map<String, Dataobjecttype> dotMap = getDataObjectTypeMap(con);
		Rawdata rdCR = new Rawdata();
		rdCR.setComponentid(GenUtils.toBigDecimal(visitId));
		rdCR.setSubjectid(vi.getSubjectID());
		rdCR.setNcExperimentUniqueid(GenUtils.toBigDecimal(vi.getExpId()));

		List<Rawdata> rdList = rdDAO.find(con, rdCR);
		Map<String, Rawdata> rdMap = new HashMap<String, Rawdata>();
		if (rdList != null && !rdList.isEmpty()) {
			for (Rawdata rd : rdList) {
				String name = new File(rd.getDatauri()).getName();
				rdMap.put(name, rd);
			}

			for (SegmentInfo si : vi.getSiList()) {
				if (si.getShi() != null) {
					SeriesHeaderInfo shi = si.getShi();
					if (shi.getImageType().equals(SeriesHeaderInfo.DICOM)) {
						Dataobjecttype dot = dotMap.get("3_LOC COL DICOM");
						Assertion.assertNotNull(dot);
						String dcmDirName = new File(shi.getImages().get(0))
								.getParentFile().getName();
						Rawdata rd = rdMap.get(dcmDirName);
						Assertion.assertNotNull(rd);
						registerDataObjectType(ui, con, dot, rd, doDAO);
					} else if (shi.getImageType()
							.equals(SeriesHeaderInfo.PFILE)) {
						Dataobjecttype dot = dotMap.get("3_LOC FILE PFILE");
						Assertion.assertNotNull(dot);
						for (String imgFile : shi.getImages()) {
							String name = new File(imgFile).getName();
							Rawdata rd = rdMap.get(name);
							Assertion.assertNotNull(rd);
							registerDataObjectType(ui, con, dot, rd, doDAO);
						}
					} else if (shi.getImageType().equals(SeriesHeaderInfo.AFNI)) {
						Dataobjecttype dot = dotMap.get("3_LOC FILE AFNI");
						Assertion.assertNotNull(dot);
						for (String imgFile : shi.getImages()) {
							String name = new File(imgFile).getName();
							Rawdata rd = rdMap.get(name);
							Assertion.assertNotNull(rd);
							registerDataObjectType(ui, con, dot, rd, doDAO);
						}
					} else if (shi.getImageType().equals(
							SeriesHeaderInfo.ANALYZE)) {
						Dataobjecttype dot = dotMap.get("3_LOC FILE ANALYZE");
						Assertion.assertNotNull(dot);
						for (String imgFile : shi.getImages()) {
							String name = new File(imgFile).getName();
							Rawdata rd = rdMap.get(name);
							Assertion.assertNotNull(rd);
							registerDataObjectType(ui, con, dot, rd, doDAO);
						}
					}
				}
			}

		}
	}

	protected void registerDataObjectType(UserInfo ui, Connection con,
			Dataobjecttype dot, Rawdata rd, DataobjectDAO doDAO)
			throws Exception {
		Dataobject dataObject = new Dataobject();
		dataObject.setDataid(rd.getUniqueid());
		dataObject.setObjecttypeid(dot.getUniqueid());
		dataObject.setExtensionname("rawData");
		Databaseuser adminDU = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);

		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(theDBID);
		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.DATAOBJECT_TABLE, "uniqueid");
		int tableID = DBUtils.getTableID(this.theDBID, con,
				Constants.DATAOBJECT_TABLE);

		dataObject.setOwner(adminDU.getUniqueid());
		dataObject.setModuser(adminDU.getUniqueid());
		dataObject.setModtime(new Date());
		dataObject.setUniqueid(uniqueid);
		dataObject.setTableid(GenUtils.toBigDecimal(tableID));

		File f = new File(rd.getDatauri());
		if (f.isFile()) {
			dataObject.setObjectsize(new BigDecimal(f.length()));
		} else if (f.isDirectory()) {
			long totalSizeUnderDir = clinical.utils.FileUtils
					.getTotalSizeUnderDir(f);
			dataObject.setObjectsize(new BigDecimal(totalSizeUnderDir));
		}

		doDAO.insert(con, dataObject);
	}

	protected Map<String, Dataobjecttype> getDataObjectTypeMap(Connection con)
			throws Exception {
		Map<String, Dataobjecttype> dotMap = new HashMap<String, Dataobjecttype>(
				17);
		DataobjecttypeDAO dotDAO = DAOFactory
				.createDataobjecttypeDAO(this.theDBID);

		List<Dataobjecttype> dotList = dotDAO.find(con, new Dataobjecttype());
		for (Dataobjecttype dot : dotList) {
			if (dot.getObjecttype().startsWith("3_LOC")) {
				dotMap.put(dot.getObjecttype(), dot);
			}
		}
		return dotMap;
	}
}
