package clinical.web.vo;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import clinical.utils.Assertion;
import clinical.web.download.SubjectImageDataBundle;

/**
 * holder of all {@link SubjectImageDataBundle} objects belonging to the same experiment 
 * across multiple sites as returned by the mediated query.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: ExpSubjectsIDBInfo.java 366 2011-05-05 20:06:27Z bozyurt $
 */
public class ExpSubjectsIDBInfo {
	protected String expName;
	Map<String, List<SubjectImageDataBundle>> exp2SIDBMap = new HashMap<String, List<SubjectImageDataBundle>>();
	protected List<String> runTypes;

	public ExpSubjectsIDBInfo(String expName) {
		this.expName = expName;
	}
	
	public int getSubjectCount() {
		int sum = 0;
		for (List<SubjectImageDataBundle> sidbList : exp2SIDBMap.values()) {
			sum += sidbList.size();
		}
		return sum;
	}

	public String getExpName() {
		return expName;
	}
	
	public List<SubjectImageDataBundle> getAllSIDBs() {
		List<SubjectImageDataBundle> sidbList = new ArrayList<SubjectImageDataBundle>();
		for (List<SubjectImageDataBundle> sidbl : exp2SIDBMap.values()) {
		    sidbList.addAll(sidbl);	
		}
		return sidbList;
	}
	
	public Set<String> getRemoteSiteIDSet(String curSiteID) {
		Set<String> remoteSiteIDSet = new HashSet<String>(17);
		for (List<SubjectImageDataBundle> sidbl : exp2SIDBMap.values()) {
			for(SubjectImageDataBundle sidb : sidbl) {
				if (!sidb.getSiteID().equals(curSiteID)) {
					remoteSiteIDSet.add(sidb.getSiteID());
				}
			}
		}
		return remoteSiteIDSet;
	}
	
	public void addSIDB(String expID, SubjectImageDataBundle sidb) {
		List<SubjectImageDataBundle> sidbList = exp2SIDBMap.get(expID);
		if ( sidbList == null) {
			sidbList = new ArrayList<SubjectImageDataBundle>(30);
			exp2SIDBMap.put(expID, sidbList);
		}
		sidbList.add(sidb);
	}

	public Map<String, List<SubjectImageDataBundle>> getExp2SIDBMap() {
		return exp2SIDBMap;
	}

	public void setExp2SIDBMap(Map<String, List<SubjectImageDataBundle>> exp2SIDBMap) {
		this.exp2SIDBMap = exp2SIDBMap;
	}

	public String getKey() {
		return String.valueOf( expName.hashCode() );
	}
	
	
	public void leaveOnlyRunTypes(Set<String> runtTypeSet) {
		for(List<SubjectImageDataBundle> sidbList : exp2SIDBMap.values()) {
			for(SubjectImageDataBundle sidb : sidbList) {
				sidb.removeSegmentUnless(runtTypeSet);
			}
		}
	}
	
	
	public static Map<String,String> prepExpID2NameMap(List<SubjectImageDataBundle> sidbList) {
         Map<String,String> expID2NameMap = new HashMap<String, String>(11);
         for(SubjectImageDataBundle sidb : sidbList) {
        	 for(String expID: sidb.getExpIds()) {
        		 String key = expID + "_" + sidb.getSiteID();
        		 if ( !expID2NameMap.containsKey(key)) {
        			 String name = sidb.getExperimentName(expID);
        			 Assertion.assertNotNull(name);
        			 expID2NameMap.put(key, name);
        		 }
        	 }
         }
         
         return expID2NameMap;
	}
	
	protected List<String> prepRunTypes() {
		Set<String> runTypeSet = new HashSet<String>(7);
		
		for (String expID : exp2SIDBMap.keySet()) {
			List<SubjectImageDataBundle> sidbList = exp2SIDBMap.get(expID);
			for (SubjectImageDataBundle sidb : sidbList) {
				Set<String> snSet = sidb.getSegmentNameSet();
				for (String segName : snSet) {
					if (!runTypeSet.contains(segName)) {
						if (segName.startsWith("Segment ")) {
							continue;
						}
						if (segName.equals("t1") || segName.equals("t2")) {
							runTypeSet.add(segName);
						} else {
							String runType = segName.replaceFirst("\\d+$", "");
							if (!runTypeSet.contains(runType))
								runTypeSet.add(runType);
						}
					}
				}
			}
		}
		if (!runTypeSet.isEmpty()) {
			this.runTypes = new ArrayList<String>(runTypeSet);
			Collections.sort(runTypes);			
		}
		return this.runTypes;
	}

	public List<String> getRunTypes() {
		if ( runTypes == null) {
			prepRunTypes();
		}
		return runTypes;
	}

}
