package clinical.web.vo;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.jdom.Element;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.server.vo.Jobs;
import clinical.utils.GenUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */

public class JobProvenanceInfo implements Serializable {
	private static final long serialVersionUID = 1L;
	private String name;
	private String description;
	private String dataURI;
	private Date modTime;
	private Jobs job;
	private List<JobProvenanceParamInfo> params = new ArrayList<JobProvenanceParamInfo>(
			5);
	private final static Pattern datePattern = Pattern.compile("\\d{2,2}\\/\\d{2,2}\\/\\d{4,4}");

	public JobProvenanceInfo(String name, String description, String dataURI,
			Date modTime, Jobs job) {
		super();
		this.name = name;
		this.description = description;
		this.dataURI = dataURI;
		this.modTime = modTime;
		this.job = job;
	}

	public JobProvenanceInfo(String name, String description) {
		super();
		this.name = name;
		this.description = description;
	}

	public String getName() {
		return name;
	}

	public String getDescription() {
		return description;
	}

	public String getDataURI() {
		return dataURI;
	}

	public Date getModTime() {
		return modTime;
	}

	public Jobs getJob() {
		return job;
	}

	public int getNumOfParams() {
		return params.size();
	}

	public List<JobProvenanceParamInfo> getParams() {
		return Collections.unmodifiableList(params);
	}

	public void addParam(JobProvenanceParamInfo jppi) {
		params.add(jppi);
	}

	public JobProvenanceParamInfo findParam(String paramName) {
		if (params.isEmpty()) {
			return null;
		}
		for (JobProvenanceParamInfo pi : params) {
			if (pi.getName().equals(paramName)) {
				return pi;
			}
		}
		return null;
	}

	public boolean hasSameParams(JobProvenanceInfo other) {
		if (params.size() != other.params.size()) {
			return false;
		}
		Map<String, JobProvenanceParamInfo> map = new HashMap<String, JobProvenanceParamInfo>(
				11);
		for (JobProvenanceParamInfo pi : other.params) {
			map.put(pi.getName(), pi);
		}
		for (JobProvenanceParamInfo pi : params) {
			JobProvenanceParamInfo opi = map.get(pi.getName());
			if (opi == null)
				return false;
			if (!pi.equals(opi)) {
				return false;
			}
		}
		return true;
	}

	public Element toXML() {
		Element e = new Element("job-provenance");
		e.setAttribute("name", getName());
		if (modTime == null) {
			e.setAttribute("modTime", new Date().toString());
		} else {
			e.setAttribute("modTime", modTime.toString());
		}
		if (description == null) {
			Element de = new Element("description");
			e.setText(description);
			e.addContent(de);
		}
		// don't save dataURI
		// Element due = new Element("dataURI");
		// due.setText(getDataURI());
		// e.addContent(due);
		for (JobProvenanceParamInfo pi : params) {
			e.addContent(pi.toXML());
		}
		return e;
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder(128);
		sb.append("JobProvenanceInfo::[");
		sb.append("name:").append(name);
		sb.append(",desc:").append(description);
		sb.append(",dataURI:").append(dataURI);
		sb.append("\nJob:").append(this.job);
		for (JobProvenanceParamInfo pi : params) {
			sb.append("\n\t").append(pi);
		}
		sb.append(']');
		return sb.toString();
	}

	public static enum JobProvenanceParamType {
		STRING, REAL, INT, BOOL, DATE
	}// ;

	public static class JobProvenanceParamInfo {
		String name;
		String value;

		public JobProvenanceParamInfo(String name, String value) {
			this.name = name;
			this.value = value;
		}

		public String getName() {
			return name;
		}

		public String getValue() {
			return value;
		}

		public JobProvenanceParamType getTypeFromValue() {
			if (value == null) {
				return null;
			}
			if (value.equals("yes") || value.equals("no")) {
				return JobProvenanceParamType.BOOL;
			} else {
				Number number = GenUtils.toNumber(value);
				if (number == null) {
					if (value.indexOf('/') != -1) {
						if (datePattern.matcher(value).find()) {
							return JobProvenanceParamType.DATE;
						}
					}
					return JobProvenanceParamType.STRING;
				} else {
					return (number instanceof Integer) ? JobProvenanceParamType.INT
							: JobProvenanceParamType.REAL;
				}
			}
		}

		public Element toXML() {
			Element e = new Element("param");
			e.setAttribute("name", name);
			e.setAttribute("value", value);
			return e;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((name == null) ? 0 : name.hashCode());
			result = prime * result + ((value == null) ? 0 : value.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			JobProvenanceParamInfo other = (JobProvenanceParamInfo) obj;
			if (name == null) {
				if (other.name != null)
					return false;
			} else if (!name.equals(other.name))
				return false;
			if (value == null) {
				if (other.value != null)
					return false;
			} else if (!value.equals(other.value))
				return false;
			return true;
		}

		@Override
		public String toString() {
			return "JobProvenanceParamInfo::[name=" + name + ", value=" + value
					+ "]";
		}

		public JSONObject toJSON() throws JSONException {
			JSONObject js = new JSONObject();
			js.put("name", name);
			js.put("value", value);
			return js;
		}

		public static JobProvenanceParamInfo initializeFormJSON(JSONObject js)
				throws JSONException {
			String name = js.getString("name");
			String value = js.getString("value");
			return new JobProvenanceParamInfo(name, value);
		}

	}// ;

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((dataURI == null) ? 0 : dataURI.hashCode());
		result = prime * result + ((name == null) ? 0 : name.hashCode());
		result = prime * result
				+ ((description == null) ? 0 : description.hashCode());
		result = prime * result
				+ ((job == null) ? 0 : job.getUniqueid().hashCode());
		if (params == null || params.isEmpty()) {
			result = prime * result;
		} else {
			for (JobProvenanceParamInfo pi : params) {
				result = prime * result + pi.hashCode();
			}
		}
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		JobProvenanceInfo other = (JobProvenanceInfo) obj;
		if (this.job == null) {
			if (other.job != null)
				return false;
		} else if (!job.getUniqueid().equals(other.job.getUniqueid())) {
			return false;
		}

		if (dataURI == null) {
			if (other.dataURI != null)
				return false;
		} else if (!dataURI.equals(other.dataURI))
			return false;
		if (description == null) {
			if (other.description != null)
				return false;
		} else if (!description.equals(other.description))
			return false;
		if (name == null) {
			if (other.name != null)
				return false;
		} else if (!name.equals(other.name))
			return false;
		if (params == null) {
			if (other.params != null)
				return false;
		} else {
			if (params.size() != other.params.size()) {
				return false;
			}
			int len = params.size();
			for (int i = 0; i < len; i++) {
				JobProvenanceParamInfo pi = params.get(i);
				JobProvenanceParamInfo opi = other.params.get(i);
				if (!pi.equals(opi)) {
					return false;
				}
			}
		}
		return true;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public void setDataURI(String dataURI) {
		this.dataURI = dataURI;
	}

	public void setModTime(Date modTime) {
		this.modTime = modTime;
	}

	public void setJob(Jobs job) {
		this.job = job;
	}
}
