package clinical.web.workflow.cbf;

import java.io.File;
import java.io.Serializable;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.web.vo.upload.SegmentInfo;
import clinical.web.workflow.common.JSONUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: CBFProcessInfo.java 612 2012-05-19 02:38:37Z bozyurt $
 */
public class CBFProcessInfo implements Serializable {
	private static final long serialVersionUID = 1L;
	File cbfBrik;
	File csfBrik;
	File minconBrik;
	File anatBrik;
	File outputDir;
	SegmentInfo cbfSI;
	SegmentInfo csfSI;
	SegmentInfo minconSI;
	SegmentInfo anatSI;
	File interimMatFile;
	File matFile;
	File binaryMaskFile;
	File selectedSliceFile;
	List<File> resultImageFiles = new ArrayList<File>(2);
	List<File> anatRegImageFiles = new ArrayList<File>(2);
	List<File> cbfRegImageFiles = new ArrayList<File>(2);
	private List<IFileInfo> additionalResultImageFiles = new ArrayList<IFileInfo>(
			2);
	private List<IFileInfo> intermediateFiles = new ArrayList<IFileInfo>();

	public CBFProcessInfo(File cbfBrik, File csfBrik, File minconBrik,
			File anatBrik, SegmentInfo cbfSI, SegmentInfo csfSI,
			SegmentInfo minconSI, SegmentInfo anatSI, File outputDir) {
		super();
		this.cbfBrik = cbfBrik;
		this.csfBrik = csfBrik;
		this.minconBrik = minconBrik;
		this.anatBrik = anatBrik;
		this.outputDir = outputDir;
		this.cbfSI = cbfSI;
		this.csfSI = csfSI;
		this.minconSI = minconSI;
		this.anatSI = anatSI;
	}

	public CBFProcessInfo(File cbfBrik, File csfBrik, File minconBrik,
			SegmentInfo cbfSI, SegmentInfo csfSI, SegmentInfo minconSI,
			File outputDir) {
		super();
		this.cbfBrik = cbfBrik;
		this.csfBrik = csfBrik;
		this.minconBrik = minconBrik;
		this.cbfSI = cbfSI;
		this.csfSI = csfSI;
		this.minconSI = minconSI;
		this.outputDir = outputDir;
	}

	public JSONObject toJSON() throws JSONException {
		JSONObject js = new JSONObject();
		JSONUtils.addJSONField("cbfBrik", cbfBrik, js);
		JSONUtils.addJSONField("csfBrik", csfBrik, js);
		JSONUtils.addJSONField("minconBrik", minconBrik, js);
		JSONUtils.addJSONField("anatBrik", anatBrik, js);
		JSONUtils.addJSONField("outputDir", outputDir, js);
		JSONUtils.addJSONField("interimMatFile", interimMatFile, js);
		JSONUtils.addJSONField("matFile", matFile, js);
		JSONUtils.addJSONField("binaryMaskFile", binaryMaskFile, js);
		JSONUtils.addJSONField("selectedSliceFile", selectedSliceFile, js);

		JSONUtils.addJSONSI("cbfSI", cbfSI, js);
		JSONUtils.addJSONSI("csfSI", csfSI, js);
		JSONUtils.addJSONSI("minconSI", minconSI, js);
		JSONUtils.addJSONSI("anatSI", anatSI, js);
		JSONUtils.addJSONArray("resultImageFiles", resultImageFiles, js);
		JSONUtils.addJSONArray("anatRegImageFiles", anatRegImageFiles, js);
		JSONUtils.addJSONArray("cbfRegImageFiles", cbfRegImageFiles, js);
		JSONArray arr = new JSONArray();
		for (IFileInfo bi : intermediateFiles) {
			arr.put(bi.toJSON());
		}
		js.put("intermediateFiles", arr);
		JSONArray arr2 = new JSONArray();
		for (IFileInfo bi : additionalResultImageFiles) {
			arr2.put(bi.toJSON());
		}
		js.put("otherResultFiles", arr2);
		return js;
	}

	public static CBFProcessInfo initializeFromJSON(JSONObject js)
			throws JSONException, ParseException {
		File cbfBrik = JSONUtils.getFile("cbfBrik", js);
		File csfBrik = JSONUtils.getFile("csfBrik", js);
		File minconBrik = JSONUtils.getFile("minconBrik", js);
		File anatBrik = JSONUtils.getFile("anatBrik", js);
		File outputDir = JSONUtils.getFile("outputDir", js);
		File interimMatFile = JSONUtils.getFile("interimMatFile", js);
		File matFile = JSONUtils.getFile("matFile", js);
		File binaryMaskFile = JSONUtils.getFile("binaryMaskFile", js);
		File selectedSliceFile = JSONUtils.getFile("selectedSliceFile", js);

		SegmentInfo cbfSI = JSONUtils.getSI("cbfSI", js);
		SegmentInfo csfSI = JSONUtils.getSI("csfSI", js);
		SegmentInfo minconSI = JSONUtils.getSI("minconSI", js);
		SegmentInfo anatSI = JSONUtils.getSI("anatSI", js);

		CBFProcessInfo cpi = new CBFProcessInfo(cbfBrik, csfBrik, minconBrik,
				anatBrik, cbfSI, csfSI, minconSI, anatSI, outputDir);

		cpi.interimMatFile = interimMatFile;
		cpi.matFile = matFile;
		cpi.binaryMaskFile = binaryMaskFile;
		cpi.selectedSliceFile = selectedSliceFile;

		cpi.resultImageFiles = JSONUtils.getFiles("resultImageFiles", js);
		cpi.anatRegImageFiles = JSONUtils.getFiles("anatRegImageFiles", js);
		cpi.cbfRegImageFiles = JSONUtils.getFiles("cbfRegImageFiles", js);
		if (js.has("intermediateFiles")) {
			JSONArray arr = js.getJSONArray("intermediateFiles");
			for (int i = 0; i < arr.length(); i++) {
				IFileInfo ifi = FileInfoFactory.initializeFromJSON(arr
						.getJSONObject(i));
				cpi.addIntermediateFile(ifi);
			}
		}
		if (js.has("otherResultFiles")) {
			JSONArray arr = js.getJSONArray("otherResultFiles");
			for (int i = 0; i < arr.length(); i++) {
				IFileInfo ifi = FileInfoFactory.initializeFromJSON(arr
						.getJSONObject(i));
				cpi.addAdditionalResultFile(ifi);
			}
		}
		return cpi;
	}

	public File getCbfBrik() {
		return cbfBrik;
	}

	public File getCsfBrik() {
		return csfBrik;
	}

	public File getMinconBrik() {
		return minconBrik;
	}

	public File getAnatBrik() {
		return anatBrik;
	}

	public File getOutputDir() {
		return outputDir;
	}

	public SegmentInfo getCbfSI() {
		return cbfSI;
	}

	public SegmentInfo getCsfSI() {
		return csfSI;
	}

	public SegmentInfo getMinconSI() {
		return minconSI;
	}

	public SegmentInfo getAnatSI() {
		return anatSI;
	}

	public File getMatFile() {
		return matFile;
	}

	public void setMatFile(File matFile) {
		this.matFile = matFile;
	}

	public File getBinaryMaskFile() {
		return binaryMaskFile;
	}

	public void setBinaryMaskFile(File binaryMaskFile) {
		this.binaryMaskFile = binaryMaskFile;
	}

	public File getInterimMatFile() {
		return interimMatFile;
	}

	public void setInterimMatFile(File interimMatFile) {
		this.interimMatFile = interimMatFile;
	}

	public File getSelectedSliceFile() {
		return selectedSliceFile;
	}

	public void setSelectedSliceFile(File selectedSliceFile) {
		this.selectedSliceFile = selectedSliceFile;
	}

	public List<File> getResultImageFiles() {
		return resultImageFiles;
	}

	public void addResultImage(File resultImg) {
		if (!this.resultImageFiles.contains(resultImg)) {
			this.resultImageFiles.add(resultImg);
		}
	}

	public void addRegAnatImage(File anatRegImg) {
		if (!anatRegImageFiles.contains(anatRegImg)) {
			this.anatRegImageFiles.add(anatRegImg);
		}
	}

	public void addCbfRegImage(File cbfRegImg) {
		if (!this.cbfRegImageFiles.contains(cbfRegImg)) {
			this.cbfRegImageFiles.add(cbfRegImg);
		}
	}

	public void addIntermediateFile(IFileInfo bi) {
		if (!intermediateFiles.contains(bi)) {
			intermediateFiles.add(bi);
		}
	}

	public List<IFileInfo> getIntermediateFiles() {
		return intermediateFiles;
	}

	public void addAdditionalResultFile(IFileInfo ifi) {
		if (!additionalResultImageFiles.contains(ifi)) {
			additionalResultImageFiles.add(ifi);
		}
	}

	public List<File> getAnatRegImageFiles() {
		return anatRegImageFiles;
	}

	public List<File> getCbfRegImageFiles() {
		return cbfRegImageFiles;
	}

	public boolean canDoSkullStripping() {
		if (anatRegImageFiles.size() != 2 || cbfRegImageFiles.size() != 2) {
			return false;
		}
		if (hasAllBrikFiles(this.cbfRegImageFiles)
				&& hasAllBrikFiles(this.anatRegImageFiles)) {
			return true;
		}
		return false;
	}

	public static boolean hasAllBrikFiles(List<File> files) {
		boolean hasBRIK = false;
		boolean hasHEAD = false;
		if (files.size() != 2)
			return false;
		String commonPart = null;
		for (File f : files) {
			String name = f.getName();
			if (commonPart == null) {
				commonPart = name.replaceFirst("\\.\\w+$", "");
			}
			if (name.endsWith(".BRIK")) {
				if (name.startsWith(commonPart)) {
					hasBRIK = true;
				}
			}
			if (name.endsWith(".HEAD")) {
				if (name.startsWith(commonPart)) {
					hasHEAD = true;
				}
			}

		}
		return hasBRIK && hasHEAD;
	}

	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("CBFProcessInfo::[");
		sb.append("cbfBrik:").append(cbfBrik);
		sb.append("\ncsfBrik:").append(csfBrik);
		sb.append("\nminconBrik:").append(minconBrik);
		sb.append("\n]");
		return sb.toString();
	}

	public List<IFileInfo> getAdditionalResultImageFiles() {
		return additionalResultImageFiles;
	}

}