package clinical.web.workflow.cbf;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.LogFactory;

import clinical.server.vo.Deriveddata;
import clinical.utils.Assertion;
import clinical.utils.FileUtils;
import clinical.web.CBFBIRNConstants;
import clinical.web.IAppConfigService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.download.FileBundleConfig;
import clinical.web.download.Packager;
import clinical.web.download.PathWrapper;
import clinical.web.exception.BaseException;
import clinical.web.scheduler.JobException;
import clinical.web.vo.JobProvenanceInfo;
import clinical.web.vo.upload.VisitInfo;
import clinical.web.workflow.common.WFGenUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: CBFProcessingJob.java 786 2013-03-20 02:21:25Z bozyurt $
 */
public class CBFProcessingJob extends AbstractCBFProcessingJob {
	private CBFProcJobHelper helper;
	private boolean testMode = false;
	@SuppressWarnings("unused")
	private FieldMapProcessingStub fmStub;
	private boolean useMultiWFOptimization = true;
	private volatile boolean shortcut = false; 

	public CBFProcessingJob(String id, UserInfo ui, String description,
			CBFWFContext context, String templateDir, String matlabDir,
			String emailTo, VisitProcessInfo vpi) throws BaseException {
		super();
		log = LogFactory.getLog(CBFProcessingJob.class);
		this.id = id;
		this.ui = ui;
		this.description = description;
		this.context = context;
		this.templateDir = templateDir;
		this.matlabDir = matlabDir;
		this.emailTo = emailTo;
		this.helper = new CBFProcJobHelper(ui, context, templateDir);

		VisitInfo vi = context.getViList().get(0);
		String subjectID = vi.getSubjectID();

		String bundleName = subjectID + "_Visit_"
				+ sdf.format(vi.getVisitDate()) + "_" + vi.getVisitId();
		File f = new File(context.getCacheDir(), bundleName + ".tar.gz");
		this.resultTarFile = f.getAbsolutePath();

		// for testing
		// this.fmStub = new FieldMapProcessingStub(context);

		this.vpi = vpi;
	}

	public CBFProcessingJob(String id, UserInfo ui, String description,
			CBFWFContext context, String templateDir, String matlabDir,
			String emailTo) throws BaseException {
		this(id, ui, description, context, templateDir, matlabDir, emailTo,
				null);
	}

	public void execute(int stageId) throws JobException {
		try {
			if (context.isUseAutomatedSegmentation() || context.isUseLTMethod()) {
				handleAutoCSF();
			} else {
				handleProcessing(stageId);
			}
			// testStub(stageId);
		} catch (Exception e) {
			e.printStackTrace();
			throw new JobException(e);
		}
	}

	@SuppressWarnings("unused")
	private void testStub(int stageId) {
		// test stub
		long start = System.currentTimeMillis();
		int count = 1;
		while (true) {
			synchronized (this) {
				try {
					this.wait(2000);
				} catch (InterruptedException e) {
				}
			}
			if (isCanceled()) {
				// TODO do cleanup
				break;
			}
			sendMessage("cbf proc ( " + stageId + ") " + count);
			long now = System.currentTimeMillis();
			if ((now - start) / 1000.0 > 10) {
				break;
			}
			count++;
		}
	}

	protected void handleAutoCSF() throws Exception {
		IAppConfigService configService = ServiceFactory.getAppConfigService();

		prepareEnvironment(configService);
		Map<String, WFStep> wfStepMap = findWFSteps2Skip();
		if (useMultiWFOptimization
				&& wfStepMap.containsKey(CBFBIRNConstants.KEY_SMOOTHING)) {
			handleOptimizedProcessing(wfStepMap);
			return;
		}

		File matlabDirFile = new File(matlabDir);
		sendMessage("preparing for CBF processing");
		this.vpi = helper.prepare4Job(matlabDirFile);
		if (isCanceled()) {
			return;
		}
		if (context.isDoB0Correction()) {
			if (useMultiWFOptimization
					&& wfStepMap.containsKey(CBFBIRNConstants.KEY_FIELDMAP)) {
				// optimization: use results from previous run
				// TODO 22x support
				WFStep fmStep = wfStepMap.get(CBFBIRNConstants.KEY_FIELDMAP);
				helper.updateCPIListWithPreviousFieldMapWFStep(vpi, fmStep);
			} else {
				for (FieldMapInfo fmi : vpi.getFmList()) {
					helper.handleFieldMapProcessing(fmi, matlabDirFile,
							context.isDoReg());

					if (isCanceled()) {
						return;
					}
				}
				helper.updateCPIListWithFieldMapCorrection(vpi, matlabDirFile);
			}
			if (isCanceled()) {
				return;
			}
			Assertion.assertTrue(!vpi.getCpiList().isEmpty());
			for (int i = 0; i < vpi.getCpiList().size(); i++) {
				CBFProcessInfo cpi = vpi.getCpiList().get(i);
				if (isCanceled()) {
					return;
				}
				sendMessage("processing " + cpi.getCbfSI().getProtocolId());

				helper.handleCBFProcessingAutomated(cpi, matlabDirFile,
						context.isDoSkullStripping(), context.getGmThreshold(),
						context.isDoPartialVolumeCorrection(),
						context.getAlignment(), context.getCsfMethod(),
						context.isUseLTMethod(),
						context.getGaussianFilterParam());

			}
		} else if (context.isDoSenseval()) {
			log.info("doing SENSE");
			if (useMultiWFOptimization
					&& wfStepMap.containsKey(CBFBIRNConstants.KEY_SENSE)) {
				// optimization: use results from previous run
				WFStep senseStep = wfStepMap.get(CBFBIRNConstants.KEY_SENSE);
				helper.updateCPIListWithPreviousSenseWFStep(vpi, senseStep);
			} else {
				for (SenseInfo se : vpi.getSeList()) {
					log.info("Handling SENSE for " + se.getPrefix());
					helper.handleSENSEProcessing(se, matlabDirFile);

					if (isCanceled()) {
						return;
					}
				}

				helper.updateCPIListWithSense(vpi, matlabDirFile);
			}
			if (isCanceled()) {
				return;
			}
			Assertion.assertTrue(!vpi.getCpiList().isEmpty());
			for (int i = 0; i < vpi.getCpiList().size(); i++) {
				CBFProcessInfo cpi = vpi.getCpiList().get(i);
				if (isCanceled()) {
					return;
				}
				sendMessage("processing " + cpi.getCbfSI().getProtocolId());

				helper.handleCBFProcessingAutomated(cpi, matlabDirFile,
						context.isDoSkullStripping(), context.getGmThreshold(),
						context.isDoPartialVolumeCorrection(),
						context.getAlignment(), context.getCsfMethod(),
						context.isUseLTMethod(),
						context.getGaussianFilterParam());
			}
		} else {
			if (isCanceled()) {
				return;
			}

			Assertion.assertTrue(!vpi.getCpiList().isEmpty());
			for (int i = 0; i < vpi.getCpiList().size(); i++) {
				CBFProcessInfo cpi = vpi.getCpiList().get(i);
				if (isCanceled()) {
					return;
				}
				sendMessage("processing " + cpi.getCbfSI().getProtocolId());

				helper.handleCBFProcessingAutomated(cpi, matlabDirFile,
						context.isDoSkullStripping(), context.getGmThreshold(),
						context.isDoPartialVolumeCorrection(),
						context.getAlignment(), context.getCsfMethod(),
						context.isUseLTMethod(),
						context.getGaussianFilterParam());
			}
		}
		// save derived data and create a bundle of results for download
		postprocess();
	}

	protected void prepareEnvironment(IAppConfigService configService) {
		String envPath = configService.getParamValue("cbfbirn.env.path");
		helper.setEnvPath(envPath);
		helper.addEnvironmentVar("FSLDIR",
				configService.getParamValue("FSLDIR"));
		helper.addEnvironmentVar("FSLOUTPUTTYPE",
				configService.getParamValue("FSLOUTPUTTYPE"));
		helper.addEnvironmentVar("AFNI_PLUGINPATH",
				configService.getParamValue("AFNI_PLUGINPATH"));
		helper.addEnvironmentVar("FSLMULTIFILEQUIT",
				configService.getParamValue("FSLMULTIFILEQUIT"));
		helper.addEnvironmentVar("FSLCONFDIR",
				configService.getParamValue("FSLCONFDIR"));
		helper.addEnvironmentVar("FSLMACHTYPE",
				configService.getParamValue("FSLMACHTYPE"));
	}

	/**
	 * skips all the processing till the final step
	 * 
	 * @param wfStepMap
	 * @throws Exception
	 */
	void handleOptimizedProcessing(Map<String, WFStep> wfStepMap)
			throws Exception {
		if (isCanceled()) {
			return;
		}
		this.shortcut = true;
		WFStep skipStep = wfStepMap.get(CBFBIRNConstants.KEY_SMOOTHING);
		File matlabDirFile = new File(matlabDir);
		sendMessage("preparing for CBF processing");
		this.vpi = helper.prepare4Job(matlabDirFile);
		if (isCanceled()) {
			return;
		}
		Assertion.assertTrue(!vpi.getCpiList().isEmpty());
		CBFProcessInfo cpi = vpi.getCpiList().get(0);
		sendMessage("processing " + cpi.getCbfSI().getProtocolId());
		if (context.isDoB0Correction()) {
			if (useMultiWFOptimization
					&& wfStepMap.containsKey(CBFBIRNConstants.KEY_FIELDMAP)) {
				// optimization: use results from previous run
				WFStep fmStep = wfStepMap.get(CBFBIRNConstants.KEY_FIELDMAP);
				helper.updateCPIListWithPreviousFieldMapWFStep(vpi, fmStep);
			} else {
				throw new Exception("No previous field map correction to use!");
			}

			if (isCanceled()) {
				return;
			}

			// all but last step is same, so use results from a matching job
			helper.handleCBFProcessingUsingResultFrom(cpi, skipStep.getJpi(),
					matlabDirFile);
		} else if (context.isDoSenseval()) {
			log.info("doing SENSE");
			if (useMultiWFOptimization
					&& wfStepMap.containsKey(CBFBIRNConstants.KEY_SENSE)) {
				// optimization: use results from previous run
				WFStep senseStep = wfStepMap.get(CBFBIRNConstants.KEY_SENSE);
				helper.updateCPIListWithPreviousSenseWFStep(vpi, senseStep);
			} else {
				throw new Exception("No previous SENSE correction to use!");
			}
			if (isCanceled()) {
				return;
			}
			// all but last step is same, so use results from a matching job
			helper.handleCBFProcessingUsingResultFrom(cpi, skipStep.getJpi(),
					matlabDirFile);
		} else {
			if (isCanceled()) {
				return;
			}
			// all but last step is same, so use results from a matching job
			helper.handleCBFProcessingUsingResultFrom(cpi, skipStep.getJpi(),
					matlabDirFile);
		}

		if (isCanceled()) {
			return;
		}
		// save derived data and create a bundle of results for download
		postprocess();

	}

	protected void handleProcessing(int stageId) throws Exception {
		IAppConfigService configService = ServiceFactory.getAppConfigService();

		prepareEnvironment(configService);

		Map<String, WFStep> wfStepMap = findWFSteps2Skip();
		if (useMultiWFOptimization
				&& wfStepMap.containsKey(CBFBIRNConstants.KEY_SMOOTHING)) {
			handleOptimizedProcessing(wfStepMap);
			return;
		}

		File matlabDirFile = new File(matlabDir);
		if (stageId == 1) {
			if (isCanceled()) {
				return;
			}
			sendMessage("preparing for CBF processing");
			this.vpi = helper.prepare4Job(matlabDirFile);
			if (isCanceled()) {
				return;
			}
			if (context.isDoB0Correction()) {

				if (useMultiWFOptimization
						&& wfStepMap.containsKey(CBFBIRNConstants.KEY_FIELDMAP)) {
					// optimization: use results from previous run
					WFStep fmStep = wfStepMap
							.get(CBFBIRNConstants.KEY_FIELDMAP);
					helper.updateCPIListWithPreviousFieldMapWFStep(vpi, fmStep);
				} else {
					for (FieldMapInfo fmi : vpi.getFmList()) {

						helper.handleFieldMapProcessing(fmi, matlabDirFile,
								context.isDoReg());

						// for testing
						// fmStub.handleFieldMapProcessing(fmi, matlabDirFile);

						if (isCanceled()) {
							return;
						}
					}
					helper.updateCPIListWithFieldMapCorrection(vpi,
							matlabDirFile);
				}
				if (isCanceled()) {
					return;
				}
				Assertion.assertTrue(!vpi.getCpiList().isEmpty());
				CBFProcessInfo cpi = vpi.getCpiList().get(0);
				helper.handleCBFProcessingStep1(cpi, matlabDirFile,
						context.isDoPartialVolumeCorrection(),
						context.getAlignment());
				// also set vpi for recovery
				prep4Recovery();

				handleEmail(USER_ACTION_MAIL);
			} else if (context.isDoSenseval()) {
				log.info("doing SENSE");
				if (useMultiWFOptimization
						&& wfStepMap.containsKey(CBFBIRNConstants.KEY_SENSE)) {
					// optimization: use results from previous run
					WFStep senseStep = wfStepMap
							.get(CBFBIRNConstants.KEY_SENSE);
					helper.updateCPIListWithPreviousSenseWFStep(vpi, senseStep);
				} else {
					for (SenseInfo se : vpi.getSeList()) {
						log.info("Handling SENSE for " + se.getPrefix());
						helper.handleSENSEProcessing(se, matlabDirFile);

						if (isCanceled()) {
							return;
						}
					}

					helper.updateCPIListWithSense(vpi, matlabDirFile);
				}
				if (isCanceled()) {
					return;
				}
				Assertion.assertTrue(!vpi.getCpiList().isEmpty());
				CBFProcessInfo cpi = vpi.getCpiList().get(0);
				helper.handleCBFProcessingStep1(cpi, matlabDirFile,
						context.isDoPartialVolumeCorrection(),
						context.getAlignment());
				// also set vpi for recovery
				prep4Recovery();
				if (isCanceled()) {
					return;
				}
				handleEmail(USER_ACTION_MAIL);
			} else {
				if (isCanceled()) {
					return;
				}
				Assertion.assertTrue(!vpi.getCpiList().isEmpty());

				CBFProcessInfo cpi = vpi.getCpiList().get(0);
				sendMessage("processing " + cpi.getCbfSI().getProtocolId());
				helper.handleCBFProcessingStep1(cpi, matlabDirFile,
						context.isDoPartialVolumeCorrection(),
						context.getAlignment());
				// also set vpi for recovery
				prep4Recovery();
				if (isCanceled()) {
					return;
				}
				handleEmail(USER_ACTION_MAIL);
			}
		} else {

			String cacheDir = context.getCacheDir();
			File maskFile = new File(cacheDir, "mask.txt");
			File sliceNoFile = new File(cacheDir, "slice_no.txt");
			if (!maskFile.exists()) {
				// multiple slice annotation
				maskFile = new File(cacheDir);
			}
			Assertion.assertTrue(sliceNoFile.isFile());
			CBFProcessInfo cpi = vpi.getCpiList().get(0);
			cpi.setBinaryMaskFile(maskFile);
			cpi.setSelectedSliceFile(sliceNoFile);

			if (isCanceled()) {
				return;
			}

			sendMessage("processing (Step 2) " + cpi.getCbfSI().getProtocolId());
			helper.handleCBFProcessingStep2(cpi, matlabDirFile,
					context.isDoSkullStripping(), context.getGmThreshold(),
					context.isDoPartialVolumeCorrection(),
					context.getCsfMethod(), context.getGaussianFilterParam());

			// the rest of the segments/briks do not need user intervention
			for (int i = 1; i < vpi.getCpiList().size(); i++) {
				cpi = vpi.getCpiList().get(i);
				if (isCanceled()) {
					return;
				}
				sendMessage("processing " + cpi.getCbfSI().getProtocolId());
				helper.handleCBFProcessingStep1(cpi, matlabDirFile,
						context.isDoPartialVolumeCorrection(),
						context.getAlignment());
				cpi.setBinaryMaskFile(maskFile);
				cpi.setSelectedSliceFile(sliceNoFile);
				if (isCanceled()) {
					return;
				}
				sendMessage("processing (Step 2) "
						+ cpi.getCbfSI().getProtocolId());
				helper.handleCBFProcessingStep2(cpi, matlabDirFile,
						context.isDoSkullStripping(), context.getGmThreshold(),
						context.isDoPartialVolumeCorrection(),
						context.getCsfMethod(),
						context.getGaussianFilterParam());
			}

			// for TEST serialize the state for further tests
			if (testMode) {
				String homeDir = System.getProperty("user.home");
				File vpiSerFile = new File(homeDir, "vpi.ser");
				FileUtils.serialize(this.vpi, vpiSerFile.getAbsolutePath());
				File serFile = new File(homeDir, "CBFWFContext.ser");
				FileUtils.serialize(this.context, serFile.getAbsolutePath());
			}
			if (isCanceled()) {
				return;
			}
			// save derived data and create a bundle of results for download
			postprocess();
		}
	}

	private Map<String, WFStep> findWFSteps2Skip() throws Exception {
		List<JobProvenanceInfo> jpiList = helper.getExistingWFResults();
		JobProvenanceInfo theJPI = MultipleWFHelper.prepJobProvenanceInfo("",
				"", context);
		Map<String, WFStep> wfStepMap = MultipleWFHelper.findWFSteps2Skip(
				theJPI, jpiList);
		return wfStepMap;
	}

	protected void postprocess() throws Exception {
		// derived data persistence
		List<Deriveddata> ddList = helper.saveDerivedData(this.vpi);

		// bundle the results for download
		List<Deriveddata> downloadDDList = new ArrayList<Deriveddata>(10);
		for (Deriveddata dd : ddList) {
			if (dd.getDatauri().endsWith(".mat")
					|| dd.getDatauri().endsWith(".png")
					|| dd.getDatauri().indexOf("intermediate") != -1
					|| dd.getDatauri().indexOf("CBF+orig") != -1
					|| dd.getDatauri().indexOf("MPPCASL_Mask+orig") != -1
					|| dd.getDatauri().indexOf("process_summary.txt") != -1
					|| dd.getDatauri().indexOf("cbfbirn_path1_summary.txt") != -1) {
				downloadDDList.add(dd);
			}
		}

		VisitInfo vi = context.getViList().get(0);
		String subjectID = vi.getSubjectID();

		String bundleName = subjectID + "_Visit_"
				+ sdf.format(vi.getVisitDate()) + "_" + vi.getVisitId();

		List<PathWrapper> allFiles = new ArrayList<PathWrapper>();
		String srcRootDir = null;
		for (Deriveddata dd : downloadDDList) {
			String datauri = dd.getDatauri();
			if (srcRootDir == null) {
				srcRootDir = WFGenUtils.extractRootDir(datauri, subjectID);
			}
			String relativePath = FileUtils
					.getRelativePath(srcRootDir, datauri);
			PathWrapper pw = new PathWrapper(relativePath, srcRootDir);
			allFiles.add(pw);
		}
		// final packaging

		Packager p = new Packager(context.getCacheDir(), bundleName,
				FileBundleConfig.GZIPPED);
		p.includeFiles(allFiles);
		p.pack();

		if (isCanceled()) {
			return;
		}
		handleEmail(JOB_FINISHED_MAIL);
	}

	public void setTestMode(boolean testMode) {
		this.testMode = testMode;
	}

	public boolean isTestMode() {
		return testMode;
	}

	@Override
	public int getNumberOfStages() {
		if (this.context.isUseAutomatedSegmentation()
				|| this.context.isUseLTMethod() || shortcut) {
			return 1;
		} else {
			return 2;
		}
	}

	@Override
	public synchronized void cancel() {
		if (isCanceled()) {
			return;
		}
		super.cancel();
	}

}
