package clinical.web.workflow.cbf;

import java.io.Serializable;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.web.vo.upload.VisitInfo;

/**
 * Context object passed in to the CBF Workflow job containing the context
 * information about the specific workflow for the visit.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: CBFWFContext.java 767 2013-02-06 01:59:36Z bozyurt $
 */
public class CBFWFContext implements Serializable {
	private static final long serialVersionUID = 1L;
	private String dbID;
	private boolean doB0Correction = false;
	private boolean doSenseval = false;
	private boolean doReg = true;
	private boolean doSkullStripping = true;
	private double gmThreshold = 0.6;
	private String cacheDir;
	private List<VisitInfo> viList = new ArrayList<VisitInfo>(1);
	private List<VisitProcessInfo> vpiList = new ArrayList<VisitProcessInfo>(1);
	private String matlabDir;
	private String templateDir;
	private String emailTo;
	private boolean doPartialVolumeCorrection = false;
	private boolean useAutomatedSegmentation = false;
	private String jobID;
	private int alignment = 3;
	private int csfMethod = PVM;
	private boolean useLTMethod = false;
	private double gaussianFilterParam = 4.0;
	
	public final static int AUTOMATIC = 0;
	public final static int TOP5 = 1;
	public final static int PVM = 2;
			

	public CBFWFContext(String dbID, boolean doB0Correction, String cacheDir,
			List<VisitInfo> viList, boolean doSense, boolean doReg,
			boolean doSkullStripping, double gmThreshold,
			List<VisitProcessInfo> vpiList, String jobID) {
		this.dbID = dbID;
		this.doB0Correction = doB0Correction;
		this.viList = viList;
		this.cacheDir = cacheDir;
		this.doSenseval = doSense;
		this.doReg = doReg;
		this.doSkullStripping = doSkullStripping;
		this.gmThreshold = gmThreshold;
		this.vpiList = vpiList;
		this.jobID = jobID;
	}

	public String getDbID() {
		return dbID;
	}

	public boolean isDoB0Correction() {
		return doB0Correction;
	}

	public List<VisitInfo> getViList() {
		return viList;
	}

	public String getCacheDir() {
		return cacheDir;
	}

	public JSONObject toJSON() throws JSONException {
		JSONObject js = new JSONObject();
		js.put("dbID", dbID);
		js.put("doB0Correction", doB0Correction);
		js.put("doSenseval", doSenseval);
		js.put("cacheDir", cacheDir);
		js.put("doReg", doReg);
		js.put("doSkullStripping", doSkullStripping);
		js.put("gmThreshold", gmThreshold);
		js.put("doPVC", doPartialVolumeCorrection);
		js.put("autoCSF", useAutomatedSegmentation);
		js.put("align", alignment);
		js.put("csfMethod", csfMethod);
		if (jobID != null) {
			js.put("jobID", jobID);
		}
		JSONArray jsArr = new JSONArray();
		for (VisitInfo vi : viList) {
			jsArr.put(vi.toJSON());
		}
		js.put("viList", jsArr);

		jsArr = new JSONArray();
		for (VisitProcessInfo vpi : vpiList) {
			jsArr.put(vpi.toJSON());
		}
		js.put("vpiList", jsArr);

		if (templateDir != null) {
			js.put("templateDir", templateDir);
		}
		if (matlabDir != null) {
			js.put("matlabDir", matlabDir);
		}
		if (emailTo != null) {
			js.put("emailTo", emailTo);
		}

		return js;
	}
	public static CBFWFContext fromJSON(JSONObject js) throws JSONException,
			ParseException {
		List<VisitInfo> viList = new ArrayList<VisitInfo>(1);
		String dbID = js.getString("dbID");
		boolean doB0 = js.getBoolean("doB0Correction");
		boolean doSenseval = false;
		if (js.has("doSenseval")) {
			doSenseval = js.getBoolean("doSenseval");
		}
		String cacheDir = js.getString("cacheDir");
		boolean doReg = false;
		if (js.has("doReg")) {
			doReg = js.getBoolean("doReg");
		}
		boolean doSS = false;
		if (js.has("doSkullStripping")) {
			doSS = js.getBoolean("doSkullStripping");
		}
		double gmThreshold = 0.6;
		if (js.has("gmThreshold")) {
			gmThreshold = js.getDouble("gmThreshold");
		}
		int alignment = 3;
		if (js.has("align")) {
			alignment = js.getInt("align");
		}
		int csfMethod = AUTOMATIC;
		if (js.has("csfMethod")) {
			csfMethod = js.getInt("csfMethod");
		}

		JSONArray jsArr = js.getJSONArray("viList");
		for (int i = 0; i < jsArr.length(); i++) {
			JSONObject jo = jsArr.getJSONObject(i);
			viList.add(VisitInfo.initializeFromJSON(jo));
		}

		List<VisitProcessInfo> vpiList = new ArrayList<VisitProcessInfo>(1);
		if (js.has("vpiList")) {
			jsArr = js.getJSONArray("vpiList");
			for (int i = 0; i < jsArr.length(); i++) {
				vpiList.add(VisitProcessInfo.initializeFromJSON(jsArr
						.getJSONObject(i)));
			}
		}
		
		String jobID = null;
		if (js.has("jobID")) {
			jobID = js.getString("jobID");
		}

		CBFWFContext ctx = new CBFWFContext(dbID, doB0, cacheDir, viList,
				doSenseval, doReg, doSS, gmThreshold, vpiList, jobID);
		if (js.has("templateDir")) {
			ctx.templateDir = js.getString("templateDir");
		}
		if (js.has("matlabDir")) {
			ctx.matlabDir = js.getString("matlabDir");
		}
		if (js.has("emailTo")) {
			ctx.emailTo = js.getString("emailTo");
		}

		boolean doPVC = false;
		if (js.has("doPVC")) {
			doPVC = js.getBoolean("doPVC");
		}
		boolean autoCSF = false;
		if (js.has("autoCSF")) {
			autoCSF = js.getBoolean("autoCSF");
		}
		ctx.doPartialVolumeCorrection = doPVC;
		ctx.useAutomatedSegmentation = autoCSF;
		ctx.alignment = alignment;
		ctx.csfMethod = csfMethod;

		return ctx;
	}

	public void setDoB0Correction(boolean doB0Correction) {
		this.doB0Correction = doB0Correction;
	}

	public boolean isDoSenseval() {
		return doSenseval;
	}

	public void setDoSenseval(boolean doSenseval) {
		this.doSenseval = doSenseval;
	}

	public boolean isDoReg() {
		return doReg;
	}

	public boolean isDoSkullStripping() {
		return doSkullStripping;
	}

	public double getGmThreshold() {
		return gmThreshold;
	}

	public List<VisitProcessInfo> getVpiList() {
		return vpiList;
	}

	void setVpiList(List<VisitProcessInfo> vpiList) {
		this.vpiList = vpiList;
	}

	public String getMatlabDir() {
		return matlabDir;
	}

	public void setMatlabDir(String matlabDir) {
		this.matlabDir = matlabDir;
	}

	public String getTemplateDir() {
		return templateDir;
	}

	public void setTemplateDir(String templateDir) {
		this.templateDir = templateDir;
	}

	public String getEmailTo() {
		return emailTo;
	}

	public void setEmailTo(String emailTo) {
		this.emailTo = emailTo;
	}

	public boolean isDoPartialVolumeCorrection() {
		return doPartialVolumeCorrection;
	}

	public void setDoPartialVolumeCorrection(boolean doPartialVolumeCorrection) {
		this.doPartialVolumeCorrection = doPartialVolumeCorrection;
	}

	public boolean isUseAutomatedSegmentation() {
		return useAutomatedSegmentation;
	}

	public void setUseAutomatedSegmentation(boolean useAutomatedSegmentation) {
		this.useAutomatedSegmentation = useAutomatedSegmentation;
	}

	public String getJobID() {
		return jobID;
	}

	public int getAlignment() {
		return alignment;
	}

	public void setAlignment(int alignment) {
		this.alignment = alignment;
	}

	public int getCsfMethod() {
		return csfMethod;
	}

	public void setCsfMethod(int csfMethod) {
		this.csfMethod = csfMethod;
	}

	public boolean isUseLTMethod() {
		return useLTMethod;
	}

	public void setUseLTMethod(boolean useLTMethod) {
		this.useLTMethod = useLTMethod;
	}

	public double getGaussianFilterParam() {
		return gaussianFilterParam;
	}

	public void setGaussianFilterParam(double gaussianFilterParam) {
		this.gaussianFilterParam = gaussianFilterParam;
	}

}
